/* mxbt - btree functions that operate on an index file compatible
   with that used by Mix Database Toolchest (or something like that). */
   
/* Written by Paul Edwards and released to the public domain */

/* The btree index file is comprised of three different record types,
   stored as a fixed length (e.g. 512 bytes), even though the record
   types may not require that full length.

   1. A control record.  This is the first record in the file, and
   contains important information such as the fixed length that all
   the other records are.
   
   2. Leaf nodes.  These are identified by having "-1" in the first
   field (a "short").  Every search key will be found in one of these
   records.
   
   3. Index nodes.  These are identified by not having "-1" in the
   first field.  Some, but not all, of the keys will be found in
   one of these fields. 
   
   Note that if you wish to update records in this index file, you
   should update the leaf nodes before the index nodes.
*/

/* The algorithm we employ is:

   1. fetch the control record
   
   2. Search the index nodes for our key until we reach a leaf
      node.
      
   3. Search the leaf node for our key. */

#include <stdio.h>

#include "mxbt.h"

static void mxbtInit(MXBT *mxbt);
static void mxbtOpen(MXBT *mxbt, char *indexFile);
static void mxbtClose(MXBT *mxbt);
static void mxbtFetchControl(MXBT *mxbt);
static void mxbtSetKey(MXBT *mxbt, void *searchKey);
static void mxbtSetCompare(MXBT *mxbt, 
            int (*compare)(void *testKey, void *searchKey, int len));
static void mxbtReadRec(MXBT *mxbt);
static void mxbtFindLeaf(MXBT *mxbt);
static void mxbtSearchLeaf(MXBT *mxbt);

long mxbtOneSearch(MXBT *mxbt, 
                   char *indexFile, 
                   void *searchKey,
                   int (*compare)(void *testKey, void *searchKey, int len))
{
    mxbt->error = 0;
    mxbtInit(mxbt);
    if (!mxbt->error)
    {
        mxbtOpen(mxbt, indexFile);
        if (!mxbt->error)
        {
            mxbtFetchControl(mxbt);
            if (!mxbt->error)
            {
                mxbtSetKey(mxbt, searchKey);
                mxbtSetCompare(mxbt, compare);
                mxbtFindLeaf(mxbt);
                if (!mxbt->error)
                {
                    mxbtSearchLeaf(mxbt);
                }
            }
            mxbtClose(mxbt);
        }
    }
    if (mxbt->error)
    {
        return (-1L);
    }     
    else
    {
        return (mxbt->value);
    }
}

static void mxbtInit(MXBT *mxbt)
{
    mxbt->buf = mxbt->myunion.intbuf;
    mxbt->index = (struct mxbt_indexrec *)mxbt->buf;
    mxbt->leaf = (struct mxbt_leafrec *)mxbt->buf;
    return;
}

static void mxbtOpen(MXBT *mxbt, char *indexFile)
{
    mxbt->fp = fopen(indexFile, "rb");
    if (mxbt->fp == NULL)
    {
        mxbt->error = 1;
    }
    return;
}

static void mxbtClose(MXBT *mxbt)
{
    if (fclose(mxbt->fp) != 0)
    {
        mxbt->error = 1;
    }
    return;
}

static void mxbtFetchControl(MXBT *mxbt)
{
    if (fread(&mxbt->recSize, sizeof(unsigned short), 1, mxbt->fp) != 1)
    {
        mxbt->error = 1;
    }
    else if (fread(&mxbt->control, sizeof mxbt->control, 1, mxbt->fp) != 1)
    {
        mxbt->error = 1;
    }
    return;
}

static void mxbtSetKey(MXBT *mxbt, void *searchKey)
{
    mxbt->searchK = searchKey;
    return;
}

static void mxbtSetCompare(MXBT *mxbt, 
            int (*compare)(void *testKey, void *searchKey, int len))
{
    mxbt->compareF = compare;
    return;
}

static void mxbtReadRec(MXBT *mxbt)
{
    size_t x;
    int y;
    
    y = fseek(mxbt->fp, mxbt->recordNum * mxbt->recSize, SEEK_SET);
    if (y != 0)
    {
        mxbt->error = 1;
    }
    else
    {
        x = fread(mxbt->buf, mxbt->recSize, 1, mxbt->fp);
        if (x != 1)
        {
            mxbt->error = 1;
        }
    }
    return;
}

static void mxbtFindLeaf(MXBT *mxbt)
{
    int cnt;
    int x;
    
    mxbt->recordNum = mxbt->control.indexStart;
    mxbtReadRec(mxbt);
    while (!mxbt->error && (mxbt->index->recType != -1))
    {
        cnt = mxbt->index->keyCount;
        if (cnt == 0)
        {
            mxbt->error = 1;
        }
        else
        {
            for (x = 0; x < cnt; x++)
            {
                if (mxbt->compareF((char *)mxbt->index 
                                   + mxbt->index->keys[x].offset,
                                   mxbt->searchK,
                                   mxbt->index->keys[x].len) > 0)
                {
                    break;
                }
            }
            if (x == 0)
            {
                mxbt->recordNum = mxbt->index->recType;
            }
            else
            {
                mxbt->recordNum = mxbt->index->keys[x-1].lower;
            }
            mxbtReadRec(mxbt);
        }
    }
    return;
}

static void mxbtSearchLeaf(MXBT *mxbt)
{
    int cnt;
    int x;
    int ret;
    
    cnt = mxbt->leaf->keyCount;
    if (cnt == 0)
    {
        mxbt->error = 1;
    }
    else
    {
        for (x = 0; x < cnt; x++)
        {
            ret = mxbt->compareF((char *)mxbt->leaf 
                                 + mxbt->leaf->keys[x].offset,
                                 mxbt->searchK,
                                 mxbt->leaf->keys[x].len);
            if (ret > 0)
            {
                mxbt->error = 1;
                break;
            }
            else if (ret == 0)
            {
                mxbt->value = mxbt->leaf->keys[x].value;
                break;
            }
        }
    }
    return;
}

