/* mux.c -- mlink protocol multiplexing / demultiplexing
 *
 * Copyright (c) 1994 by Ezra Story.  All rights reserved.  Explicit
 * permission is given to use this source for any purpose as long as
 * the author (Ezra Story) is notified and this copyright notice is
 * included.
 *
 */
static char rcsid[] = "$Id: mux.c 1.18 1995/03/07 03:02:05 Ezra_Story Exp $";

#define INC_SYS

#include "defs.h"

export int      InitMux P((VOID));
export VOID     DeinitMux P((VOID));
export VOID     DeMux P((ubyte *, int));
export VOID     ReturnError P((ubyte, ubyte, ubyte));
export VOID     ReturnSuccess P((ubyte, ubyte, ubyte, ubyte *));
export VOID     WriteData P((ubyte, ubyte *, int));
export VOID     SendCommand P((ubyte, ubyte, ubyte, ubyte *));

export Fifo     writeFifo;

local VOID      DoCommand P((int, int, ubyte *));

local ubyte     *sockInBuf;
local ubyte     *cmdInBuf;
local ubyte     *cIndex;
local int       muxState;


int
InitMux(VOID)
{
    if (writeFifo = CreateFifo(2000))
        {
        if (sockInBuf = (ubyte *)malloc(CINBUF + SINBUF))
            {
            cIndex = cmdInBuf = sockInBuf + SINBUF;
            muxState = 0;
            SetErrnoMap();
            return(1);
            }
        DeleteFifo(writeFifo);
        }
    return(0);
}

VOID
DeinitMux(VOID)
{
    free(cmdInBuf);
    free(sockInBuf);
    DeleteFifo(writeFifo);
}

VOID
DeMux(buf, len)
ubyte *buf;
int len;
{
    int         st;
    ubyte       c;
    ubyte       *inbuf;
    mlsocket    *ms;

    /* realloc buffer if needed */
    if (len > SINBUF)
        {
        DeMux(buf, SINBUF);
        DeMux(buf+SINBUF, len-SINBUF);
        return;
        }

    /* set quikee vars */
    inbuf = sockInBuf;
    st    = muxState;

    /* loop over buffer */
    for(;len;len--,buf++)
        {
        c = *buf;

        switch (st)
            {
            case 0: /* just add bytes to buffer, check for SOC */
                if (c == SOC) st++;
                else *inbuf++ = c;
                break;

            case 1: /* first byte after a SOC */
                if (c == SOC)
                    {
                    *inbuf++ = c;
                    st = 0;
                    }
                else
                    {
                    if (c & 0x40)
                        {
                        if (ms = sockArray[inSock])
                            {
                            WriteSock(ms, sockInBuf, inbuf-sockInBuf);
                            inbuf = sockInBuf;
                            }
                        inSock = c & 0x3f;
                        st = 0;
                        }
                    else if (c & 0x80)
                        {
                        cIndex = cmdInBuf;
                        *cIndex++ = c & 0x3f;
                        st++;
                        }
                    else
                        st = 0;
                    }
                break;

            case 2: /* size byte of command packet */
                *cIndex++ = c;
                st = -((int)(c));
                break;

            default:  /* getting command packet */
                *cIndex++ = c;
                st++;
                if (st == 0)
                    {
                    /* flush socket buffer first */
                    if (ms = sockArray[inSock])
                        {
                        WriteSock(ms, sockInBuf, inbuf-sockInBuf);
                        inbuf = sockInBuf;
                        }

                    /* then do command */
                    cIndex = cmdInBuf;
                    DoCommand(cmdInBuf[0], cmdInBuf[1], cmdInBuf+2);
                    }
                break;
            }
        }

    if (ms = sockArray[inSock])
        {
        WriteSock(ms, sockInBuf, inbuf-sockInBuf);
        }
    muxState = st;

}

/* translate the serial command into a function call */
VOID
DoCommand(c, len, data)
int c, len;
ubyte *data;
{
    switch (c)
        {
        case CMD_NEWSOCKET:
            s_NewSocket(data[0], data[1], data[2]);
            break;
        case CMD_CLOSE:
            s_Close(data[0]);
            break;
        case CMD_SHUTDOWN:
            s_Shutdown(data[0], data[1]);
            break;
        case CMD_BIND:
            s_Bind(data[0], DATAWORD(data+1));
            break;
        case CMD_CONNECT:
            s_Connect(data[0], DATAWORD(data+1), DATALONG(data+3));
            break;
        case CMD_HBYADDR:
            s_HostByAddr(data[0], DATALONG(data+1));
            break;
        case CMD_HBYNAME:
            s_HostByName(data[0], data+1);
            break;
        case CMD_SBYNAME:
            s_ServByName(data[0], data[1], data+2);
            break;
        case CMD_SBYPORT:
            s_ServByPort(data[0], data[1], DATAWORD(data+2));
            break;
        case CMD_LISTEN:
            s_Listen(data[0]);
            break;
        case CMD_ACCEPT:
            s_Accept(data[0], data[1]);
            break;
        case CMD_PRIORITY:
            s_Priority(data[0], data[1]);
            break;
        case CMD_OOBDATA:
            s_OOBData(data[0], data[1]);
            break;
        case CMD_CONFIG:
            ReConfig(data[0], len-1, data+1);
            break;
        case CMD_GETHOSTINFO:
            GetHostInfo(data[0], data[1]);
            break;
        case CMD_SETSOCKOPT:
            s_SetSockOpt(data[0], data[1], data[2]);
            break;
        case CMD_EXITNOW:
            done();
            break;
        case CMD_BLOCK:
            s_Block(data[0], data[1]);
            break;
        case CMD_PING:
        default:  /* huh? :-) */
            break;
        }
}

VOID
ReturnError(cmd, desc, err)
ubyte cmd, desc, err;
{
    ubyte out[5];

    out[0] = SOC;
    out[1] = cmd | 0x80;
    out[2] = 2;
    out[3] = desc;
    out[4] = GetErrno(err);
    WriteFifo(writeFifo, out, 5, FIFO_STRM);
}

VOID
ReturnSuccess(cmd, desc, len, blk)
ubyte cmd, desc, len;
ubyte *blk;
{
    ubyte out[5];

    out[0] = SOC;
    out[1] = cmd | 0x80;
    out[2] = 2+len;
    out[3] = desc;
    out[4] = 0;
    WriteFifo(writeFifo, out, 5, FIFO_STRM);
    WriteFifo(writeFifo, blk, len, FIFO_STRM);
}

VOID
SendCommand(cmd, desc, len, blk)
ubyte cmd, desc, len;
ubyte *blk;
{
    ubyte out[4];

    out[0] = SOC;
    out[1] = cmd | 0x80;
    out[2] = 1+len;
    out[3] = desc;
    WriteFifo(writeFifo, out, 4, FIFO_STRM);
    WriteFifo(writeFifo, blk, len, FIFO_STRM);
}

VOID
WriteData(desc, buf, len)
ubyte desc;
ubyte *buf;
int   len;
{
    ubyte out[2];
    ubyte *a;

    /* do a channel switch if needed */
    out[0] = SOC;
    if (outSock != desc)
        {
        outSock = desc;
        out[1] = desc | 0x40;
        WriteFifo(writeFifo, out, 2, FIFO_STRM);
        }
    out[1] = SOC;

    /* output buffer, shifting any SOC bytes in stream */
    a = buf;
    while(len)
        {
        if (*buf == SOC)
            {
            if (buf-a) WriteFifo(writeFifo, a, buf-a, FIFO_STRM);
            WriteFifo(writeFifo, out, 2, FIFO_STRM);
            a = buf + 1;
            }
        buf++;
        len--;
        }
    if (buf-a) WriteFifo(writeFifo, a, buf-a, FIFO_STRM);
}

