#include <stdio.h>
#include <stdlib.h>

#include "mtypes.h"

#include "forte.h"
#include "gf1proto.h"
#include "gf1hware.h"

#include "extern.h"
#include "ultraerr.h"
#include "ultraext.h"

#include "mdriver.h"


extern ULTRA_CFG config;

#define MAXHANDLE 125		// should be enough for now

ULONG Ultra[MAXHANDLE];

/* Ultra[] holds the sample dram adresses
   of the samples of a module */


extern UNIMOD *pf;


void SetOther(UBYTE voice,ULONG frq,UBYTE ivol,UBYTE pan)
{
	UWORD vol=((UWORD)md_mainvol*ivol)/0xd;

	UltraSetFrequency(voice,frq);
	UltraVectorLinearVolume(voice,vol,0x3f,0);
	UltraSetBalance(voice,pan>>4);
}


void HandleVolume(int voice)
{
	UWORD vol;
	ULONG base,start,size,reppos,repend;
	GHOLD *aud;

	aud=&ghld[voice];

	if(aud->kick){

		aud->kick=0;

		base=Ultra[aud->handle];

		start=aud->start;
		reppos=aud->reppos;
		repend=aud->repend;
		size=aud->size;

		if(aud->flags&SF_16BITS){
			start<<=1;
			reppos<<=1;
			repend<<=1;
			size<<=1;
		}

		/* Stop current sample and start a new one */

		UltraStopVoice(voice);

		SetOther(voice,aud->frq,aud->vol,aud->pan);

		if(aud->flags&SF_LOOP){

			// Start a looping sample

			UltraStartVoice(voice,
							base+start,
							base+reppos,
							base+repend,0x8|((aud->flags&SF_16BITS)?4:0));
		}
		else{

			// Start a one-shot sample

			UltraStartVoice(voice,
							base+start,
							base+start,
							base+size+2,(aud->flags&SF_16BITS)?4:0);
		}
	}
}


void SetBPM(int bpm)
{
	/* The player routine has to be called (bpm*50)/125 times a second,
	   so the interval between calls takes 125/(bpm*50) seconds (amazing!).

	   The Timer1 handler has a resolution of 240 microseconds.

	   So the timer value to program:

	   (125/(bpm*50)) / 2.4e-4 = 10417/bpm
	*/

	UltraStartTimer(1,10417/bpm);
}



void HandleTimer1()
{
	int t;
	static int odd=0;

	/* Only service every 3rd call to this handler .. This
	   effectively makes this a 3*80=240 microsecond handler */

	odd++;
	if(odd==3){
		if(md_tickhandler!=NULL) md_tickhandler();
		SetBPM(md_bpm);		// Update beats-per-minute

		for(t=0;t<md_numchn;t++){
			GHOLD *aud;

			aud=&ghld[t];

			if(aud->kick){
				if(UltraReadVolume(t)<100){
					HandleVolume(t);
				}
				else{
					UltraVectorLinearVolume(t,0,0x3f,VL_WAVE_IRQ);
				}
			}
			else{
				SetOther(t,aud->frq,aud->vol,aud->pan);
			}
		}
		odd=0;
	}
}



BOOL GUS_Load(FILE *fp,SAMPLEINFO *smp)
/*
	callback routine for the MODLOAD module.

	fp			:file ptr to that sample
	smp			:Sampleinfo of the sample that is being loaded.
*/
{
	int handle,t;
	long length,loopstart,loopend,p,l;
	char *buffer[1024];

	SL_Init(fp,smp->flags,smp->flags|SF_SIGNED);

	// Find empty slot to put sample address in

	for(handle=0;handle<MAXHANDLE;handle++){
		if(Ultra[handle]==0) break;
	}

	if(handle==MAXHANDLE){
		myerr=ERROR_OUT_OF_HANDLES;
		return 0;
	}

	length=smp->length;
	loopstart=smp->loopstart;
	loopend=smp->loopend;

	if(smp->flags&SF_16BITS){
		length<<=1;
		loopstart<<=1;
		loopend<<=1;
	}

	// Allocate GUS dram and store the address in Ultra[handle]
	// Alloc 8 bytes more for anticlick measures. see below.

	if(UltraMemAlloc(length+8,&Ultra[handle])!=ULTRA_OK){
		myerr=ERROR_SAMPLE_TOO_BIG;
		return 0;
	}

	// Load the sample

	p=Ultra[handle];
	l=length;

	while(l>0){
		char buffer[512];
		long todo;

		todo=(l>512) ? 512 : l;

		SL_Load(buffer,todo);

		UltraDownload(buffer,0,p,todo+8,TRUE);

		p+=todo;
		l-=todo;
	}

	if(smp->flags&SF_LOOP){	// looping sample ?

		/*	Anticlick for looping samples:
			Copy the first bytes in the loop
			beyond the end of the loop */

		for(t=0;t<8;t++){
			UltraPoke(Ultra[handle]+loopend+t,
					  UltraPeek(Ultra[handle]+loopstart+t));
		}
	}
	else{

		/* 	Anticlick for one-shot samples:
			Zero the bytes beyond the end of the sample.
		*/

		for(t=0;t<8;t++){
			UltraPoke(Ultra[handle]+length+t,0);
		}
	}

	smp->handle=handle;
	return 1;
}



void GUS_UnLoad(SAMPLEINFO *smp)
/*
	callback routine to unload samples

	smp			:sampleinfo of sample that is being freed
*/
{
	long length;

	length=smp->length;
	if(smp->flags&SF_16BITS) length<<=1;

	UltraMemFree(length+8,Ultra[smp->handle]);
	Ultra[smp->handle]=0;
}



BOOL GUS_Init(void)
{
	if(!(md_mode&DMODE_16BITS)){
		md_mode|=DMODE_16BITS;		// gus can't do 8 bit mixing
	}

	if(!(md_mode&DMODE_STEREO)){
		md_mode|=DMODE_STEREO;		// gus can't do mono mixing
	}

	/* get gus config */

	if(!UltraGetCfg(&config)){
		myerr="Ultrasound env. string not found..";
		return 0;
	}

	/* Set up 14 channels */

	if(UltraOpen(&config,14)==NO_ULTRA){
		myerr="No ultrasound card found";
		return 0;
	}

	/* Grab the 80 microsecond timer handler */

	UltraTimer1Handler(HandleTimer1);
	UltraVolumeHandler(HandleVolume);
	return 1;
}



void GUS_Exit(void)
{
	UltraStopTimer(1);
	UltraDisableOutput();
	UltraClose();
}



void GUS_PlayStart(void)
{
	// init number of voices

	UltraNumVoices((md_numchn<14)?14:md_numchn);
	md_mixfreq=(md_numchn<=14) ? 44100 : (617400L/md_numchn);

	// Let's make some noise !

	UltraEnableOutput();
	SetBPM(md_bpm);			// Kickstart the timer
}



void GUS_PlayStop(void)
{
	UltraStopTimer(1);
	UltraDisableOutput();
}


BOOL GUS_IsThere(void)
{
	return(getenv("ULTRASND")!=NULL);
}


DRIVER gusdriver={
	NULL,
	"Gravis Ultrasound",
	"MikMod GUS Driver v0.2 (uses the GUS-SDK 2.11)",
	GUS_IsThere,
	GUS_Load,
	GUS_UnLoad,
	GUS_Init,
	GUS_Exit,
	GUS_PlayStart,
	GUS_PlayStop
};
