//***************************************************************************
//
//  VisualKB demonstrates basic mouse and keyboard handling techniques in
//  a text-entry application and echoes keyboard messages to the screen.
//
//***************************************************************************

#include <afxwin.h>
#include "visualkb.h"

CMyApp myApp;

/////////////////////////////////////////////////////////////////////////////
// CMyApp member functions

BOOL CMyApp::InitInstance ()
{
    m_pMainWnd = new CMainWindow ();
    m_pMainWnd->ShowWindow (m_nCmdShow);
    m_pMainWnd->UpdateWindow ();
    return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
// CMainWindow message map and member functions

BEGIN_MESSAGE_MAP (CMainWindow, CWnd)
    ON_WM_CREATE ()
    ON_WM_PAINT ()
    ON_WM_SETFOCUS ()
    ON_WM_KILLFOCUS ()
    ON_WM_LBUTTONDOWN ()
    ON_WM_MOUSEMOVE ()
    ON_WM_KEYDOWN ()
    ON_WM_KEYUP ()
    ON_WM_SYSKEYDOWN ()
    ON_WM_SYSKEYUP ()
    ON_WM_CHAR ()
    ON_WM_SYSCHAR ()
END_MESSAGE_MAP ()

CMainWindow::CMainWindow ()
{
    m_nTextPos = 0;
    m_nMsgPos = 0;

    m_hCursorArrow = myApp.LoadStandardCursor (IDC_ARROW);
    m_hCursorIBeam = myApp.LoadStandardCursor (IDC_IBEAM);

    CString myClass = AfxRegisterWndClass (0, NULL,
        (HBRUSH) ::GetStockObject (LTGRAY_BRUSH),
        myApp.LoadStandardIcon (IDI_APPLICATION));

    CreateEx (WS_EX_DLGMODALFRAME, myClass, "Visual Keyboard",
        WS_OVERLAPPED | WS_SYSMENU | WS_CAPTION | WS_MINIMIZEBOX,
        CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT,
        NULL, NULL, NULL);
}

int CMainWindow::OnCreate (LPCREATESTRUCT lpCreateStruct)
{
    TEXTMETRIC tm;

    CDC* pDC = GetDC ();
    pDC->GetTextMetrics (&tm);
    m_cxChar = tm.tmAveCharWidth;
    m_cyChar = tm.tmHeight;
    m_cyLine = tm.tmHeight + tm.tmExternalLeading;
    ReleaseDC (pDC);

    m_rectTextBox.SetRect (16, 16, (m_cxChar * 64) + 16,
        (m_cyChar * 2) + 16);

    m_rectMsgBox.SetRect (16, (m_cyChar * 5) + 16, (m_cxChar * 64) + 16,
        (m_cyLine * MAX_STRINGS) + (m_cyChar * 7) + 16);

    m_rectScroll.SetRect (m_cxChar + 16, (m_cyChar * 7) + 16,
        (m_cxChar * 63) + 16, (m_cyLine * MAX_STRINGS) + (m_cyChar * 6) + 16);

    m_pointTextOrigin.x = m_cxChar + 16;
    m_pointTextOrigin.y = (m_cyChar / 2) + 16;

    m_pointCaretPos = m_pointTextOrigin;
    m_nTextLimit = (m_cxChar * 63) + 16;

    m_pointHeaderOrigin.x = m_cxChar + 16;
    m_pointHeaderOrigin.y = (m_cyChar * 4) + 16;

    m_pointUpperMsgOrigin.x = m_cxChar + 16;
    m_pointUpperMsgOrigin.y = (m_cyChar * 6) + 16;

    m_pointLowerMsgOrigin.x = m_cxChar + 16;
    m_pointLowerMsgOrigin.y = (m_cyChar * 6) +
        (m_cyLine * (MAX_STRINGS - 1)) + 16;

    m_nTabStops[0] = (m_cxChar * 24) + 16;
    m_nTabStops[1] = (m_cxChar * 30) + 16;
    m_nTabStops[2] = (m_cxChar * 36) + 16;
    m_nTabStops[3] = (m_cxChar * 42) + 16;
    m_nTabStops[4] = (m_cxChar * 46) + 16;
    m_nTabStops[5] = (m_cxChar * 50) + 16;
    m_nTabStops[6] = (m_cxChar * 54) + 16;

    CRect rect (0, 0, m_rectMsgBox.right + 16, m_rectMsgBox.bottom + 16);
    CalcWindowRect (&rect);

    SetWindowPos (NULL, 0, 0, rect.Width (), rect.Height (),
        SWP_NOZORDER | SWP_NOMOVE | SWP_NOREDRAW);
    return 0;
}

void CMainWindow::PostNcDestroy ()
{
    delete this;
}

void CMainWindow::OnPaint ()
{
    CPaintDC dc (this);
    static CString header = "Message\tChar\tRep\tScan\tExt\tCon\tPrv\tTran";

    dc.SetBkMode (TRANSPARENT);
    dc.TabbedTextOut (m_pointHeaderOrigin.x, m_pointHeaderOrigin.y,
        header, header.GetLength (), sizeof (m_nTabStops), m_nTabStops,
        m_pointHeaderOrigin.x);

    Draw3DRectangle (&dc, m_rectTextBox);
    Draw3DRectangle (&dc, m_rectMsgBox);

    DrawInputText (&dc);
    DrawMessages (&dc);
}

void CMainWindow::OnSetFocus (CWnd* pWnd)
{
    CreateSolidCaret (2, m_cyChar);
    SetCaretPos (m_pointCaretPos);
    ShowCaret ();
}

void CMainWindow::OnKillFocus (CWnd* pWnd)
{
    HideCaret ();
    m_pointCaretPos = GetCaretPos ();
    ::DestroyCaret ();
}

void CMainWindow::OnLButtonDown (UINT nFlags, CPoint point)
{
    if (m_rectTextBox.PtInRect (point)) {
        m_nTextPos = GetNearestPos (point);
        PositionCaret ();
    }
}

void CMainWindow::OnMouseMove (UINT nFlags, CPoint point)
{
    ::SetCursor (m_rectTextBox.PtInRect (point) ?
        m_hCursorIBeam : m_hCursorArrow);
}

void CMainWindow::OnKeyDown (UINT nChar, UINT nRepCnt, UINT nFlags)
{
    CString msg = "WM_KEYDOWN";
    ShowMessage (msg, nChar, nRepCnt, nFlags);

    switch (nChar) {

    case VK_LEFT:
        if (m_nTextPos != 0) {
            m_nTextPos--;
            PositionCaret ();
        }
        break;

    case VK_RIGHT:
        if (m_nTextPos != m_text.GetLength ()) {
            m_nTextPos++;
            PositionCaret ();
        }
        break;

    case VK_HOME:
        m_nTextPos = 0;
        PositionCaret ();
        break;

    case VK_END:
        m_nTextPos = m_text.GetLength ();
        PositionCaret ();
        break;
    }
}

void CMainWindow::OnChar (UINT nChar, UINT nRepCnt, UINT nFlags)
{
    CString msg = "WM_CHAR";
    ShowMessage (msg, nChar, nRepCnt, nFlags);

    CClientDC dc (this);

    switch (nChar) {

    case VK_ESCAPE:
    case VK_RETURN:
        m_text.Empty ();
        m_nTextPos = 0;
        break;

    case VK_BACK:
        if (m_nTextPos != 0) {
            m_text = m_text.Left (m_nTextPos - 1) +
                m_text.Right (m_text.GetLength () - m_nTextPos);
            m_nTextPos--;
        }
        break;

    default:
        if ((nChar >= 0) && (nChar <= 31))
            return;

        if (m_nTextPos == m_text.GetLength ()) {
            m_text += nChar;
            m_nTextPos++;
        }
        else
            m_text.SetAt (m_nTextPos++, nChar);

        CSize size = dc.GetTextExtent (m_text, m_text.GetLength ());
        if ((m_pointTextOrigin.x + size.cx) > m_nTextLimit) {
            m_text = nChar;
            m_nTextPos = 1;
        }
        break;
    }

    HideCaret ();
    DrawInputText (&dc);
    PositionCaret (&dc);
    ShowCaret ();
}

void CMainWindow::OnKeyUp (UINT nChar, UINT nRepCnt, UINT nFlags)
{
    CString msg = "WM_KEYUP";
    ShowMessage (msg, nChar, nRepCnt, nFlags);
    CWnd::OnKeyUp (nChar, nRepCnt, nFlags);
}

void CMainWindow::OnSysKeyDown (UINT nChar, UINT nRepCnt, UINT nFlags)
{
    CString msg = "WM_SYSKEYDOWN";
    ShowMessage (msg, nChar, nRepCnt, nFlags);
    CWnd::OnSysKeyDown (nChar, nRepCnt, nFlags);
}

void CMainWindow::OnSysChar (UINT nChar, UINT nRepCnt, UINT nFlags)
{
    CString msg = "WM_SYSCHAR";
    ShowMessage (msg, nChar, nRepCnt, nFlags);
    CWnd::OnSysChar (nChar, nRepCnt, nFlags);
}

void CMainWindow::OnSysKeyUp (UINT nChar, UINT nRepCnt, UINT nFlags)
{
    CString msg = "WM_SYSKEYUP";
    ShowMessage (msg, nChar, nRepCnt, nFlags);
    CWnd::OnSysKeyUp (nChar, nRepCnt, nFlags);
}

void CMainWindow::PositionCaret (CDC* pDC)
{
    BOOL bRelease = FALSE;

    if (pDC == NULL) {
        pDC = GetDC ();
        bRelease = TRUE;
    }

    CPoint point = m_pointTextOrigin;
    CString temp = m_text.Left (m_nTextPos);
    point.x += (pDC->GetTextExtent (temp, temp.GetLength ())).cx;
    SetCaretPos (point);

    if (bRelease)
        ReleaseDC (pDC);
}

int CMainWindow::GetNearestPos (CPoint point)
{
    if (point.x <= m_pointTextOrigin.x)
        return 0;

    CClientDC dc (this);
    int nLen = m_text.GetLength ();
    if (point.x >= (m_pointTextOrigin.x +
        (dc.GetTextExtent (m_text, nLen)).cx))
        return nLen;

    int i = 0;
    int nPrevChar = m_pointTextOrigin.x;
    int nNextChar = m_pointTextOrigin.x;

    while (nNextChar < point.x) {
        i++;
        nPrevChar = nNextChar;      
        nNextChar = m_pointTextOrigin.x +
            (dc.GetTextExtent (m_text.Left (i), i)).cx;
    }
    return ((point.x - nPrevChar) < (nNextChar - point.x)) ? i - 1: i;
}

void CMainWindow::DrawInputText (CDC* pDC)
{
    pDC->SelectStockObject (NULL_PEN);
    pDC->SelectStockObject (LTGRAY_BRUSH);
    pDC->SetBkColor (RGB (192, 192, 192));

    pDC->TextOut (m_pointTextOrigin.x, m_pointTextOrigin.y, m_text);

    pDC->Rectangle (m_pointTextOrigin.x +
        (pDC->GetTextExtent (m_text, m_text.GetLength ())).cx,
        m_rectTextBox.top + 1, m_rectTextBox.right - 1,
        m_rectTextBox.bottom - 1);
}

void CMainWindow::Draw3DRectangle (CDC* pDC, CRect& rect)
{
    CPen* pPen = new CPen (PS_SOLID, 1, RGB (128, 128, 128));
    pDC->SelectObject (pPen);

    pDC->MoveTo (rect.left, rect.bottom - 2);
    pDC->LineTo (rect.left, rect.top);
    pDC->LineTo (rect.right - 1, rect.top);

    delete pDC->SelectStockObject (WHITE_PEN);

    pDC->MoveTo (rect.right - 1, rect.top + 1);
    pDC->LineTo (rect.right - 1, rect.bottom - 1);
    pDC->LineTo (rect.left, rect.bottom - 1);
}

void CMainWindow::ShowMessage (CString& msg, UINT nChar, UINT nRepCnt,
                               UINT nFlags)
{
    CString string;
    string.Format ("\t %u\t  %u\t  %u\t  %u\t  %u\t  %u\t   %u",
        nChar, nRepCnt, nFlags & 0xFF,
        (nFlags >> 8) & 0x01,
        (nFlags >> 13) & 0x01,
        (nFlags >> 14) & 0x01,
        (nFlags >> 15) & 0x01);

    ScrollWindow (0, -m_cyLine, &m_rectScroll);
    ValidateRect (m_rectScroll);

    CClientDC dc (this);
    dc.SetBkColor (RGB (192, 192, 192));

    m_messages[m_nMsgPos] = msg + string;
    dc.TabbedTextOut (m_pointLowerMsgOrigin.x, m_pointLowerMsgOrigin.y,
        m_messages[m_nMsgPos], m_messages[m_nMsgPos].GetLength (),
        sizeof (m_nTabStops), m_nTabStops, m_pointLowerMsgOrigin.x);

    if (++m_nMsgPos == MAX_STRINGS)
        m_nMsgPos = 0;  
}

void CMainWindow::DrawMessages (CDC* pDC)
{
    int pos = m_nMsgPos;

    for (int i=0; i<MAX_STRINGS; i++) {

        pDC->TabbedTextOut (m_pointUpperMsgOrigin.x,
            m_pointUpperMsgOrigin.y + (m_cyLine * i),
            m_messages[pos], m_messages[pos].GetLength (),
            sizeof (m_nTabStops), m_nTabStops,
            m_pointUpperMsgOrigin.x);

        if (++pos == MAX_STRINGS)
            pos = 0;    
    }
}
