/****************************************************************** ITEMS.CPP
 *                                                                          *
 *                     Display Item Class Functions                         *
 *                                                                          *
 ****************************************************************************/

#define INCL_BASE
#define INCL_DOSDEVIOCTL
#define INCL_PM
#include <os2.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "Config.h"
#include "Debug.h"
#include "DQPS.h"
#include "MemSize.h"
#include "Support.h"
#include "ReString.h"

#include "Items.h"

static int MeasureValue ( char *Text, char *Label, BOOL ShowK, BOOL ShowM ) {
   int Result ;
   if ( ShowK == SHOWK_NEVER ) {
      Result = sprintf ( Text, "%s 1,234,567,890 ", Label ) ;
   } else if ( !ShowM ) {
      Result = sprintf ( Text, "%s 1,234,567K", Label ) ;
   } else {
      Result = sprintf ( Text, "%s 1,234M", Label ) ;
   } /* endif */
   return ( Result ) ;
} 

inline ULONG KiloScale ( ULONG Value, int ShowTrueK=TRUE, int Scale=1 ) {
   if ( ShowTrueK ) {
      return ( (Value+(0x200/Scale))/(0x400/Scale) ) ;
   } else {
      return ( (Value+(500/Scale))/(1000/Scale) ) ;
   } /* endif */
}

inline ULONG MegaScale ( ULONG Value, int ShowTrueK=TRUE, int Scale=1 ) {
   if ( ShowTrueK ) {
      return ( (Value+(0x80000/Scale))/(0x100000/Scale) ) ;
   } else {
      return ( (Value+(500000/Scale))/(1000000/Scale) ) ;
   } /* endif */
}

static void FormatValue ( ULONG Value, char Text[], COUNTRYINFO &CountryInfo, int ShowK=SHOWK_ABOVE512, int ShowTrueK=TRUE, int ShowM=TRUE, int Scale=1 ) {

   int MegaBreak = ShowTrueK ? 0x800000/Scale : 8000000/Scale ;
   int KiloBreak = ShowTrueK ? 0x80000/Scale : 500000/Scale ;

   switch ( ShowK ) {
      case SHOWK_NEVER:
         sprintf ( Text, "%lu", Value*Scale ) ;
         break;
      case SHOWK_ALWAYS:
         if ( !ShowM || ( Value < MegaBreak ) )
            sprintf ( Text, "%lu", KiloScale(Value,ShowTrueK,Scale) ) ;
         else 
            sprintf ( Text, "%lu", MegaScale(Value,ShowTrueK,Scale) ) ;
         break;
      case SHOWK_ABOVE512:
      default:
         if ( Value < KiloBreak )
            sprintf ( Text, "%lu", Value*Scale ) ;
         else if ( !ShowM || ( Value < MegaBreak ) )
            sprintf ( Text, "%lu", KiloScale(Value,ShowTrueK,Scale) ) ;
         else
            sprintf ( Text, "%lu", MegaScale(Value,ShowTrueK,Scale) ) ;
         break;
   } /* endswitch */

   char Work[100] ;
   char *p1 = Text ;
   char *p2 = Work ;
   while ( *p1 ) {
      *p2 = *p1 ;
      p1 ++ ;
      p2 ++ ;
      if ( *p1 ) {
         if ( strlen(p1) % 3 == 0 ) {
            *p2 = CountryInfo.szThousandsSeparator [0] ;
            p2 ++ ;
         } /* endif */
      } /* endif */
   } /* endwhile */
   *p2 = 0 ;
   strcpy ( Text, Work ) ;

   p2 = Text + strlen(Text) ;
   *(p2+1) = 0 ;
   switch ( ShowK ) {
      case SHOWK_NEVER:
         *p2 = ' ' ;
         break;
      case SHOWK_ALWAYS:
         if ( !ShowM || ( Value < MegaBreak ) )
            *p2 = (char) ( ShowTrueK ? 'K' : 'k' ) ;
         else 
            *p2 = (char) ( ShowTrueK ? 'M' : 'm' ) ;
         break;
      case SHOWK_ABOVE512:
      default:
         if ( Value < KiloBreak )
            *p2 = ' ' ;
         else if ( !ShowM || ( Value < MegaBreak ) )
            *p2 = (char) ( ShowTrueK ? 'K' : 'k' ) ;
         else 
            *p2 = (char) ( ShowTrueK ? 'M' : 'm' ) ;
         break;
   } /* endswitch */
}


Item::Item ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, Dde_Server *ddeserver, char *topic ) :
   Id(id), Flag(TRUE), Value(0), pDdeServer(ddeserver), LevelSense(0) {
   strcpy ( Name, pName ) ;
   strcpy ( CurrentLabel, pCurrentLabel ) ;
   strcpy ( DefaultLabel, pDefaultLabel ) ;
   strcpy ( Topic, topic ) ;
   pDdeServer->AddItem ( Topic, DefaultLabel, DDEFMT_TEXT, "", 1 ) ;
   pDdeItem = pDdeServer->FindItem ( Topic, DefaultLabel ) ;
   NormalColors [0] = 0xFFFFFF ;
   NormalColors [1] = 0x000000 ;
   WarningColors[0] = 0xFFFF00 ;
   WarningColors[1] = 0x000000 ;
   ErrorColors  [0] = 0xFF0000 ;
   ErrorColors  [1] = 0xFFFFFF ;
} /* endmethod */

Item::~Item ( ) {
   pDdeServer->RemoveItem ( Topic, DefaultLabel ) ;
} /* endmethod */

void Item::GetProfile ( HINI IniHandle ) {

     Flag = TRUE ;  ULONG Size ;
     if ( PrfQueryProfileSize ( IniHandle, PSZ(PROGRAM_NAME), PSZ(Name), &Size )
        AND ( ( Size == sizeof(Flag) ) OR ( Size == sizeof(short) ) )
        AND PrfQueryProfileData ( IniHandle, PSZ(PROGRAM_NAME), PSZ(Name), &Flag, &Size ) ) {
     } /* endif */

     char Tag [80] ;
     strcpy ( Tag, PCHAR(Name) ) ;
     strcat ( Tag, " Label" ) ;
     if ( PrfQueryProfileSize ( IniHandle, PSZ(PROGRAM_NAME), PSZ(Tag), &Size )
        AND ( Size == sizeof(CurrentLabel) )
        AND PrfQueryProfileData ( IniHandle, PSZ(PROGRAM_NAME), PSZ(Tag), &CurrentLabel, &Size ) ) {
     } /* endif */

     strcpy ( Tag, PCHAR(Name) ) ;
     strcat ( Tag, " Warning" ) ;
     if ( PrfQueryProfileSize ( IniHandle, PSZ(PROGRAM_NAME), PSZ(Tag), &Size )
        AND ( Size == sizeof(WarningLevel) )
        AND PrfQueryProfileData ( IniHandle, PSZ(PROGRAM_NAME), PSZ(Tag), &WarningLevel, &Size ) ) {
     } /* endif */

     strcpy ( Tag, PCHAR(Name) ) ;
     strcat ( Tag, " Error" ) ;
     if ( PrfQueryProfileSize ( IniHandle, PSZ(PROGRAM_NAME), PSZ(Tag), &Size )
        AND ( Size == sizeof(ErrorLevel) )
        AND PrfQueryProfileData ( IniHandle, PSZ(PROGRAM_NAME), PSZ(Tag), &ErrorLevel, &Size ) ) {
     } /* endif */

} /* endmethod */

void Item::PutProfile ( HINI IniHandle ) {

   PrfWriteProfileData ( IniHandle, PSZ(PROGRAM_NAME), PSZ(Name), &Flag, sizeof(Flag) ) ;

   // Save custom label, if any.
   char Tag [80] ;
   strcpy ( Tag, PCHAR(Name) ) ;
   strcat ( Tag, " Label" ) ;
   if ( strcmp ( CurrentLabel, DefaultLabel ) ) {
      PrfWriteProfileData ( IniHandle, PSZ(PROGRAM_NAME), PSZ(Tag), CurrentLabel, sizeof(CurrentLabel) ) ;
   } else {
      PrfWriteProfileData ( IniHandle, PSZ(PROGRAM_NAME), PSZ(Tag), 0, 0 ) ;
   } /* endif */

   // If the item uses a warning/error threshold at all . . .
   if ( LevelSense ) {

      // Save the warning threshold value.
      ULONG Threshold = WarningLevel ;
      strcpy ( Tag, PCHAR(Name) ) ;
      strcat ( Tag, " Warning" ) ;
      PrfWriteProfileData ( IniHandle, PSZ(PROGRAM_NAME), PSZ(Tag), &Threshold, sizeof(Threshold) ) ;

      // Save the error threshold value.
      Threshold = ErrorLevel ;
      strcpy ( Tag, PCHAR(Name) ) ;
      strcat ( Tag, " Error" ) ;
      PrfWriteProfileData ( IniHandle, PSZ(PROGRAM_NAME), PSZ(Tag), &Threshold, sizeof(Threshold) ) ;

   } /* endif */

} /* endmethod */

void Item::FormatLine ( char *Buffer, int MaxWidth ) {

   // Get the left and right texts for the line.
   char Label[100], Text[100] ;
   FormatText ( Label, Text, QueryValue() ) ;

   // Format the line.
   strcat ( Buffer, Label ) ;
   sprintf ( Buffer+strlen(Buffer), "%*s", MaxWidth-strlen(Label)-strlen(Text), "" ) ;
   strcat ( Buffer, Text ) ;
   strcat ( Buffer, "\r\n" ) ;
}


void Item::DdeUpdate ( char *Text ) {

   int Format, Size ; PVOID Data ;
   pDdeItem->Query ( Format, Data, Size ) ;

   int NewSize = strlen(Text) + 1 ;
   if ( Data && memcmp ( Text, Data, NewSize ) )
      pDdeItem->Update ( DDEFMT_TEXT, Text, NewSize ) ;
}


void Item::Paint ( HPS hPS, RECTL &Rectangle,
   char *Label, char *Text, ULONG NewValue, int Scale ) {

   COLOR *Colors = NormalColors ;
   if ( LevelSense > 0 ) {
      if ( NewValue >= ErrorLevel/Scale ) {
         Colors = ErrorColors ;
      } else if ( NewValue >= WarningLevel/Scale ) {
         Colors = WarningColors ;
      } /* endif */
   } else if ( LevelSense < 0 ) {
      if ( NewValue*Scale <= ErrorLevel/Scale ) {
         Colors = ErrorColors ;
      } else if ( NewValue <= WarningLevel/Scale ) {
         Colors = WarningColors ;
      } /* endif */
   } /* endif */

   WinDrawText ( hPS, strlen(Text), PSZ(Text), &Rectangle,
      Colors[1], Colors[0], DT_RIGHT | DT_BOTTOM | DT_ERASERECT ) ;

   WinDrawText ( hPS, strlen(Label), PSZ(Label), &Rectangle,
      Colors[1], Colors[0], DT_LEFT | DT_BOTTOM ) ;

   Value = NewValue ;
}


Clock::Clock ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
   Dde_Server *pDdeServer, char *Topic, COUNTRYINFO &countryinfo,
   char *szam, char *szpm, ResourceString *daysofweek, BOOL showseconds, BOOL hour24 )
   : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ),
   DaysOfWeek(daysofweek), ShowSeconds(showseconds), Hour24(hour24) {
   CountryInfo = countryinfo ;
   memcpy ( szAm, szam, min(strlen(szam),sizeof(szAm)-1) ) ;
   szAm [ sizeof(szAm) - 1 ] = 0 ;
   memcpy ( szPm, szpm, min(strlen(szpm),sizeof(szPm)-1) ) ;
   szPm [ sizeof(szPm) - 1 ] = 0 ;
} /* endmethod */

int Clock::Measure ( HPS hPS, RECTL &Rectangle ) {
   int Result ;
   char Text [100] ;
   if ( CountryInfo.fsDateFmt != DATEFMT_YY_MM_DD ) {
      Result = sprintf ( Text, " MMM, MM-DD HH:MM%s%s ", ShowSeconds?":SS":"", CountryInfo.fsTimeFmt?"":"xx" ) ;
   } else {
      Result = sprintf ( Text, " MM-DD MMM HH:MM%s%s ", ShowSeconds?":SS":"", CountryInfo.fsTimeFmt?"":"xx" ) ;
   } /* endif */
   WinQueryWindowRect ( HWND_DESKTOP, &Rectangle ) ;
   WinDrawText ( hPS, strlen(Text), PSZ(Text), &Rectangle, 0, 0, DT_LEFT | DT_BOTTOM | DT_QUERYEXTENT ) ;
   return ( Result ) ;
}

struct Date_Time {
   unsigned Weekday : 3 ;
   unsigned Month : 4 ;
   unsigned Day : 5 ;
   unsigned Hours : 5 ;
   unsigned Minutes : 6 ;
   unsigned Seconds : 6 ;
} ;

ULONG Clock::NewValue ( void ) {

  DATETIME DateTime ;
  DosGetDateTime ( &DateTime ) ;

  struct Date_Time Dttm = { 
     DateTime.weekday, 
     DateTime.month, 
     DateTime.day, 
     DateTime.hours, 
     DateTime.minutes, 
     0 
  } ;

  if ( ShowSeconds ) 
     Dttm.Seconds = DateTime.seconds ;

  ULONG Time = * ( (PULONG) &Dttm ) ;

  return ( Time ) ;
}


void Clock::FormatText ( char *Label, char *Text, ULONG Time ) {

   strcpy ( Label, CurrentLabel ) ;

   struct Date_Time Dttm = * ( (struct Date_Time*) &Time ) ;
   ULONG Dow    = Dttm.Weekday ;
   ULONG Month  = Dttm.Month ;
   ULONG Day    = Dttm.Day ;
   ULONG Hour   = Dttm.Hours ;
   ULONG Minute = Dttm.Minutes ;
   ULONG Second = Dttm.Seconds ;

   switch ( CountryInfo.fsDateFmt ) {

      case DATEFMT_DD_MM_YY:
         sprintf ( Text, "%0.*s, %02lu%s%02lu ",
            strlen(PCHAR(*DaysOfWeek))/7, PSZ(*DaysOfWeek) + Dow*(strlen(PCHAR(*DaysOfWeek))/7),
            Day, CountryInfo.szDateSeparator, Month ) ;
         break ;

      case DATEFMT_YY_MM_DD:
         sprintf ( Text, "%02lu%s%02lu %0.*s ",
            Month, CountryInfo.szDateSeparator, Day,
            strlen(PCHAR(*DaysOfWeek))/7, PSZ(*DaysOfWeek) + Dow*(strlen(PCHAR(*DaysOfWeek))/7) ) ;
         break ;

      case DATEFMT_MM_DD_YY:
      default:
         sprintf ( Text, "%0.*s, %02lu%s%02lu ",
            strlen(PCHAR(*DaysOfWeek))/7, PSZ(*DaysOfWeek) + Dow*(strlen(PCHAR(*DaysOfWeek))/7),
            Month, CountryInfo.szDateSeparator, Day ) ;
         break ;

   } /* endswitch */

   if ( Hour24 ) {
      if ( ShowSeconds ) {
         sprintf ( Text+strlen(Text), "%02lu%s%02lu%s%02lu",
             Hour, CountryInfo.szTimeSeparator, Minute, CountryInfo.szTimeSeparator, Second ) ;
      } else {
         sprintf ( Text+strlen(Text), "%02lu%s%02lu",
             Hour, CountryInfo.szTimeSeparator, Minute ) ;
      } /* endif */
   } else {
      PCHAR AmPm ;
      if ( Hour ) {
         if ( Hour < 12 ) {
            AmPm = szAm ;
         } else if ( Hour == 12 ) {
            if ( Minute )
               AmPm = szPm ;
            else
               AmPm = szAm ;
         } else if ( Hour > 12 ) {
            Hour -= 12 ;
            AmPm = szPm ;
         } /* endif */
      } else {
         Hour = 12 ;
         if ( Minute )
            AmPm = szAm ;
         else
            AmPm = szPm ;
      } /* endif */
      if ( ShowSeconds )
         sprintf ( Text+strlen(Text), "%02lu%s%02lu%s%02lu%s",
             Hour, CountryInfo.szTimeSeparator, Minute, CountryInfo.szTimeSeparator, Second, AmPm ) ;
      else
         sprintf ( Text+strlen(Text), "%02lu%s%02lu%s",
             Hour, CountryInfo.szTimeSeparator, Minute, AmPm ) ;
   } /* endif */

   DdeUpdate ( Text ) ;
}


void Clock::FormatLine ( char *Buffer, int MaxWidth ) {

   // Get the left and right texts for the line.
   char Label[100], Text[100] ;
   FormatText ( Label, Text, QueryValue() ) ;

   // Format the line.
   sprintf ( Buffer+strlen(Buffer), "%*s%s%*s", 
      (MaxWidth-strlen(Text))/2, "", Text, 
      MaxWidth-strlen(Text)-(MaxWidth-strlen(Text))/2, "" ) ;
   strcat ( Buffer, "\r\n" ) ;
}


void Clock::Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) {
   ULONG Time = NewValue ( ) ;
   if ( Mandatory || ( Time != Value ) ) {
      char Label[100], Text [100] ;
      FormatText ( Label, Text, Time ) ;
      WinDrawText ( hPS, strlen(Text), PSZ(Text), &Rectangle,
         NormalColors[1], NormalColors[0], DT_CENTER | DT_BOTTOM | DT_ERASERECT ) ;
      Value = Time ;
   } /* endif */
}


ElapsedTime::ElapsedTime ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
   Dde_Server *pDdeServer, char *Topic, COUNTRYINFO &countryinfo,
   ResourceString *day, ResourceString *days, BOOL showseconds )
   : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ),
   Day(day), Days(days), ShowSeconds(showseconds) {
   CountryInfo = countryinfo ;
} /* endmethod */

int ElapsedTime::Measure ( HPS hPS, RECTL &Rectangle ) {
   int Result ;
   char Text [100] ;
   Result = sprintf ( Text, "%s XX %s, XX:XX%s", QueryCurrentLabel(), PSZ(*Days), ShowSeconds?":XX":"" ) ;
   WinQueryWindowRect ( HWND_DESKTOP, &Rectangle ) ;
   WinDrawText ( hPS, strlen(Text), PSZ(Text), &Rectangle, 0, 0, DT_LEFT | DT_BOTTOM | DT_QUERYEXTENT ) ;
   return ( Result ) ;
}

ULONG ElapsedTime::NewValue ( void ) {
   ULONG Milliseconds ;
   DosQuerySysInfo ( QSV_MS_COUNT, QSV_MS_COUNT, &Milliseconds, sizeof(Milliseconds) ) ;
   return ( Milliseconds / 1000 ) ;
}

void ElapsedTime::FormatText ( char *Label, char *Text, ULONG Time ) {

   strcpy ( Label, CurrentLabel ) ;

   *Text = 0 ;

   ULONG NumberOfDays = Time / ( 60 * 60 * 24 ) ;
   Time -= NumberOfDays * 60 * 60 * 24 ;
   ULONG Hours   = Time / ( 60 * 60 ) ;
   Time -= Hours * 60 * 60 ;
   ULONG Minutes = Time / 60 ;
   ULONG Seconds = Time % 60 ;

   if ( NumberOfDays ) {
      sprintf ( Text, "%lu %s, ",
         NumberOfDays, NumberOfDays > 1 ? PSZ(*Days) : PSZ(*Day) ) ;
   } /* endif */

   if ( ShowSeconds )
      sprintf ( Text+strlen(Text), "%lu%s%02lu%s%02lu",
         Hours, CountryInfo.szTimeSeparator, Minutes,
         CountryInfo.szTimeSeparator, Seconds ) ;
   else
      sprintf ( Text+strlen(Text), "%lu%s%02lu",
         Hours, CountryInfo.szTimeSeparator, Minutes ) ;

   DdeUpdate ( Text ) ;
}

void ElapsedTime::Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) {
   ULONG Time = NewValue ( ) ;
   if ( Mandatory || ( Time != Value ) ) {
      char Label[100], Text[100] ;
      FormatText ( Label, Text, Time ) ;
      Paint ( hPS, Rectangle, Label, Text, Time ) ;
   } /* endif */
}


MemoryFree::MemoryFree ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
   Dde_Server *pDdeServer, char *Topic, COUNTRYINFO &countryinfo, USHORT sk, USHORT stk, USHORT sm ) 
   : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ), 
   ShowK(sk), ShowTrueK(stk), ShowM(sm) {
   CountryInfo = countryinfo ;
   LevelSense = -1 ;
   MinLevel = 0 ;
   ULONG TotalPhysicalMemory ;
   DosQuerySysInfo ( QSV_TOTPHYSMEM, QSV_TOTPHYSMEM, &TotalPhysicalMemory, sizeof(TotalPhysicalMemory) ) ;
   MaxLevel = TotalPhysicalMemory ;
   DefaultLevels [0] = WarningLevel = TotalPhysicalMemory / 8 ;
   DefaultLevels [1] = ErrorLevel = TotalPhysicalMemory / 16 ;
} /* endmethod */

int MemoryFree::Measure ( HPS hPS, RECTL &Rectangle ) {
   char Text [100] ;
   int Result = MeasureValue ( Text, QueryCurrentLabel(), ShowK, ShowM ) ;
   WinQueryWindowRect ( HWND_DESKTOP, &Rectangle ) ;
   WinDrawText ( hPS, strlen(Text), PSZ(Text), &Rectangle, 0, 0, DT_LEFT | DT_BOTTOM | DT_QUERYEXTENT ) ;
   return ( Result ) ;
}

ULONG MemoryFree::NewValue ( void ) {
   APIRET16 APIENTRY16 Dos16MemAvail ( PULONG pulAvailMem ) ;
   ULONG Value ;
   Dos16MemAvail ( &Value ) ;
   return ( Value ) ;
}

void MemoryFree::FormatText ( char *Label, char *Text, ULONG Value ) {
   strcpy ( Label, CurrentLabel ) ;
   FormatValue ( Value, Text, CountryInfo, ShowK, ShowTrueK, ShowM ) ;
   DdeUpdate ( Text ) ;
}

void MemoryFree::Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) {
   ULONG Size = NewValue ( ) ;
   if ( Mandatory || ( Size != Value ) ) {
      char Label[100], Text [100] ;
      FormatText ( Label, Text, Size ) ;
      Paint ( hPS, Rectangle, Label, Text, Size ) ;
   } /* endif */
}


VirtualFree::VirtualFree ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
   Dde_Server *pDdeServer, char *Topic, COUNTRYINFO &countryinfo, USHORT sk, USHORT stk, USHORT sm ) 
   : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ), 
   ShowK(sk), ShowTrueK(stk), ShowM(sm) {
   CountryInfo = countryinfo ;
   LevelSense = -1 ;
   MinLevel = 0 ;
   MaxLevel = 64*1024*1024 ;
   DefaultLevels [0] = WarningLevel = 8*1024*1024 ;
   DefaultLevels [1] = ErrorLevel = 4*1024*1024 ;
} /* endmethod */

int VirtualFree::Measure ( HPS hPS, RECTL &Rectangle ) {
   char Text [100] ;
   int Result = MeasureValue ( Text, QueryCurrentLabel(), ShowK, ShowM ) ;
   WinQueryWindowRect ( HWND_DESKTOP, &Rectangle ) ;
   WinDrawText ( hPS, strlen(Text), PSZ(Text), &Rectangle, 0, 0, DT_LEFT | DT_BOTTOM | DT_QUERYEXTENT ) ;
   return ( Result ) ;
}

ULONG VirtualFree::NewValue ( void ) {
   ULONG Space ;
   DosQuerySysInfo ( QSV_TOTAVAILMEM, QSV_TOTAVAILMEM, &Space, sizeof(Space) ) ;
   return ( Space ) ;
}

void VirtualFree::FormatText ( char *Label, char *Text, ULONG Value ) {
   strcpy ( Label, CurrentLabel ) ;
   FormatValue ( Value, Text, CountryInfo, ShowK, ShowTrueK, ShowM ) ;
   DdeUpdate ( Text ) ;
}

void VirtualFree::Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) {
   ULONG Size = NewValue ( ) ;
   if ( Mandatory || ( Size != Value ) ) {
      char Label[100], Text [100] ;
      FormatText ( Label, Text, Size ) ;
      Paint ( hPS, Rectangle, Label, Text, Size ) ;
   } /* endif */
}


SwapSize::SwapSize ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
   Dde_Server *pDdeServer, char *Topic, COUNTRYINFO countryinfo, 
   USHORT sk, USHORT stk, USHORT sm, PSZ swappath, ULONG initialsize )
   : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ), 
   ShowK(sk), ShowTrueK(stk), ShowM(sm) {
   CountryInfo = countryinfo ;
   SwapPath = new BYTE [ strlen(PCHAR(swappath)) + 1 ] ;
   strcpy ( PCHAR(SwapPath), PCHAR(swappath) ) ;
   LevelSense = +1 ;
   MinLevel = 0 ;
   MaxLevel = 64*1024*1024 ;
   DefaultLevels [0] = WarningLevel = initialsize > 0x2000 ? initialsize * 2 * 1024 : 16 * 1024 * 1024 ;
   DefaultLevels [1] = ErrorLevel = initialsize > 0x2000 ? initialsize * 2 * 1024 : 32 * 1024 * 1024 ;
} /* endmethod */

SwapSize::~SwapSize ( void ) {
   delete [] SwapPath ;
} /* endmethod */

int SwapSize::Measure ( HPS hPS, RECTL &Rectangle ) {
   char Text [100] ;
   int Result = MeasureValue ( Text, QueryCurrentLabel(), ShowK, ShowM ) ;
   WinQueryWindowRect ( HWND_DESKTOP, &Rectangle ) ;
   WinDrawText ( hPS, strlen(Text), PSZ(Text), &Rectangle, 0, 0, DT_LEFT | DT_BOTTOM | DT_QUERYEXTENT ) ;
   return ( Result ) ;
}

ULONG SwapSize::NewValue ( void ) {

   char Path [_MAX_PATH+1] ;
   strcpy ( Path, (PCHAR)SwapPath ) ;

   if ( Path[strlen(Path)-1] != '\\' )
      strcat ( Path, "\\" ) ;

   strcat ( Path, "SWAPPER.DAT" ) ;

   ULONG SwapSize = 0 ;
   FILESTATUS3 Status ;
   if ( DosQueryPathInfo ( (PSZ)Path, FIL_STANDARD, &Status, sizeof(Status) ) == 0 ) 
      SwapSize = Status.cbFile ;

   return ( SwapSize ) ;
}

void SwapSize::FormatText ( char *Label, char *Text, ULONG Value ) {
   strcpy ( Label, CurrentLabel ) ;
   FormatValue ( Value, Text, CountryInfo, ShowK, ShowTrueK, ShowM ) ;
   DdeUpdate ( Text ) ;
}

void SwapSize::Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) {
   ULONG Size = NewValue ( ) ;
   if ( Mandatory || ( Size != Value ) ) {
      char Label[100], Text [100] ;
      FormatText ( Label, Text, Size ) ;
      Paint ( hPS, Rectangle, Label, Text, Size ) ;
   } /* endif */
}


SwapFree::SwapFree ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
   Dde_Server *pDdeServer, char *Topic, COUNTRYINFO &countryinfo,
   USHORT sk, USHORT stk, USHORT sm, PSZ swappath, ULONG minfree )
   : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ),
   ShowK(sk), ShowTrueK(stk), ShowM(sm), MinFree(minfree) {
   CountryInfo = countryinfo ;
   SwapPath = new BYTE [ strlen(PCHAR(swappath)) + 1 ] ;
   strcpy ( PCHAR(SwapPath), PCHAR(swappath) ) ;
   LevelSense = -1 ;
   MinLevel = 0 ;
   MaxLevel = 64*1024*1024 ;
   DefaultLevels [0] = WarningLevel = 8*1024*1024 ;
   DefaultLevels [1] = ErrorLevel = 4*1024*1024 ;
} /* endmethod */

SwapFree::~SwapFree ( void ) {
   delete [] SwapPath ;
} /* endmethod */

int SwapFree::Measure ( HPS hPS, RECTL &Rectangle ) {
   char Text [100] ;
   int Result = MeasureValue ( Text, QueryCurrentLabel(), ShowK, ShowM ) ;
   WinQueryWindowRect ( HWND_DESKTOP, &Rectangle ) ;
   WinDrawText ( hPS, strlen(Text), PSZ(Text), &Rectangle, 0, 0, DT_LEFT | DT_BOTTOM | DT_QUERYEXTENT ) ;
   return ( Result ) ;
}

ULONG SwapFree::NewValue ( void ) {

   DosError ( FERR_DISABLEHARDERR ) ;
   FSALLOCATE Allocation ;
   DosQueryFSInfo ( SwapPath[0]-'A'+1, FSIL_ALLOC, PBYTE(&Allocation), sizeof(Allocation) ) ;
   DosError ( FERR_ENABLEHARDERR ) ;

   ULONG SwapFree = Allocation.cUnitAvail*Allocation.cSectorUnit*Allocation.cbSector ;

   SwapFree -= MinFree * 1024 ;                 // The swap-file will not allocate the space required free.

   SwapFree -= ( SwapFree % ( 1024 * 1024 ) ) ; // The swap-file only expands by 1M increments.

   return ( SwapFree ) ;
}

void SwapFree::FormatText ( char *Label, char *Text, ULONG Value ) {
   strcpy ( Label, CurrentLabel ) ;
   FormatValue ( Value, Text, CountryInfo, ShowK, ShowTrueK, ShowM ) ;
   DdeUpdate ( Text ) ;
}

void SwapFree::Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) {
   ULONG Size = NewValue ( ) ;
   if ( Mandatory || ( Size != Value ) ) {
      char Label[100], Text [100] ;
      FormatText ( Label, Text, Size ) ;
      Paint ( hPS, Rectangle, Label, Text, Size ) ;
   } /* endif */
}


SpoolSize::SpoolSize ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
   Dde_Server *pDdeServer, char *Topic, COUNTRYINFO &countryinfo, 
   USHORT sk, USHORT stk, USHORT sm, PSZ spoolpath )
   : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ), 
   ShowK(sk), ShowTrueK(stk), ShowM(sm) {
   CountryInfo = countryinfo ;
   SpoolPath = new BYTE [ strlen(PCHAR(spoolpath)) + 1 ] ;
   strcpy ( PCHAR(SpoolPath), PCHAR(spoolpath) ) ;
   LevelSense = +1 ;
   MinLevel = 0 ;
   MaxLevel = 64*1024*1024 ;
   DefaultLevels [0] = WarningLevel = 8*1024*1024 ;
   DefaultLevels [1] = ErrorLevel = 16*1024*1024 ;
} /* endmethod */

SpoolSize::~SpoolSize ( void ) {
   delete [] SpoolPath ;
} /* endmethod */

int SpoolSize::Measure ( HPS hPS, RECTL &Rectangle ) {
   char Text [100] ;
   int Result = MeasureValue ( Text, QueryCurrentLabel(), ShowK, ShowM ) ;
   WinQueryWindowRect ( HWND_DESKTOP, &Rectangle ) ;
   WinDrawText ( hPS, strlen(Text), PSZ(Text), &Rectangle, 0, 0, DT_LEFT | DT_BOTTOM | DT_QUERYEXTENT ) ;
   return ( Result ) ;
}

ULONG SpoolSize::NewValue ( void ) {

   ULONG PathSize ;
   DosQuerySysInfo ( QSV_MAX_PATH_LENGTH, QSV_MAX_PATH_LENGTH, &PathSize, sizeof(PathSize) ) ;

   PBYTE Path = PBYTE ( malloc ( size_t(PathSize) ) ) ;
   if ( Path == NULL ) 
      return ( 0 ) ;

   PFILEFINDBUF3 Found = PFILEFINDBUF3 ( malloc ( size_t( PathSize + sizeof(FILEFINDBUF3) ) ) ) ;
   if ( Found == NULL ) {
      free ( Path ) ;
      return ( 0 ) ;
   } /* endif */

   strcpy ( (PCHAR)Path, (PCHAR)SpoolPath ) ;
   strcat ( (PCHAR)Path, "\\*.*" ) ;

   HDIR hDir = (HDIR) HDIR_CREATE ;
   ULONG Count = 1 ;
   ULONG TotalSize = 0 ;

   if ( !DosFindFirst2 ( Path, &hDir,
      FILE_NORMAL | FILE_READONLY | FILE_DIRECTORY | FILE_ARCHIVED,
      Found, PathSize+sizeof(FILEFINDBUF3), &Count, FIL_STANDARD ) ) {

      do {

         if ( !strcmp ( (PCHAR)Found->achName, "." ) OR !strcmp ( (PCHAR)Found->achName, ".." ) )
            continue ;

         if ( Found->attrFile & FILE_DIRECTORY ) {
 
            HDIR hDir2 = (HDIR) HDIR_CREATE ;

            strcpy ( (PCHAR)Path, (PCHAR)SpoolPath ) ;
            strcat ( (PCHAR)Path, "\\" ) ;
            strcat ( (PCHAR)Path, (PCHAR)Found->achName ) ;
            strcat ( (PCHAR)Path, "\\*.*" ) ;

            ULONG Count2 = 1 ;
            if ( !DosFindFirst2 ( Path, &hDir2,
               FILE_NORMAL | FILE_READONLY | FILE_ARCHIVED,
               Found, PathSize+sizeof(FILEFINDBUF3), &Count2, FIL_STANDARD ) ) {
               do {
                  TotalSize += Found->cbFile ;
               } while ( !DosFindNext ( hDir2, Found, PathSize+sizeof(FILEFINDBUF3), &Count2 ) ) ;
               DosFindClose ( hDir2 ) ;
            } /* endif */
         } else {
            TotalSize += Found->cbFile ;
         } /* endif */
      } while ( !DosFindNext ( hDir, Found, PathSize+sizeof(FILEFINDBUF3), &Count ) ) ;

      DosFindClose ( hDir ) ;

   } /* enddo */

   free ( Path ) ;
   free ( Found ) ;

   return ( TotalSize ) ;
}

void SpoolSize::FormatText ( char *Label, char *Text, ULONG Value ) {
   strcpy ( Label, CurrentLabel ) ;
   FormatValue ( Value, Text, CountryInfo, ShowK, ShowTrueK, ShowM ) ;
   DdeUpdate ( Text ) ;
}

void SpoolSize::Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) {
   ULONG Size = NewValue ( ) ;
   if ( Mandatory || ( Size != Value ) ) {
      char Label[100], Text [100] ;
      FormatText ( Label, Text, Size ) ;
      Paint ( hPS, Rectangle, Label, Text, Size ) ;
   } /* endif */
}


CpuLoad::CpuLoad ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
   Dde_Server *pDdeServer, char *Topic, ULONG maxcount, PULONG idlecount, ResourceString *error )
   : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ), 
   MaxCount(maxcount), IdleCount(idlecount), Error(error) { 
   LevelSense = +1 ;
   MinLevel = 0 ;
   MaxLevel = 100 ;
   DefaultLevels [0] = WarningLevel = 75 ;
   DefaultLevels [1] = ErrorLevel = 90 ;
} /* endmethod */

int CpuLoad::Measure ( HPS hPS, RECTL &Rectangle ) {
   int Result ;
   char Text [100] ;
   Result = sprintf ( Text, "%s 100%", QueryCurrentLabel() ) ;
   WinQueryWindowRect ( HWND_DESKTOP, &Rectangle ) ;
   WinDrawText ( hPS, strlen(Text), PSZ(Text), &Rectangle, 0, 0, DT_LEFT | DT_BOTTOM | DT_QUERYEXTENT ) ;
   return ( Result ) ;
}

ULONG CpuLoad::NewValue ( void ) {
   if ( *IdleCount == 0xFFFFFFFF ) 
      return ( 0xFFFFFFFF ) ;
   MaxCount = (ULONG) max ( MaxCount, *IdleCount ) ;
   ULONG Load = ( ( MaxCount - *IdleCount ) * 100 ) / MaxCount ;
   return ( Load ) ;
}

void CpuLoad::FormatText ( char *Label, char *Text, ULONG Value ) {
   if ( Value == 0xFFFFFFFF ) {
      strcpy ( Text, PCHAR(*Error) ) ;
   } else {
      strcpy ( Label, CurrentLabel ) ;
      sprintf ( Text, "%lu%%", Value ) ;
   } /* endif */
   DdeUpdate ( Text ) ;
}

void CpuLoad::Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) {
   ULONG Load = NewValue ( ) ;
   if ( Mandatory || ( Load != Value ) ) {
      char Label[100], Text [100] ;
      FormatText ( Label, Text, Load ) ;
      Paint ( hPS, Rectangle, Label, Text, Load ) ;
   } /* endif */
}


TaskCount::TaskCount ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
   Dde_Server *pDdeServer, char *Topic, HAB anchor )
   : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ), 
   Anchor(anchor) { 
   LevelSense = +1 ;
   MinLevel = 0 ;
   MaxLevel = 256 ;     
   DefaultLevels [0] = WarningLevel = 30 ;  
   DefaultLevels [1] = ErrorLevel = 40 ;    
} /* endmethod */

int TaskCount::Measure ( HPS hPS, RECTL &Rectangle ) {
   int Result ;
   char Text [100] ;
   Result = sprintf ( Text, "%s 999 ", QueryCurrentLabel() ) ;
   WinQueryWindowRect ( HWND_DESKTOP, &Rectangle ) ;
   WinDrawText ( hPS, strlen(Text), PSZ(Text), &Rectangle, 0, 0, DT_LEFT | DT_BOTTOM | DT_QUERYEXTENT ) ;
   return ( Result ) ;
}

ULONG TaskCount::NewValue ( void ) {
   return ( WinQuerySwitchList ( Anchor, PSWBLOCK(NULL), 0 ) ) ;
}

void TaskCount::FormatText ( char *Label, char *Text, ULONG Value ) {
   strcpy ( Label, CurrentLabel ) ;
   sprintf ( Text, "%lu ", Value ) ;
   DdeUpdate ( Text ) ;
}

void TaskCount::Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) {
   ULONG Count = NewValue ( ) ;
   if ( Mandatory || ( Count != Value ) ) {
      char Label[100], Text [100] ;
      FormatText ( Label, Text, Count ) ;
      Paint ( hPS, Rectangle, Label, Text, Count ) ;
   } /* endif */
}


DriveFree::DriveFree ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
   Dde_Server *pDdeServer, char *Topic, COUNTRYINFO &countryinfo, 
   USHORT sk, USHORT stk, USHORT sm, USHORT drivenumber, ResourceString *driveerror, BOOL showfilesystemname, 
   PSZ filesystem, BOOL showdisklabel, PSZ disklabel ) 
   : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ), 
   ShowK(sk), ShowTrueK(stk), ShowM(sm), DriveError(driveerror), DriveNumber(drivenumber),
   ShowFileSystemName(showfilesystemname), ShowDiskLabel(showdisklabel), 
   Error(FALSE) {
   CountryInfo = countryinfo ;
   strcpy ( PCHAR(FileSystem), PCHAR(filesystem) ) ;
   strcpy ( PCHAR(DiskLabel), PCHAR(disklabel) ) ;
   LevelSense = -1 ;
   MinLevel = 0 ;
   DosError ( FERR_DISABLEHARDERR ) ;
   FSALLOCATE Allocation ;
   DosQueryFSInfo ( DriveNumber, FSIL_ALLOC, (PBYTE)&Allocation, sizeof(Allocation) ) ;
   DosError ( FERR_ENABLEHARDERR ) ;
   MaxLevel = Allocation.cUnit*Allocation.cSectorUnit*Allocation.cbSector ;
   DefaultLevels [0] = WarningLevel = MaxLevel / 5 ;
   DefaultLevels [1] = ErrorLevel = MaxLevel / 10 ;
} /* endmethod */

int DriveFree::Measure ( HPS hPS, RECTL &Rectangle ) {
   char Label[100] ;
   if ( ShowFileSystemName && *FileSystem ) {
      if ( ShowDiskLabel && *DiskLabel ) {
         sprintf ( Label, "%s (%s,%s)", QueryCurrentLabel(), DiskLabel, FileSystem ) ;
      } else {
         sprintf ( Label, "%s (%s)", QueryCurrentLabel(), FileSystem ) ;
      } /* endif */
   } else {
      if ( ShowDiskLabel && *DiskLabel ) {
         sprintf ( Label, "%s (%s)", QueryCurrentLabel(), DiskLabel ) ;
      } else {
         sprintf ( Label, "%s", QueryCurrentLabel() ) ;
      } /* endif */
   } /* endif */
   char Text[100] ;
   int Result = MeasureValue ( Text, Label, ShowK, ShowM ) ;
   WinQueryWindowRect ( HWND_DESKTOP, &Rectangle ) ;
   WinDrawText ( hPS, strlen(Text), PSZ(Text), &Rectangle, 0, 0, DT_LEFT | DT_BOTTOM | DT_QUERYEXTENT ) ;
   return ( Result ) ;
}

ULONG DriveFree::NewValue ( void ) {

   if ( Error ) 
      return ( 0 ) ;

   DosError ( FERR_DISABLEHARDERR ) ;

   FSALLOCATE Allocation ;
   APIRET Status = DosQueryFSInfo ( DriveNumber, FSIL_ALLOC, (PBYTE)&Allocation, sizeof(Allocation) ) ;

   DosError ( FERR_ENABLEHARDERR ) ;

   if ( Status ) {
      Error = TRUE ;
      return ( 0 ) ;
   } /* endif */

   return ( Allocation.cUnitAvail*Allocation.cSectorUnit*Allocation.cbSector ) ;
}

void DriveFree::FormatText ( char *Label, char *Text, ULONG Value ) {

   if ( ShowFileSystemName && *FileSystem ) {
      if ( ShowDiskLabel && *DiskLabel ) {
         sprintf ( Label, "%s (%s,%s)", QueryCurrentLabel(), DiskLabel, FileSystem ) ;
      } else {
         sprintf ( Label, "%s (%s)", QueryCurrentLabel(), FileSystem ) ;
      } /* endif */
   } else {
      if ( ShowDiskLabel && *DiskLabel ) {
         sprintf ( Label, "%s (%s)", QueryCurrentLabel(), DiskLabel ) ;
      } else {
         sprintf ( Label, "%s", QueryCurrentLabel() ) ;
      } /* endif */
   } /* endif */

   if ( Error )
      strcpy ( Text, PCHAR(*DriveError) ) ;
   else
      FormatValue ( Value, Text, CountryInfo, ShowK, ShowTrueK, ShowM ) ;

   DdeUpdate ( Text ) ;
}

void DriveFree::Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) {
   ULONG Size = NewValue ( ) ;
   if ( Mandatory || ( Size != Value ) ) {
      char Label[100], Text [100] ;
      FormatText ( Label, Text, Size ) ;
      Paint ( hPS, Rectangle, Label, Text, Size ) ;
   } /* endif */
}


TotalFree::TotalFree ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
   Dde_Server *pDdeServer, char *Topic, COUNTRYINFO &countryinfo, 
   USHORT sk, USHORT stk, USHORT sm, ULONG drives )
   : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ), 
   ShowK(sk), ShowTrueK(stk), ShowM(sm), Drives(drives) {
   CountryInfo = countryinfo ;
   LevelSense = -1 ;
   MinLevel = 0 ;
   MaxLevel = 64*1024*1024 ;    
   DefaultLevels [0] = WarningLevel = 8*1024*1024 ;
   DefaultLevels [1] = ErrorLevel = 4*1024*1024 ;
} /* endmethod */

int TotalFree::Measure ( HPS hPS, RECTL &Rectangle ) {
   char Text [100] ;
   int Result = MeasureValue ( Text, QueryCurrentLabel(), ShowK, ShowM ) ;
   WinQueryWindowRect ( HWND_DESKTOP, &Rectangle ) ;
   WinDrawText ( hPS, strlen(Text), PSZ(Text), &Rectangle, 0, 0, DT_LEFT | DT_BOTTOM | DT_QUERYEXTENT ) ;
   return ( Result ) ;
}

ULONG TotalFree::NewValue ( void ) {

   ULONG Free = 0 ;
   ULONG Mask = Drives >> 2 ;

   for ( int Drive=3; Drive<=26; Drive++ ) {
      if ( Mask & 1 ) {

         DosError ( FERR_DISABLEHARDERR ) ;

         FSALLOCATE Allocation ;
         APIRET Status = DosQueryFSInfo ( Drive, FSIL_ALLOC, (PBYTE)&Allocation, sizeof(Allocation) ) ;

         DosError ( FERR_ENABLEHARDERR ) ;

         if ( Status )
            Drives &= ~ ( 1 << (Drive-1) ) ;
         else
            Free += Allocation.cUnitAvail*Allocation.cSectorUnit*(Allocation.cbSector/256) ;
      } /* endif */
      Mask >>= 1 ;

   } /* endfor */

   return ( Free ) ;
}

void TotalFree::FormatText ( char *Label, char *Text, ULONG Value ) {
   strcpy ( Label, CurrentLabel ) ;
   FormatValue ( Value, Text, CountryInfo, ShowK, ShowTrueK, ShowM, 256 ) ;
   DdeUpdate ( Text ) ;
}

void TotalFree::Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) {
   ULONG Size = NewValue ( ) ;
   if ( Mandatory || ( Size != Value ) ) {
      char Label[100], Text [100] ;
      FormatText ( Label, Text, Size ) ;
      Paint ( hPS, Rectangle, Label, Text, Size, 256 ) ;
   } /* endif */
}


SwapSlack::SwapSlack ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
   Dde_Server *pDdeServer, char *Topic, COUNTRYINFO &countryinfo, 
   USHORT sk, USHORT stk, USHORT sm, VirtualFree *pvf, SwapFree *psf, MemoryFree *pmf ) 
   : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ), 
   ShowK(sk), ShowTrueK(stk), ShowM(sm), pVirtualFree(pvf), pSwapFree(psf), pMemFree(pmf) {
   CountryInfo = countryinfo ;
} /* endmethod */

int SwapSlack::Measure ( HPS hPS, RECTL &Rectangle ) {
   char Text [100] ;
   int Result = MeasureValue ( Text, QueryCurrentLabel(), ShowK, ShowM ) ;
   WinQueryWindowRect ( HWND_DESKTOP, &Rectangle ) ;
   WinDrawText ( hPS, strlen(Text), PSZ(Text), &Rectangle, 0, 0, DT_LEFT | DT_BOTTOM | DT_QUERYEXTENT ) ;
   return ( Result ) ;
}

ULONG SwapSlack::NewValue ( void ) {
   LONG Slack = pVirtualFree->QueryFlag() ? pVirtualFree->QueryValue() : pVirtualFree->NewValue() ;
   Slack -= pSwapFree->QueryFlag() ? pSwapFree->QueryValue() : pSwapFree->NewValue() ;
   Slack -= pMemFree->QueryFlag() ? pMemFree->QueryValue() : pMemFree->NewValue() ;
   return ( ULONG ( max ( 0, Slack ) ) ) ;
}

void SwapSlack::FormatText ( char *Label, char *Text, ULONG Value ) {
   strcpy ( Label, CurrentLabel ) ;
   FormatValue ( Value, Text, CountryInfo, ShowK, ShowTrueK, ShowM ) ;
   DdeUpdate ( Text ) ;
}

void SwapSlack::Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) {
   ULONG Size = NewValue ( ) ;
   if ( Mandatory || ( Size != Value ) ) {
      char Label[100], Text[100] ;
      FormatText ( Label, Text, Size ) ;
      Paint ( hPS, Rectangle, Label, Text, Size ) ;
   } /* endif */
} /* endmethod */


ProcessCount::ProcessCount ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel,
   Dde_Server *pDdeServer, char *Topic )
   : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ) { 
   DosAllocMem ( (PPVOID)&DQPS_Buffer, 0x10000, PAG_READ | PAG_WRITE | PAG_COMMIT | OBJ_TILE ) ;
   LevelSense = +1 ;
   MinLevel = 0 ;
   MaxLevel = 256 ;    
   DefaultLevels [0] = WarningLevel = 30 ; 
   DefaultLevels [1] = ErrorLevel = 40 ;   
} /* endmethod */

ProcessCount::~ProcessCount ( ) { 
   DosFreeMem ( DQPS_Buffer ) ; 
} /* endmethod */

int ProcessCount::Measure ( HPS hPS, RECTL &Rectangle ) {
   int Result ;
   char Text [100] ;
   Result = sprintf ( Text, "%s 123 ", QueryCurrentLabel() ) ;
   WinQueryWindowRect ( HWND_DESKTOP, &Rectangle ) ;
   WinDrawText ( hPS, strlen(Text), PSZ(Text), &Rectangle, 0, 0, DT_LEFT | DT_BOTTOM | DT_QUERYEXTENT ) ;
   return ( Result ) ;
} /* endmethod */

ULONG ProcessCount::NewValue ( void ) {

   if ( DosQProcStatus ( DQPS_Buffer, 0xFFFF ) ) 
      return ( 0 ) ;

   qsPrec_s *pProcRecord = ((qsPtrRec_s*)DQPS_Buffer)->pProcRec ;
   int ProcessCount = 0 ;
   while ( pProcRecord->RecType == 1 ) {
      int Size = sizeof(qsPrec_s) ;
      Size += pProcRecord->cTCB * sizeof(qsTrec_s) ;
      Size += pProcRecord->c16Sem * sizeof(short) ;
      Size += pProcRecord->cLib * sizeof(short) ;
      Size += pProcRecord->cShrMem * sizeof(short) ;
      pProcRecord = (qsPrec_s*) ( (char*)pProcRecord + Size ) ;
      ProcessCount ++ ;
   } /* endwhile */
   return ( ProcessCount ) ;
} /* endmethod */

void ProcessCount::FormatText ( char *Label, char *Text, ULONG Value ) {
   strcpy ( Label, CurrentLabel ) ;
   sprintf ( Text, "%lu ", Value ) ;
   DdeUpdate ( Text ) ;
} /* endmethod */

void ProcessCount::Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) {
   ULONG Count = NewValue ( ) ;
   if ( Mandatory || ( Count != Value ) ) {
      char Label[100], Text[100] ;
      FormatText ( Label, Text, Count ) ;
      Paint ( hPS, Rectangle, Label, Text, Count ) ;
   } /* endif */
} /* endmethod */


ThreadCount::ThreadCount ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel,
   Dde_Server *pDdeServer, char *Topic )
   : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ) { 
   DosAllocMem ( (PPVOID)&DQPS_Buffer, 0x10000, PAG_READ | PAG_WRITE | PAG_COMMIT | OBJ_TILE ) ;
   LevelSense = +1 ;
   MinLevel = 0 ;
   PSZ Threads = ScanSystemConfig ( 0, "THREADS" ) ;
   if ( Threads == 0 ) 
      Threads = (PSZ) "64" ;
   int MaxThreads = atoi ( Threads ) ;
   MaxLevel = MaxThreads ;
   DefaultLevels [0] = WarningLevel = ( MaxThreads * 3 ) / 4 ; 
   DefaultLevels [1] = ErrorLevel = ( MaxThreads * 9 ) / 10 ;   
} /* endmethod */

ThreadCount::~ThreadCount ( ) { 
   DosFreeMem ( DQPS_Buffer ) ; 
} /* endmethod */

int ThreadCount::Measure ( HPS hPS, RECTL &Rectangle ) {
   int Result ;
   char Text [100] ;
   Result = sprintf ( Text, "%s 123 ", QueryCurrentLabel() ) ;
   WinQueryWindowRect ( HWND_DESKTOP, &Rectangle ) ;
   WinDrawText ( hPS, strlen(Text), PSZ(Text), &Rectangle, 0, 0, DT_LEFT | DT_BOTTOM | DT_QUERYEXTENT ) ;
   return ( Result ) ;
} /* endmethod */

ULONG ThreadCount::NewValue ( void ) {

   if ( DosQProcStatus ( DQPS_Buffer, 0xFFFF ) )
      return ( 0 ) ;

   return ( ((qsPtrRec_s*)DQPS_Buffer)->pGlobalRec->cThrds ) ;
} /* endmethod */

void ThreadCount::FormatText ( char *Label, char *Text, ULONG Value ) {
   strcpy ( Label, CurrentLabel ) ;
   sprintf ( Text, "%lu ", Value ) ;
   DdeUpdate ( Text ) ;
} /* endmethod */

void ThreadCount::Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) {
   ULONG Count = NewValue ( ) ;
   if ( Mandatory || ( Count != Value ) ) {
      char Label[100], Text[100] ;
      FormatText ( Label, Text, Count ) ;
      Paint ( hPS, Rectangle, Label, Text, Count ) ;
   } /* endif */
} /* endmethod */


Battery::Battery ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel,
   Dde_Server *pDdeServer, char *Topic, ResourceString *error, ResourceString *charging, ResourceString *ac )
   : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ), 
   Handle(-1), Error(error), Charging(charging), AC(ac) { 
   LevelSense = -1 ;
   MinLevel = 0 ;
   MaxLevel = 100 ;
   DefaultLevels [0] = WarningLevel = 25 ; 
   DefaultLevels [1] = ErrorLevel = 10 ;   
   ULONG Action = 0;
   APIRET Status = DosOpen ( (PSZ) "APM$", &Handle, &Action, 0, FILE_NORMAL,
         FILE_OPEN, OPEN_ACCESS_READWRITE | OPEN_SHARE_DENYREADWRITE | OPEN_FLAGS_FAIL_ON_ERROR, 0 ) ;
   if ( Status ) 
      Handle = 0xFFFFFFFF ;
} /* endmethod */

Battery::~Battery ( ) {
   if ( Handle != 0xFFFFFFFF ) 
      DosClose ( Handle ) ;
} /* endmethod */

int Battery::Measure ( HPS hPS, RECTL &Rectangle ) {
   int Result ;
   char Text [100] ;
   Result = sprintf ( Text, "%s %s ", QueryCurrentLabel(), PCHAR(*Charging) ) ;
   WinQueryWindowRect ( HWND_DESKTOP, &Rectangle ) ;
   WinDrawText ( hPS, strlen(Text), PSZ(Text), &Rectangle, 0, 0, DT_LEFT | DT_BOTTOM | DT_QUERYEXTENT ) ;
   return ( Result ) ;
} /* endmethod */

ULONG Battery::NewValue ( void ) {

   if ( Handle != 0xFFFFFFFF ) 
      return ( -1 ) ;

   struct {
      USHORT ParmLength ;
      USHORT PowerFlags ;
      UCHAR  ACStatus ;
      UCHAR  BatteryStatus ;
      UCHAR  BatteryLife ;
   } Parm ;
   Parm.ParmLength = sizeof(Parm) ;
   ULONG ParmLen = sizeof(Parm) ;
   USHORT Data ;
   ULONG DataLen = sizeof(Data) ;
   APIRET Status = DosDevIOCtl ( Handle, IOCTL_POWER, POWER_GETPOWERSTATUS, &Parm, sizeof(Parm), &ParmLen, &Data, sizeof(Data), &DataLen ) ;
   if ( Status ) {
//    Log ( "Unable to call APM$ for power status.  Status %i.", Status ) ;
      DosClose ( Handle ) ;
      Handle = 0xFFFFFFFF ;
      return ( -1 ) ;
   } else {
      if ( Parm.PowerFlags & 1 ) {                       // If APM enabled,
         if ( Parm.BatteryStatus == 3 ) {                //   If battery charging,
            if ( Parm.BatteryLife < 100 )                //     If not fully charged,
               return ( 101 ) ;                          //       Report CHARGING.
            else                                         //     Else
               return ( 100 ) ;                          //       Report 100%.
         } else if ( Parm.ACStatus == 1 ) {              //   Else if A/C,
            return ( 201 ) ;                             //     Report A/C.
         } else if ( Parm.BatteryLife <= 100 ) {         //   Else if not fully charged,
            return ( Parm.BatteryLife ) ;                //     Report percentage full.
         } else {                                        //   Else,
            return ( -1 ) ;                              //     Report error.
         } /* endif */                                   //
      } else {                                           // Else,
         return ( -1 ) ;                                 //   Report error.
      } /* endif */
   } /* endif */
} /* endmethod */

void Battery::FormatText ( char *Label, char *Text, ULONG Value ) {
   strcpy ( Label, CurrentLabel ) ;
   if ( (long)Value < 0 ) {
      strcpy ( Text, PCHAR(*Error) ) ;
      strcat ( Text, " " ) ;
   } else if ( Value > 200 ) {
      strcpy ( Text, PCHAR(*AC) ) ;
      strcat ( Text, " " ) ;
   } else if ( Value > 100 ) {
      strcpy ( Text, PCHAR(*Charging) ) ;
      strcat ( Text, " " ) ;
   } else { 
      sprintf ( Text, "%lu%%", Value ) ;
   } /* endif */
   DdeUpdate ( Text ) ;
} /* endmethod */

void Battery::Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) {
   ULONG Count = NewValue ( ) ;
   if ( Mandatory || ( Count != Value ) ) {
      char Label[100], Text[100] ;
      FormatText ( Label, Text, Count ) ;
      Paint ( hPS, Rectangle, Label, Text, Count ) ;
   } /* endif */
} /* endmethod */

int Battery::Discharging ( ) {

   if ( Handle != 0xFFFFFFFF ) 
      return ( FALSE ) ;

   struct {
      USHORT ParmLength ;
      USHORT PowerFlags ;
      UCHAR  ACStatus ;
      UCHAR  BatteryStatus ;
      UCHAR  BatteryLife ;
   } Parm ;
   Parm.ParmLength = sizeof(Parm) ;
   ULONG ParmLen = sizeof(Parm) ;
   USHORT Data ;
   ULONG DataLen = sizeof(Data) ;
   APIRET Status = DosDevIOCtl ( Handle, IOCTL_POWER, POWER_GETPOWERSTATUS, &Parm, sizeof(Parm), &ParmLen, &Data, sizeof(Data), &DataLen ) ;
   if ( Status ) {
//    Log ( "Unable to call APM$ for power status.  Status %i.", Status ) ;
      DosClose ( Handle ) ;
      Handle = 0xFFFFFFFF ;
   } else {
      if ( Parm.PowerFlags & 1 ) {
         if ( Parm.BatteryStatus == 3 ) {
            ;
         } else if ( Parm.ACStatus == 1 ) {     
            ;
         } else if ( Parm.BatteryLife <= 100 ) {
            return ( TRUE ) ;
         } /* endif */                          
      } /* endif */
   } /* endif */

   return ( FALSE ) ;
}

