/******************************************************************** ITEMS.H
 *                                                                          *
 *                     Display Item Class Definition                        *
 *                                                                          *
 ****************************************************************************/

#ifndef ITEMS_H
#define ITEMS_H

#ifndef OS2_INCLUDED
   #define INCL_BASE
   #define INCL_PM
   #include <os2.h>
#endif

#include <time.h>

#include "Dde.h"
#include "ReString.h"

extern VOID Log ( char *Message, ... ) ;

class Item {

   private:
      USHORT Id ;                  // Item ID.
      BOOL   Flag ;                // Flag: Show this item at this time?
      char   Name [80] ;           // Text for items profile name.
      char   DefaultLabel [80] ;   // Text to display on left part of line (default).
      Dde_Server *pDdeServer ;     // -> DDE Server object
      char Topic [80] ;            // DDE Topic name
      Dde_Item *pDdeItem ;         // -> DDE Item object

   protected:
      char   CurrentLabel [80] ;   // Text to display on left part of line.
      ULONG  Value ;               // Current value.

   public:
      Item ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, Dde_Server *ddeserver, char *topic ) :
         Id(id), Flag(TRUE), Value(0), pDdeServer(ddeserver) {
         strcpy ( Name, pName ) ;
         strcpy ( CurrentLabel, pCurrentLabel ) ;
         strcpy ( DefaultLabel, pDefaultLabel ) ;
         strcpy ( Topic, topic ) ;
         pDdeServer->AddItem ( Topic, DefaultLabel, DDEFMT_TEXT, "", 1 ) ;
         pDdeItem = pDdeServer->FindItem ( Topic, DefaultLabel ) ;
      }

      ~Item ( ) {
         pDdeServer->RemoveItem ( Topic, DefaultLabel ) ;
      }

      USHORT QueryId           ( void ) { return ( Id   ) ; }
      BOOL   QueryFlag         ( void ) { return ( Flag ) ; }
      PCHAR  QueryName         ( void ) { return ( Name ) ; }
      PCHAR  QueryCurrentLabel ( void ) { return ( CurrentLabel ) ; }
      PCHAR  QueryDefaultLabel ( void ) { return ( DefaultLabel ) ; }
      ULONG  QueryValue        ( void ) { return ( Value ) ; }

      void SetLabel ( char *label ) { strcpy ( CurrentLabel, label ) ; }

      void SetFlag   ( void ) { Flag = TRUE ; }
      void ResetFlag ( void ) { Flag = FALSE ; }

      void DdeUpdate ( char *Text ) ;

      void Paint ( HPS hPS, RECTL &Rectangle, 
         COLOR TextColor, COLOR BackColor,
         char *Label, char *Text, ULONG NewValue ) ;

      // The following functions must be provided by all subclasses.

      virtual int Measure ( HPS hPS, RECTL &Rectangle ) = 0 ;

      virtual ULONG NewValue ( void ) = 0 ;

      virtual void FormatText ( char *Label, char *Text, ULONG Value ) = 0 ;

      virtual void FormatLine ( char *Buffer, int MaxWidth ) ;

      virtual void Repaint ( HPS hPS, RECTL &Rectangle,
         COLOR TextColor, COLOR BackColor, BOOL Mandatory ) = 0 ;
} ;

class Clock : public Item {

   private:
      COUNTRYINFO CountryInfo ;
      char szAm [3] ;
      char szPm [3] ;
      ResourceString *DaysOfWeek ;
      BOOL ShowSeconds ;

   public:
      Clock ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
         Dde_Server *pDdeServer, char *Topic, COUNTRYINFO &countryinfo,
         char *szam, char *szpm, ResourceString *daysofweek, BOOL showseconds )

         : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ),
         DaysOfWeek(daysofweek), ShowSeconds(showseconds) {

         CountryInfo = countryinfo ;
         strcpy ( szAm, szam ) ;
         strcpy ( szPm, szpm ) ;
      }

      void SetShowSeconds ( BOOL showseconds ) { ShowSeconds = showseconds ; }
 
      int Measure ( HPS hPS, RECTL &Rectangle ) ;

      ULONG NewValue ( void ) ;

      void FormatText ( char *Label, char *Text, ULONG Value ) ;

      void FormatLine ( char *Buffer, int MaxWidth ) ;

      void Repaint ( HPS hPS, RECTL &Rectangle,
         COLOR TextColor, COLOR BackColor, BOOL Mandatory ) ;
} ;

class ElapsedTime : public Item {

   private:
      COUNTRYINFO CountryInfo ;
      ResourceString *Day ;
      ResourceString *Days ;
      BOOL ShowSeconds ;

   public:
      ElapsedTime ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
         Dde_Server *pDdeServer, char *Topic, COUNTRYINFO &countryinfo,
         ResourceString *day, ResourceString *days, BOOL showseconds )

         : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ),
         Day(day), Days(days), ShowSeconds(showseconds) {

         CountryInfo = countryinfo ;
      }

      void SetShowSeconds ( BOOL showseconds ) { ShowSeconds = showseconds ; }

      int Measure ( HPS hPS, RECTL &Rectangle ) ;

      ULONG NewValue ( void ) ;

      void FormatText ( char *Label, char *Text, ULONG Value ) ;

      void Repaint ( HPS hPS, RECTL &Rectangle,
         COLOR TextColor, COLOR BackColor, BOOL Mandatory ) ;
} ;

class SwapFree : public Item {

   private:
      COUNTRYINFO CountryInfo ;
      USHORT ShowK ;
      PSZ SwapPath ;
      ULONG MinFree ;

   public:
      SwapFree ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
         Dde_Server *pDdeServer, char *Topic, COUNTRYINFO &countryinfo,
         USHORT sk, PSZ swappath, ULONG minfree )

         : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ),
         ShowK(sk), MinFree(minfree) {

         CountryInfo = countryinfo ;
         SwapPath = new BYTE [ strlen(PCHAR(swappath)) + 1 ] ;
         strcpy ( PCHAR(SwapPath), PCHAR(swappath) ) ;
      }

      ~SwapFree ( void ) {
         delete [] SwapPath ;
      }

      void SetShowK ( USHORT flag ) { ShowK = flag ; }

      int Measure ( HPS hPS, RECTL &Rectangle ) ;

      ULONG NewValue ( void ) ;

      void FormatText ( char *Label, char *Text, ULONG Value ) ;

      void Repaint ( HPS hPS, RECTL &Rectangle,
         COLOR TextColor, COLOR BackColor, BOOL Mandatory ) ;
} ;

class MemoryFree : public Item {

   private:
      COUNTRYINFO CountryInfo ;
      USHORT ShowK ;
      PULONG pMemoryFree ;
      time_t *pMemoryTime ;
      ResourceString *DriveError ;

   public:
      MemoryFree ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
         Dde_Server *pDdeServer, char *Topic, COUNTRYINFO &countryinfo,
         USHORT sk, PULONG pmemoryfree, time_t *pmemorytime, ResourceString *driveerror )

         : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ), 
         ShowK(sk), pMemoryFree(pmemoryfree), pMemoryTime(pmemorytime), DriveError(driveerror) {

         CountryInfo = countryinfo ;
      }

      void SetShowK ( USHORT flag ) { ShowK = flag ; }

      int Measure ( HPS hPS, RECTL &Rectangle ) ;

      ULONG NewValue ( void ) ;

      void FormatText ( char *Label, char *Text, ULONG Value ) ;

      void Repaint ( HPS hPS, RECTL &Rectangle,
         COLOR TextColor, COLOR BackColor, BOOL Mandatory ) ;
} ;

class VirtualFree : public Item {

   private:
      COUNTRYINFO CountryInfo ;
      USHORT ShowK ;

   public:
      VirtualFree ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
         Dde_Server *pDdeServer, char *Topic, COUNTRYINFO &countryinfo, USHORT sk )

         : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ), 
         ShowK(sk) {

         CountryInfo = countryinfo ;
      }

      void SetShowK ( USHORT flag ) { ShowK = flag ; }

      int Measure ( HPS hPS, RECTL &Rectangle ) ;

      ULONG NewValue ( void ) ;

      void FormatText ( char *Label, char *Text, ULONG Value ) ;

      void Repaint ( HPS hPS, RECTL &Rectangle,
         COLOR TextColor, COLOR BackColor, BOOL Mandatory ) ;
} ;

class SwapSize : public Item {

   private:
      COUNTRYINFO CountryInfo ;
      USHORT ShowK ;
      PSZ SwapPath ;

   public:
      SwapSize ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
         Dde_Server *pDdeServer, char *Topic, COUNTRYINFO countryinfo, 
         USHORT sk, PSZ swappath )

         : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ), 
         ShowK(sk) {

         CountryInfo = countryinfo ;
         SwapPath = new BYTE [ strlen(PCHAR(swappath)) + 1 ] ;
         strcpy ( PCHAR(SwapPath), PCHAR(swappath) ) ;
      }

      ~SwapSize ( void ) {
         delete [] SwapPath ;
      }

      void SetShowK ( USHORT flag ) { ShowK = flag ; }

      int Measure ( HPS hPS, RECTL &Rectangle ) ;

      ULONG NewValue ( void ) ;

      void FormatText ( char *Label, char *Text, ULONG Value ) ;

      void Repaint ( HPS hPS, RECTL &Rectangle,
         COLOR TextColor, COLOR BackColor, BOOL Mandatory ) ;
} ;

class SpoolSize : public Item {

   private:
      COUNTRYINFO CountryInfo ;
      USHORT ShowK ;
      PSZ SpoolPath ;

   public:
      SpoolSize ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
         Dde_Server *pDdeServer, char *Topic, COUNTRYINFO &countryinfo, 
         USHORT sk, PSZ spoolpath )

         : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ), 
         ShowK(sk) {

         CountryInfo = countryinfo ;
         SpoolPath = new BYTE [ strlen(PCHAR(spoolpath)) + 1 ] ;
         strcpy ( PCHAR(SpoolPath), PCHAR(spoolpath) ) ;
      }

      ~SpoolSize ( void ) {
         delete [] SpoolPath ;
      }

      void SetShowK ( USHORT flag ) { ShowK = flag ; }

      int Measure ( HPS hPS, RECTL &Rectangle ) ;

      ULONG NewValue ( void ) ;

      void FormatText ( char *Label, char *Text, ULONG Value ) ;

      void Repaint ( HPS hPS, RECTL &Rectangle,
         COLOR TextColor, COLOR BackColor, BOOL Mandatory ) ;
} ;

class CpuLoad : public Item {

   private:
      PULONG IdleCount ;
      ULONG MaxCount ;

   public:
      CpuLoad ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
         Dde_Server *pDdeServer, char *Topic, ULONG maxcount, PULONG idlecount )

         : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ), 
         MaxCount(maxcount), IdleCount(idlecount) { }

      int Measure ( HPS hPS, RECTL &Rectangle ) ;

      ULONG NewValue ( void ) ;

      void FormatText ( char *Label, char *Text, ULONG Value ) ;

      void Repaint ( HPS hPS, RECTL &Rectangle,
         COLOR TextColor, COLOR BackColor, BOOL Mandatory ) ;
} ;

class TaskCount : public Item {

   private:
      HAB Anchor ;

   public:
      TaskCount ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
         Dde_Server *pDdeServer, char *Topic, HAB anchor )

         : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ), 
         Anchor(anchor) { }

      int Measure ( HPS hPS, RECTL &Rectangle ) ;

      ULONG NewValue ( void ) ;

      void FormatText ( char *Label, char *Text, ULONG Value ) ;

      void Repaint ( HPS hPS, RECTL &Rectangle,
         COLOR TextColor, COLOR BackColor, BOOL Mandatory ) ;
} ;

class DriveFree : public Item {

   private:
      COUNTRYINFO CountryInfo ;
      USHORT ShowK ;
      ResourceString *DriveError ;
      USHORT DriveNumber ;
      BOOL ShowFileSystemName ;
      BYTE FileSystem [80] ;
      BOOL ShowDiskLabel ;
      BYTE DiskLabel [12] ;
      BOOL Error ;

   public:
      DriveFree ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
         Dde_Server *pDdeServer, char *Topic, COUNTRYINFO &countryinfo, 
         USHORT sk, USHORT drivenumber, ResourceString *driveerror, BOOL showfilesystemname, 
         PSZ filesystem, BOOL showdisklabel, PSZ disklabel )

         : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ), 
         ShowK(sk), DriveError(driveerror), DriveNumber(drivenumber),
         ShowFileSystemName(showfilesystemname), ShowDiskLabel(showdisklabel), 
         Error(FALSE) {

         CountryInfo = countryinfo ;
         strcpy ( PCHAR(FileSystem), PCHAR(filesystem) ) ;
         strcpy ( PCHAR(DiskLabel), PCHAR(disklabel) ) ;
      }

      void SetShowK ( USHORT flag ) { ShowK = flag ; }

      void SetShowFileSystemName ( BOOL showfilesystemname ) { ShowFileSystemName = showfilesystemname ; }

      void SetShowDiskLabel ( BOOL showdisklabel ) { ShowDiskLabel = showdisklabel ; }

      int Measure ( HPS hPS, RECTL &Rectangle ) ;
 
      ULONG NewValue ( void ) ;

      void FormatText ( char *Label, char *Text, ULONG Value ) ;

      void Repaint ( HPS hPS, RECTL &Rectangle,
         COLOR TextColor, COLOR BackColor, BOOL Mandatory ) ;
} ;

class TotalFree : public Item {

   private:
      COUNTRYINFO CountryInfo ;
      USHORT ShowK ;
      ULONG Drives ;

   public:
      TotalFree ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
         Dde_Server *pDdeServer, char *Topic, COUNTRYINFO &countryinfo, 
         USHORT sk, ULONG drives )

         : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ), 
         ShowK(sk), Drives(drives) {

         CountryInfo = countryinfo ;
      }

      void SetShowK ( USHORT flag ) { ShowK = flag ; }

      void ResetMask ( ULONG drives ) { Drives = drives ; }

      int Measure ( HPS hPS, RECTL &Rectangle ) ;

      ULONG NewValue ( void ) ;

      void FormatText ( char *Label, char *Text, ULONG Value ) ;

      void Repaint ( HPS hPS, RECTL &Rectangle,
         COLOR TextColor, COLOR BackColor, BOOL Mandatory ) ;
} ;

class SwapSlack : public Item {

   private:
      COUNTRYINFO  CountryInfo ;
      USHORT       ShowK ;
      VirtualFree *pVirtualFree ;
      SwapFree    *pSwapFree ;
      MemoryFree  *pMemFree ;

   public:
      SwapSlack ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
         Dde_Server *pDdeServer, char *Topic, COUNTRYINFO &countryinfo, 
         USHORT sk, VirtualFree *pvf, SwapFree *psf, MemoryFree *pmf )

         : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ), 
         ShowK(sk), pVirtualFree(pvf), pSwapFree(psf), pMemFree(pmf) {

         CountryInfo = countryinfo ;
      }

      void SetShowK ( USHORT flag ) { ShowK = flag ; }

      int Measure ( HPS hPS, RECTL &Rectangle ) ;

      ULONG NewValue ( void ) ;

      void FormatText ( char *Label, char *Text, ULONG Value ) ;

      void Repaint ( HPS hPS, RECTL &Rectangle,
         COLOR TextColor, COLOR BackColor, BOOL Mandatory ) ;
} ;

#endif
