/****************************************************************** MEMSIZE.C
 *									    *
 * System Resources Monitor						    *
 *									    *
 * Revision 1.60 (08 March 1993)					    *
 *									    *
 * (C) Copyright 1991-1993 by Richard W. Papo.				    *
 *									    *
 * This is 'FreeWare'.	As such, it may be copied and distributed	    *
 * freely.  If you want to use part of it in your own program, please	    *
 * give credit where credit is due.  If you want to change the		    *
 * program, please refer the change request to me or send me the	    *
 * modified source code.  I can be reached at CompuServe 72607,3111.	    *
 *									    *
 ****************************************************************************/

#define INCL_BASE
#define INCL_PM
#include <os2.h>

#include <direct.h>
#include <process.h>
#include <stdarg.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <sys\types.h>
#include <sys\stat.h>

#include "debug.h"
#include "support.h"
#include "about.h"
#include "settimer.h"

#include "memsize.h"


/****************************************************************************
 *									    *
 *			 Definitions & Declarations			    *
 *									    *
 ****************************************************************************/

  // Constants

#define PROGRAM_NAME	    "MEMSIZE"
#define CLASS_NAME	    "MEMSIZE"

enum
{
  ITEM_CLOCK,
  ITEM_MEMORYFREE,
  ITEM_SWAPFILESIZE,
  ITEM_SWAPDISKFREE,
  ITEM_SPOOLFILESIZE,
  ITEM_CPULOAD,
  ITEM_TASKCOUNT,
  ITEM_BASE_COUNT
} ;

#define WM_REFRESH	    WM_USER

#define MAX_DRIVES	  26
#define DRIVE_ERROR	  0xFFFFFFFFL


  // Data Types

typedef struct	      // Data structure for item to be monitored.
{
  CHAR	 Name [80] ;	       // Text for item's profile name.
  BOOL	 Flag ; 	       // Flag: Show this item at this time?
  CHAR	 Label [80] ;	       // Text to display on left part of line.
  ULONG  Value ;	       // Value to display on right part of line.
  CHAR	 MenuOption [80] ;     // Text to display in system menu.
  USHORT MenuId ;	       // ID for use in system menu.
  ULONG  (*NewValue)	       // Function to determine new value.
    (PVOID,USHORT) ;
  USHORT Parm ; 	       // Parameter to pass to NewValue function.
  USHORT Divisor ;	       // Amount to divide value by before display.
  CHAR	 Suffix ;	       // Character to place after value.
}
ITEM, *PITEM ;

typedef struct	      // Parameters saved to system.
{
  PITEM  Items ;		// Items to display.
  int	 ItemCount ;

  SWP	 Position ;		// Window size & location.
  BOOL	 fPosition ;

  BOOL	 HideControls ; 	// User options.
  BOOL	 fHideControls ;

  USHORT TimerInterval ;
  BOOL	 fTimerInterval ;

  CHAR	 FontNameSize [80] ;	// Presentation Parameters
  BOOL	 fFontNameSize ;

  COLOR  BackColor ;
  BOOL	 fBackColor ;

  COLOR  TextColor ;
  BOOL	 fTextColor ;
}
PROFILE, *PPROFILE ;

typedef struct        // Data structure for window.
{
  HAB		 Anchor ;
  HMODULE	 Library ;

  ULONG 	 MaxCount ;
  ULONG 	 IdleCounter ;
  ULONG 	 IdleCount ;
  TID		 IdleLoopTID ;
  TID		 MonitorLoopTID ;

  PROFILE	 Profile ;

  HWND           hwndTitleBar ;
  HWND           hwndSysMenu ;
  HWND		 hwndMinMax ;

  ULONG 	 Drives ;

  char           SwapPath [_MAX_PATH] ;
  int		 MinFree ;

  PCHAR 	 SpoolPath ;

  long		 Width ;
  long		 Height ;

  COUNTRYINFO	 CountryInfo ;
}
DATA, *PDATA ;

typedef struct
{
  HAB Anchor ;
  HMODULE Library ;
}
PARMS, *PPARMS ;

typedef struct
{
  volatile PULONG Counter ;
  PUSHORT Interval ;
  HWND Owner ;
}
MONITOR_PARMS, *PMONITOR_PARMS ;


  // Function Prototypes

extern void main ( int argc, PCHAR argv[] ) ;

static MRESULT EXPENTRY MessageProcessor
(
  HWND hwnd,
  USHORT msg,
  MPARAM mp1,
  MPARAM mp2
) ;

static METHODFUNCTION Create ;
static METHODFUNCTION Destroy ;
static METHODFUNCTION Size ;
static METHODFUNCTION SaveApplication ;
static METHODFUNCTION Paint ;
static METHODFUNCTION Command ;
static METHODFUNCTION ResetDefaults ;
static METHODFUNCTION HideControlsCmd ;
static METHODFUNCTION SetTimer ;
static METHODFUNCTION About ;
static METHODFUNCTION ButtonDown ;
static METHODFUNCTION ButtonDblClick ;
static METHODFUNCTION PresParamChanged ;
static METHODFUNCTION SysColorChange ;
static METHODFUNCTION QueryKeysHelp ;
static METHODFUNCTION HelpError ;
static METHODFUNCTION ExtHelpUndefined ;
static METHODFUNCTION HelpSubitemNotFound ;
static METHODFUNCTION Refresh ;

static void GetProfile ( HAB Anchor, HMODULE Library, PPROFILE Profile ) ;
static void PutProfile ( PPROFILE Profile ) ;

static PCHAR ScanSystemConfig ( PCHAR Keyword ) ;

static void ResizeWindow ( HWND hwnd, PPROFILE Profile ) ;

static void HideControls
(
  BOOL fHide,
  HWND hwndFrame,
  HWND hwndSysMenu,
  HWND hwndTitleBar,
  HWND hwndMinMax
) ;

static void UpdateWindow ( HWND hwnd, PDATA Data, BOOL All ) ;

static ULONG ComputeTime       ( PDATA Data, USHORT Dummy ) ;
static ULONG ComputeFreeMemory ( PDATA Data, USHORT Dummy ) ;
static ULONG ComputeSwapSize   ( PDATA Data, USHORT Dummy ) ;
static ULONG ComputeSwapFree   ( PDATA Data, USHORT Dummy ) ;
static ULONG ComputeSpoolSize  ( PDATA Data, USHORT Dummy ) ;
static ULONG ComputeCpuLoad    ( PDATA Data, USHORT Dummy ) ;
static ULONG ComputeTaskCount  ( PDATA Data, USHORT Dummy ) ;
static ULONG ComputeDriveFree  ( PDATA Data, USHORT Drive ) ;

static void _far _cdecl MonitorLoopThread ( PMONITOR_PARMS Parms ) ;

static VOID UpdateDriveList
(
  HAB Anchor,
  HMODULE Library,
  PPROFILE Profile,
  ULONG OldDrives
) ;

static BOOL CheckDrive ( USHORT Drive ) ;

static VOID RebuildDisplayItems ( HWND hwnd, PDATA Data ) ;

static ULONG CalibrateLoadMeter ( void ) ;

static void _far _cdecl CounterThread ( PULONG Counter ) ;


  // Global Data (private)

static ITEM Items [ ITEM_BASE_COUNT + MAX_DRIVES ] =
{
  {
    "ShowTime",      TRUE, "", 0L, "", IDM_SHOW_CLOCK,
    ComputeTime,       0,    0, ' '
  },

  {
    "ShowMemory",    TRUE, "", 0L, "", IDM_SHOW_MEMORY,
    ComputeFreeMemory, 0, 1024, 'K'
  },

  {
    "ShowSwapsize",  TRUE, "", 0L, "", IDM_SHOW_SWAPSIZE,
    ComputeSwapSize,   0, 1024, 'K'
  },

  {
    "ShowSwapfree",  TRUE, "", 0L, "", IDM_SHOW_SWAPFREE,
    ComputeSwapFree,   0, 1024, 'K'
  },

  {
    "ShowSpoolSize", TRUE, "", 0L, "", IDM_SHOW_SPOOLSIZE,
    ComputeSpoolSize,  0, 1024, 'K'
  },

  {
    "ShowCpuLoad",   TRUE, "", 0L, "", IDM_SHOW_CPULOAD,
    ComputeCpuLoad,    0,    0, '%'
  },

  {
    "ShowTaskCount", TRUE, "", 0L, "", IDM_SHOW_TASKCOUNT,
    ComputeTaskCount,  0,    0, ' '
  }
} ;


/****************************************************************************
 *									    *
 *	Program Mainline						    *
 *									    *
 ****************************************************************************/

extern void main ( int argc, PCHAR argv[] )
{
 /***************************************************************************
  * Local Declarations							    *
  ***************************************************************************/

  HELPINIT HelpInit =
  {
    sizeof ( HELPINIT ),
    0L,
    NULL,
    MAKEP ( 0xFFFF, ID_MAIN ),
    0,
    0,
    0,
    0,
    NULL,
    CMIC_HIDE_PANEL_ID,
    PROGRAM_NAME ".HLP"
  } ;

  HAB Anchor ;
  FRAMECDATA fcdata ;
  CHAR HelpTitle [80] ;
  HWND hwndClient ;
  HWND hwndFrame ;
  HWND hwndHelp ;
  HMODULE Library ;
  HMQ MessageQueue ;
  PARMS Parms ;
  QMSG QueueMessage ;
  BOOL Reset ;
  CHAR ResetCommand [40] ;
  CHAR Title [80] ;

 /***************************************************************************
  * Initialize for PM.	Abort if unable to do so.			    *
  ***************************************************************************/

  Anchor = WinInitialize ( 0 ) ;
  if ( Anchor == NULL )
    return ;

 /***************************************************************************
  * Create the application message queue.  Abort if unable to do so.	    *
  ***************************************************************************/

  MessageQueue = WinCreateMsgQueue ( Anchor, 0 ) ;
  if ( MessageQueue == NULL )
  {
    WinTerminate ( Anchor ) ;
    return ;
  }

 /***************************************************************************
  * Now WIN and GPI calls will work.  Open the language DLL.		    *
  ***************************************************************************/

  if ( DosLoadModule ( NULL, 0, PROGRAM_NAME, &Library ) )
  {
    Debug ( HWND_DESKTOP, "ERROR: Unable to load " PROGRAM_NAME ".DLL." ) ;
    WinDestroyMsgQueue ( MessageQueue ) ;
    WinTerminate ( Anchor ) ;
    return ;
  }

 /***************************************************************************
  * Register the window class.						    *
  ***************************************************************************/

  if ( NOT WinRegisterClass ( Anchor, CLASS_NAME, MessageProcessor,
    CS_MOVENOTIFY, sizeof(PVOID) ) )
  {
    CHAR Message [200] ;
    char Title [80] ;

    WinLoadString ( Anchor, Library, IDS_TITLE,
      sizeof(Title), Title ) ;
    WinLoadString ( Anchor, Library, IDS_ERROR_WINREGISTERCLASS,
      sizeof(Message), Message ) ;
    WinMessageBox ( HWND_DESKTOP, HWND_DESKTOP, Message,
      Title, 0, MB_ENTER | MB_ICONEXCLAMATION ) ;

    WinDestroyMsgQueue ( MessageQueue ) ;
    WinTerminate ( Anchor ) ;
    return ;
  }

 /***************************************************************************
  * Decipher command-line parameters.					    *
  ***************************************************************************/

  WinLoadString ( Anchor, Library, IDS_PARMS_RESET, sizeof(ResetCommand), ResetCommand ) ;
  Reset = FALSE ;

  while ( --argc )
  {
    argv ++ ;
    strupr ( *argv ) ;
    if ( *argv[0] == '?' )
    {
      CHAR Message [200] ;
      CHAR Title [80] ;

      WinLoadString ( Anchor, Library, IDS_TITLE,
	sizeof(Title), Title ) ;
      WinLoadString ( Anchor, Library, IDS_PARAMETERLIST,
	sizeof(Message), Message ) ;
      WinMessageBox ( HWND_DESKTOP, HWND_DESKTOP, Message,
	Title, 0, MB_ENTER | MB_NOICON ) ;

      WinDestroyMsgQueue ( MessageQueue ) ;
      WinTerminate ( Anchor ) ;
      return ;
    }

    if ( !strcmp ( *argv, ResetCommand ) )
    {
      Reset = TRUE ;
      continue ;
    }

    {
      CHAR Format [200] ;
      CHAR Message [200] ;
      CHAR Title [80] ;

      WinLoadString ( Anchor, Library, IDS_TITLE,
	sizeof(Title), Title ) ;
      WinLoadString ( Anchor, Library, IDS_ERROR_INVALIDPARM,
	sizeof(Format), Format ) ;
      sprintf ( Message, Format, *argv ) ;
      WinMessageBox ( HWND_DESKTOP, HWND_DESKTOP, Message,
	Title, 0, MB_ENTER | MB_ICONEXCLAMATION ) ;

      WinDestroyMsgQueue ( MessageQueue ) ;
      WinTerminate ( Anchor ) ;
      return ;
    }
  }

 /***************************************************************************
  * If we're going to reset the program's profile, do it now.               *
  ***************************************************************************/

  if ( Reset )
  {
    PrfWriteProfileData ( HINI_USERPROFILE, PROGRAM_NAME, NULL, NULL, 0 ) ;
  }

 /***************************************************************************
  * Create the help instance.						    *
  ***************************************************************************/

  WinLoadString ( Anchor, Library, IDS_HELPTITLE, sizeof(HelpTitle), HelpTitle ) ;
  HelpInit.pszHelpWindowTitle = HelpTitle ;

  hwndHelp = WinCreateHelpInstance ( Anchor, &HelpInit ) ;

  if ( hwndHelp == NULL )
  {
    CHAR Message [200] ;
    char Title [80] ;

    WinLoadString ( Anchor, Library, IDS_TITLE,
      sizeof(Title), Title ) ;
    WinLoadString ( Anchor, Library, IDS_ERROR_WINCREATEHELPINSTANCE,
      sizeof(Message), Message ) ;
    WinMessageBox ( HWND_DESKTOP, HWND_DESKTOP, Message,
      Title, 0, MB_ENTER | MB_ICONEXCLAMATION ) ;
  }

 /***************************************************************************
  * Create the frame window.						    *
  ***************************************************************************/

  fcdata.cb = sizeof(fcdata) ;
  fcdata.flCreateFlags =
    FCF_TITLEBAR | FCF_SYSMENU | FCF_BORDER |
      FCF_ICON | FCF_MINBUTTON | FCF_NOBYTEALIGN | FCF_ACCELTABLE ;
  fcdata.hmodResources = 0 ;
  fcdata.idResources   = ID_MAIN ;

  hwndFrame = WinCreateWindow
  (
    HWND_DESKTOP,
    WC_FRAME,
    "",
    0,
    0, 0, 0, 0,
    HWND_DESKTOP,
    HWND_TOP,
    ID_MAIN,
    &fcdata,
    NULL
  ) ;

  if ( hwndFrame == NULL )
  {
    CHAR Message [200] ;
    char Title [80] ;

    WinLoadString ( Anchor, Library, IDS_TITLE,
      sizeof(Title), Title ) ;
    WinLoadString ( Anchor, Library, IDS_ERROR_WINCREATEFRAME,
      sizeof(Message), Message ) ;
    WinMessageBox ( HWND_DESKTOP, HWND_DESKTOP, Message,
      Title, 0, MB_ENTER | MB_ICONEXCLAMATION ) ;

    WinDestroyMsgQueue ( MessageQueue ) ;
    WinTerminate ( Anchor ) ;
    return ;
  }

 /***************************************************************************
  * Set the title.							    *
  ***************************************************************************/

  WinLoadString ( Anchor, Library, IDS_TITLE, sizeof(Title), Title ) ;
  WinSetWindowText ( hwndFrame, Title ) ;

 /***************************************************************************
  * Associate the help instance with the frame window.			    *
  ***************************************************************************/

  if ( hwndHelp )
  {
    WinAssociateHelpInstance ( hwndHelp, hwndFrame ) ;
  }

 /***************************************************************************
  * Create client window.  If this fails, destroy frame and return.	    *
  ***************************************************************************/

  Parms.Anchor = Anchor ;
  Parms.Library = Library ;

  hwndClient = WinCreateWindow
  (
    hwndFrame,
    CLASS_NAME,
    "",
    0,
    0, 0, 0, 0,
    hwndFrame,
    HWND_BOTTOM,
    FID_CLIENT,
    &Parms,
    NULL
  ) ;

  if ( hwndClient == NULL )
  {
    CHAR Message [200] ;
    char Title [80] ;

    WinLoadString ( Anchor, Library, IDS_TITLE,
      sizeof(Title), Title ) ;
    WinLoadString ( Anchor, Library, IDS_ERROR_WINCREATEWINDOW,
      sizeof(Message), Message ) ;
    WinMessageBox ( HWND_DESKTOP, HWND_DESKTOP, Message,
      Title, 0, MB_ENTER | MB_ICONEXCLAMATION ) ;

    WinDestroyWindow ( hwndFrame ) ;
    WinDestroyMsgQueue ( MessageQueue ) ;
    WinTerminate ( Anchor ) ;
    return ;
  }

 /***************************************************************************
  * Wait for and process messages to the window's queue.  Terminate         *
  *   when the WM_QUIT message is received.				    *
  ***************************************************************************/

  while ( WinGetMsg ( Anchor, &QueueMessage, NULL, 0, 0 ) )
  {
    WinDispatchMsg ( Anchor, &QueueMessage ) ;
  }

 /***************************************************************************
  * Destroy the window. 						    *
  ***************************************************************************/

  WinDestroyWindow ( hwndFrame ) ;

 /***************************************************************************
  * If help instance has been created, get rid of it.			    *
  ***************************************************************************/

  if ( hwndHelp )
  {
    WinDestroyHelpInstance ( hwndHelp ) ;
  }

 /***************************************************************************
  * Release the library, if one's been loaded.                              *
  ***************************************************************************/

  if ( Library )
  {
    DosFreeModule ( Library ) ;
  }

 /***************************************************************************
  * Discard all that was requested of the system and terminate. 	    *
  ***************************************************************************/

  WinDestroyMsgQueue ( MessageQueue ) ;

  WinTerminate ( Anchor ) ;
}

/****************************************************************************
 *									    *
 *	Window Message Processor					    *
 *									    *
 ****************************************************************************/

static MRESULT EXPENTRY MessageProcessor
(
  HWND hwnd,
  USHORT msg,
  MPARAM mp1,
  MPARAM mp2
)
{
 /***************************************************************************
  * Local Declarations							    *
  ***************************************************************************/

  static METHOD Methods [] =
  {
    { WM_CREATE,		Create		    },
    { WM_DESTROY,		Destroy 	    },
    { WM_SIZE,			Size		    },
    { WM_MOVE,			Size		    },
    { WM_SAVEAPPLICATION,	SaveApplication     },
    { WM_PAINT, 		Paint		    },
    { WM_BUTTON1DOWN,		ButtonDown	    },
    { WM_BUTTON2DOWN,		ButtonDown	    },
    { WM_BUTTON1DBLCLK, 	ButtonDblClick	    },
    { WM_BUTTON2DBLCLK, 	ButtonDblClick	    },
    { WM_PRESPARAMCHANGED,	PresParamChanged    },
    { WM_SYSCOLORCHANGE,	SysColorChange	    },
    { WM_COMMAND,		Command 	    },
    { HM_QUERY_KEYS_HELP,	QueryKeysHelp	    },
    { HM_ERROR, 		HelpError	    },
    { HM_EXT_HELP_UNDEFINED,	ExtHelpUndefined    },
    { HM_HELPSUBITEM_NOT_FOUND, HelpSubitemNotFound },
    { WM_REFRESH,		Refresh 	    }
  } ;

 /***************************************************************************
  * Dispatch the message according to the method table and return the	    *
  *   result.  Any messages not defined above get handled by the system     *
  *   default window processor. 					    *
  ***************************************************************************/

  return ( DispatchMessage ( hwnd, msg, mp1, mp2, Methods, sizeof(Methods)/sizeof(Methods[0]), WinDefWindowProc ) ) ;
}

/****************************************************************************
 *									    *
 *	Create the main window. 					    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY Create
(
  HWND hwnd,
  USHORT msg,
  MPARAM mp1,
  MPARAM mp2
)
{
 /***************************************************************************
  *				Declarations				    *
  ***************************************************************************/

  static MENUITEM MenuItems [] =
  {
    { MIT_END, MIS_TEXT,      0, IDM_SAVE_APPLICATION, NULL, 0 },
    { MIT_END, MIS_TEXT,      0, IDM_RESET_DEFAULTS,   NULL, 0 },
    { MIT_END, MIS_TEXT,      0, IDM_HIDE_CONTROLS,    NULL, 0 },
    { MIT_END, MIS_TEXT,      0, IDM_SET_TIMER,        NULL, 0 },
    { MIT_END, MIS_SUBMENU,   0, IDM_DISPLAY_ITEMS,    NULL, 0 },
  } ;

  static MENUITEM MenuSeparator =
    { MIT_END, MIS_SEPARATOR, 0, 0, NULL, 0 } ;

  static MENUITEM MenuAbout =
    { MIT_END, MIS_TEXT, 0, IDM_ABOUT, NULL, 0 } ;

  static MENUITEM MenuHelp =
    { MIT_END, MIS_HELP, 0, 0, NULL, 0 } ;

  static MENUITEM MenuItem =
    { MIT_END, MIS_TEXT, 0, 0, NULL, 0 } ;

  char AboutText [80] ;
  PDATA Data ;
  USHORT Drive ;
  char HelpText [80] ;
  HPS hPS ;
  HWND hwndFrame ;
  SHORT i ;
  PMONITOR_PARMS MonitorParms ;
  PPARMS Parms ;
  RECTL Rectangle ;
  ULONG Size ;
  PCHAR Swappath ;
  char Title [80] ;

 /***************************************************************************
  * Allocate instance data.						    *
  ***************************************************************************/

  Data = malloc ( sizeof(DATA) ) ;

  memset ( Data, 0, sizeof(DATA) ) ;

  WinSetWindowPtr ( hwnd, QWL_USER, Data ) ;

 /***************************************************************************
  * Grab any parameters from the WM_CREATE message.			    *
  ***************************************************************************/

  Parms = (PPARMS) PVOIDFROMMP ( mp1 ) ;

  Data->Anchor = Parms->Anchor ;
  Data->Library = Parms->Library ;

 /***************************************************************************
  * Get the current drive mask. 					    *
  ***************************************************************************/

  DosQCurDisk ( &Drive, &Data->Drives ) ;

 /***************************************************************************
  * Get profile data.							    *
  ***************************************************************************/

  GetProfile ( Data->Anchor, Data->Library, &Data->Profile ) ;

 /***************************************************************************
  * Get country information.						    *
  ***************************************************************************/

  {
    COUNTRYCODE CountryCode ;
    USHORT Count ;
    USHORT Status ;

    CountryCode.country = 0 ;
    CountryCode.codepage = 0 ;

    Status = DosGetCtryInfo ( sizeof(Data->CountryInfo), &CountryCode,
      &Data->CountryInfo, &Count ) ;
    if ( Status )
    {
      char Message [80] ;
      WinLoadMessage ( Data->Anchor, Data->Library, IDS_ERROR_DOSGETCTRYINFO,
	sizeof(Message), Message ) ;
      Debug ( hwnd, Message, Status ) ;
      Data->CountryInfo.fsDateFmt = DATEFMT_MM_DD_YY ;
      Data->CountryInfo.fsTimeFmt = 0 ;
      Data->CountryInfo.szDateSeparator[0] = '/' ;
      Data->CountryInfo.szDateSeparator[1] = 0 ;
      Data->CountryInfo.szTimeSeparator[0] = ':' ;
      Data->CountryInfo.szTimeSeparator[1] = 0 ;
      Data->CountryInfo.szThousandsSeparator[0] = ',' ;
      Data->CountryInfo.szThousandsSeparator[1] = 0 ;
    }
  }

 /***************************************************************************
  * Get the frame handle.						    *
  ***************************************************************************/

  hwndFrame = WinQueryWindow ( hwnd, QW_PARENT, FALSE ) ;

 /***************************************************************************
  * Get the control window handles.					    *
  ***************************************************************************/

  Data->hwndSysMenu  = WinWindowFromID ( hwndFrame, FID_SYSMENU  ) ;
  Data->hwndTitleBar = WinWindowFromID ( hwndFrame, FID_TITLEBAR ) ;
  Data->hwndMinMax   = WinWindowFromID ( hwndFrame, FID_MINMAX   ) ;

 /***************************************************************************
  * Create the submenu window for Display Items.			    *
  ***************************************************************************/

  {
    MENUITEM MenuItem ;
    HWND hwndSysSubMenu ;
    HWND hwndSubMenu ;
    SHORT idSysMenu ;

    idSysMenu = SHORT1FROMMR ( WinSendMsg ( Data->hwndSysMenu, MM_ITEMIDFROMPOSITION, NULL, NULL ) ) ;
    WinSendMsg ( Data->hwndSysMenu, MM_QUERYITEM, MPFROM2SHORT(idSysMenu,FALSE), MPFROMP(&MenuItem) ) ;
    hwndSysSubMenu = MenuItem.hwndSubMenu ;

    hwndSubMenu = WinCreateWindow ( hwndSysSubMenu, WC_MENU, "",
      WS_VISIBLE | WS_CLIPCHILDREN | WS_CLIPSIBLINGS,
      0, 0, 0, 0, hwndSysSubMenu, HWND_TOP, IDM_DISPLAY_ITEMS, NULL, NULL ) ;

    MenuItems[IDM_DISPLAY_ITEMS-MenuItems[0].id].hwndSubMenu = hwndSubMenu ;
  }

 /***************************************************************************
  * Add basic extensions to the system menu.				    *
  ***************************************************************************/

  AddSysMenuItem ( hwndFrame, &MenuSeparator, NULL ) ;

  for ( i=0; i<sizeof(MenuItems)/sizeof(MenuItems[0]); i++ )
  {
    char MenuText [80] ;
    WinLoadString ( Data->Anchor, Data->Library, i+IDS_SAVE_APPLICATION, sizeof(MenuText), MenuText ) ;
    AddSysMenuItem ( hwndFrame, MenuItems+i, MenuText ) ;
  }

 /***************************************************************************
  * Add 'About' to the system menu.					    *
  ***************************************************************************/

  AddSysMenuItem ( hwndFrame, &MenuSeparator, NULL ) ;
  WinLoadString ( Data->Anchor, Data->Library, IDS_ABOUT, sizeof(AboutText), AboutText ) ;
  AddSysMenuItem ( hwndFrame, &MenuAbout, AboutText ) ;

 /***************************************************************************
  * Add 'Help' to the system menu.					    *
  ***************************************************************************/

  WinLoadString ( Data->Anchor, Data->Library, IDS_HELP, sizeof(HelpText), HelpText ) ;
  AddSysMenuItem ( hwndFrame, &MenuHelp, HelpText ) ;

 /***************************************************************************
  * Build the display items menu.					    *
  ***************************************************************************/

  RebuildDisplayItems ( hwnd, Data ) ;

 /***************************************************************************
  * Get the SWAPPATH statement from CONFIG.SYS. 			    *
  ***************************************************************************/

  Swappath = ScanSystemConfig ( "SWAPPATH" ) ;

  if ( Swappath == NULL )
  {
    Swappath = "C:\\OS2\\SYSTEM 0" ;
  }

  strupr ( Swappath ) ;

  sscanf ( Swappath, "%s %i", Data->SwapPath, &Data->MinFree ) ;

 /***************************************************************************
  * Find out where the spool work directory is. 			    *
  ***************************************************************************/

  Data->SpoolPath = NULL ;

  if ( PrfQueryProfileSize ( HINI_PROFILE, "PM_SPOOLER", "DIR", &Size ) )
  {
    Data->SpoolPath = malloc ( (int)Size ) ;
    if ( Data->SpoolPath )
    {
      if ( PrfQueryProfileData ( HINI_PROFILE, "PM_SPOOLER", "DIR", Data->SpoolPath, &Size ) )
      {
	BYTE *p ;
	p = strchr ( Data->SpoolPath, ';' ) ;
	if ( p )
	{
	  *p = 0 ;
	}
      }
      else
      {
	free ( Data->SpoolPath ) ;
	Data->SpoolPath = FALSE ;
      }
    }
  }

 /***************************************************************************
  * Calibrate the old-style load meter, if the high resolution timer's      *
  *   available.							    *
  ***************************************************************************/

  Data->MaxCount = CalibrateLoadMeter ( ) ;
  Data->MaxCount = (ULONG) max ( 1L, Data->MaxCount ) ;

 /***************************************************************************
  * Start the new load meter.						    *
  ***************************************************************************/

  Data->IdleLoopTID = _beginthread ( CounterThread, NULL, 1024, &Data->IdleCounter ) ;
  DosSetPrty ( PRTYS_THREAD, PRTYC_IDLETIME, PRTYD_MINIMUM, Data->IdleLoopTID ) ;
  DosSuspendThread ( Data->IdleLoopTID ) ;

  Data->IdleCount = 0 ;
  Data->IdleCounter = 0 ;

  if ( Data->Profile.Items[ITEM_CPULOAD].Flag )
  {
    DosResumeThread ( Data->IdleLoopTID ) ;
  }

  MonitorParms = malloc ( sizeof(*MonitorParms) ) ;
  MonitorParms->Counter = & Data->IdleCounter ;
  MonitorParms->Interval = & Data->Profile.TimerInterval ;
  MonitorParms->Owner = hwnd ;
  Data->MonitorLoopTID = _beginthread ( MonitorLoopThread, NULL, 8192, MonitorParms ) ;

 /***************************************************************************
  * Add the program to the system task list.				    *
  ***************************************************************************/

  WinLoadString ( Data->Anchor, Data->Library, IDS_TITLE, sizeof(Title), Title ) ;
  Add2TaskList ( hwndFrame, Title ) ;

 /***************************************************************************
  * Position & size the window.  For some reason, we must move and size     *
  *   the window to the saved position before applying the resizing	    *
  *   function as fine-tuning.	Maybe the positioning request fails if	    *
  *   the window has no size?						    *
  ***************************************************************************/

  WinSetWindowPos ( hwndFrame, HWND_BOTTOM,
    Data->Profile.Position.x, Data->Profile.Position.y,
    Data->Profile.Position.cx, Data->Profile.Position.cy,
    SWP_SIZE | SWP_MOVE | SWP_ZORDER |
    ( Data->Profile.Position.fs & SWP_MINIMIZE ) |
    ( Data->Profile.Position.fs & SWP_RESTORE ) ) ;

  ResizeWindow ( hwnd, &Data->Profile ) ;

 /***************************************************************************
  * Hide the controls if so configured. 				    *
  ***************************************************************************/

  if ( Data->Profile.HideControls
    AND NOT ( Data->Profile.Position.fs & SWP_MINIMIZE ) )
  {
    CheckMenuItem ( hwndFrame, FID_SYSMENU, IDM_HIDE_CONTROLS, Data->Profile.HideControls ) ;

    HideControls
    (
      TRUE,
      hwndFrame,
      Data->hwndSysMenu,
      Data->hwndTitleBar,
      Data->hwndMinMax
    ) ;
  }

 /***************************************************************************
  * Get the saved presentation parameters and reinstate them.		    *
  ***************************************************************************/

  if ( Data->Profile.fFontNameSize )
  {
    WinSetPresParam ( hwnd, PP_FONTNAMESIZE,
      strlen(Data->Profile.FontNameSize)+1, Data->Profile.FontNameSize ) ;
  }

  if ( Data->Profile.fBackColor )
  {
    WinSetPresParam ( hwnd, PP_BACKGROUNDCOLOR,
      sizeof(Data->Profile.BackColor), &Data->Profile.BackColor ) ;
  }

  if ( Data->Profile.fTextColor )
  {
    WinSetPresParam ( hwnd, PP_FOREGROUNDCOLOR,
      sizeof(Data->Profile.TextColor), &Data->Profile.TextColor ) ;
  }

 /***************************************************************************
  * Determine our font size.						    *
  ***************************************************************************/

  hPS = WinGetPS ( hwnd ) ;
  WinQueryWindowRect ( HWND_DESKTOP, &Rectangle ) ;
  WinDrawText ( hPS, 1, " ", &Rectangle, 0L, 0L, DT_LEFT | DT_BOTTOM | DT_QUERYEXTENT ) ;
  Data->Width  = Rectangle.xRight - Rectangle.xLeft ;
  Data->Height = Rectangle.yTop - Rectangle.yBottom ;
  WinReleasePS ( hPS ) ;

 /***************************************************************************
  * Now that the window's in order, make it visible.                        *
  ***************************************************************************/

  WinShowWindow ( hwndFrame, TRUE ) ;

 /***************************************************************************
  * Success?  Return no error.						    *
  ***************************************************************************/

  return ( 0 ) ;

  hwnd = hwnd ;
  msg = msg ;
  mp1 = mp1 ;
  mp2 = mp2 ;
}

/****************************************************************************
 *									    *
 *	Destroy main window.						    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY Destroy
(
  HWND hwnd,
  USHORT msg,
  MPARAM mp1,
  MPARAM mp2
)
{
 /***************************************************************************
  *				Declarations				    *
  ***************************************************************************/

  PDATA Data ;

 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * Release the instance memory.					    *
  ***************************************************************************/

  free ( Data ) ;

 /***************************************************************************
  * We're done.                                                             *
  ***************************************************************************/

  return ( MRFROMSHORT ( 0 ) ) ;

  hwnd = hwnd ;  msg = msg ;  mp1 = mp1 ;  mp2 = mp2 ;
}

/****************************************************************************
 *									    *
 *	Process window resize message.					    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY Size
(
  HWND hwnd,
  USHORT msg,
  MPARAM mp1,
  MPARAM mp2
)
{
 /***************************************************************************
  *				Declarations				    *
  ***************************************************************************/

  PDATA Data ;

  HWND hwndFrame ;
  SWP Position ;

 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * Find out the window's new position and size.                            *
  ***************************************************************************/

  hwndFrame = WinQueryWindow ( hwnd, QW_PARENT, FALSE ) ;

  WinQueryWindowPos ( hwndFrame, &Position ) ;

  if ( NOT ( Position.fs & SWP_MINIMIZE )
    AND NOT ( Position.fs & SWP_MAXIMIZE ) )
  {
    Data->Profile.Position.x = Position.x ;
    Data->Profile.Position.y = Position.y ;

    Data->Profile.Position.cx = Position.cx ;
    Data->Profile.Position.cy = Position.cy ;
  }

 /***************************************************************************
  * If hiding the controls . . .					    *
  ***************************************************************************/

  if ( Data->Profile.HideControls )
  {

   /*************************************************************************
    * If changing to or from minimized state . . .			    *
    *************************************************************************/

    if ( ( Position.fs & SWP_MINIMIZE ) != ( Data->Profile.Position.fs & SWP_MINIMIZE ) )
    {

     /***********************************************************************
      * Hide the controls if no longer minimized.			    *
      ***********************************************************************/

      HideControls
      (
	NOT ( Position.fs & SWP_MINIMIZE ),
	hwndFrame,
	Data->hwndSysMenu,
	Data->hwndTitleBar,
	Data->hwndMinMax
      ) ;
    }
  }

  Data->Profile.Position.fs = Position.fs ;

 /***************************************************************************
  * We're done.                                                             *
  ***************************************************************************/

  return ( 0 ) ;

  hwnd = hwnd ;
  msg = msg ;
  mp1 = mp1 ;
  mp2 = mp2 ;
}

/****************************************************************************
 *									    *
 *	Process SAVE APPLICATION message.				    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY SaveApplication
(
  HWND hwnd,
  USHORT msg,
  MPARAM mp1,
  MPARAM mp2
)
{
 /***************************************************************************
  *				Declarations				    *
  ***************************************************************************/

  PDATA Data ;

 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * Call function to put all profile data out to the system.		    *
  ***************************************************************************/

  PutProfile ( &Data->Profile ) ;

 /***************************************************************************
  * We're done.  Let the system complete default processing.                *
  ***************************************************************************/

  return ( WinDefWindowProc ( hwnd, WM_SAVEAPPLICATION, 0, 0 ) ) ;

  hwnd = hwnd ;
  msg = msg ;
  mp1 = mp1 ;
  mp2 = mp2 ;
}

/****************************************************************************
 *									    *
 *	Repaint entire window.						    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY Paint
(
  HWND hwnd,
  USHORT msg,
  MPARAM mp1,
  MPARAM mp2
)
{
 /***************************************************************************
  *				Declarations				    *
  ***************************************************************************/

  PDATA Data ;
  HPS hPS ;
  RECTL Rectangle ;

 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * Get presentation space and make it use RGB colors.			    *
  ***************************************************************************/

  hPS = WinBeginPaint ( hwnd, NULL, NULL ) ;
  GpiCreateLogColorTable ( hPS, LCOL_RESET, LCOLF_RGB, 0L, 0L, NULL ) ;

 /***************************************************************************
  * Clear the window.							    *
  ***************************************************************************/

  WinQueryWindowRect ( hwnd, &Rectangle ) ;

  GpiMove ( hPS, (PPOINTL) &Rectangle.xLeft ) ;
  GpiSetColor ( hPS, Data->Profile.BackColor ) ;
  GpiBox ( hPS, DRO_FILL, (PPOINTL) &Rectangle.xRight, 0L, 0L ) ;

 /***************************************************************************
  * Release presentation space. 					    *
  ***************************************************************************/

  WinEndPaint ( hPS ) ;

 /***************************************************************************
  * Update the window and return.					    *
  ***************************************************************************/

  UpdateWindow ( hwnd, Data, TRUE ) ;

  return ( 0 ) ;

  hwnd = hwnd ;
  msg = msg ;
  mp1 = mp1 ;
  mp2 = mp2 ;
}

/****************************************************************************
 *									    *
 *	Process commands received by Main Window			    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY Command
(
  HWND hwnd,
  USHORT msg,
  MPARAM mp1,
  MPARAM mp2
)
{
 /***************************************************************************
  * Local Declarations							    *
  ***************************************************************************/

  static METHOD Methods [] =
  {
    { IDM_SAVE_APPLICATION, SaveApplication },
    { IDM_RESET_DEFAULTS,   ResetDefaults   },
    { IDM_HIDE_CONTROLS,    HideControlsCmd },
    { IDM_SET_TIMER,	    SetTimer	    },
    { IDM_EXIT, 	    Exit	    },
    { IDM_ABOUT,	    About	    },
  } ;

  USHORT Command ;
  PDATA Data ;
  short i ;

 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * Process indicated command . . .					    *
  ***************************************************************************/

  Command = SHORT1FROMMP ( mp1 ) ;

 /***************************************************************************
  * Process display item commands.					    *
  ***************************************************************************/

  for ( i=0; i<Data->Profile.ItemCount; i++ )
  {
    PITEM Item = & Data->Profile.Items [i] ;

    if ( Command == Item->MenuId )
    {
      HWND hwndFrame = WinQueryWindow ( hwnd, QW_PARENT, FALSE ) ;

      Item->Flag = Item->Flag ? FALSE : TRUE ;

      if ( i == ITEM_CPULOAD )
      {
	if ( Item->Flag )
	  DosResumeThread ( Data->IdleLoopTID ) ;
	else
	  DosSuspendThread ( Data->IdleLoopTID ) ;
      }

      CheckMenuItem ( hwndFrame, FID_SYSMENU, Item->MenuId, Item->Flag ) ;

      ResizeWindow ( hwnd, &Data->Profile ) ;

      return ( MRFROMSHORT ( 0 ) ) ;
    }
  }

 /***************************************************************************
  * Dispatch all other commands through the method table.		    *
  ***************************************************************************/

  return ( DispatchMessage ( hwnd, SHORT1FROMMP(mp1), mp1, mp2, Methods, sizeof(Methods)/sizeof(Methods[0]), NULL ) ) ;

  hwnd = hwnd ;  msg = msg ;  mp1 = mp1 ;  mp2 = mp2 ;
}

/****************************************************************************
 *									    *
 *	Process Reset Defaults menu command.				    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY ResetDefaults
( 
  HWND hwnd, 
  USHORT msg, 
  MPARAM mp1, 
  MPARAM mp2
)
{
 /***************************************************************************
  * Reset all profile data for this program.				    *
  ***************************************************************************/

  PrfWriteProfileData ( HINI_USERPROFILE, PROGRAM_NAME, NULL, NULL, 0 ) ;

 /***************************************************************************
  * Reset the program's presentation parameters.                            *
  ***************************************************************************/

  WinRemovePresParam ( hwnd, PP_FONTNAMESIZE ) ;
  WinRemovePresParam ( hwnd, PP_FOREGROUNDCOLOR ) ;
  WinRemovePresParam ( hwnd, PP_BACKGROUNDCOLOR ) ;

 /***************************************************************************
  * Done.								    *
  ***************************************************************************/

  return ( MRFROMSHORT ( 0 ) ) ;

  hwnd = hwnd ;  msg = msg ;  mp1 = mp1 ;  mp2 = mp2 ;
}

/****************************************************************************
 *									    *
 *	Process Hide Controls menu command.				    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY HideControlsCmd
( 
  HWND hwnd, 
  USHORT msg, 
  MPARAM mp1, 
  MPARAM mp2
)
{
 /***************************************************************************
  * Local Declarations. 						    *
  ***************************************************************************/

  PDATA Data ;
  HWND hwndFrame = WinQueryWindow ( hwnd, QW_PARENT, FALSE ) ;

 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * Toggle the Hide Controls setting.					    *
  ***************************************************************************/

  Data->Profile.HideControls = Data->Profile.HideControls ? FALSE : TRUE ;
  Data->Profile.fHideControls = TRUE ;

 /***************************************************************************
  * If controls aren't hidden yet, update the menu check-mark.              *
  ***************************************************************************/

  if ( Data->Profile.HideControls )
    CheckMenuItem ( hwndFrame, FID_SYSMENU, IDM_HIDE_CONTROLS, Data->Profile.HideControls ) ;

 /***************************************************************************
  * If not minimized right now, hide or reveal the controls.		    *
  ***************************************************************************/

  if ( NOT ( Data->Profile.Position.fs & SWP_MINIMIZE ) )
  {
    HideControls
    (
      Data->Profile.HideControls,
      hwndFrame,
      Data->hwndSysMenu,
      Data->hwndTitleBar,
      Data->hwndMinMax
    ) ;
  }

 /***************************************************************************
  * If controls are no longer hidden, update the menu check-mark.	    *
  ***************************************************************************/

  if ( NOT Data->Profile.HideControls )
    CheckMenuItem ( hwndFrame, FID_SYSMENU, IDM_HIDE_CONTROLS, Data->Profile.HideControls ) ;

 /***************************************************************************
  * Done.								    *
  ***************************************************************************/

  return ( MRFROMSHORT ( 0 ) ) ;

  hwnd = hwnd ;  msg = msg ;  mp1 = mp1 ;  mp2 = mp2 ;
}

/****************************************************************************
 *									    *
 *	Process Set Timer menu command. 				    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY SetTimer
( 
  HWND hwnd, 
  USHORT msg, 
  MPARAM mp1, 
  MPARAM mp2
)
{
 /***************************************************************************
  * Local Declarations. 						    *
  ***************************************************************************/

  PDATA Data ;
  SETTIMER_PARMS Parms ;

 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * Invoke the Set Timer dialog.					    *
  ***************************************************************************/

  Parms.id = IDD_SET_TIMER ;
  Parms.hwndHelp = WinQueryHelpInstance ( hwnd ) ;
  Parms.TimerInterval = & Data->Profile.TimerInterval ;

  WinDlgBox ( HWND_DESKTOP, hwnd, SetTimerProcessor,
    Data->Library, IDD_SET_TIMER, &Parms ) ;

 /***************************************************************************
  * Done.								    *
  ***************************************************************************/

  return ( MRFROMSHORT ( 0 ) ) ;

  hwnd = hwnd ;  msg = msg ;  mp1 = mp1 ;  mp2 = mp2 ;
}

/****************************************************************************
 *									    *
 *	Process About menu command.					    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY About
( 
  HWND hwnd, 
  USHORT msg, 
  MPARAM mp1, 
  MPARAM mp2
)
{
 /***************************************************************************
  * Local Declarations							    *
  ***************************************************************************/

  PDATA Data ;
  ABOUT_PARMS Parms ;

 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * Invoke the About dialog.						    *
  ***************************************************************************/

  Parms.id = IDD_ABOUT ;
  Parms.hwndHelp = WinQueryHelpInstance ( hwnd ) ;

  WinDlgBox ( HWND_DESKTOP, hwnd, AboutProcessor,
    Data->Library, IDD_ABOUT, &Parms ) ;

 /***************************************************************************
  * Done.								    *
  ***************************************************************************/

  return ( MRFROMSHORT ( 0 ) ) ;

  hwnd = hwnd ;  msg = msg ;  mp1 = mp1 ;  mp2 = mp2 ;
}

/****************************************************************************
 *									    *
 *	Process Mouse Button being pressed.				    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY ButtonDown
(
  HWND hwnd,
  USHORT msg,
  MPARAM mp1,
  MPARAM mp2
)
{
 /***************************************************************************
  * Local Declarations							    *
  ***************************************************************************/

  PDATA Data ;
  HWND hwndFrame ;
  SWP Position ;
  TRACKINFO TrackInfo ;

 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * Determine the new window position.					    *
  ***************************************************************************/

  memset ( &TrackInfo, 0, sizeof(TrackInfo) ) ;

  TrackInfo.cxBorder = 1 ;
  TrackInfo.cyBorder = 1 ;
  TrackInfo.cxGrid = 1 ;
  TrackInfo.cyGrid = 1 ;
  TrackInfo.cxKeyboard = 8 ;
  TrackInfo.cyKeyboard = 8 ;

  hwndFrame = WinQueryWindow ( hwnd, QW_PARENT, FALSE ) ;

  WinQueryWindowPos ( hwndFrame, &Position ) ;
  TrackInfo.rclTrack.xLeft   = Position.x ;
  TrackInfo.rclTrack.xRight  = Position.x + Position.cx ;
  TrackInfo.rclTrack.yBottom = Position.y ;
  TrackInfo.rclTrack.yTop    = Position.y + Position.cy ;

  WinQueryWindowPos ( HWND_DESKTOP, &Position ) ;
  TrackInfo.rclBoundary.xLeft   = Position.x ;
  TrackInfo.rclBoundary.xRight  = Position.x + Position.cx ;
  TrackInfo.rclBoundary.yBottom = Position.y ;
  TrackInfo.rclBoundary.yTop    = Position.y + Position.cy ;

  TrackInfo.ptlMinTrackSize.x = 0 ;
  TrackInfo.ptlMinTrackSize.y = 0 ;
  TrackInfo.ptlMaxTrackSize.x = Position.cx ;
  TrackInfo.ptlMaxTrackSize.y = Position.cy ;

  TrackInfo.fs = TF_MOVE | TF_STANDARD | TF_ALLINBOUNDARY ;

  if ( WinTrackRect ( HWND_DESKTOP, NULL, &TrackInfo ) )
  {
    WinSetWindowPos ( hwndFrame, NULL,
      (SHORT) TrackInfo.rclTrack.xLeft,
      (SHORT) TrackInfo.rclTrack.yBottom,
      0, 0, SWP_MOVE ) ;
  }

 /***************************************************************************
  * Return through the default processor, letting window activation	    *
  *   and other system functions occur. 				    *
  ***************************************************************************/

  return ( WinDefWindowProc ( hwnd, msg, mp1, mp2 ) ) ;

  hwnd = hwnd ;  msg = msg ;  mp1 = mp1 ;  mp2 = mp2 ;
}

/****************************************************************************
 *									    *
 *	Process Mouse Button having been double-clicked.		    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY ButtonDblClick
(
  HWND hwnd,
  USHORT msg,
  MPARAM mp1,
  MPARAM mp2
)
{
 /***************************************************************************
  * Send message to self to stop hiding the controls.			    *
  ***************************************************************************/

  WinPostMsg ( hwnd, WM_COMMAND,
    MPFROM2SHORT ( IDM_HIDE_CONTROLS, 0 ),
    MPFROM2SHORT ( CMDSRC_OTHER, TRUE ) ) ;

 /***************************************************************************
  * Return through the default processor, letting window activation	    *
  *   and other system functions occur. 				    *
  ***************************************************************************/

  return ( WinDefWindowProc ( hwnd, msg, mp1, mp2 ) ) ;

  hwnd = hwnd ;  msg = msg ;  mp1 = mp1 ;  mp2 = mp2 ;
}

/****************************************************************************
 *									    *
 *	Process Presentation Parameter Changed notification.		    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY PresParamChanged
(
  HWND hwnd,
  USHORT msg,
  MPARAM mp1,
  MPARAM mp2
)
{
 /***************************************************************************
  * Local Declarations							    *
  ***************************************************************************/

  PDATA Data ;
  HPS hPS ;
  ULONG ppid ;
  RECTL Rectangle ;

 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * Get the presentation parameter that changed.			    *
  ***************************************************************************/

  switch ( LONGFROMMP(mp1) )
  {

   /*************************************************************************
    * If font, note the fact that we now have a font to be saved as	    *
    *	part of the configuration.  Get the font metrics and resize	    *
    *	the window appropriately.					    *
    *************************************************************************/

    case PP_FONTNAMESIZE:
    {
      if ( WinQueryPresParam ( hwnd, PP_FONTNAMESIZE, 0, &ppid,
	sizeof(Data->Profile.FontNameSize), &Data->Profile.FontNameSize,
	0 ) )
      {
	Data->Profile.fFontNameSize = TRUE ;
      }
      else
      {
	strcpy ( Data->Profile.FontNameSize, "" ) ;
	Data->Profile.fFontNameSize = FALSE ;
	PrfWriteProfileData ( HINI_USERPROFILE, PROGRAM_NAME, "FontNameSize", NULL, 0 ) ;
      }

      hPS = WinGetPS ( hwnd ) ;
      WinQueryWindowRect ( HWND_DESKTOP, &Rectangle ) ;
      WinDrawText ( hPS, 1, " ", &Rectangle, 0L, 0L, DT_LEFT | DT_BOTTOM | DT_QUERYEXTENT ) ;
      Data->Width  = Rectangle.xRight - Rectangle.xLeft ;
      Data->Height = Rectangle.yTop - Rectangle.yBottom ;
      WinReleasePS ( hPS ) ;
      ResizeWindow ( hwnd, &Data->Profile ) ;
      break ;
    }

   /*************************************************************************
    * If background color, note the fact and repaint the window.	    *
    *************************************************************************/

    case PP_BACKGROUNDCOLOR:
    {
      if ( WinQueryPresParam ( hwnd, PP_BACKGROUNDCOLOR, 0, &ppid,
	sizeof(Data->Profile.BackColor), &Data->Profile.BackColor, 0 ) )
      {
	Data->Profile.fBackColor = TRUE ;
      }
      else
      {
	Data->Profile.BackColor = WinQuerySysColor ( HWND_DESKTOP, SYSCLR_WINDOW, 0L ) ;
	Data->Profile.fBackColor = FALSE ;
	PrfWriteProfileData ( HINI_USERPROFILE, PROGRAM_NAME, "BackgroundColor", NULL, 0 ) ;
      }
      WinInvalidateRect ( hwnd, NULL, TRUE ) ;
      break ;
    }

   /*************************************************************************
    * If foreground color, note the fact and repaint the window.	    *
    *************************************************************************/

    case PP_FOREGROUNDCOLOR:
    {
      if ( WinQueryPresParam ( hwnd, PP_FOREGROUNDCOLOR, 0, &ppid,
	sizeof(Data->Profile.TextColor), &Data->Profile.TextColor, 0 ) )
      {
	Data->Profile.fTextColor = TRUE ;
      }
      else
      {
	Data->Profile.TextColor = WinQuerySysColor ( HWND_DESKTOP, SYSCLR_OUTPUTTEXT, 0L ) ;
	Data->Profile.fTextColor = FALSE ;
	PrfWriteProfileData ( HINI_USERPROFILE, PROGRAM_NAME, "ForegroundColor", NULL, 0 ) ;
      }
      WinInvalidateRect ( hwnd, NULL, TRUE ) ;
      break ;
    }
  }

 /***************************************************************************
  * Return through the default processor, letting window activation	    *
  *   and other system functions occur. 				    *
  ***************************************************************************/

  return ( WinDefWindowProc ( hwnd, msg, mp1, mp2 ) ) ;

  hwnd = hwnd ;  msg = msg ;  mp1 = mp1 ;  mp2 = mp2 ;
}

/****************************************************************************
 *									    *
 *	Process System Color Change notification.			    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY SysColorChange
(
  HWND hwnd,
  USHORT msg,
  MPARAM mp1,
  MPARAM mp2
)
{
 /***************************************************************************
  * Local Declarations							    *
  ***************************************************************************/

  PDATA Data ;

 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * If we aren't using custom colors, then query for the new defaults.      *
  ***************************************************************************/

  if ( NOT Data->Profile.fBackColor )
  {
    Data->Profile.BackColor = WinQuerySysColor ( HWND_DESKTOP, SYSCLR_WINDOW, 0L ) ;
  }

  if ( NOT Data->Profile.fTextColor )
  {
    Data->Profile.TextColor = WinQuerySysColor ( HWND_DESKTOP, SYSCLR_OUTPUTTEXT, 0L ) ;
  }

 /***************************************************************************
  * Return value must be NULL, according to the documentation.		    *
  ***************************************************************************/

  return ( MRFROMP ( NULL ) ) ;

  hwnd = hwnd ;  msg = msg ;  mp1 = mp1 ;  mp2 = mp2 ;
}

/****************************************************************************
 *									    *
 *	Process Query for Keys Help resource id.			    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY QueryKeysHelp
(
  HWND hwnd,
  USHORT msg,
  MPARAM mp1,
  MPARAM mp2
)
{
 /***************************************************************************
  * Simply return the ID of the Keys Help panel.			    *
  ***************************************************************************/

  return ( (MRESULT) IDM_KEYS_HELP ) ;

  hwnd = hwnd ;  msg = msg ;  mp1 = mp1 ;  mp2 = mp2 ;
}

/****************************************************************************
 *									    *
 *	Process Help Manager Error					    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY HelpError
( 
  HWND hwnd, 
  USHORT msg, 
  MPARAM mp1, 
  MPARAM mp2
)
{
 /***************************************************************************
  * Local Declarations							    *
  ***************************************************************************/

  static struct
  {
    ULONG Error ;
    USHORT StringId ;
  }
  HelpErrors [] =
  {
    { HMERR_NO_FRAME_WND_IN_CHAIN,     IDS_HMERR_NO_FRAME_WND_IN_CHAIN },
    { HMERR_INVALID_ASSOC_APP_WND,     IDS_HMERR_INVALID_ASSOC_APP_WND },
    { HMERR_INVALID_ASSOC_HELP_INST,   IDS_HMERR_INVALID_ASSOC_HELP_IN },
    { HMERR_INVALID_DESTROY_HELP_INST, IDS_HMERR_INVALID_DESTROY_HELP_ },
    { HMERR_NO_HELP_INST_IN_CHAIN,     IDS_HMERR_NO_HELP_INST_IN_CHAIN },
    { HMERR_INVALID_HELP_INSTANCE_HDL, IDS_HMERR_INVALID_HELP_INSTANCE },
    { HMERR_INVALID_QUERY_APP_WND,     IDS_HMERR_INVALID_QUERY_APP_WND },
    { HMERR_HELP_INST_CALLED_INVALID,  IDS_HMERR_HELP_INST_CALLED_INVA },
    { HMERR_HELPTABLE_UNDEFINE,        IDS_HMERR_HELPTABLE_UNDEFINE    },
    { HMERR_HELP_INSTANCE_UNDEFINE,    IDS_HMERR_HELP_INSTANCE_UNDEFIN },
    { HMERR_HELPITEM_NOT_FOUND,        IDS_HMERR_HELPITEM_NOT_FOUND    },
    { HMERR_INVALID_HELPSUBITEM_SIZE,  IDS_HMERR_INVALID_HELPSUBITEM_S },
    { HMERR_HELPSUBITEM_NOT_FOUND,     IDS_HMERR_HELPSUBITEM_NOT_FOUND },
    { HMERR_INDEX_NOT_FOUND,	       IDS_HMERR_INDEX_NOT_FOUND       },
    { HMERR_CONTENT_NOT_FOUND,	       IDS_HMERR_CONTENT_NOT_FOUND     },
    { HMERR_OPEN_LIB_FILE,	       IDS_HMERR_OPEN_LIB_FILE	       },
    { HMERR_READ_LIB_FILE,	       IDS_HMERR_READ_LIB_FILE	       },
    { HMERR_CLOSE_LIB_FILE,	       IDS_HMERR_CLOSE_LIB_FILE        },
    { HMERR_INVALID_LIB_FILE,	       IDS_HMERR_INVALID_LIB_FILE      },
    { HMERR_NO_MEMORY,		       IDS_HMERR_NO_MEMORY	       },
    { HMERR_ALLOCATE_SEGMENT,	       IDS_HMERR_ALLOCATE_SEGMENT      },
    { HMERR_FREE_MEMORY,	       IDS_HMERR_FREE_MEMORY	       },
    { HMERR_PANEL_NOT_FOUND,	       IDS_HMERR_PANEL_NOT_FOUND       },
    { HMERR_DATABASE_NOT_OPEN,	       IDS_HMERR_DATABASE_NOT_OPEN     },
    { 0,			       IDS_HMERR_UNKNOWN	       }
  } ;

  PDATA Data ;
  ULONG ErrorCode = (ULONG) LONGFROMMP ( mp1 ) ;
  int Index ;
  char Message [200] ;
  char Title [80] ;

 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * Find the error code in the message table.				    *
  ***************************************************************************/

  Index = 0 ;
  while ( HelpErrors[Index].Error
    AND ( HelpErrors[Index].Error != ErrorCode ) )
  {
    Index ++ ;
  }

 /***************************************************************************
  * Get the message texts.						    *
  ***************************************************************************/

  WinLoadString ( Data->Anchor, Data->Library, IDS_HMERR, sizeof(Title), Title ) ;

  WinLoadString ( Data->Anchor, Data->Library, HelpErrors[Index].StringId, sizeof(Message), Message ) ;

 /***************************************************************************
  * Display the error message.						    *
  ***************************************************************************/

  WinMessageBox
  (
    HWND_DESKTOP,
    hwnd,
    Message,
    Title,
    0,
    MB_OK | MB_WARNING
  ) ;

 /***************************************************************************
  * Return zero, indicating that the message was processed.		    *
  ***************************************************************************/

  return ( MRFROMSHORT ( 0 ) ) ;

  hwnd = hwnd ;  msg = msg ;  mp1 = mp1 ;  mp2 = mp2 ;
}

/****************************************************************************
 *									    *
 *	Process "Extended Help Undefined" notification			    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY ExtHelpUndefined
( 
  HWND hwnd, 
  USHORT msg, 
  MPARAM mp1, 
  MPARAM mp2
)
{
 /***************************************************************************
  * Local Declarations							    *
  ***************************************************************************/

  PDATA Data ;
  char Message [200] ;
  char Title [80] ;

 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * Get the message texts.						    *
  ***************************************************************************/

  WinLoadString ( Data->Anchor, Data->Library, IDS_HMERR, sizeof(Title), Title ) ;

  WinLoadString ( Data->Anchor, Data->Library, IDS_HMERR_EXTHELPUNDEFINED, sizeof(Message), Message ) ;

 /***************************************************************************
  * Display the error message.						    *
  ***************************************************************************/

  WinMessageBox
  (
    HWND_DESKTOP,
    hwnd,
    Message,
    Title,
    0,
    MB_OK | MB_WARNING
  ) ;

 /***************************************************************************
  * Return zero, indicating that the message was processed.		    *
  ***************************************************************************/

  return ( MRFROMSHORT ( 0 ) ) ;

  hwnd = hwnd ;  msg = msg ;  mp1 = mp1 ;  mp2 = mp2 ;
}

/****************************************************************************
 *									    *
 *	Process "Help Subitem Not Found" notification			    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY HelpSubitemNotFound
( 
  HWND hwnd, 
  USHORT msg, 
  MPARAM mp1, 
  MPARAM mp2
)
{
 /***************************************************************************
  * Local Declarations							    *
  ***************************************************************************/

  PDATA Data ;
  int i ;
  BYTE Format [200] ;
  BYTE Message [200] ;
  BYTE Mode [40] ;
  USHORT Subtopic ;
  char Title [80] ;
  USHORT Topic ;

 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * Get the title text. 						    *
  ***************************************************************************/

  WinLoadString ( Data->Anchor, Data->Library, IDS_HMERR, sizeof(Title), Title ) ;

 /***************************************************************************
  * Format the error message.						    *
  ***************************************************************************/

  Topic = (USHORT) SHORT1FROMMP ( mp2 ) ;
  Subtopic = (USHORT) SHORT2FROMMP ( mp2 ) ;

  i = SHORT1FROMMP ( mp1 ) ;

  switch ( i )
  {
    case HLPM_FRAME:
      WinLoadString ( Data->Anchor, Data->Library, IDS_HELPMODE_FRAME, sizeof(Mode), Mode ) ;
      break ;

    case HLPM_MENU:
      WinLoadString ( Data->Anchor, Data->Library, IDS_HELPMODE_MENU, sizeof(Mode), Mode ) ;
      break ;

    case HLPM_WINDOW:
      WinLoadString ( Data->Anchor, Data->Library, IDS_HELPMODE_WINDOW, sizeof(Mode), Mode ) ;
      break ;

    default:
      WinLoadString ( Data->Anchor, Data->Library, IDS_HELPMODE_UNKNOWN, sizeof(Mode), Mode ) ;
  }

  WinLoadString ( Data->Anchor, Data->Library, IDS_HELPSUBITEMNOTFOUND, sizeof(Format), Format ) ;

  sprintf ( Message, Format, Mode, Topic, Subtopic ) ;

 /***************************************************************************
  * Display the error message.						    *
  ***************************************************************************/

  WinMessageBox
  (
    HWND_DESKTOP,
    hwnd,
    Message,
    Title,
    0,
    MB_OK | MB_WARNING
  ) ;

 /***************************************************************************
  * Return zero, indicating that the message was processed.		    *
  ***************************************************************************/

  return ( MRFROMSHORT ( 0 ) ) ;

  hwnd = hwnd ;  msg = msg ;  mp1 = mp1 ;  mp2 = mp2 ;
}

/****************************************************************************
 *									    *
 *	Process Refresh message.					    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY Refresh
( 
  HWND hwnd, 
  USHORT msg, 
  MPARAM mp1, 
  MPARAM mp2
)
{
 /***************************************************************************
  * Local Declarations							    *
  ***************************************************************************/

  PDATA Data ;
  USHORT Drive ;
  ULONG Drives ;

 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * Save the idle counter.						    *
  ***************************************************************************/

  Data->IdleCount = LONGFROMMP ( mp1 ) ;

 /***************************************************************************
  * Determine if drive mask has changed.				    *
  ***************************************************************************/

  DosQCurDisk ( &Drive, &Drives ) ;

  if ( Drives != Data->Drives )
  {
   /*************************************************************************
    * It has.  First save the display options.				    *
    *************************************************************************/

    SaveApplication ( hwnd, WM_SAVEAPPLICATION, 0, 0 ) ;

   /*************************************************************************
    * Next, update the drive item list. 				    *
    *************************************************************************/

    UpdateDriveList ( Data->Anchor, Data->Library, &Data->Profile, Data->Drives ) ;

   /*************************************************************************
    * If the controls are hidden, hide the whole window and reveal the	    *
    *	controls.  Otherwise the menu wouldn't get updated correctly.       *
    *************************************************************************/

    if ( Data->Profile.HideControls )
    {
      WinShowWindow ( WinQueryWindow(hwnd,QW_PARENT,FALSE), FALSE ) ;
      HideControls
      (
	FALSE,
	WinQueryWindow ( hwnd, QW_PARENT, FALSE ),
	Data->hwndSysMenu,
	Data->hwndTitleBar,
	Data->hwndMinMax
      ) ;
    }

   /*************************************************************************
    * Update the menu.							    *
    *************************************************************************/

    RebuildDisplayItems ( hwnd, Data ) ;

   /*************************************************************************
    * If the controls were supposed to be hidden, hide them once more and   *
    *	show the window to the world again.				    *
    *************************************************************************/

    if ( Data->Profile.HideControls )
    {
      HideControls
      (
	TRUE,
	WinQueryWindow ( hwnd, QW_PARENT, FALSE ),
	Data->hwndSysMenu,
	Data->hwndTitleBar,
	Data->hwndMinMax
      ) ;
      WinShowWindow ( WinQueryWindow(hwnd,QW_PARENT,FALSE), TRUE ) ;
    }

   /*************************************************************************
    * Save the updated drive mask.					    *
    *************************************************************************/

    Data->Drives = Drives ;

   /*************************************************************************
    * Resize the window to accommodate the new option list.		    *
    *************************************************************************/

    ResizeWindow ( hwnd, &Data->Profile ) ;
  }

 /***************************************************************************
  * Update the statistics.						    *
  ***************************************************************************/

  UpdateWindow ( hwnd, Data, FALSE ) ;

 /***************************************************************************
  * Return zero, indicating that the message was processed.		    *
  ***************************************************************************/

  return ( MRFROMSHORT ( 0 ) ) ;

  hwnd = hwnd ;  msg = msg ;  mp1 = mp1 ;  mp2 = mp2 ;
}


/****************************************************************************
 *									    *
 *			     Get Profile Data				    *
 *									    *
 ****************************************************************************/

static void GetProfile ( HAB Anchor, HMODULE Library, PPROFILE Profile )
{
 /***************************************************************************
  * Local Declarations							    *
  ***************************************************************************/

  int i ;
  ULONG Size ;

 /***************************************************************************
  * Build the fixed portion of the item list.				    *
  ***************************************************************************/

  for ( i=0; i<ITEM_BASE_COUNT; i++ )
  {
    WinLoadString ( Anchor, Library, i*2+IDS_SHOW_CLOCK_LABEL,
      sizeof(Items[i].Label), Items[i].Label ) ;

    WinLoadString ( Anchor, Library, i*2+IDS_SHOW_CLOCK_OPTION,
      sizeof(Items[i].MenuOption), Items[i].MenuOption ) ;

    Items[i].Flag = TRUE ;
    if
    (
      PrfQueryProfileSize ( HINI_PROFILE, PROGRAM_NAME, Items[i].Name, &Size )
      AND
      ( Size == sizeof(Items[i].Flag) )
      AND
      PrfQueryProfileData ( HINI_PROFILE, PROGRAM_NAME, Items[i].Name, &Items[i].Flag, &Size )
    )
    {
      ;
    }
  }

 /***************************************************************************
  * Add items for each drive on the system.				    *
  ***************************************************************************/

  UpdateDriveList ( Anchor, Library, Profile, 0 ) ;

 /***************************************************************************
  * Get the window's current size and position.                             *
  ***************************************************************************/

  memset ( &Profile->Position, 0, sizeof(Profile->Position) ) ;
  Profile->fPosition = FALSE ;
  if
  (
    PrfQueryProfileSize ( HINI_PROFILE, PROGRAM_NAME, "Position", &Size )
    AND
    ( Size == sizeof(Profile->Position) )
    AND
    PrfQueryProfileData ( HINI_PROFILE, PROGRAM_NAME, "Position", &Profile->Position, &Size )
  )
  {
    Profile->fPosition = TRUE ;
  }

 /***************************************************************************
  * Get the program options.						    *
  ***************************************************************************/

  Profile->HideControls = FALSE ;
  if
  (
    PrfQueryProfileSize ( HINI_PROFILE, PROGRAM_NAME, "HideControls", &Size )
    AND
    ( Size == sizeof(Profile->HideControls) )
    AND
    PrfQueryProfileData ( HINI_PROFILE, PROGRAM_NAME, "HideControls", &Profile->HideControls, &Size )
  )
  {
    Profile->fHideControls = TRUE ;
  }

  Profile->TimerInterval = 1000 ;
  if
  (
    PrfQueryProfileSize ( HINI_PROFILE, PROGRAM_NAME, "TimerInterval", &Size )
    AND
    ( Size == sizeof(Profile->TimerInterval) )
    AND
    PrfQueryProfileData ( HINI_PROFILE, PROGRAM_NAME, "TimerInterval", &Profile->TimerInterval, &Size )
  )
  {
    Profile->fTimerInterval = TRUE ;
  }

 /***************************************************************************
  * Get the presentation parameters.					    *
  ***************************************************************************/

  strcpy ( Profile->FontNameSize, "" ) ;
  Profile->fFontNameSize = FALSE ;
  if
  (
    PrfQueryProfileSize ( HINI_PROFILE, PROGRAM_NAME, "FontNameSize", &Size )
    AND
    ( Size == sizeof(Profile->FontNameSize) )
    AND
    PrfQueryProfileData ( HINI_PROFILE, PROGRAM_NAME, "FontNameSize", &Profile->FontNameSize, &Size )
  )
  {
    Profile->fFontNameSize = TRUE ;
  }

  Profile->BackColor = WinQuerySysColor ( HWND_DESKTOP, SYSCLR_WINDOW, 0L ) ;
  Profile->fBackColor = FALSE ;
  if
  (
    PrfQueryProfileSize ( HINI_PROFILE, PROGRAM_NAME, "BackgroundColor", &Size )
    AND
    ( Size == sizeof(Profile->BackColor) )
    AND
    PrfQueryProfileData ( HINI_PROFILE, PROGRAM_NAME, "BackgroundColor", &Profile->BackColor, &Size )
  )
  {
    Profile->fBackColor = TRUE ;
  }

  Profile->TextColor = WinQuerySysColor ( HWND_DESKTOP, SYSCLR_OUTPUTTEXT, 0L ) ;
  Profile->fTextColor = FALSE ;
  if
  (
    PrfQueryProfileSize ( HINI_PROFILE, PROGRAM_NAME, "ForegroundColor", &Size )
    AND
    ( Size == sizeof(Profile->TextColor) )
    AND
    PrfQueryProfileData ( HINI_PROFILE, PROGRAM_NAME, "ForegroundColor", &Profile->TextColor, &Size )
  )
  {
    Profile->fTextColor = TRUE ;
  }
}

/****************************************************************************
 *									    *
 *			     Put Profile Data				    *
 *									    *
 ****************************************************************************/

static void PutProfile ( PPROFILE Profile )
{
 /***************************************************************************
  * Local Declarations							    *
  ***************************************************************************/

  int i ;

 /***************************************************************************
  * Save the window's current size and position.                            *
  ***************************************************************************/

  PrfWriteProfileData
  (
    HINI_USERPROFILE,
    PROGRAM_NAME,
    "Position",
    &Profile->Position,
    (ULONG)sizeof(Profile->Position)
  ) ;

 /***************************************************************************
  * Save the program options.						    *
  ***************************************************************************/

  if ( Profile->fHideControls )
  {
    PrfWriteProfileData
    (
      HINI_USERPROFILE,
      PROGRAM_NAME,
      "HideControls",
      &Profile->HideControls,
      (ULONG)sizeof(Profile->HideControls)
    ) ;
  }

  if ( Profile->fTimerInterval )
  {
    PrfWriteProfileData
    (
      HINI_USERPROFILE,
      PROGRAM_NAME,
      "TimerInterval",
      &Profile->TimerInterval,
      (ULONG)sizeof(Profile->TimerInterval)
    ) ;
  }

 /***************************************************************************
  * Save the item options.						    *
  ***************************************************************************/

  for ( i=0; i<Profile->ItemCount; i++ )
  {
    PITEM Item = & Profile->Items [i] ;

    PrfWriteProfileData
    (
      HINI_USERPROFILE,
      PROGRAM_NAME,
      Item->Name,
      &Item->Flag,
      (ULONG)sizeof(Item->Flag)
    ) ;
  }

 /***************************************************************************
  * Save the presentation parameters.					    *
  ***************************************************************************/

  if ( Profile->fFontNameSize )
  {
    PrfWriteProfileData
    (
      HINI_USERPROFILE,
      PROGRAM_NAME,
      "FontNameSize",
      Profile->FontNameSize,
      (ULONG)sizeof(Profile->FontNameSize)
    ) ;
  }

  if ( Profile->fBackColor )
  {
    PrfWriteProfileData
    (
      HINI_USERPROFILE,
      PROGRAM_NAME,
      "BackgroundColor",
      &Profile->BackColor,
      (ULONG)sizeof(Profile->BackColor)
    ) ;
  }

  if ( Profile->fTextColor )
  {
    PrfWriteProfileData
    (
      HINI_USERPROFILE,
      PROGRAM_NAME,
      "ForegroundColor",
      &Profile->TextColor,
      (ULONG)sizeof(Profile->TextColor)
    ) ;
  }
}

/****************************************************************************
 *									    *
 *	Scan CONFIG.SYS for a keyword.	Return the value.		    *
 *									    *
 ****************************************************************************/

static PCHAR ScanSystemConfig ( PCHAR Keyword )
{
 /***************************************************************************
  * Local Declarations							    *
  ***************************************************************************/

  static char Buffer [500] ;
  FILE *File ;
  SEL GlobalSelector ;
  SEL LocalSelector ;
  char Path [_MAX_PATH] ;
  GINFOSEG FAR *pGlobalInfoSeg ;

 /***************************************************************************
  * Get the boot drive number from the global information segment.	    *
  ***************************************************************************/

  DosGetInfoSeg ( &GlobalSelector, &LocalSelector ) ;

  pGlobalInfoSeg = MAKEPGINFOSEG ( GlobalSelector ) ;

 /***************************************************************************
  * Build the CONFIG.SYS path.						    *
  ***************************************************************************/

  Path[0] = (char) ( pGlobalInfoSeg->bootdrive + 'A' - 1 ) ;
  Path[1] = 0 ;
  strcat ( Path, ":\\CONFIG.SYS" ) ;

 /***************************************************************************
  * Open CONFIG.SYS for reading.					    *
  ***************************************************************************/

  File = fopen ( Path, "rt" ) ;
  if ( NOT File )
  {
    return ( NULL ) ;
  }

 /***************************************************************************
  * While there're more lines in CONFIG.SYS, read a line and check it.      *
  ***************************************************************************/

  while ( fgets ( Buffer, sizeof(Buffer), File ) )
  {

   /*************************************************************************
    * Clean any trailing newline character from the input string.	    *
    *************************************************************************/

    if ( Buffer[strlen(Buffer)-1] == '\n' )
    {
      Buffer[strlen(Buffer)-1] = 0 ;
    }

   /*************************************************************************
    * If keyword starts the line, we've found the line we want.  Close      *
    *	the file and return a pointer to the parameter text.		    *
    *************************************************************************/

    if ( NOT strnicmp ( Buffer, Keyword, strlen(Keyword) )
      AND ( Buffer[strlen(Keyword)] == '=' ) )
    {
      fclose ( File ) ;
      return ( Buffer + strlen(Keyword) + 1 ) ;
    }
  }

 /***************************************************************************
  * Close the file.  We haven't found the line we wanted.                   *
  ***************************************************************************/

  fclose ( File ) ;

  return ( NULL ) ;
}

/****************************************************************************
 *									    *
 *			 Resize Client Window				    *
 *									    *
 ****************************************************************************/

static void ResizeWindow ( HWND hwnd, PPROFILE Profile )
{
 /***************************************************************************
  * Local Declarations							    *
  ***************************************************************************/

  SHORT Count ;
  SHORT fHadToHide = FALSE ;
  SHORT fHadToRestore = FALSE ;
  LONG Height ;
  HPS hPS ;
  HWND hwndFrame ;
  short i ;
  RECTL Rectangle ;
  char Text [100] ;
  LONG Widest ;

 /***************************************************************************
  * If the window is visible and minimized, restore it invisibly.	    *
  ***************************************************************************/

  hwndFrame = WinQueryWindow ( hwnd, QW_PARENT, FALSE ) ;

  if ( Profile->Position.fs & SWP_MINIMIZE )
  {
    if ( WinIsWindowVisible ( hwndFrame ) )
    {
      WinShowWindow ( hwndFrame, FALSE ) ;
      fHadToHide = TRUE ;
    }
    WinSetWindowPos ( hwndFrame, NULL, 0, 0, 0, 0, SWP_RESTORE ) ;
    fHadToRestore = TRUE ;
  }

 /***************************************************************************
  * Determine how many items are to be displayed.			    *
  ***************************************************************************/

  hPS = WinGetPS ( hwnd ) ;

  Count = 0 ;
  Widest = 0 ;
  Height = 0 ;

  for ( i=0; i<Profile->ItemCount; i++ )
  {
    PITEM Item = & Profile->Items [i] ;

    if ( Item->Flag )
    {
      Count ++ ;

      sprintf ( Text, "%s 1,234,567K", Item->Label ) ;

      WinQueryWindowRect ( HWND_DESKTOP, &Rectangle ) ;

      WinDrawText ( hPS, strlen(Text), Text,
	&Rectangle, 0L, 0L, DT_LEFT | DT_BOTTOM | DT_QUERYEXTENT ) ;

      Widest = max ( Widest, (Rectangle.xRight-Rectangle.xLeft+1) ) ;

      Height += Rectangle.yTop - Rectangle.yBottom ;
    }
  }

  WinReleasePS ( hPS ) ;

 /***************************************************************************
  * Get the window's current size & position.                               *
  ***************************************************************************/

  WinQueryWindowRect ( hwndFrame, &Rectangle ) ;

  WinCalcFrameRect ( hwndFrame, &Rectangle, TRUE ) ;

 /***************************************************************************
  * Adjust the window's width & height.                                     *
  ***************************************************************************/

  Rectangle.xRight  = Rectangle.xLeft + Widest ;

  Rectangle.yTop    = Rectangle.yBottom + Height ;

 /***************************************************************************
  * Compute new frame size and apply it.				    *
  ***************************************************************************/

  WinCalcFrameRect ( hwndFrame, &Rectangle, FALSE ) ;

  WinSetWindowPos ( hwndFrame, NULL, 0, 0,
    (SHORT) (Rectangle.xRight-Rectangle.xLeft),
    (SHORT) (Rectangle.yTop-Rectangle.yBottom),
    SWP_SIZE ) ;

 /***************************************************************************
  * Return the window to its original state.				    *
  ***************************************************************************/

  if ( fHadToRestore )
  {
    WinSetWindowPos ( hwndFrame, NULL,
      Profile->Position.x, Profile->Position.y,
      Profile->Position.cx, Profile->Position.cy,
      SWP_MOVE | SWP_SIZE | SWP_MINIMIZE ) ;
  }

  if ( fHadToHide )
  {
    WinShowWindow ( hwndFrame, TRUE ) ;
  }

 /***************************************************************************
  * Invalidate the window so that it gets repainted.			    *
  ***************************************************************************/

  WinInvalidateRect ( hwnd, NULL, TRUE ) ;
}

/****************************************************************************
 *									    *
 *			Hide Window Controls				    *
 *									    *
 ****************************************************************************/

static void HideControls
(
  BOOL fHide,
  HWND hwndFrame,
  HWND hwndSysMenu,
  HWND hwndTitleBar,
  HWND hwndMinMax
)
{
 /***************************************************************************
  * Local Declarations							    *
  ***************************************************************************/

  SWP OldPosition ;
  SWP Position ;
  RECTL Rectangle ;
  BOOL WasVisible ;

 /***************************************************************************
  * Get original window position and state.				    *
  ***************************************************************************/

  WinQueryWindowPos ( hwndFrame, &OldPosition ) ;

  WasVisible = WinIsWindowVisible ( hwndFrame ) ;

 /***************************************************************************
  * Restore and hide the window.					    *
  ***************************************************************************/

  WinSetWindowPos ( hwndFrame, NULL, 0, 0, 0, 0, SWP_RESTORE | SWP_HIDE ) ;

 /***************************************************************************
  * Determine client window and location.				    *
  ***************************************************************************/

  WinQueryWindowPos ( hwndFrame, &Position ) ;

  Rectangle.xLeft   = Position.x ;
  Rectangle.xRight  = Position.x + Position.cx ;
  Rectangle.yBottom = Position.y ;
  Rectangle.yTop    = Position.y + Position.cy ;

  WinCalcFrameRect ( hwndFrame, &Rectangle, TRUE ) ;

 /***************************************************************************
  * Hide or reveal the controls windows by changing their parentage.	    *
  ***************************************************************************/

  if ( fHide )
  {
    WinSetParent ( hwndSysMenu,  HWND_OBJECT, FALSE ) ;
    WinSetParent ( hwndTitleBar, HWND_OBJECT, FALSE ) ;
    WinSetParent ( hwndMinMax,	 HWND_OBJECT, FALSE ) ;
  }
  else
  {
    WinSetParent ( hwndSysMenu,  hwndFrame, TRUE ) ;
    WinSetParent ( hwndTitleBar, hwndFrame, TRUE ) ;
    WinSetParent ( hwndMinMax,	 hwndFrame, TRUE ) ;
  }

 /***************************************************************************
  * Tell the frame that things have changed.  Let it update the window.     *
  ***************************************************************************/

  WinSendMsg ( hwndFrame, WM_UPDATEFRAME,
    MPFROMSHORT ( FCF_TITLEBAR | FCF_SYSMENU | FCF_MINBUTTON ), 0L ) ;

 /***************************************************************************
  * Reposition the frame around the client window, which is left be.	    *
  ***************************************************************************/

  WinCalcFrameRect ( hwndFrame, &Rectangle, FALSE ) ;

  WinSetWindowPos ( hwndFrame, NULL,
    (SHORT) Rectangle.xLeft,  (SHORT) Rectangle.yBottom,
    (SHORT) (Rectangle.xRight-Rectangle.xLeft),
    (SHORT) (Rectangle.yTop-Rectangle.yBottom),
    SWP_SIZE | SWP_MOVE ) ;

 /***************************************************************************
  * If window was maximized, put it back that way.			    *
  ***************************************************************************/

  if ( OldPosition.fs & SWP_MAXIMIZE )
  {
    WinSetWindowPos ( hwndFrame, NULL,
      (SHORT) Rectangle.xLeft,	(SHORT) Rectangle.yBottom,
      (SHORT) (Rectangle.xRight-Rectangle.xLeft),
      (SHORT) (Rectangle.yTop-Rectangle.yBottom),
      SWP_SIZE | SWP_MOVE |
      ( OldPosition.fs & SWP_MAXIMIZE ) ) ;
  }

 /***************************************************************************
  * If the window was visible in the first place, show it.		    *
  ***************************************************************************/

  if ( WasVisible )
  {
    WinShowWindow ( hwndFrame, TRUE ) ;
  }
}

/****************************************************************************
 *									    *
 *    Update Window							    *
 *									    *
 ****************************************************************************/

#pragma optimize ( "gle", off )

static void UpdateWindow ( HWND hwnd, PDATA Data, BOOL All )
{
 /***************************************************************************
  *				Declarations				    *
  ***************************************************************************/

  int Count ;
  HPS hPS ;
  short i ;
  RECTL Rectangle ;
  char Text [100] ;

 /***************************************************************************
  * Determine how many items are to be displayed.			    *
  ***************************************************************************/

  Count = 0 ;
  for ( i=0; i<Data->Profile.ItemCount; i++ )
  {
    if ( Data->Profile.Items[i].Flag )
    {
      Count ++ ;
    }
  }

 /***************************************************************************
  * Get presentation space and make it use RGB colors.			    *
  ***************************************************************************/

  hPS = WinGetPS ( hwnd ) ;
  GpiCreateLogColorTable ( hPS, LCOL_RESET, LCOLF_RGB, 0L, 0L, NULL ) ;

 /***************************************************************************
  * Get the window's size and determine the initial position.               *
  ***************************************************************************/

  WinQueryWindowRect ( hwnd, &Rectangle ) ;

  Rectangle.xLeft += Data->Width / 2 ;
  Rectangle.xRight -= Data->Width / 2 ;

  Rectangle.yBottom = Data->Height * ( Count - 1 ) ;
  Rectangle.yTop = Rectangle.yBottom + Data->Height ;

 /***************************************************************************
  * Review all items.  Display those changed, or all.			    *
  ***************************************************************************/

  for ( i=0; i<Data->Profile.ItemCount; i++ )
  {
    PITEM Item = & Data->Profile.Items [i] ;
    ULONG NewValue ;

    if ( Item->Flag )
    {
      NewValue = Item->NewValue ( Data, Item->Parm ) ;

      if ( All OR ( NewValue != Item->Value ) )
      {
	switch ( i )
	{
	  case ITEM_CLOCK:
	  {
	    ULONG Month  = ( NewValue % 100000000L ) / 1000000L ;
	    ULONG Day	 = ( NewValue % 1000000L ) / 10000L ;
	    ULONG Hour	 = ( NewValue % 10000L ) / 100L ;
	    ULONG Minute = ( NewValue % 100L ) ;

	    switch ( Data->CountryInfo.fsDateFmt )
	    {
	      case DATEFMT_DD_MM_YY:
		sprintf ( Text, "%02lu%s%02lu ",
		  Day, Data->CountryInfo.szDateSeparator, Month ) ;
		break ;

	      case DATEFMT_YY_MM_DD:
	      case DATEFMT_MM_DD_YY:
	      default:
		sprintf ( Text, "%02lu%s%02lu ",
		  Month, Data->CountryInfo.szDateSeparator, Day ) ;
		break ;
	    }

	    if ( Data->CountryInfo.fsTimeFmt )
	    {
	      sprintf ( Text+strlen(Text), "%02lu%s%02lu",
		Hour,
		Data->CountryInfo.szTimeSeparator,
		Minute ) ;
	    }
	    else
	    {
	      PCHAR AmPm ;

	      if ( Hour )
	      {
		if ( Hour < 12 )
		{
		  AmPm = "a" ;
		}
		else if ( Hour == 12 )
		{
		  if ( Minute )
		    AmPm = "p" ;
		  else
		    AmPm = "a" ;
		}
		else if ( Hour > 12 )
		{
		  Hour -= 12 ;
		  AmPm = "p" ;
		}
	      }
	      else
	      {
		Hour = 12 ;
		if ( Minute )
		  AmPm = "a" ;
		else
		  AmPm = "p" ;
	      }
	      sprintf ( Text+strlen(Text), "%02lu%s%02lu%s",
		Hour, Data->CountryInfo.szTimeSeparator, Minute, AmPm ) ;
	    }
	    break ;
	  }

	  default:
	  {
	    memset ( Text, 0, sizeof(Text) ) ;

	    if ( NewValue == DRIVE_ERROR )
	    {
	      WinLoadString ( Data->Anchor, Data->Library, IDS_DRIVEERROR, sizeof(Text), Text ) ;
	    }
	    else
	    {
	      if ( Item->Divisor )
	      {
		if ( NewValue < Item->Divisor * 1000 )
		  sprintf ( Text, "%lu", NewValue ) ;
		else
		  sprintf ( Text, "%lu", (NewValue+Item->Divisor/2)/Item->Divisor ) ;
	      }
	      else
	      {
		sprintf ( Text, "%lu", NewValue ) ;
	      }

	      {
		PCHAR p1, p2 ;
		CHAR Work[100] ;

		p1 = Text ;
		p2 = Work ;
		while ( *p1 )
		{
		  *p2 = *p1 ;
		  p1 ++ ;
		  p2 ++ ;
		  if ( *p1 )
		  {
		    if ( strlen(p1) % 3 == 0 )
		    {
		      *p2 = Data->CountryInfo.szThousandsSeparator [0] ;
		      p2 ++ ;
		    }
		  }
		}
		*p2 = 0 ;
		strcpy ( Text, Work ) ;
	      }

	      if ( Item->Divisor )
	      {
		if ( NewValue < Item->Divisor * 1000 )
		  Text[strlen(Text)] = ' ' ;
		else
		  Text[strlen(Text)] = Item->Suffix ;
	      }
	      else
	      {
		Text[strlen(Text)] = Item->Suffix ;
	      }
	    }
	  }
	}

	WinDrawText ( hPS, strlen(Text), Text, &Rectangle,
	  Data->Profile.TextColor, Data->Profile.BackColor,
	  DT_RIGHT | DT_BOTTOM | DT_ERASERECT ) ;

	WinDrawText ( hPS, strlen(Item->Label), Item->Label, &Rectangle,
	  Data->Profile.TextColor, Data->Profile.BackColor,
	  DT_LEFT | DT_BOTTOM ) ;

	Item->Value = NewValue ;
      }
      Rectangle.yBottom -= Data->Height ;
      Rectangle.yTop	-= Data->Height ;
    }
  }

 /***************************************************************************
  * Release the presentation space and return.				    *
  ***************************************************************************/

  WinReleasePS ( hPS ) ;
}

#pragma optimize ( "gle", on )


/****************************************************************************
 *									    *
 *    Compute Time							    *
 *									    *
 ****************************************************************************/

static ULONG ComputeTime ( PDATA Data, USHORT Dummy )
{
  char DateString [9] ;
  short Hour, Minute ;
  short Month, Day ;
  ULONG Time ;
  char TimeString [9] ;

  _strdate ( DateString ) ;
  _strtime ( TimeString ) ;

  Hour	 = atoi ( TimeString + 0 ) ;
  Minute = atoi ( TimeString + 3 ) ;

  Month  = atoi ( DateString + 0 ) ;
  Day	 = atoi ( DateString + 3 ) ;

  Time = Month ;
  Time *= 100 ;
  Time += Day ;
  Time *= 100 ;
  Time += Hour ;
  Time *= 100 ;
  Time += Minute ;

  return ( Time ) ;
}

/****************************************************************************
 *									    *
 *    Compute Available Memory						    *
 *									    *
 ****************************************************************************/

static ULONG ComputeFreeMemory ( PDATA Data, USHORT Dummy )
{
  ULONG MemFree ;

  DosMemAvail ( &MemFree ) ;

  return ( MemFree ) ;
}

/****************************************************************************
 *									    *
 *    Compute Swap-File Size						    *
 *									    *
 ****************************************************************************/

static ULONG ComputeSwapSize ( PDATA Data, USHORT Dummy )
{
 /***************************************************************************
  *				Declarations				    *
  ***************************************************************************/

  char Path [_MAX_PATH+1] ;
  struct stat Status ;
  ULONG SwapSize ;

 /***************************************************************************
  * Find the swap file and find its size.				    *
  ***************************************************************************/

  strcpy ( Path, Data->SwapPath ) ;

  if ( Path[strlen(Path)-1] != '\\' )
  {
    strcat ( Path, "\\" ) ;
  }

  strcat ( Path, "SWAPPER.DAT" ) ;

  SwapSize = 0 ;
  if ( stat ( Path, &Status ) == 0 )
  {
    SwapSize = Status.st_size ;
  }

  return ( SwapSize ) ;
}

/****************************************************************************
 *									    *
 *    Compute Available Swap Space					    *
 *									    *
 ****************************************************************************/

static ULONG ComputeSwapFree ( PDATA Data, USHORT Dummy )
{
 /***************************************************************************
  *				Declarations				    *
  ***************************************************************************/

  FSALLOCATE Allocation ;
  char Path [_MAX_PATH+1] ;
  ULONG SwapFree ;

 /***************************************************************************
  * Find the swap file and find its size.				    *
  ***************************************************************************/

  strcpy ( Path, Data->SwapPath ) ;
  strcat ( Path, "\\SWAPPER.DAT" ) ;

 /***************************************************************************
  * Compute swap device free space.					    *
  ***************************************************************************/

  SwapFree = 0 ;
  if ( Path[0] )
  {
    DosQFSInfo ( Path[0]-'A'+1, FSIL_ALLOC,
      (PBYTE)&Allocation, sizeof(Allocation) ) ;

    SwapFree = Allocation.cUnitAvail*Allocation.cSectorUnit*Allocation.cbSector ;
  }

 /***************************************************************************
  * Return swap device's free space, less the minimum free space.           *
  ***************************************************************************/

  if ( SwapFree < (ULONG)Data->MinFree*1024L )
    return ( 0L ) ;
  else
    return ( SwapFree - (ULONG)Data->MinFree * 1024L ) ;
}

/****************************************************************************
 *									    *
 *    Compute Spool-file Size						    *
 *									    *
 ****************************************************************************/

static ULONG ComputeSpoolSize ( PDATA Data, USHORT Dummy )
{
 /***************************************************************************
  *				Declarations				    *
  ***************************************************************************/

  USHORT Count ;
  FILEFINDBUF *Found ;
  HDIR hDir = HDIR_CREATE ;
  BYTE *Path ;
  USHORT PathSize ;
  ULONG TotalSize = 0 ;

 /***************************************************************************
  * Build file specifier for the spool directory.			    *
  ***************************************************************************/

  DosQSysInfo ( 0, (PBYTE)&PathSize, sizeof(PathSize) ) ;

  Path = malloc ( PathSize ) ;
  if ( Path == NULL )
  {
    return ( 0 ) ;
  }

  Found = malloc ( PathSize + sizeof(FILEFINDBUF) ) ;
  if ( Found == NULL )
  {
    free ( Path ) ;
    return ( 0 ) ;
  }

  strcpy ( Path, Data->SpoolPath ) ;
  strcat ( Path, "\\*.*" ) ;

 /***************************************************************************
  * If there are any files/directories in the spool directory . . .	    *
  ***************************************************************************/

  Count = 1 ;
  if ( !DosFindFirst2 ( Path, &hDir,
    FILE_NORMAL | FILE_READONLY | FILE_DIRECTORY | FILE_ARCHIVED,
    Found, PathSize+sizeof(FILEFINDBUF), &Count, FIL_STANDARD, 0L ) )
  {

   /*************************************************************************
    * Loop through every entry in the spool directory.			    *
    *************************************************************************/

    do
    {

     /***********************************************************************
      * Ignore the parent and current directory entries.		    *
      ***********************************************************************/

      if ( !strcmp ( Found->achName, "." )
	OR !strcmp ( Found->achName, ".." ) )
      {
	continue ;
      }

     /***********************************************************************
      * If the entry is a subdirectory . . .				    *
      ***********************************************************************/

      if ( Found->attrFile & FILE_DIRECTORY )
      {

       /*********************************************************************
	* Scan the subdirectory and add every file's size to the total.     *
	*********************************************************************/

	HDIR hDir = HDIR_CREATE ;

	strcpy ( Path, Data->SpoolPath ) ;
	strcat ( Path, "\\" ) ;
	strcat ( Path, Found->achName ) ;
	strcat ( Path, "\\*.*" ) ;

	Count = 1 ;
	if ( !DosFindFirst2 ( Path, &hDir,
	  FILE_NORMAL | FILE_READONLY | FILE_ARCHIVED,
	  Found, PathSize+sizeof(FILEFINDBUF), &Count, FIL_STANDARD, 0L ) )
	{
	  do
	  {
	    TotalSize += Found->cbFileAlloc ;
	  }
	  while ( !DosFindNext ( hDir, Found, PathSize+sizeof(FILEFINDBUF), &Count ) ) ;
	  DosFindClose ( hDir ) ;
	}

	Count = 1 ;
      }

     /***********************************************************************
      * Else if it was a file, add its size to the total.		    *
      ***********************************************************************/

      else
      {
	TotalSize += Found->cbFileAlloc ;
      }
    }
    while ( !DosFindNext ( hDir, Found, PathSize+sizeof(FILEFINDBUF), &Count ) ) ;

   /*************************************************************************
    * Close the directory scan. 					    *
    *************************************************************************/

    DosFindClose ( hDir ) ;
  }

  free ( Path ) ;
  free ( Found ) ;

  return ( TotalSize ) ;
}

/****************************************************************************
 *									    *
 *    Compute CPU Load							    *
 *									    *
 ****************************************************************************/

static ULONG ComputeCpuLoad ( PDATA Data, USHORT Dummy )
{
  ULONG Load ;

  Data->MaxCount = (ULONG) max ( Data->MaxCount, Data->IdleCount ) ;

  Load = ( ( Data->MaxCount - Data->IdleCount ) * 100 ) / Data->MaxCount ;

  return ( Load ) ;
}

/****************************************************************************
 *									    *
 *    Compute Active Task Count 					    *
 *									    *
 ****************************************************************************/

static ULONG ComputeTaskCount ( PDATA Data, USHORT Dummy )
{
  return ( WinQuerySwitchList ( Data->Anchor, NULL, 0 ) ) ;
}

/****************************************************************************
 *									    *
 *    Compute Drive Free Space						    *
 *									    *
 ****************************************************************************/

static ULONG ComputeDriveFree ( PDATA Data, USHORT Drive )
{
  FSALLOCATE Allocation ;

  if ( DosQFSInfo ( Drive, FSIL_ALLOC, (PBYTE)&Allocation, sizeof(Allocation) ) )
    return ( DRIVE_ERROR ) ;

  return ( Allocation.cUnitAvail*Allocation.cSectorUnit*Allocation.cbSector ) ;
}

/****************************************************************************
 *									    *
 *    Monitor Loop Thread						    *
 *									    *
 ****************************************************************************/

static void _far _cdecl MonitorLoopThread ( PMONITOR_PARMS Parms )
{
 /***************************************************************************
  * Local Declarations							    *
  ***************************************************************************/

  ULONG     Counter ;
  ULONG     CurrentCounter ;
  ULONG     DeltaMilliseconds ;
  BOOL	    HiResTimer ;
  ULONG     LastCounter ;
  ULONG     LastMilliseconds ;
  ULONG     Nanoseconds ;
  PGINFOSEG pGlobalInfoSeg = NULL ;
  PLINFOSEG pLocalInfoSeg = NULL ;
  TIMESTAMP Time [2] ;

 /***************************************************************************
  * Set this thread's priority as high as it can go.                        *
  ***************************************************************************/

  DosSetPrty ( PRTYS_THREAD, PRTYC_TIMECRITICAL, PRTYD_MAXIMUM, *_threadid  ) ;

 /***************************************************************************
  * Get pointer to global information segment, where the time is kept.	    *
  ***************************************************************************/

  DosGetInfoSeg ( &SELECTOROF(pGlobalInfoSeg), &SELECTOROF(pLocalInfoSeg) ) ;

 /***************************************************************************
  * Start up the high resolution timer, if it is available.		    *
  ***************************************************************************/

  HiResTimer = OpenTimer ( ) ;

 /***************************************************************************
  * Loop forever . . .							    *
  ***************************************************************************/

  while ( 1 )
  {

   /*************************************************************************
    * Reset the last time and count seen.				    *
    *************************************************************************/

    if ( HiResTimer )
      GetTime ( &Time[0] ) ;
    else
      LastMilliseconds = pGlobalInfoSeg->msecs ;

    LastCounter = *Parms->Counter ;

   /*************************************************************************
    * Sleep for a bit.							    *
    *************************************************************************/

    DosSleep ( *Parms->Interval ) ;

   /*************************************************************************
    * Find out how much time and counts went by.			    *
    *************************************************************************/

    CurrentCounter = *Parms->Counter ;

    if ( HiResTimer )
    {
      GetTime ( &Time[1] ) ;
      DeltaMilliseconds = ElapsedTime ( &Time[0], &Time[1], &Nanoseconds ) ;
      if ( Nanoseconds >= 500000L )
	DeltaMilliseconds ++ ;
    }
    else
    {
      DeltaMilliseconds = pGlobalInfoSeg->msecs - LastMilliseconds ;
    }

   /*************************************************************************
    * Find out how much idle time was counted.	Adjust it to persecond.     *
    *************************************************************************/

    Counter = ( (CurrentCounter-LastCounter) * 1000 ) / DeltaMilliseconds ;

   /*************************************************************************
    * Tell the owner window to refresh its statistics.			    *
    *************************************************************************/

    WinPostMsg ( Parms->Owner, WM_REFRESH, MPFROMLONG(Counter), 0L ) ;
  }
}

/****************************************************************************
 *									    *
 *	Update the Item List to reflect changes in the available drives.    *
 *									    *
 ****************************************************************************/

static VOID UpdateDriveList
(
  HAB Anchor,
  HMODULE Library,
  PPROFILE Profile,
  ULONG OldDrives
)
{
 /***************************************************************************
  * Local Declarations							    *
  ***************************************************************************/

  USHORT Count ;
  USHORT Drive ;
  ULONG Drives ;
  SHORT i ;
  USHORT OldCount ;
  USHORT OldIndex ;
  ITEM OldItems [ ITEM_BASE_COUNT + MAX_DRIVES ] ;
  ULONG Size ;

  CHAR LabelFormat [80] ;
  CHAR OptionFormat [80] ;

 /***************************************************************************
  * Get format strings. 						    *
  ***************************************************************************/

  WinLoadString ( Anchor, Library, IDS_SHOW_DRIVE_FREE_LABEL, sizeof(LabelFormat), LabelFormat ) ;
  WinLoadString ( Anchor, Library, IDS_SHOW_DRIVE_FREE_OPTION, sizeof(OptionFormat), OptionFormat ) ;

 /***************************************************************************
  * Save the old item list for comparison.				    *
  ***************************************************************************/

  OldCount = 0 ;
  memset ( OldItems, 0, sizeof(OldItems) ) ;

  if ( OldDrives )
  {
    OldCount = Profile->ItemCount ;
    memcpy ( OldItems, Items, sizeof(OldItems) ) ;
  }

 /***************************************************************************
  * Add items for each drive on the system.				    *
  ***************************************************************************/

  Count = ITEM_BASE_COUNT ;
  OldIndex = ITEM_BASE_COUNT ;

  DosQCurDisk ( &Drive, &Drives ) ;

  Drives >>= 2 ;
  OldDrives >>= 2 ;

  for ( Drive=3; Drive<MAX_DRIVES; Drive++ )
  {
    while ( ( OldIndex < OldCount )
      AND ( (SHORT)OldItems[OldIndex].MenuId < IDM_SHOW_DRIVE_FREE + Drive ) )
    {
      OldIndex ++ ;
    }

    if ( Drives & 1 )
    {
      if ( OldDrives & 1 )
      {
	Items[Count++] = OldItems[OldIndex++] ;
      }
      else
      {
	if ( CheckDrive ( Drive ) )
	{
	  sprintf ( Items[Count].Name,	     "ShowDrive%c:", Drive+'A'-1 ) ;
	  sprintf ( Items[Count].Label,      LabelFormat,    Drive+'A'-1 ) ;
	  sprintf ( Items[Count].MenuOption, OptionFormat,   Drive+'A'-1 ) ;

	  Items[Count].MenuId = IDM_SHOW_DRIVE_FREE + Drive ;
	  Items[Count].NewValue = ComputeDriveFree ;
	  Items[Count].Parm = Drive ;
	  Items[Count].Divisor = 1024 ;
	  Items[Count].Suffix = 'K' ;
	  Count ++ ;
	}
      }
    }

    Drives >>= 1 ;
    OldDrives >>= 1 ;
  }

 /***************************************************************************
  * Save pointer to fixed configuration information.			    *
  ***************************************************************************/

  Profile->Items = Items ;
  Profile->ItemCount = Count ;

 /***************************************************************************
  * Fetch the display flags for the drives.				    *
  ***************************************************************************/

  for ( i=ITEM_BASE_COUNT; i<Profile->ItemCount; i++ )
  {
    PITEM Item = & Profile->Items [i] ;

    Item->Flag = TRUE ;
    if
    (
      PrfQueryProfileSize ( HINI_PROFILE, PROGRAM_NAME, Item->Name, &Size )
      AND
      ( Size == sizeof(Item->Flag) )
      AND
      PrfQueryProfileData ( HINI_PROFILE, PROGRAM_NAME, Item->Name, &Item->Flag, &Size )
    )
    {
      ;
    }
  }
}

/****************************************************************************
 *									    *
 *	Check to see if drive should be added to display list.		    *
 *									    *
 ****************************************************************************/

static BOOL CheckDrive ( USHORT Drive )
{
 /***************************************************************************
  * Local Declarations							    *
  ***************************************************************************/

  USHORT Action ;
  BOOL	 Addit = FALSE ;
  USHORT Handle ;
  USHORT Status ;
  BYTE	 Path [3] ;

 /***************************************************************************
  * Attempt to open the drive as an entire device.			    *
  ***************************************************************************/

  Path[0] = (BYTE) ( Drive + 'A' - 1 ) ;
  Path[1] = ':' ;
  Path[2] = 0 ;

  Status = DosOpen ( Path, &Handle, &Action, 0, 0, FILE_OPEN,
    OPEN_ACCESS_READONLY | OPEN_SHARE_DENYNONE |
    OPEN_FLAGS_DASD | OPEN_FLAGS_FAIL_ON_ERROR, 0 ) ;

 /***************************************************************************
  * Process return code.						    *
  ***************************************************************************/

  switch ( Status )
  {

   /*************************************************************************
    * No error?  Check to see if removable media (floppies, CDs).  If media *
    *	can be removed, don't try to monitor it.                            *
    *************************************************************************/

    case 0:
    {
      BIOSPARAMETERBLOCK BiosParms ;
      BYTE Command = 0 ;

      if ( !DosDevIOCtl ( &BiosParms, &Command, DSK_GETDEVICEPARAMS, IOCTL_DISK, Handle ) )
      {
	if ( BiosParms.fsDeviceAttr & 1 )  // Must be FIXED drive.
	{
	  Addit = TRUE ;
	}
      }
      DosClose ( Handle ) ;
      break ;
    }

   /*************************************************************************
    * Drive not ready?	Had to be a floppy or CD-ROM without a disk in it.  *
    *	We don't monitor floppies nor CDs.                                  *
    *************************************************************************/

    case ERROR_NOT_READY:
      break ;

   /*************************************************************************
    * Path not found, or bad network name?  Had to be a network drive	    *
    *	that was still defined but had lost its session.  Don't watch it,   *
    *	since it will have to be disconnected and then reconnected in order *
    *	to work.							    *
    *************************************************************************/

    case ERROR_PATH_NOT_FOUND:
    case ERROR_BAD_NET_NAME:
      break ;

   /*************************************************************************
    * Not allowed access to the drive?	Must have been a live network drive,*
    *	which rarely allows a remote user to open the drive as a device.    *
    *	We can, however, monitor the thing.				    *
    *************************************************************************/

    case ERROR_NETWORK_ACCESS_DENIED:	// IBM LAN Manager
    case ERROR_INVALID_ACCESS:		// Novell File Server
    case ERROR_NOT_SUPPORTED:		// TCP/IP
    case ERROR_OPEN_FAILED:
      Addit = TRUE ;
      break ;

   /*************************************************************************
    * Some other error?  Tell me about it.				    *
    *************************************************************************/

    default:
      Debug ( HWND_DESKTOP, "Unable to open drive '%s'.  Status=%04X.  "
	"Tell the author about it at CIS 72607,3111.",
	Path, Status ) ;
  }

 /***************************************************************************
  * Return the final verdict.						    *
  ***************************************************************************/

  return ( Addit ) ;
}

/****************************************************************************
 *									    *
 *	Rebuild the Display Items submenu.				    *
 *									    *
 ****************************************************************************/

static VOID RebuildDisplayItems ( HWND hwnd, PDATA Data )
{
 /***************************************************************************
  * Local Declarations							    *
  ***************************************************************************/

  HWND	   Frame ;
  SHORT    i ;
  HWND	   ItemMenu ;
  MENUITEM MenuItem ;
  HWND	   SysMenu ;

 /***************************************************************************
  * Find the item menu's handle.                                            *
  ***************************************************************************/

  Frame = WinQueryWindow ( hwnd, QW_PARENT, FALSE ) ;

  SysMenu = WinWindowFromID ( Frame, FID_SYSMENU ) ;

  WinSendMsg ( SysMenu, MM_QUERYITEM,
    MPFROM2SHORT ( IDM_DISPLAY_ITEMS, TRUE ),
    (MPARAM) &MenuItem ) ;
  ItemMenu = MenuItem.hwndSubMenu ;

 /***************************************************************************
  * Remove all items from the menu.					    *
  ***************************************************************************/

  if ( WinSendMsg ( ItemMenu, MM_QUERYITEMCOUNT, 0, 0 ) )
  {
    USHORT Id ;
    SHORT ItemsLeft ;

    do
    {
      Id = SHORT1FROMMR ( WinSendMsg ( ItemMenu, MM_ITEMIDFROMPOSITION,
	0, 0 ) ) ;

      ItemsLeft = SHORT1FROMMR ( WinSendMsg ( ItemMenu, MM_DELETEITEM,
	MPFROM2SHORT(Id,FALSE), 0 ) ) ;
    }
    while ( ItemsLeft ) ;
  }

 /***************************************************************************
  * Prepare menu item structure for use.				    *
  ***************************************************************************/

  MenuItem.iPosition = MIT_END ;
  MenuItem.afStyle = MIS_TEXT ;
  MenuItem.afAttribute = 0 ;
  MenuItem.hwndSubMenu = NULL ;
  MenuItem.hItem = 0L ;

 /***************************************************************************
  * Add all menu items called for.					    *
  ***************************************************************************/

  for ( i=0; i<Data->Profile.ItemCount; i++ )
  {
    PITEM Item = & Data->Profile.Items [i] ;

    MenuItem.id = Item->MenuId ;

    AddSysSubMenuItem ( Frame, IDM_DISPLAY_ITEMS, &MenuItem, Item->MenuOption ) ;

    CheckMenuItem ( Frame, FID_SYSMENU, Item->MenuId, Item->Flag ) ;
  }

 /***************************************************************************
  * Split the menu if it's too tall.                                        *
  ***************************************************************************/

  if ( Data->Profile.ItemCount > 15 )
  {
    USHORT Midpoint = ( Data->Profile.ItemCount + 1 ) / 2 ;
    USHORT Id = Data->Profile.Items[Midpoint].MenuId ;
    MENUITEM MenuItem ;

    if ( WinSendMsg ( ItemMenu, MM_QUERYITEM, MPFROM2SHORT(Id,FALSE), &MenuItem ) )
    {
      MenuItem.afStyle |= MIS_BREAKSEPARATOR ;
      WinSendMsg ( ItemMenu, MM_SETITEM, MPFROM2SHORT(0,FALSE), &MenuItem ) ;
    }
  }
}

/****************************************************************************
 *									    *
 *			 Calibrate the Load Meter			    *
 *									    *
 ****************************************************************************/

static ULONG CalibrateLoadMeter ( void )
{
 /***************************************************************************
  * Local Declarations							    *
  ***************************************************************************/

  double	AdjustedMaxLoad = 0.0 ;
  ULONG 	MaxLoad ;
  ULONG 	Milliseconds ;
  ULONG 	Nanoseconds ;
  TID		tidCalibrate ;
  TIMESTAMP	Time[2] ;

 /***************************************************************************
  * If HRTIMER.SYS has been installed . . .				    *
  ***************************************************************************/

  if ( OpenTimer ( ) )
  {
   /*************************************************************************
    * Increase this thread's priority to the maximum.                       *
    *************************************************************************/

    DosSetPrty ( PRTYS_THREAD, PRTYC_TIMECRITICAL, PRTYD_MAXIMUM, *_threadid ) ;

   /*************************************************************************
    * Create the calibration thread and set its priority next highest.	    *
    *************************************************************************/

    tidCalibrate = _beginthread ( CounterThread, NULL, 1024, &MaxLoad ) ;
    DosSetPrty ( PRTYS_THREAD, PRTYC_TIMECRITICAL, PRTYD_MAXIMUM-1, tidCalibrate ) ;
    DosSuspendThread ( tidCalibrate ) ;

   /*************************************************************************
    * Reset the calibration count, get the time, and let the counter go.    *
    *************************************************************************/

    MaxLoad = 0 ;
    GetTime ( &Time[0] ) ;
    DosResumeThread ( tidCalibrate ) ;

   /*************************************************************************
    * Sleep for one second.						    *
    *************************************************************************/

    DosSleep ( 1000 ) ;

   /*************************************************************************
    * Suspend the calibration counter and get the time. 		    *
    *************************************************************************/

    DosSuspendThread ( tidCalibrate ) ;
    GetTime ( &Time[1] ) ;

   /*************************************************************************
    * Return priorities to normal.					    *
    *************************************************************************/

    DosSetPrty ( PRTYS_THREAD, PRTYC_REGULAR, 0, *_threadid ) ;

   /*************************************************************************
    * Get the elapsed time and adjust the calibration count.		    *
    *************************************************************************/

    Milliseconds = ElapsedTime ( &Time[0], &Time[1], &Nanoseconds ) ;

    AdjustedMaxLoad = (double)MaxLoad * 1.0E9 ;
    AdjustedMaxLoad /= (double)Milliseconds*1.0E6L + (double)Nanoseconds ;

   /*************************************************************************
    * Close down the connection to HRTIMER.SYS. 			    *
    *************************************************************************/

    CloseTimer ( ) ;
  }

 /***************************************************************************
  * Return the adjusted calibration count.  If HRTIMER was not there, it    *
  *   will be zero.							    *
  ***************************************************************************/

  return ( (ULONG)AdjustedMaxLoad ) ;
}

/****************************************************************************
 *									    *
 *		      General Purpose Counter Thread			    *
 *									    *
 ****************************************************************************/

static void _far _cdecl CounterThread ( PULONG Counter )
{
  while ( 1 )
  {
    (*Counter) ++ ;
  }
}
