/*********************************************************************
** FILENAME: mamove.c                       VERSION: 1.00
**
** DESCRIPTION: RemoteAccess message area positioning utility
**
** NOTES: See wb_fcopy.c for file_copy() details.
**
** AUTHOR: John Kristoff                START DATE: 04/29/94
**         Internet: jkristof@xroads.chigate.com
**                   jkristof@mica.meddean.luc.edu
**         FidoNet:  1:115/743
**      CompuServe:  74111,3652
**
** VERSION  DATE     WHO  DETAIL
** 1.00     22May94  JK   Initial design and coding
**
**      Copyright John Kristoff, 1994.  All rights reserved.
**      You may use this program or any part there-of as desired
**      without restriction.
*/

#include <stdio.h>                      /* Standard i/o */
#include <stdlib.h>                     /* Standard library */
#include <limits.h>                     /* Variable max/min macros */
#include "ra.h"                         /* RemoteAccess structures */
#include "wb_fcopy.c"                   /* Copy a file */

#define VERS       "1.00"               /* Version of this program */
#define FNAME      "MESSAGES.RA"        /* Data file */
#define FNAMEBAK   "MESSAGES.BAK"       /* Backup data file */
#define FNAMELEN   81                   /* Fully qualified filepath + '\0' */
#define AREALEN    40                   /* Message area string length */
#define ENVVAR     "RA"                 /* Environment variable of RA dir */
#define FALSE      0                    /* Boolean false flag */
#define TRUE       1                    /* Boolean true flag */

#define HELP       1                    /* Display help screen */
#define OPEN       10                   /* Error opening file */
#define CLOSE      20                   /* Error closing file */
#define READ       30                   /* Error reading file */
#define WRITE      40                   /* Error writing file */
#define SEEK       50                   /* Error seeking in file */
#define MEMORY     60                   /* Error allocating memory */

#define USAGE      "   Usage:  MAMOVE <source> <dest>\n"\
                   "Examples:  MAMOVE 23 2, MAMOVE 15 382, MAMOVE 1 6\n"\
                   "\n"\
                   "<source>   area # to move\n"\
                   "  <dest>   area # <source> will precede\n"\
                   "\n"\
                   "   Note:   area # range is 1 to %ld\n", LONG_MAX

int main( int argc, char * argv[] );    /* If you don't know, YUSC */
void Error( int ErrorLevel,             /* Error handling function */
            int Line,
            char * Filename );

int
main( int argc, char * argv[] )
{
    long int AreaToMove  = 0;           /* Area # to be moved */
    long int CurrentArea = 0;           /* The Area # we're at */
    long int InsertPoint = 0;           /* AreaToMove placed before this */
    long int SeekPoint   = 0;           /* How many structs til MoveArea */

    int Moved = FALSE;                  /* Flag trips when area is moved */

    struct MESSAGE MoveArea;            /* Structure to be moved */
    struct MESSAGE Area;                /* Current struct in processing */

    char * DataDir = NULL;              /* Pointer to ENVVAR */
    char Filename[FNAMELEN] = FNAME;    /* Fully qualified data filename */
    char BakFile[FNAMELEN]  = FNAMEBAK; /* Fully qualified backup filename */

    FILE * fpFilename = NULL;           /* Pointer to data file */
    FILE * fpBakFile  = NULL;           /* Pointer to backup data file */

    printf( "\n"
            "MAMOVE v" VERS ", " __DATE__ ".\n"
            "RemoteAccess message area positioning utility\n"
            "Copyright John Kristoff, 1994.  All rights reserved.\n"
            "\n" );

    if( argc < 3 )
    {
        Error( HELP, __LINE__, NULL );
    }

    /*
    ** Build fully qualified filenames if ENVVAR exists.
    */
    if( (DataDir = getenv(ENVVAR)) != NULL )
    {
        if( DataDir[strlen(DataDir) - 1] != '\\')
        {
            sprintf( Filename, "%s\\%s", DataDir, FNAME );
            sprintf( BakFile, "%s\\%s", DataDir, FNAMEBAK );
        }
        else
        {
            sprintf( Filename, "%s%s", DataDir, FNAME );
            sprintf( BakFile, "%s%s", DataDir, FNAMEBAK );
        }
    }

    /*
    ** Get area to be moved.
    */
    AreaToMove  = atol( argv[1] );
    if( AreaToMove < 1 || AreaToMove > LONG_MAX )
    {
        Error( HELP, __LINE__, NULL );
    }

    /*
    ** Get insertion point of area to be moved.
    */
    InsertPoint = atol( argv[2] );
    if( InsertPoint < 1 || InsertPoint > LONG_MAX )
    {
        Error( HELP, __LINE__, NULL );
    }

    /*
    ** Make backup of data file.
    */
    if( file_copy( Filename, BakFile ) == 1 )
    {
        fprintf( stderr, "ERROR (%d): Cannot copy %s to %s\n",
                         __LINE__, Filename, BakFile );
        exit( EXIT_FAILURE );
    }

    /*
    ** We'll be reading from the backup file in order to re-create
    ** a new data file.
    */
    fpBakFile = fopen( BakFile, "rb" );
    if( fpBakFile == NULL )
    {
        Error( OPEN, __LINE__, BakFile );
    }

    /*
    ** Go to the area to move and read struct.
    */
    SeekPoint = sizeof(struct MESSAGE) * ( AreaToMove - 1 );
    if( fseek(fpBakFile, SeekPoint, SEEK_SET) != 0 )
    {
        Error( SEEK, __LINE__, BakFile );
    }

    if( fread(&MoveArea, sizeof(struct MESSAGE), 1, fpBakFile) == NULL )
    {
        Error( READ, __LINE__, BakFile );
    }

    rewind( fpBakFile );

    fpFilename = fopen( Filename, "wb" );
    if( fpFilename == NULL )
    {
        Error( OPEN, __LINE__, Filename );
    }

    if( fread(&Area, sizeof(struct MESSAGE), 1, fpBakFile) == NULL )
    {
        Error( READ, __LINE__, BakFile );
    }

    /*
    ** Read backup data file until EOF, writing new data file as we go.
    */
    while( !feof(fpBakFile) )
    {
        ++CurrentArea;

        /*
        ** If this is the insertion point, add the area to move.
        */
        if( CurrentArea == InsertPoint )
        {
            /*
            ** Blank area name strings that are reported to stdout.
            */
            char MoveName[AREALEN + 1]   = "";
            char InsertName[AREALEN + 1] = "";

            /*
            ** Write the area to be moved into the data file.
            */
            if( fwrite(&MoveArea, sizeof(struct MESSAGE),
                       1, fpFilename) == NULL )
            {
                Error( WRITE, __LINE__, Filename );
            }

            /*
            ** Create area name string reported to stdout.
            */
            if( MoveArea.NameSize > 0 )
            {
                strncpy( MoveName, MoveArea.Name, AREALEN );
                MoveName[MoveArea.NameSize] = '\0';
            }
            else
            {
                strcpy( MoveName, "[Unused]" );
            }
        
            if( Area.NameSize > 0 )
            {
                strncpy( InsertName, Area.Name, AREALEN );
                InsertName[Area.NameSize] = '\0';
            }
            else
            {
                strcpy( InsertName, "[Unused]" );
            }

            printf( "Inserting: %s\n"
                    "   before: %s\n"
                    "\n",
                   MoveName, InsertName );

            Moved = TRUE;
        }

        /*
        ** Unless this struct is where the area to be moved used to
        ** be, write it to the data file.
        */
        if( CurrentArea != AreaToMove )
        {
            if( fwrite(&Area, sizeof(struct MESSAGE),
                       1, fpFilename) == NULL )
            {
                Error( WRITE, __LINE__, Filename );
            }
        }

        fread(&Area, sizeof(struct MESSAGE), 1, fpBakFile);
    }

    fclose( fpBakFile );
    if( ferror(fpBakFile) )
    {
        Error( CLOSE, __LINE__, BakFile );
    }

    /*
    ** If the insertion point specified was higher than what existed
    ** in the backup file, the area to be moved will be written to the
    ** end of the data file now.
    */
    if( Moved == FALSE )
    {
        char MoveName[AREALEN + 1]   = "";
 
        if( fwrite(&MoveArea, sizeof(struct MESSAGE),
                   1, fpFilename) == NULL )
        {
            Error( WRITE, __LINE__, Filename );
        }

        if( MoveArea.NameSize > 0 )
        {
            strncpy( MoveName, MoveArea.Name, AREALEN );
            MoveName[MoveArea.NameSize] = '\0';
        }
        else
        {
            strcpy( MoveName, "[Unused]" );
        }

        printf( "Moved to last area: %s\n"
                "\n",
                MoveName );
    }

    fclose( fpFilename );
    if( ferror(fpFilename) )
    {
        Error( CLOSE, __LINE__, Filename );
    }

    printf( "Successfully re-created %s\n", Filename );
    return( EXIT_SUCCESS );
}



/*********************************************************************
** FUNCTION: Error
**
** DESCRIPTION: Error handling and exit function
**
** PARAMS: int ErrorLevel       errorlevel to exit with
**         int Line             Line number where error occured
**         char * File          Filename where I/O error occured
**
** RETURN: void
**
** NOTES: NULL passed to char * File, if error was not file i/o
**        related.
*/

void
Error( int ErrorLevel, int Line, char * File )
{
    switch( ErrorLevel )
    {
        case HELP:   fprintf( stderr, USAGE );
                     break;
    
        case OPEN:   fprintf( stderr, "ERROR (%d): Cannot open: %s\n",
                             Line, File );
                     break;
    
        case CLOSE:  fprintf( stderr, "ERROR (%d): Cannot close %s\n",
                             Line, File );
                     break;
    
        case READ:   fprintf( stderr, "ERROR (%d): Cannot read %s\n",
                             Line, File );
                     break;
   
        case WRITE:  fprintf( stderr, "ERROR (%d): Cannot write %s\n",
                             Line, File );
                     break;
 
        case SEEK:   fprintf( stderr, "ERROR (%d): Seek error in %s\n",
                              Line, File );
                     break;

        case MEMORY: fprintf( stderr, "ERROR (%d): Cannot allocate memory\n",
                             Line );
                     break;

        default:     fprintf( stderr, "ERROR (%d): Invalid instruction\n",
                             Line );
    }
    exit( ErrorLevel );
}
