//@doc      VIEW
//@module   VIEW.H - View declarations |
//  
//  This file contains the declaration of the main view classes.
//  
//@devnote  Version 1.2, jcw@meta4.nl
//@normal   Copyright <cp> 1996 Meta Four Software. All rights reserved.

#ifndef __K4VIEW_H__
#define __K4VIEW_H__

#ifndef __K4CONF_H__
    #error Please include "k4conf.h" before this header file
#endif
    
/////////////////////////////////////////////////////////////////////////////
// Declarations in this file

    class c4_View;                      // a view on underlying data
    class c4_Cursor;                    // an index into a view
    class c4_RowRef;                    // a reference to a row
		class c4_Row;                   // one row in a view
    class c4_Storage;                   // manages view persistence

    class c4_Property;                  // for access inside rows
        class c4_IntProp;
        class c4_FloatProp;
        class c4_StringProp;
        class c4_ViewProp;

        // defined in "k4viewx.h", included at the end of this file    
    class c4_Sequence;
    class c4_Reference;
        class c4_IntRef;
        class c4_FloatRef;
        class c4_StringRef;
        class c4_ViewRef;

    class c4_Persist;                   // not defined here
    class c4_Table;                     // not defined here

/////////////////////////////////////////////////////////////////////////////
//@class A general class of objects to represent a collection of data rows.
//
// Views act like pointers to the actual collections, setting a view to a new
// value does not alter the collection to which this view pointed previously.
//
// The protocol used for this class mimics the way many other collection
// classes are defined. For example, when used with MFC, you will see member
// definitions such as <mf ::GetSize>, <mf ::GetAt>, <mf ::InsertAt>, etc.
//
// The elements of views can be referred to by their 0-based index, which
// produces a row-reference of type <c c4_RowRef>. These row references can
// be copied, used to get or set properties, or dereferenced (in which case
// an object of class <c c4_Row> is returned). Taking the address of a row
// reference produces a <c c4_Cursor>, which acts very much like a pointer.
//
//@ex The following code creates a view with 1 row and 2 properties: |
//  c4_StringProp pName ("Name");
//  c4_IntProp pAge ("Age");
//
//  c4_Row data;
//  pName (data) = "John Williams";
//  pAge (data) = 43;
//
//  c4_View myView;
//  myView.Add(row);

class c4_View
{
public: //@access Public members
//@group Construction / destruction / assignment
    //@cmember Constructs a view based on a sequence.
    c4_View (c4_Sequence* =0);
    //@cmember Constructs an empty view with one property.
    c4_View (const c4_Property&);
    //@cmember Constructs a view from another one.
    c4_View (const c4_View&);
    //@cmember Destructor.
    ~c4_View ();
    
    //@cmember Makes this view the same as another one.
    c4_View& operator= (const c4_View&);

//@group Getting / setting the number of rows
    //@cmember Returns the number of entries.
    int GetSize() const;    
    //@cmember Returns highest index (size - 1).
    int GetUpperBound() const;
   
    //@cmember Changes the size of this view.
    void SetSize(int, int =-1); 
    //@cmember Removes all entries (sets size to zero).
    void RemoveAll();

//@group Getting / setting individual elements
    //@cmember Returns a reference to specified entry.
    c4_RowRef GetAt(int) const;
    //@cmember Shorthand for <mf c4_View::GetAt>.
    c4_RowRef operator[] (int) const;
    
    //@cmember Changes the value of the specified entry.
    void SetAt(int, c4_RowRef);
    //@cmember Element access, for use as RHS or LHS.
    c4_RowRef ElementAt(int);
    
//@group These can increase the number of rows
    //@cmember Sets an entry, growing the view if needed.
    void SetAtGrow(int, c4_RowRef);   
    //@cmember Adds a new entry at the end.
    int Add(c4_RowRef);

//@group Insertion / deletion of rows
    //@cmember Inserts one or more copies of an entry.
    void InsertAt(int, c4_RowRef, int =1);
    //@cmember Removes entries starting at the given index.
    void RemoveAt(int, int =1);
    //@cmember Inserts a copy of the contents of another view.
    void InsertAt(int, c4_View*);

//@group Dealing with the properties of this view
    //@cmember Returns the number of properties.
    int NumProperties() const; 
    //@cmember Returns the id of the N-th property.
    int NthProperty(int) const;
    //@cmember Finds the index of a property, given its id.
    int FindProperty(int);
    //@cmember Returns a description of the structure.
    c4_String Describe() const;

    //@cmember Returns a view like the first, with a property appended to it.
    friend c4_View operator, (const c4_View&, const c4_Property&);
    
//@group Derived views
    //@cmember Creates view with all rows in natural (property-wise) order.
    c4_View Sort() const;
    //@cmember Creates view sorted according to the specified properties.
    c4_View SortOn(const c4_View&) const;
    //@cmember Creates sorted view, with some properties sorted in reverse.
    c4_View SortOnReverse(const c4_View&, const c4_View&) const;

    //@cmember Creates a view with rows matching the specified value.
    c4_View Select(c4_RowRef) const;
    //@cmember Creates a view with row values within the specified range.
    c4_View SelectRange(c4_RowRef, c4_RowRef) const;

    //@cmember Creates a view with the specified property arrangement.
    c4_View Project(const c4_View&) const;
    //@cmember Creates a derived view with some properties omitted.
    c4_View ProjectWithout(const c4_View&) const;

    //@cmember Returns the index of the specified row in this view (or -1).
    int GetIndexOf(c4_RowRef) const;

//@group Searching
    //@cmember Finds index of the the next entry matching the specified key.
    int Find(c4_RowRef, int start_ =0) const;
    //@cmember Searches for key, assuming the view is sorted accordingly.
    int Search(c4_RowRef) const;
    
protected:
    void _IncSeqRef();
    void _DecSeqRef();

public: // ouch!
    // Pointer to the sequence associated with this view.
    c4_Sequence* _seq;
};

/////////////////////////////////////////////////////////////////////////////
//@class A wrapper class of objects to iterate over sequences of rows.
//
// Cursor objects can be used to point to specific entries in a view.
// A cursor acts very much like a pointer to a row in a view, and is 
// returned when taking the address of a <c c4_RowRef>. Dereferencing
// a cursor leads to the original row reference again. You can construct a
// cursor for a <c c4_Row>, but since such rows are not part of a collection,
// incrementing or decrementing these cursors is meaningless (and wrong). 
//
// The usual range of pointer operations can be applied to these objects:
// pre/post-increment and decrement, adding or subtracting integer offsets,
// as well as with the full range of comparison operators. If two cursors
// point to entries in the same view, their difference can be calculated.
//
// As with regular pointers, care must be taken to avoid running off of
// either end of a view (the debug build includes assertions to check this).

class c4_Cursor
{
public: //@access Public members
//@group Construction / destruction / dereferencing
    //@cmember Constructs a new cursor.
    c4_Cursor (c4_Sequence&, int);
    
    //@cmember Dereferences this cursor to "almost" a row.
    c4_RowRef operator* () const;
    
    //@cmember This is the same as *(cursor + offset).
    c4_RowRef operator[] (int) const;

//@group Stepping the iterator forwards / backwards
    //@cmember Pre-increments the cursor.
    c4_Cursor& operator++ ();
    //@cmember Post-increments the cursor.
    c4_Cursor operator++ (int);
    //@cmember Pre-decrements the cursor.
    c4_Cursor& operator-- ();
    //@cmember Post-decrements the cursor.
    c4_Cursor operator-- (int);

    //@cmember Advances by a given offset.
    c4_Cursor& operator+= (int);
    //@cmember Backs up by a given offset.
    c4_Cursor& operator-= (int);

    //@cmember Subtracts a specified offset.
    c4_Cursor operator- (int) const;
    //@cmember Returns the distance between two cursors.
    int operator- (c4_Cursor) const;
    
    //@cmember Adds specified offset.
    friend c4_Cursor operator+ (c4_Cursor, int);
    //@cmember Adds specified offset to cursor.
    friend c4_Cursor operator+ (int, c4_Cursor);

//@group Comparing row positions
    //@cmember Returns true if both cursors are equal.
    friend bool operator== (c4_Cursor, c4_Cursor);
    //@cmember Returns true if both cursors are not equal.
    friend bool operator!= (c4_Cursor, c4_Cursor);
    //@cmember True if first cursor is less than second cursor.
    friend bool operator< (c4_Cursor, c4_Cursor);
    //@cmember True if first cursor is greater than second cursor.
    friend bool operator> (c4_Cursor, c4_Cursor);
    //@cmember True if first cursor is less or equal to second cursor.
    friend bool operator<= (c4_Cursor, c4_Cursor);
    //@cmember True if first cursor is greater or equal to second cursor.
    friend bool operator>= (c4_Cursor, c4_Cursor);

//!private:
    c4_Sequence* _seq;
    int _index;
};

/////////////////////////////////////////////////////////////////////////////
//@class Row references can be used on either side of an assignment.

class c4_RowRef
{
public: //@access Public members
//@group General operations
    //@cmember Assigns the value of another row to this one.
    c4_RowRef operator= (const c4_RowRef&);
    //@cmember Returns the cursor associated to this row.
    c4_Cursor operator& () const;
    //@cmember Returns the underlying container view.
    c4_View Container() const;

//@group Comparing row contents
    //@cmember Returns true if the contents of both rows is equal.
    friend bool operator== (c4_RowRef, c4_RowRef);
    //@cmember Returns true if the contents of both rows is not equal.
    friend bool operator!= (c4_RowRef, c4_RowRef);
    //@cmember True if first row is less than second row.
    friend bool operator< (c4_RowRef, c4_RowRef);
    //@cmember True if first row is greater than second row.
    friend bool operator> (c4_RowRef, c4_RowRef);
    //@cmember True if first row is less or equal to second row.
    friend bool operator<= (c4_RowRef, c4_RowRef);
    //@cmember True if first row is greater or equal to second row.
    friend bool operator>= (c4_RowRef, c4_RowRef);
    
protected:
    c4_RowRef (c4_Cursor);
    
    c4_Cursor _cursor;
    
    friend class c4_Cursor;
};

/////////////////////////////////////////////////////////////////////////////
//@class A row is an entry in a sequence with copy semantics. Rows can exist
// by themselves and as the contents of views. Row assignment implies that
// a copy of the contents of the originating row is made.
//@base public | <c c4_RowRef>

class c4_Row : public c4_RowRef 
{
public: //@access Public members
    //@cmember Constructs a row with no properties.
    c4_Row ();
    //@cmember Constructs a row from another one.
    c4_Row (const c4_Row&);
    //@cmember Constructs a row copy from a row reference.
    c4_Row (const c4_RowRef&);
    //@cmember Destructor.
    ~c4_Row ();
    
    //@cmember Assigns a copy of another row to this one.
    c4_Row& operator= (const c4_Row&);
    //@cmember Copies another row to this one.
    c4_Row& operator= (const c4_RowRef&);
    
    //@cmember Adds all properties and values into this row.
    void ConcatRow(const c4_RowRef&);
    //@cmember Returns a new row which is the concatenation of two others.
    friend c4_Row operator+ (const c4_RowRef&, const c4_RowRef&);
};

/////////////////////////////////////////////////////////////////////////////
//@class Objects of this class manage persistent storage of view structures.
//
// The storage class uses a view, with additional functionality to be able 
// to store and reload the data it contains (including nested subviews).
//
// By default, data is loaded on demand, i.e. whenever data which has
// not yet been referenced is used for the first time. Loading is limited
// to the lifetime of this storage object, since the storage object carries
// the file descriptor with it that is needed to access the data file.
//
// To save changes, call the <mf ::Commit> member. This is the only time
// that data is written to file - when using a read-only file simply avoid
// calling <mf ::Commit>.
//
// The <mf ::LoadFromStream> and <mf ::SaveToStream> members can be used to
// serialize the contents of this storage row using only sequential I/O
// (no seeking, only read or write calls).
//
// The data storage mechanism implementation provides fail-safe operation:
// if anything prevents <mf ::Commit> from completing its task, the last
// succesfully committed version of the saved data will be recovered on
// the next open. This includes all changes made to the table structure. 
//
//@ex The following code creates a view with 1 row and stores it on file: |
//  c4_StringProp pName ("Name");
//  c4_IntProp pAge ("Age");
//
//  c4_View myView;
//  myView.Add(pName ["John Williams"] + pAge [43]);
//
//  c4_Storage storage ("myfile.dat", "Musicians[Name:S,Age:I]");
//  storage.View("Musicians") = myView;

class c4_Storage 
{
public: //@access Public members
    //@cmember Constructs a storage object for given file and format.
    c4_Storage (const char*, const char*);
	//@cmember Constructs a storage object, keeping the current structure.
	c4_Storage (const char*, bool canModify_ =false);
    //@cmember Constructs a storage object based on the given file.
    c4_Storage (c4_File* =0); 
    //@cmember Destructor.
    ~c4_Storage ();
    
    //@cmember Gives access to the stored data as a single row.
    c4_RowRef Contents() const;

    //@cmember Returns the file associated with this object.
    c4_File* File() const;
    //@cmember Returns the root table entry.
    c4_Table& RootTable() const;
    //@cmember Returns a description of the data structure.
    c4_String Description() const;
    
    //@cmember Flushes pending changes to file right now.
    bool Commit();
    //@cmember (Re)initializes for on-demand loading.
    // This will cancel any uncommitted changes.
    bool Rollback();
    
    //@cmember Used to get or set a named view in this storage object.
    c4_ViewRef View(const char* name_) const;
    //@cmember (Re-) defines the structure of one of the views.
	void Define(const char* description_);
    //@cmember Retrieves a named view from this storage object.
    c4_View Get(const char* name_) const;
    //@cmember Stores a view under a specified name in this storage object.
    void Set(const char* name_, const c4_View& view_);

    //@cmember Loads contents from the specified input stream.
    void LoadFromStream(c4_File&);
    //@cmember Saves contents to the specified output stream.
    void SaveToStream(c4_File&);

private:
    c4_Storage (const c4_Storage&);     // not implemented
    void operator= (const c4_Storage&); // not implemented

	void Initialize(const char*, const char*, int);
	void SetStructure(const char*);

    c4_Persist* _persist;
    bool _cleanupFile;
	c4_String _structure;
};

/////////////////////////////////////////////////////////////////////////////
//@class Properties correspond to the subfields of data rows.
//
// Property objects exist independently of view, row, and storage objects.
// They have a name and type, and can be used by multiple views.
// You will normally only use derived classes for strong typing.
//@xref <c c4_IntProp>, <c c4_FloatProp>, <c c4_StringProp>, <c c4_ViewProp>

class c4_Property
{
public: //@access Public members
    //@cmember Constructs a new property with the give type and name.
    c4_Property (char, const char*);
    //@cmember Constructs a property from its id.
    c4_Property (int);
    //@cmember Destructor.
    ~c4_Property ();
    
    //@cmember Returns the name of this property.
    c4_String Name() const;
    //@cmember Returns the type of this property.
    char Type() const;

    //@cmember Returns a unique id for this property.
    // Properties with the same type and name always share the same id.
    int GetId() const;

    //@cmember Gets or sets this untyped property in a row.
    c4_Reference operator() (c4_RowRef) const;

    //@cmember Adjusts the reference count.
    void Refs(int) const;

    //@cmember Removes unused property info and deallocates the cache.
    static void CleanupCache();

private:
    c4_Property (const c4_Property&);   // not implemented
    void operator= (const c4_Property&); // not implemented

    int _id;
};

    //@class Class of objects representing integer properties.
    //@base public | <c c4_Property>
class c4_IntProp : public c4_Property 
{
public: //@access Public members
    //@cmember Constructs a new property.
    c4_IntProp (const char*);
    //@cmember Creates a row with a single integer.
    c4_Row operator[] (long);
    //@cmember Gets or sets an integer property of a row.
    c4_IntRef operator() (c4_RowRef);
};

    //@class Class of objects representing floating point properties.
    //@base public | <c c4_Property>
class c4_FloatProp : public c4_Property 
{
public: //@access Public members
    //@cmember Constructs a new property.
    c4_FloatProp (const char*);
    //@cmember Creates a row with a single floating point value.
    c4_Row operator[] (double);
    //@cmember Gets or sets an floating point property of a row.
    c4_FloatRef operator() (c4_RowRef);
};

    //@class Class of objects representing string properties.
    //@base public | <c c4_Property>
class c4_StringProp : public c4_Property
{
public: //@access Public members
    //@cmember Constructs a new property.
    c4_StringProp (const char*);
    //@cmember Create a row with a single string.
    c4_Row operator[] (const char*);
    //@cmember Gets or sets a string property of a row.
    c4_StringRef operator() (c4_RowRef);
};

    //@class Class of objects representing view properties.
    //@base public | <c c4_Property>
class c4_ViewProp : public c4_Property
{
public: //@access Public members
    //@cmember Constructs a new property.
    c4_ViewProp (const char*);
    //@cmember Create a row with a single view.
    c4_Row operator[] (const c4_View&);
    //@cmember Gets or sets a view property of a row.
    c4_ViewRef operator() (c4_RowRef);
};

/////////////////////////////////////////////////////////////////////////////

#include "k4viewx.h"

#if q4_INLINE
    #include "k4view.inl"
#endif

/////////////////////////////////////////////////////////////////////////////

#endif
