#!/usr/bin/perl
##
## vi:ts=4
##
##---------------------------------------------------------------------------##
##    
##  Author:
##      Markus F.X.J. Oberhumer         markus.oberhumer@jk.uni-linz.ac.at
##    
##  Description:
##      Convert the output of the LZO test program into a nice table
##    
##  Copyright (C) 1996 Markus Franz Xaver Johannes Oberhumer
##    
##---------------------------------------------------------------------------##

require 'ctime.pl';


while ($_ = $ARGV[ $[ ], /^-/) {
	shift(@ARGV);
	/^--$/ && ($opt_last = 1, last);

	/^-D(.*)/ && ($opt_debug = $1 ? $1 : 1);
	/^-n/ && $opt_sort_summary_by_name++;
	/^-r/ && $opt_sort_summary_by_ratio++;
	/^-t/ && $opt_clear_time++;
}


$alg = '';
$sep = "+--------------------------------------------------------------------+\n";

@algs = ();
%average = ();
%total = ();
$block_size = -1;


while (<>) {

	if (/\s+(\d+)\s+block\-size/) {
		if ($block_size < 0) {
			$block_size = $1;
			&start($block_size);
		} elsif ($block_size != $1) {
			die "$0: block-size: $block_size $1\n";
		}
	}

	if (/^\s*(\S+)\s*(\|.*\|)\s*$/i) {
		if ($1 ne $alg) {
			&footer($1);
			&header($1);
		}
		$line = $2;
		&stats(*line);
		print "$line\n";
	}
}
&footer($1);

&summary();
exit(0);


##---------------------------------------------------------------------------##

sub stats {
	local (*l) = @_;

	if ($opt_clear_time) {
		substr($l,50) = '     0.00     0.00 |';
	}

	if ($l !~ /^\|\s*([\w\~\!\-\.\+]+)\s+(\d+)\s+(\d+)\s+(\d+)\s+([\d\.]+)\s+([\d\.]+)\s+([\d\.]+)\s+([\d\.]+)\s*\|/) {
		die;
	}

	$n++;
	$s[0] += $2;
	$s[1] += $3;
	$s[2] += $4;
	$s[3] += $5;
	$s[4] += $6;
	$s[5] += $7;
	$s[6] += $8;
}


##---------------------------------------------------------------------------##

sub header {
	local ($t) = @_;

	$alg = $t;

	# reset stats
	$n = 0;
	@s = (0, 0, 0, 0.0, 0.0, 0.0, 0.0);

	print "\n$alg\n\n";
	print $sep;
print <<Header;
| File Name       Length  CxB  ComLen  %Remn  Bits  Com K/s  Dec K/s |
| ---------       ------  ---  ------  -----  ----  -------  ------- |
Header
}


##---------------------------------------------------------------------------##

sub footer {
	local ($t) = @_;

	return unless $alg;
	die if $n <= 0;
	die if $s[0] <= 0;

	push(@algs,$alg);

	$average{$alg} = 
		sprintf("| %-14s %7d %4d %7d %6.1f %5.2f %8.2f %8.2f |\n",
			"Average", $s[0]/$n, $s[1]/$n, $s[2]/$n,
			$s[3]/$n, $s[4]/$n,
			$s[5]/$n, $s[6]/$n );

	$total{$alg} = 
		sprintf("| %-14s %7d %4d %7d %6.1f %5.2f %8.2f %8.2f |\n",
			"Total", $s[0], $s[1], $s[2],
			$s[2]/$s[0]*100, $s[2]/$s[0]*8,
			$s[5]/$n, $s[6]/$n );

	print $sep;
	print $average{$alg};
	print $total{$alg};
	print $sep, "\n";
}

##---------------------------------------------------------------------------##

$sort_mode = 0;

sub cmp_by_ratio {
	local ($aa, $bb);

	if ($sort_mode == 0) {
		$aa = $average{$a};
		$bb = $average{$b};
	} elsif ($sort_mode == 1) {
		$aa = $total{$a};
		$bb = $total{$b};
	} else {
		die;
	}

	($aa =~ m%^\s*\|\s+\S+\s+\d+\s+\d+\s+\d+\s+(\S+)%) || die;
	$aa = $1;
	($bb =~ m%^\s*\|\s+\S+\s+\d+\s+\d+\s+\d+\s+(\S+)%) || die;
	$bb = $1;

	# $aa < $bb;
	$aa cmp $bb;
}


sub summary {
	local ($l);
	local (@k);

	$sort_mode = 0;
	if ($opt_sort_summary_by_name) {
		@k = sort(@algs);
	} elsif ($opt_sort_summary_by_ratio) {
		@k = sort(cmp_by_ratio @algs);
	} else {
		@k = @algs;
	}

	print "\n\n";
	print "Summary of average values\n\n";
	print $sep;
print <<Summary;
| Algorithm       Length  CxB  ComLen  %Remn  Bits  Com K/s  Dec K/s |
| ---------       ------  ---  ------  -----  ----  -------  ------- |
Summary

	for (@k) {
		$l = $average{$_};
		$l =~ s/Average[\s]{7}/sprintf("%-14s",$_)/e;
		print $l;
	}
	print $sep;



	$sort_mode = 1;
	if ($opt_sort_summary_by_name) {
		@k = sort(@algs);
	} elsif ($opt_sort_summary_by_ratio) {
		@k = sort(cmp_by_ratio @algs);
	} else {
		@k = @algs;
	}

	print "\n\n";
	print "Summary of total values\n\n";
	print $sep;
print <<Summary;
| Algorithm       Length  CxB  ComLen  %Remn  Bits  Com K/s  Dec K/s |
| ---------       ------  ---  ------  -----  ----  -------  ------- |
Summary

	for (@k) {
		$l = $total{$_};
		$l =~ s/Total[\s]{9}/sprintf("%-14s",$_)/e;
		print $l;
	}
	print $sep;


print <<Summary;


Notes:
- CxB is the number of blocks 
- K/s is the speed measured in 1000 uncompressed bytes per second
- all averages are calculated from the un-rounded values

Summary
}


##---------------------------------------------------------------------------##

sub start {
	local ($bs) = @_;
	local ($t, $x);

	$t = &ctime(time); chop($t);
	$t = sprintf("%-47s |", $t);

	$x = sprintf("%d (= %d kB)", $bs, $bs / 1024);
	$x = sprintf("%-47s |", $x);

print <<Start;

+--------------------------------------------------------------------+
| DATA COMPRESSION TEST                                              |
| =====================                                              | 
| Time of run      : $t
| Context length   : $x
| Timing accuracy  : One part in 100                                 |
+--------------------------------------------------------------------+


Start
}

__END__


# insert something like this after 'Time of run':

| Hardware         : Intel 80486 DX2/66, 20 MB RAM, 256 kB Cache     |
| Operating system : MSDOS 5.0, QEMM 7.04, CWSDPMI 0.90+ (r2)        |
| Compiler         : djgpp v2 + gcc 2.7.2                            |
| Compiler flags   : -O2 -fforce-addr -fomit-frame-pointer -m486     |
| Test suite       : Calgary Corpus Suite                            |
| Files in suite   : 14                                              |


