/* lzo_util.h -- utilities for the the LZO library

   This file is part of the LZO real-time data compression library.

   Copyright (C) 1996 Markus Franz Xaver Johannes Oberhumer

   The LZO library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   The LZO library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the LZO library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation, Inc.,
   675 Mass Ave, Cambridge, MA 02139, USA.

   Markus F.X.J. Oberhumer
   markus.oberhumer@jk.uni-linz.ac.at
 */


/* WARNING: this file should *not* be used by applications. It is
   part of the implementation of the library and is subject
   to change.
 */


#ifndef __LZO_UTIL_H
#define __LZO_UTIL_H

#ifdef __cplusplus
extern "C" {
#endif


/***********************************************************************
// some prototypes
************************************************************************/

extern const lzo_byte __lzo_copyright[];


/***********************************************************************
// fast memcpy that copies multiples of 8 byte chunks.
// len is the number of bytes.
// note: all parameters must be lvalues, len > 0
//       dest and src advance, len is undefined afterwards
************************************************************************/

#if 1 && !defined(__LZO_MSDOS16)

#define MEMCPY8_DS(dest,src,len) \
	memcpy(dest,src,len); \
	dest += len; \
	src += len

#else

#define MEMCPY8_DS(dest,src,len) \
	{ register lzo_uint __l = (len) / 8; \
	do { \
		*dest++ = *src++; \
		*dest++ = *src++; \
		*dest++ = *src++; \
		*dest++ = *src++; \
		*dest++ = *src++; \
		*dest++ = *src++; \
		*dest++ = *src++; \
		*dest++ = *src++; \
	} while (--__l > 0); }

#endif



/***********************************************************************
// memcpy and memmove
// len is the number of bytes.
// note: all parameters must be lvalues, len > 0
//       dest and src advance, len is undefined afterwards
************************************************************************/

#if defined(LZO_ALWAYS_USE_MEMCPY)

#define MEMCPY_DS(dest,src,len) \
	memcpy(dest,src,len); \
	dest += len; \
	src += len;

#else

#define MEMCPY_DS(dest,src,len) \
	do *dest++ = *src++; \
	while (--len > 0)

#endif


#define MEMMOVE_DS(dest,src,len) \
	do *dest++ = *src++; \
	while (--len > 0)



/***********************************************************************
// fast bzero that clears multiples of 8 pointers
// n is the number of pointers.
// note: n > 0
//       s and n are undefined afterwards
************************************************************************/

#if defined(LZO_OPTIMIZE_GNUC_i386)

#define BZERO8_PTR(s,n) \
__asm__ __volatile__( \
	"movl  %0,%%eax \n"             \
	"movl  %1,%%edi \n"             \
	"movl  %2,%%ecx \n"             \
	"cld \n"                        \
	"rep \n"                        \
	"stosl %%eax,(%%edi) \n"        \
	: /* no outputs */              \
	:"g" (0),"g" (s),"g" (n)	  	\
	:"eax","edi","ecx", "cc"        \
)

#elif (LZO_UINT_MAX <= UINT_MAX)

#define BZERO8_PTR(s,n) 		memset(s,0,(n)*sizeof(lzo_byte *))

#else

#define BZERO8_PTR(s,n) 		lzo_memset(s,0,(n)*lzo_sizeof(lzo_byte *))

#endif


/***********************************************************************
// rotate (not used at the moment)
************************************************************************/

#if defined(__GNUC__) && defined(__i386__)

unsigned char lzo_rotr8(unsigned char value, int shift);
extern __inline__ unsigned char lzo_rotr8(unsigned char value, int shift)
{
	unsigned char result;

	__asm__ __volatile__ ("movb %b1, %b0; rorb %b2, %b0"
                      	: "=a"(result) : "g"(value), "c"(shift));
	return result;
}

unsigned short lzo_rotr16(unsigned short value, int shift);
extern __inline__ unsigned short lzo_rotr16(unsigned short value, int shift)
{
	unsigned short result;

	__asm__ __volatile__ ("movw %b1, %b0; rorw %b2, %b0"
                      	: "=a"(result) : "g"(value), "c"(shift));
	return result;
}

#endif



#ifdef __cplusplus
} /* extern "C" */
#endif

#endif /* already included */

/*
vi:ts=4
*/
