// File ImageButton.java jlouie 6.97
// Based on FancyButtonBean from "Presenting Java Beans" by Michael Morrison
// SAMS Net and GrayFilter from "Java in a Nutshell" David Flanagan O'Reilly
// A JDK1.1/BDK1.0 release version of Michael Morrisons FancyButtonBean
// that supports a gif image.
// Compile GrayFilter.java and then ImageButton.java and ImageButtonBeanInfo.java
// using: javac -d . .\ImageButton\Source\GrayFilter.java
// javac -d . .\ImageButton\Source\ImageButton.java
// javac -d . .\ImageButton\Source\ImageButtonBeanInfo.java
// Test the component using: java jlouie.beans.ImageButton
// Jar the bean using: 
// jar cfm .\imagebutton.jar jlouie\beans\ImageButton.mf 
// jlouie\beans\*.class jlouie\beans\*.gif
// Where default.gif is in the folder jlouie\beans along with the compiled class files,
// the manifest file and the BeanInfo file.
// Place the jar file in the BDK jars folder
// Test the component with the BeanBox using: c:\bdk\beanbox>run
// Add the ImageButton to the form.
// Set the isSticky property to true!
// Have fun with Java.

package jlouie.beans;

import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;
import java.beans.*;
import java.io.*;
import java.net.*;

public class ImageButton extends Canvas implements Serializable {
      
      private final static String DEFAULT_FILE= "default.gif";
      private final static Dimension DEFAULT_DIMENSION= new Dimension(30,10);

      private  Image image= null;
      private  Image gray= null;
      private  boolean isSticky= false;
      private  boolean isDown= false;
      private  transient boolean isFocus= false;
      private  String imageName= DEFAULT_FILE;
      private  transient ActionListener actionListener= null;

      public static void main(String[] args) {
             Frame f= new Frame();
             f.addNotify();
             f.setLayout(new GridLayout(1,2));
             ImageButton ib= new ImageButton(DEFAULT_FILE,true);
             f.add(ib);
             f.add(new ImageButton());
             f.setSize(200,200);
             f.show();
      }
           
      public ImageButton(String imageName, boolean isSticky) {
            this.isSticky= isSticky;
            image= loadImage(imageName);
            sizeToFit();
            ImageFilter filter= new GrayFilter();
            ImageProducer producer= new FilteredImageSource(image.getSource(), filter);
            gray= this.createImage(producer);
            enableEvents(AWTEvent.FOCUS_EVENT_MASK | AWTEvent.MOUSE_EVENT_MASK |
                  AWTEvent.MOUSE_MOTION_EVENT_MASK | AWTEvent.KEY_EVENT_MASK);
      }
      
      public ImageButton(String imageName) {
            this(imageName, false);
      }
      
      public ImageButton() {
            this(DEFAULT_FILE, false);     
      }
      
      private void sizeToFit() {
            Dimension d= getPreferredSize();
            setSize(d.width, d.height);
            Component p= getParent();
            if (p != null) {
                  p.invalidate();
                  p.doLayout();
            }
      }
      
      public String getImageName() {
            return imageName;
      }
      
      public void setImageName(String imageName) {
            if (imageName != null) {
                  this.imageName= imageName;
                  image= loadImage(imageName);
                  sizeToFit();
            }
      }
      
      public boolean isSticky() {
         return isSticky;
      }
      
      public void setSticky(boolean isSticky) {
         this.isSticky= isSticky;
      }
      
      public boolean isFocus() {
         return isFocus;
      }
      
      public boolean isDown() {     // returns false is not a sticky button
         return (isSticky && isDown);
      }
      
      public synchronized void AddActionListener(ActionListener l) {
            actionListener= AWTEventMulticaster.add(actionListener, l);
      }
      
      public synchronized void removeActionListener(ActionListener l) {
            actionListener= AWTEventMulticaster.remove(actionListener, l);
      }
      
      protected void processActionEvent(ActionEvent e) {
            if (actionListener != null) {
                  actionListener.actionPerformed(e);
            }
      }
      
      protected void processFocusEvent(FocusEvent e) {
            switch(e.getID()) {
                  case FocusEvent.FOCUS_GAINED:
                        isFocus = true;
                        update(getGraphics());
                        break;
                  case FocusEvent.FOCUS_LOST:
                        isFocus = false;
                        update(getGraphics());
                        break;    
            }
            super.processFocusEvent(e);
      }
      
      protected void processMouseEvent(MouseEvent e) {
            switch(e.getID()) {
                  case MouseEvent.MOUSE_PRESSED:
                       isDown= !isDown;
                       update(getGraphics());
                       break;
                  case MouseEvent.MOUSE_RELEASED:
                       if (isDown && !isSticky) {
                          processActionEvent(new ActionEvent(this,
                              ActionEvent.ACTION_PERFORMED, null));
                          isDown= false;
                          update(getGraphics());
                       }
            }
            super.processMouseEvent(e);
      }
      
      protected void processKeyEvent(KeyEvent e) {
            if ((e.getKeyCode() == KeyEvent.VK_ENTER) ||
                   (e.getKeyChar() == KeyEvent.VK_SPACE)) {
                if (isSticky) {
                   isDown= !isDown;
                   update(getGraphics());
                }
                else {
                    isDown= true;
                    update(getGraphics());
                    processActionEvent(new ActionEvent(this,
                        ActionEvent.ACTION_PERFORMED, null));
                    isDown= false;
                    update(getGraphics());
                }        
            }
            super.processKeyEvent(e);
      }
      
      protected void processMouseMotionEvent(MouseEvent e) {
            if (e.getID() == MouseEvent.MOUSE_DRAGGED && !isSticky) {
                  Point p= e.getPoint();
                  if ((p.x<0 || p.x>getSize().width) ||
                        (p.y<0 || p.y>getSize().height)) {
                        if (isDown) {
                              isDown= false;
                              update(getGraphics());
                        }     
                  }
                  else if (!isDown) {
                     isDown= true;
                     update(getGraphics());
                  }
            }
            super.processMouseMotionEvent(e);
      }
      
      public Image loadImage(String name) {
         try {
            URL url = this.getClass().getResource(name);
            Object o= url.getContent();
            if (o instanceof ImageProducer) {
               return createImage((ImageProducer)o);
            }
            else return null;
         }                
         catch (Exception e) {
            System.out.println(e.toString());
            return null;
         }
      }  
      
      public Dimension preferredSize() {
            if (image != null) {
                  return new Dimension(image.getWidth(null), image.getHeight(null));
            }
            else return DEFAULT_DIMENSION;
      }
      
      public void update(Graphics g) {
            paint(g);
      }
   
      public synchronized void paint(Graphics g) {
            Dimension d= preferredSize();
            if ((image != null) && (gray != null)) {
                  if (isDown) {
                     g.drawImage(gray,0,0,this);
                  }
                  else g.drawImage(image,0,0,this);
            }
            else g.drawString("ImageButton",10,10);
            if (isFocus) {
               g.setColor(Color.black);
            }
            else g.setColor(Color.darkGray);
            g.draw3DRect(0,0,d.width-1,d.height-1,!isDown);
      }
      
}     

 

