/****************************************************************************
	FILE:		writelib.c


	CONTENTS:	ChooseDest() 		window procedure for "Send Box" window.
				ChooseTo()			window procedure for "Choose Box" window.
				ExpandNames()		utility for expanding a addressbook entries
				SaveAttachInfo()	saves info from attach edit control
				SaveAddressInfo()	saves info from address edit controls
				SaveBodyInfo()		saves info from message body edit control
				SaveEnvelopeInfo()	saves info from various envelope edit controls
				SaveMessage()		Uses above "save..." functions to convert the
									edit control contents to a Banyan Mail Message.
				StoreNames()		utility for storing the Names to a Banyan Mail
									Envelope (called by SaveAddressInfo)


	COMMENTS:	This file contains the window procedures and other utilities
				used in composing a mail message.
****************************************************************************/
#include "MyMail.h"
#include "io.h"
#include "string.h"
#include "dos.h"
#include "stdlib.h"


#define INCL_STDA_UI
#define INCL_STDA
#include "vnsstda.h"
#undef  INCL_STDA_UI
#undef  INCL_STDA


/*** fmode is used in opening DOS files. ***/
#define fmode OF_READ|OF_SHARE_DENY_WRITE


/****************************************************************************
	FUNCTION:	ChooseDest()

	PURPOSE:	Processes messages for "Send Box" child window.  This
				dialog box presents the destination choices: Send & Save,
				Send & Delete, & Just Save.
****************************************************************************/
BOOL FAR PASCAL _export ChooseDest(HWND hDlg, unsigned message, WORD wParam, LONG lParam)
{
	long dest;

	/*** Send Window Messages to Status Bar ***/
	StatBarCtrlMsg(hDlg, SendBox, message, wParam, lParam);

	switch (message) {
		case WM_INITDIALOG:
			/*** Center the Dialog Window in the HappyMail Main window ***/
			CenterWindow(hDlg, GetDesktopWindow());

			/*** Add a check mark to the default destination ***/
			if ((lParam == IDM_FORWARD) && (Options.defaultdest == 3))
				CheckRadioButton(hDlg, SND_SD, SND_JS, SND_SS);
			else
				CheckRadioButton(hDlg, SND_SD, SND_JS, (SND_SD - 1) + Options.defaultdest);
			return (TRUE);

		case WM_COMMAND:
			switch(wParam) {
				case IDM_OK:
					/*** User selected OK -- set the return value ***/
                    /*** based on which button was checked.       ***/
					if (IsDlgButtonChecked(hDlg, SND_SD))
						dest = 1L;
					if (IsDlgButtonChecked(hDlg, SND_SS))
						dest = 2L;
					if (IsDlgButtonChecked(hDlg, SND_JS))
						dest = 3L;

					/*** No break intentionally. ***/

				case IDCANCEL:
					/*** User selected CANCEL. ***/
					EndDialog(hDlg, (wParam == IDM_OK) ? dest : 0L);
					return (TRUE);
			} /** end switch(wParam) **/
	} /** end switch(message) **/
	return (FALSE);
}

/****************************************************************************
	FUNCTION:	ChooseTo()

	PURPOSE:	Processes messages for "Choose Box" child window.  This
				dialog box presents the destination choices: TO:, CC:, & BCC:
****************************************************************************/
BOOL FAR PASCAL _export ChooseTo(HWND hDlg, unsigned message, WORD wParam, LONG lParam)
{
	long dest;

	/*** Send Window Messages to Status Bar ***/
	StatBarCtrlMsg(hDlg, ChooseBox, message, wParam, lParam);

	switch (message) {
		case WM_INITDIALOG:
			/*** Center the Dialog Window in the HappyMail Main window ***/
			CenterWindow(hDlg, GetDesktopWindow());

			/*** Add a check mark to the first ***/
			/*** destination choice -- TO:     ***/
			if (LOWORD(lParam) == CB_CC)
				CheckRadioButton(hDlg, CHB_TO, CHB_BCC, CHB_CC);
			else if (LOWORD(lParam) == CB_BCC)
				CheckRadioButton(hDlg, CHB_TO, CHB_BCC, CHB_BCC);
			else
				CheckRadioButton(hDlg, CHB_TO, CHB_BCC, CHB_TO);
			return (TRUE);

		case WM_COMMAND:
			switch(wParam) {
				case IDM_OK:
					/*** User selected OK -- set the return value ***/
					/*** based on which button was checked.       ***/
					if (IsDlgButtonChecked(hDlg, CHB_TO) || IsDlgButtonChecked(hDlg, SND_SS))
						dest = 1L;
					if (IsDlgButtonChecked(hDlg, CHB_CC))
						dest = 2L;
					if (IsDlgButtonChecked(hDlg, CHB_BCC))
						dest = 3L;

					/*** No break intentionally. ***/

				case IDCANCEL:
                    /*** User selected CANCEL. ***/
					EndDialog(hDlg, (wParam == IDM_OK) ? dest : 0L);
					return (TRUE);
			} /** end switch(wParam) **/
	} /** end switch(message) **/
	return (FALSE);
}



/******************************************************************************
	FUNCTION:	ExpandNames()

	PURPOSE:  	A list of recipient names is examined and Addressbook labels are
				replaced with their corresponding addresses.
*******************************************************************************/
HANDLE ExpandNames(char far *input, int level)
{
	char 			tempname[512], tempstr[512], tail[512];
	char 			far *tempptr, far *newptr, far *start, far *strptr;
	GLOBALHANDLE   	hNames, hNewNames;
	int             Size;
	unsigned short 	valid, i;
	BOOL			NeedsExpanding;
	HVNSD			hVNSD;
	unsigned short 	STDA_Array[]={STDA_USERS, STDA_NICKNAMES,
					STDA_LISTS, STDA_FILES, STDA_PRINTERS, STDA_SERVICES};


	/*** Check for a high level indicating a recursive call. ***/
	if (level > 6) {
		BWCCMessageBox(hWnd, "Error interpreting addresses!\nCheck for looping address book entries.", "WARNING!", MB_ICONSTOP | MB_OK);
		return 0;
	}

	Size = MAXNAMESLEN;
	tempptr = (char far *) input;

	/*** Create a new buffer for the expanded list of names. ***/
	if ((hNames = SafeAlloc(GLOBAL, MAXNAMESLEN, hWnd)) == NULL) {
		return NULL;
	}
	newptr = (char far *) GlobalLock(hNames);

	/*** Temporary buffers set to null. ***/
	NullFill((LPSTR) &tempname, 512);
	tail[0] ='\0';

	/*** Pull one name from the list at a time and assign it to tempname. ***/
	while(VnsParseMailRecipient((LPSTR) tempptr, (LPSTR) &tempname, (LPSTR far *) &(tempptr)) != MS_NONAMEADDR) {

		if (tempname[0] != '\0') {
			/*** If the name is followed by bracketed info, ***/
			/*** peel it off and save it for later use.     ***/
			for (i=0; (tempname[i] != '\0') && (tempname[i] != '[') && (i< 512); i++);
			if (tempname[i] == '[') {
				lstrcpy((LPSTR) &tail, (LPSTR) &tempname+i);
				tempname[i] = '\0';
			}

			/*** Remove any trailing blanks from name. ***/
			for (i=lstrlen((LPSTR) &tempname)-1; (tempname[i] == 32) && (i > 0); i--)
				tempname[i] = '\0';

			/*** Set some default values. ***/
			NeedsExpanding = FALSE;
			strptr = (char far *) tempname;

			/*** InAddressBook() will convert an address   ***/
			/*** label to its corresponding list of names. ***/
			if (InAddressBook(tempname, tempname, TRUE))
				NeedsExpanding = TRUE;
			else
				if ( hSTDA) {
					tempstr[0] = '\0';
					/*** Check each of the 6 STDA classes to determine if the  ***/
					/*** address is an inclusion record.  If so, copy the mail ***/
					/*** address that corresponds to the inclusion record.     ***/
					for (valid = 0,i=0; i<6 && valid == 0; i++)
						VnsVerifyStdaName(hSTDA, (LPSTR) &tempname, STDA_Array[i], &valid, (LPSTR) &tempstr);
					if ((valid) && (tempstr[0] != '\0')) {
						lstrcpy((LPSTR) &tempname, (LPSTR) &tempstr);
						NeedsExpanding = TRUE;
					}
				}

			hNewNames = 0;
			if (NeedsExpanding) {
				lstrcpy((LPSTR) &tempstr, (LPSTR) &tempname);
				hNewNames = ExpandNames((LPSTR) &tempname, level + 1);
				if (hNewNames != 0)
					strptr = (char far *) GlobalLock(hNewNames);
				else  {
					SafeFree(GLOBAL, hNames);
					return 0;
				}
			}


			/*** Reattatch any tail info before storing name. ***/
			lstrcat((LPSTR) &tempname, (LPSTR) &tail);
			tail[0] = '\0';

			/*** Increase the buffer size if necessary. ***/
			if ((lstrlen(newptr) + lstrlen(strptr))> Size) {
				Size += lstrlen(strptr) + MAXNAMESLEN;
				if ((hNames = GlobalReAlloc(hNames, Size, GMEM_ZEROINIT))==NULL) {
					BWCCMessageBox(hWnd, "Unable to allocate enough memory!", "WARNING!", MB_ICONSTOP | MB_OK);
					return NULL;
				}
			}

			/*** Copy the name to our new list. ***/
			if (lstrlen(newptr) > 0) lstrcat(newptr, (LPSTR) ", ");
			lstrcat(newptr, strptr);

			/*** Release memory for recursive call. ***/
			if (hNewNames > 0)
				SafeFree(GLOBAL, hNewNames);

		}
		NullFill((LPSTR) &tempname, 512);
	}
	GlobalUnlock(hNames);
	return hNames;
}



/******************************************************************************
	FUNCTION:	SaveAddressInfo()

	PURPOSE:  	To store the recipient addresses
*******************************************************************************/
BOOL SaveAddressInfo(HWND hDlg, WORD menuitem, char far *MsgID)
{

	char    result[1064];
	char 	rclass, oldclass;

	/*** Store the new recipient lists (i.e. TO, CC, BCC). ***/
	/*** Update the Status Bar. ***/
	if (hStatus)
		SendMessage(hStatus, STB_STRING, 0, (LONG)"Storing Names ...");

	/*** Spin through each recipient class, delete the   ***/
	/*** old names if necessary, and save the new names. ***/
	rclass = RECIPIENTTO;
	do {
		/*** Delete one recipient at a time till none  ***/
		/*** are left for the current recipient class. ***/
		if (menuitem == IDM_EDIT) {
			VnsGetMailRecipient(hVNMAS, MsgID, rclass, (LPSTR) &result, 0);
			while (result[0] != '\0') {
				VnsDeleteMailRecipient(hVNMAS, MsgID, rclass, (LPSTR) &result);
				VnsGetMailRecipient(hVNMAS, MsgID, rclass, (LPSTR) &result, 0);
			}
		}

		/*** Alternate through all the recipient classes. ***/
		NullFill((LPSTR) &result, 1064);
		oldclass = rclass;
		switch (rclass) {
			case RECIPIENTTO:
				SendDlgItemMessage(hDlg, CB_TO, WM_GETTEXT, MAXNAMESLEN+64, (long) &result);
				rclass = RECIPIENTCC;
				break;
			case RECIPIENTCC:
				SendDlgItemMessage(hDlg, CB_CC, WM_GETTEXT, MAXNAMESLEN+64, (long) &result);
				rclass = RECIPIENTBCC;
				break;
			case RECIPIENTBCC:
				SendDlgItemMessage(hDlg, CB_BCC, WM_GETTEXT, MAXNAMESLEN+64, (long) &result);
				rclass = RECIPIENTTO;
				break;
		}
		if (result[0] != 0)
			StoreNames((LPSTR) &result, oldclass, MsgID);
	} while (rclass != RECIPIENTTO);

	return;
}


/******************************************************************************
	FUNCTION:	SaveAttachInfo()

	PURPOSE:  	To store the attachment attributes and bodypart structure
*******************************************************************************/
BOOL SaveAttachInfo(HWND hDlg, BodyPart far *Body, char far *FoldName, WORD menuitem, char far *MsgID, char far *OldMsgID)
{
	char			buff[129], result[1064];
	CallStatus		uStat;
	long			lTemp;
	int 			i, fhandle, retcount, bytes, offset;
	unsigned int	date, time, attr;


	/*** Update status bar message ***/
	if (hStatus)
		SendMessage(hStatus, STB_STRING, 0, (LONG)"Storing Attachments...");

	if (menuitem == IDM_FORWARD) {
		/*** Retrieve Body part info from the old  ***/
		/*** message and copy to the new message. ***/
		NullFill((char far *) Body, sizeof(BodyPart) * 11);
		uStat =  VnsGetMailBodyPartList(hVNMAS, OldMsgID, Body, (unsigned short far *)&retcount);
		if (Recover(uStat))
			uStat =  VnsGetMailBodyPartList(hVNMAS, OldMsgID, Body, (unsigned short far *)&retcount);

		/*** Some messages seem to lose the bodypartId field for . ***/
		/*** attachments and Banyan chokes if this field is empty. ***/
		for(i=0; i<retcount;i++) Body[i].bodypartId=MAINBODY+i;
	}
	else {
		/*** Store the Attachments' attributes and sizes.  These attibutes are ***/
		/*** used only by HappyMail. Banyan mail will ignore these fields.    ***/
		for (i=1; i <= 10 && Body[i].present == PRESENT; i++){
			if ( (fhandle = (_lopen((char far *)Body[i].label, fmode)) ) == HFILE_ERROR){
				DspMsgCode( hWnd, "Error Opening an Attached File", 0, FALSE);
			}
			else {
				/*** Use the contentAttributes field to store 	***/
				/*** the file Size date and attributes 			***/
				lTemp = fsize(fhandle);
				Body[i].contentAttributes[0] = (unsigned char) ((lTemp >> 24) & 0x000000FF);
				Body[i].contentAttributes[1] = (unsigned char) ((lTemp >> 16) & 0x000000FF);
				Body[i].contentAttributes[2] = (unsigned char) ((lTemp >>  8) & 0x000000FF);
				Body[i].contentAttributes[3] = (unsigned char) (lTemp & 0x000000FF);
				_dos_getftime(fhandle, (unsigned int *) &date, (unsigned int *) &time);
				Body[i].contentAttributes[4] = (unsigned char) ((date >> 8) & 0x00FF);
				Body[i].contentAttributes[5] = (unsigned char) (date & 0x00FF);
				Body[i].contentAttributes[6] = (unsigned char) ((time >>  8) & 0x00FF);
				Body[i].contentAttributes[7] = (unsigned char) (time & 0x00FF);
				lstrcpy((LPSTR) &buff, (LPSTR) &Body[i].label);
				_dos_getfileattr((char *) &buff, (unsigned int *) &attr);
				Body[i].contentAttributes[8] = (unsigned char) ((attr >> 8) & 0x00FF);
				Body[i].contentAttributes[9] = (unsigned char) (attr & 0x00FF);
				_lclose(fhandle);
			}
		}
	}


	/**** Store the filled Body Parts List *****/
	i = (i>10) ? 10 : i;
	uStat = VnsStoreMailBodyPartList(hVNMAS, MsgID, Body, i);
	if (Recover(uStat))
		uStat = VnsStoreMailBodyPartList(hVNMAS, MsgID, Body, i);
	if (uStat) {
		DspMsgCode( hDlg, "Error Storing Mail Message : ", uStat, FALSE);
		VnsReleaseMailEnvelope(hVNMAS, MsgID);
		if (menuitem==IDM_FORWARD) VnsReleaseMailEnvelope(hVNMAS, OldMsgID);
		return (FALSE);
	}


	/*** Store the Attachments. ***/
	for (i=1; i <= 10 && Body[i].present == PRESENT; i++){
		wsprintf((LPSTR) &result,"Storing Attachment #%d ...", (int) i);
		if (hStatus)
			SendMessage(hStatus, STB_STRING, 0, (LONG) &result);

		/*** If we're NOT forwarding open the file for reading. ***/
		if (menuitem != IDM_FORWARD) {
			if ( (fhandle = (_lopen((char far *)Body[i].label, fmode)) ) == HFILE_ERROR){
				DspMsgCode( hWnd, "Error Opening an Attached File", 0, FALSE);
				continue;
			}
		}
		offset=0;
		do {
			if (menuitem == IDM_FORWARD) {
				if (VnsGetMailBodyPart(hVNMAS, UserName, FoldName, OldMsgID, i, (LPSTR) result, offset, (unsigned short WFAR *) &bytes)!=0)
					bytes = 0;
			}
			else
				bytes = _lread(fhandle, &result, 1024);
			if(bytes > 0) {
				uStat=VnsStoreMailBodyPart(hVNMAS, (LPSTR) &UserName, FoldName, MsgID, i, (LPSTR) &result, bytes, offset);
				if (Recover(uStat))
					uStat=VnsStoreMailBodyPart(hVNMAS, (LPSTR) &UserName, FoldName, MsgID, i, (LPSTR) &result, bytes, offset);
				if (uStat)
					DspMsgCode( hWnd, "Error Storing an Attached File", 0, FALSE);
			}
			offset++;
		} while(bytes==1024);
		if (menuitem != IDM_FORWARD)
			_lclose(fhandle);
	}

	return TRUE;
}


/******************************************************************************
	FUNCTION:	SaveBodyInfo()

	PURPOSE:  	To store the Message Body of a Compose Box to a Banyan Message.
*******************************************************************************/
BOOL SaveBodyInfo(HWND hDlg, WORD menuitem, char far *FoldName, char far *MsgID, char far *OldMsgID)
{
	char 			result[1064], far *textstr, far *pText;
	int 			i, offset, size, dx, dy, retcount, index;
	unsigned int	tabset[1], bytes;
	GLOBALHANDLE	hBody;
	DWORD			tSize, textSize;
	CallStatus		uStat;
	HFONT			hFont, hSysFont;
	RECT			Rect;
	Env				TempEnv;
	GLOBALHANDLE	hMessage;
	HDC				hDC;

	/*** Update the Status Bar. ***/
	if (hStatus)
		SendMessage(hStatus, STB_STRING, 0, (LONG)"Storing Message ...");

	/*** In order for a message to appear correctly formatted in a Banyan mail    ***/
	/*** client, each line of the message should be terminated with a CR-LF pair  ***/
	/*** or a single LF.  Also each line should be approx. 80 characters or less. ***/
	/*** To simplify the re-formatting of the message to these parameters, the    ***/
	/*** edit control is modified by installing a fixed font, adding tab settings,***/
	/*** and resizing the control to Options.linelength chars wide.                 ***/

	/*** Change Font. ***/
	hFont = (HFONT) SendDlgItemMessage(hDlg, CB_MBODY, WM_GETFONT, 0, 0L);
	hSysFont = GetStockObject(ANSI_FIXED_FONT);
	SendDlgItemMessage(hDlg, CB_MBODY, WM_SETFONT, hSysFont, NULL);

	/*** Set tabs.***/
	tSize = GetDialogBaseUnits();
	tabset[0] = (LOWORD(tSize))*4;
	SendDlgItemMessage(hDlg, CB_MBODY, EM_SETTABSTOPS, 1, (LONG) &tabset);

	/*** Calculate the Options.linelength size for window -- then resize. ***/
	_fmemset((void far *) &result, 'A', Options.linelength);
	result[Options.linelength] = '\0';
	hDC = GetDC(GetDlgItem(hDlg, CB_MBODY));
	SelectObject(hDC, hSysFont);
	textSize = GetTextExtent(hDC, result,  Options.linelength);
	ReleaseDC(GetDlgItem(hDlg, CB_MBODY), hDC);
	SendDlgItemMessage(hDlg, CB_MBODY, EM_GETRECT, 0, (LPARAM) (RECT far *) &Rect);
	dx = Rect.left + Rect.right;
	dy = Rect.top + Rect.bottom;
	Rect.right = LOWORD(textSize) + Rect.left; // + Options.linelength;
	Rect.bottom = dy;
	Rect.top = Rect.left = 0;
	SendDlgItemMessage(hDlg, CB_MBODY, EM_FMTLINES, FALSE, 0L);
	SendDlgItemMessage(hDlg, CB_MBODY, EM_SETRECT, 0, (LPARAM) (RECT far *) &Rect);
	SendDlgItemMessage(hDlg, CB_MBODY, EM_FMTLINES, TRUE, 0L);
	Rect.right = dx;
	Rect.bottom = dy;


	if (menuitem == IDM_FORWARD) {

		NullFill((char far *) &result, 1024);

		/*** Pre-pend Comment info to the edit window ***/
		if (SendDlgItemMessage(hDlg, CB_MBODY, EM_LINELENGTH, 0, 0L) > 0L) {
			SendDlgItemMessage(hDlg, CB_MBODY, EM_SETSEL, 1, MAKELPARAM(0, 0));
			lstrcpy((LPSTR) &result, "Comments:\r\012");
			SendDlgItemMessage(hDlg, CB_MBODY, EM_REPLACESEL, 0, (long) (char far *) result);

			index = SendDlgItemMessage(hDlg, CB_MBODY, EM_GETLINECOUNT, 0, 0L);
			retcount = SendDlgItemMessage(hDlg, CB_MBODY, EM_LINEINDEX, index-1, 0L);
			retcount += SendDlgItemMessage(hDlg, CB_MBODY, EM_LINELENGTH, retcount, 0L);
		}
		else {
			retcount = 0;
		}

		/*** Append Original Message Header to the edit window ***/
		SendDlgItemMessage(hDlg, CB_MBODY, EM_SETSEL, 1, MAKELPARAM(retcount, retcount));
		lstrcpy((LPSTR) &result, "\r\012\r\012- - - - - - - - - - - - - - Original Message - - - - - - - - - - - - - -\r\012");
		SendDlgItemMessage(hDlg, CB_MBODY, EM_REPLACESEL, 0, (long) (char far *) result);
		retcount += lstrlen((LPSTR) &result);
		SendDlgItemMessage(hDlg, CB_MBODY, EM_SETSEL, 1, MAKELPARAM(retcount, retcount));

		/*** Append Original Envelope info and Message Body to the edit window ***/
		FormatEnvelope(FoldName, OldMsgID, (Env far *)&TempEnv, (GLOBALHANDLE far *)&hMessage, GetDlgItem(hDlg, CB_MBODY), FALSE);
		if (!FormatBody(FoldName, OldMsgID, hMessage, MAXBODY)) {
			DspMsgCode( hWnd, "Error Reading Original Mail Message", 0, FALSE);
			SafeFree(GLOBAL, hMessage);
			VnsReleaseMailEnvelope(hVNMAS, MsgID);
			VnsReleaseMailEnvelope(hVNMAS, OldMsgID);
			SendDlgItemMessage(hDlg, CB_MBODY, WM_SETFONT, hFont, NULL);
			SendDlgItemMessage(hDlg, CB_MBODY, EM_SETRECT, 0, (LPARAM) (RECT far *) &Rect);
			return FALSE;
		}

		textstr = GlobalLock(hMessage);
		SendDlgItemMessage(hDlg, CB_MBODY, EM_REPLACESEL, 0, (long) (char far *) textstr);
		retcount += lstrlen(textstr);

		/*** Release the memory for the original message envelope ***/
		SafeFree(GLOBAL, hMessage);
		VnsReleaseMailEnvelope(hVNMAS, OldMsgID);
	}


	/*** Determine the correct size for the buffer.  ***/
	bytes  = (unsigned int) SendDlgItemMessage(hDlg, CB_MBODY, WM_GETTEXTLENGTH, 0, 0L) + 1;

	/*** Grab enough memory for the message and get a pointer to it. ***/
	if ((hBody = SafeAlloc(GLOBAL, MAKELONG(bytes, 0), hWnd)) == NULL) {
		/*** Release the mail envelope and associated memory. ***/
		VnsReleaseMailEnvelope(hVNMAS, MsgID);
		SendDlgItemMessage(hDlg, CB_MBODY, WM_SETFONT, hFont, NULL);
		SendDlgItemMessage(hDlg, CB_MBODY, EM_SETRECT, 0, (LPARAM) (RECT far *) &Rect);
		return (FALSE);
	}
	pText = (char far *) GlobalLock(hBody);

	/*** Copy the edit control to the memory buffer. ***/
	bytes = SendDlgItemMessage(hDlg, CB_MBODY, WM_GETTEXT, bytes, (LPARAM) pText) + 1;

	/*** Now modify the buffer by removing excess CR characters. ***/
	for(i=0;i<bytes;i++)
		if ((pText[i] == '\r') && (pText[i+1] == '\r') && (pText[i+2]==0x0A)) {
			lstrcpy(pText+i, pText+i+2);
			bytes-=2;
		}
	pText[bytes-1] = '\0';

	/*** Translate the text to the OEM character set. ***/
	AnsiToOem(pText, pText);

	/*** Copy the memory buffer to the Banyan Mail message. ***/
	offset = 0;
	for(i=0;i<bytes;i+=1024) {
		size = min(1024, bytes - i);
		uStat = VnsStoreMailBodyPart(hVNMAS, UserName, FoldName, MsgID, MAINBODY, pText+i, size, offset);
		if (Recover(uStat))
			uStat = VnsStoreMailBodyPart(hVNMAS, UserName, FoldName, MsgID, MAINBODY, pText+i, size, offset);
		offset++;
		if (uStat) {
			/*** Release the mail envelope and associated memory. ***/
			SendDlgItemMessage(hDlg, CB_MBODY, WM_SETFONT, hFont, NULL);
			SendDlgItemMessage(hDlg, CB_MBODY, EM_SETRECT, 0, (LPARAM) (RECT far *) &Rect);
			VnsReleaseMailEnvelope(hVNMAS, MsgID);
			DspMsgCode( hWnd, "Error Storing Mail Message : ", uStat, FALSE);
			return (FALSE);
		}
	}
	SafeFree(GLOBAL, hBody);
	SendDlgItemMessage(hDlg, CB_MBODY, WM_SETFONT, hFont, NULL);
	SendDlgItemMessage(hDlg, CB_MBODY, EM_SETRECT, 0, (LPARAM) (RECT far *) &Rect);
	return (TRUE);

}


/******************************************************************************
	FUNCTION:	SaveEnvelopeInfo()

	PURPOSE:  	To store the current settings from the compose box into a
				Banyan mail envelope.
*******************************************************************************/
BOOL SaveEnvelopeInfo(HWND hDlg, WORD menuitem, char far *MsgID, char far *FoldName, Env far *MailEnv)
{
	date far 	*createDate, far *tempDate;
	char		buff[128], hr[5], min[5];
	struct tm 	time;
	CallStatus 	uStat;

	/*** Fill the fields of the Mail Envelope structure  ***/
	/*** with values taken from the dialog box controls. ***/
	createDate = &(MailEnv->creationDate);
	VnsGetTime(&createDate->gmt_seconds, &createDate->tz_off, createDate->tz);
	MailEnv->messageType 	= (char) IPMES;
	MailEnv->traceinfo[0] 	= 0x0;
	MailEnv->status			= (char) MA_UNSENT;
	MailEnv->replyTo[0] 		= 0x0;
	lstrncpy((char far *) &MailEnv->messageId, MsgID, 14);
	lstrcpy((LPSTR) &MailEnv->from, (LPSTR) &UserName);
	MailEnv->forwarder[0] 	= 0x0;
	if (menuitem == IDM_FORWARD) {
		/*** Modify Subject to Show the Fwd line ***/
		lstrcpy((LPSTR) &buff, "Fwd: ");
		lstrcat((LPSTR) &buff, (LPSTR) &MailEnv->subject);
		buff[MAXSUBSTR-1] = '\0';
		lstrcpy((LPSTR) &MailEnv->subject, (LPSTR) &buff);
	}
	else
		SendDlgItemMessage(hDlg, CB_SUBJECT, WM_GETTEXT, MAXSUBSTR, (LPARAM) &MailEnv->subject);
	if (IsDlgButtonChecked(hDlg, CB_CERTIFY))
		MailEnv->certified 	= (char) MESCONFIRM;
	else
		MailEnv->certified 	= (char) NOMESCONFIRM;



	/*** Retrieve the Priority setting. ***/
	MailEnv->priority = MED_PRIORITY;
	if (IsDlgButtonChecked(hDlg, CB_LOW))
		MailEnv->priority = (char) LOW_PRIORITY;
	if (IsDlgButtonChecked(hDlg, CB_HIGH))
		MailEnv->priority = (char) HIGH_PRIORITY;


	/*** Store the mail envelope . ***/
	uStat = VnsStoreMailEnvelope(hVNMAS, (LPSTR) &UserName, FoldName, MsgID, (Env WFAR*) MailEnv);
	if (Recover(uStat))
		uStat = VnsStoreMailEnvelope(hVNMAS, (LPSTR) &UserName, FoldName, MsgID, (Env WFAR*) MailEnv);
	if (uStat) {
		DspMsgCode( hWnd, "Error Storing Mail Envelope ", uStat, FALSE);
		return (FALSE);
	}

	/*** Release the mail envelope and associated memory. ***/
	VnsReleaseMailEnvelope(hVNMAS, MsgID);

	return TRUE;
}



/******************************************************************************
	FUNCTION:	SaveMessage()

	PURPOSE:  	To store the current settings from the compose box into a
				Banyan mail message.
*******************************************************************************/
BOOL SaveMessage(HWND hDlg,	BodyPart far *Body, WORD menuitem, char far *FoldName, char far *MsgID, char far *OldMsgID)
{

	int 			destination, size, index;
	MailFolder far	*TempFolder;
	GLOBALHANDLE	hBody, hFolder;
	Env				MailEnv;
	CallStatus		uStat;


	/*** Retrieve the Destination then put up an hourglass cursor. ***/
	if (hStatus)
		SendMessage(hStatus, STB_STRING, 0, 0L);
	if (menuitem == IDM_FORWARD)
		destination = (int) CallDlg(hDlg, (FARPROC) ChooseDest, FwdDestBox, (LPARAM) IDM_FORWARD);
	else
		destination = (int) CallDlg(hDlg, (FARPROC) ChooseDest, SendBox, 0L);
	SetCursor(LoadCursor(NULL, IDC_WAIT));

	/*** Copy fields to a Banyan mail envelope. ***/
	if (destination) {

		/*** Update Status bar. ***/
		if (hStatus)
			SendMessage(hStatus, STB_STRING, 0, (LONG)"Storing Envelope ...");

		/*** Open a Banyan Envelope -- A new one if Composing ***/
		/*** or Answering, an existing envelope if Editing.   ***/
		switch (menuitem) {
				case IDM_EDIT:
					NullFill((char far *) &MailEnv, sizeof(Env));
					uStat = VnsGetMailEnvelope(hVNMAS, (LPSTR) &UserName, FoldName, MsgID, &MailEnv);
					if (Recover(uStat))
						uStat = VnsGetMailEnvelope(hVNMAS, (LPSTR) &UserName, FoldName, MsgID, &MailEnv);
					break;

				case IDM_FORWARD:
					/*** Create a copy of the old Mail envelope for retrieving    ***/
					/*** name lists, etc.. and Create a new envelope for sending. ***/
					NullFill((char far *) &MailEnv, sizeof(Env));
					uStat = VnsGetMailEnvelope(hVNMAS, UserName, FoldName, OldMsgID, &MailEnv);
					if (Recover(uStat))
						uStat = VnsGetMailEnvelope(hVNMAS, UserName, FoldName, OldMsgID, &MailEnv);
					if (uStat) {
						DspMsgCode( hDlg, "Error Reading Original Mail Message : ", uStat, FALSE);
						return(FALSE);
					}

					/*** Change Status on old message (UNREAD to READ) ***/
					ChangeUnread(FoldName, OldMsgID);

					/*** No break intentional. ***/

				default:
					if (menuitem != IDM_FORWARD)
						NullFill((char far *) &MailEnv, sizeof(Env));
					uStat = VnsInitNewMailEnvelope(hVNMAS, MsgID);
					if (Recover(uStat))
						uStat = VnsInitNewMailEnvelope(hVNMAS, MsgID);
					if (uStat) {
						DspMsgCode( hWnd, "Error Opening Mail Envelope: ", uStat, FALSE);
						return(FALSE);
					}
		} /*** End switch(menuitem) ***/



		/*** Store the Attachments with their attributes ***/
		/*** and sizes. These attibutes are used only by ***/
		/*** HappyMail. Banyan will ignore these fields. ***/
		if (!SaveAttachInfo(hDlg, Body, FoldName, menuitem, MsgID, OldMsgID))
			return FALSE;

		/*** Save the recipient adddress info. ***/
		SaveAddressInfo(hDlg, menuitem, MsgID);

		if (!SaveBodyInfo(hDlg, menuitem, FoldName, MsgID, OldMsgID))
			return FALSE;

		/*** Fill the fields of the Mail Envelope structure  ***/
		/*** with values taken from the dialog box controls. ***/
		if (!SaveEnvelopeInfo(hDlg, menuitem, MsgID, FoldName, &MailEnv))
			return FALSE;


		/**** Send the mail message. ****/
		if (destination == 1 || destination == 2) {
			/*** Update Status Bar. ***/
			if (hStatus)
				SendMessage(hStatus, STB_STRING, 0, (LONG)"Sending Message ...");
			uStat = VnsSendMailMsg(hVNMAS, (LPSTR) &UserName, FoldName, MsgID);
			if (Recover(uStat))
				uStat = VnsSendMailMsg(hVNMAS, (LPSTR) &UserName, FoldName, MsgID);
			if (uStat) {
				DspMsgCode( hWnd, "Error Sending Mail", 0, FALSE);
				EndDialog(hDlg, TRUE);
				return (FALSE);
			}
		}


		/**** Delete Mail message -- move to wastebasket. ***/
		if (destination == 1) {
			if (hStatus)
				SendMessage(hStatus, STB_STRING, 0, (LONG)"Deleting Message ...");
			VnsDeleteMailMsg(hVNMAS, (LPSTR) &UserName, FoldName, MsgID);
		}


		/*** Update the affected Folder windows, i.e. the ***/
		/*** Wastebasket and the current Folder window.   ***/
		if (hStatus)
			SendMessage(hStatus, STB_STRING, 0, (LONG)"Updating Folders ...");
		if (destination == 1) {
			hFolder =  GetFolder2("Wastebasket");
			if (hFolder != NULL) {
				TempFolder = (MailFolder far *) GlobalLock(hFolder);
				PostMessage(TempFolder->hListMail, (int) WM_LBRESET, (int) NULL, (long) NULL);
				GlobalUnlock(hFolder);
			}
			if (menuitem == IDM_EDIT) {
				hFolder =  GetFolder2(FoldName);
				if (hFolder != NULL) {
					TempFolder = (MailFolder far *) GlobalLock(hFolder);
					index = FindID(hFolder, MsgID);
					size = (TempFolder->MsgCount < MaxMsgs) ? TempFolder->MsgCount : MaxMsgs;
					lstrncpy((LPSTR) &(TempFolder->Item[index]), (LPSTR) &(TempFolder->Item[index+1]), sizeof(ItemRec)*(size-(index+1)));
					SendMessage(TempFolder->hListBox, (int) LB_DELETESTRING, (int) index, (long) NULL);
					TempFolder->MsgCount--;
					WriteTitleBar(TempFolder);
					GlobalUnlock(hFolder);
				}
			}
		}

		if (destination == 2) {
			hFolder =  GetFolder2(FoldName);
			if (hFolder != NULL) {
				TempFolder = (MailFolder far *) GlobalLock(hFolder);
				if (menuitem == IDM_EDIT) {
					if (Options.sortorder != BYSTATUS) {
						index = FindID(hFolder, MsgID);
						TempFolder->Item[index].flags = 0x00;
						SendMessage(TempFolder->hListBox, LB_DELETESTRING, index, 0L);
						SendMessage(TempFolder->hListBox, LB_INSERTSTRING, index, (LONG) index);
					}
					else
						PostMessage(TempFolder->hListMail, (int) WM_LBRESET, (int) NULL, (long) NULL);
				}
				else {
					PostMessage(TempFolder->hListMail, (int) WM_LBRESET, (int) NULL, (long) NULL);
				}
				GlobalUnlock(hFolder);
			}
		}

		if (destination == 3) {
			hFolder =  GetFolder2(FoldName);
			if (hFolder != NULL) {
				TempFolder = (MailFolder far *) GlobalLock(hFolder);
				if (menuitem != IDM_EDIT) {
					PostMessage(TempFolder->hListMail, (int) WM_LBRESET, (int) NULL, (long) NULL);
				}
				else {
					index = FindID(hFolder, MsgID);
					TempFolder->Item[index].flags = 0x00;
					SendMessage(TempFolder->hListBox, LB_DELETESTRING, index, 0L);
					SendMessage(TempFolder->hListBox, LB_INSERTSTRING, index, (LONG) index);
				}


				GlobalUnlock(hFolder);
			}
		}

		/*** Update Status Bar. ***/
		if (hStatus)
			SendMessage(hStatus, STB_STRING, 0, 0L);
	} /*** if destination ***/
	else
		return (FALSE);

	return (TRUE);
}



/******************************************************************************
	FUNCTION:   StoreNames()

	PURPOSE:  	Store a list of recipient names to the specified class:
				TO:, CC:, or BCC:.

	COMMENTS:	Since the code for this function is very similar to the
				CheckNames() code, these two should be combined in some
				future release.
*******************************************************************************/
int StoreNames(char far *input, char RecipientClass, char far *MsgID)
{
	CallStatus 		uStat;
	char 			tempname[512];
	char far 		*tempptr;
	GLOBALHANDLE 	hNames;

	NullFill((LPSTR) &tempname, 512);

	/*** Expand any address book       ***/
	/*** entries contained in the list ***/
	if ((hNames = ExpandNames(input, 0)) == NULL)
		return;
	tempptr = (char far *) GlobalLock(hNames);

	/*** Store one name from the list at a time. ***/
	while(VnsParseMailRecipient(tempptr, (LPSTR)&tempname, (char far * far *) &tempptr) != MS_NONAMEADDR) {
		if (tempname[0] != '\0') {

			/*** Store the name in the Banyan mail message. ***/
			uStat = VnsStoreMailRecipient(hVNMAS, MsgID, RecipientClass, (LPSTR) &tempname);
			if (Recover(uStat))
				uStat = VnsStoreMailRecipient(hVNMAS, MsgID, RecipientClass, (LPSTR) &tempname);
			if (uStat) {
				DspMsgCode(hWnd, "Error Storing Name : ", uStat,  FALSE);
				SafeFree(GLOBAL, hNames);
				return;
			}
		}
		NullFill((LPSTR) &tempname, 512);
	}

	/*** Release the memory allocated by ExpandNames(). ***/
	SafeFree(GLOBAL, hNames);
	return;
}
