/****************************************************************************
	FILE:		winlib.c


	CONTENTS:	AskChange()		window procedure for the Yes-No dialog box.
				CallDlg()		creates a dialog box
				CenterWindow()	centers a window.
				DeClogDlgMsg()	unclogs queued-up window messages.
				EditProc()		window procedure for sub-classed Edit controls.
								Translates tabs into [CTRL] tabs.
				ResizeWindow()	Resize the window to the size specified and
								change the font to the ANSI FIXED font.

	COMMENTS:	This file contains window procedures and window management
				procedures which are used throughout HappyMail.
****************************************************************************/
#include "MyMail.h"
#include "string.h"

/******************************************************************************
	FUNCTION:	AskChange()

	PURPOSE:	Dialog Procedure for YesNoBox Dialog Box

	COMMENTS:	This procedure is initially passed a long value which is used to
				load the text strings used in the box. The long value is translated
				to a stringtable values with the following formulas:

				string1 = (lvalue * 10) + 8000;
				string2 = (lvalue * 10) + 8001;
				string3 = (lvalue * 10) + 8002;

*******************************************************************************/
BOOL FAR PASCAL _export AskChange(HWND hDlg, unsigned message, WORD wParam, LONG lParam)
{
	unsigned int 	stValue;
    char 			buff[80];

	switch (message) {
		case WM_INITDIALOG:
			/*** Center the Dialog Window in the HappyMail Main window ***/
			CenterWindow(hDlg, GetDesktopWindow());

			/*** Use lParam and the formula described above to ***/ 
			/*** load the proper strings into the static text  ***/
			/*** controls and caption of this dialog box.      ***/
            stValue = ((unsigned int)lParam * 10) + 8000;
			LoadString(hInst, stValue++, (LPSTR) &buff, 80);
			SendMessage(hDlg, WM_SETTEXT, NULL, (LONG) &buff);
			LoadString(hInst, stValue++, (LPSTR) &buff, 80);
			SetDlgItemText(hDlg, YN_LINE1, (LPCSTR) &buff);
			LoadString(hInst, stValue, (LPSTR) &buff, 80);
			SetDlgItemText(hDlg, YN_LINE2, (LPCSTR) &buff);
			break;

		case WM_COMMAND:                 
			switch(wParam) {
				case IDNO:              
				case IDYES:       
                    /*** Use the return value to indicate which button was selected. ***/
					EndDialog(hDlg, (wParam == IDYES) ? 1L : 0L);
					return (TRUE);
			} /** end switch(wParam) **/
			break;
	} /** end switch(message) **/
	return (FALSE);
}


/******************************************************************************
	FUNCTION:   CallDlg()

	PURPOSE:  	changes cursor to hourglass then calls a dialog box
*******************************************************************************/
LONG CallDlg(HWND hwnd, FARPROC ProcName, unsigned int BoxName, long lParam)
{
	long 	retval;
	FARPROC	lpProc;

	/*** Change to HourGlass Cursor before starting ***/
	SetCursor(LoadCursor(NULL, IDC_WAIT));
	lpProc = MakeProcInstance(ProcName, hInst);
	if (lParam != NULL)
		retval = DialogBoxParam(hInst, MAKEINTRESOURCE(BoxName), hwnd, lpProc, lParam);
	else
		retval = DialogBox(hInst, MAKEINTRESOURCE(BoxName), hwnd, lpProc);
	FreeProcInstance(lpProc);

	if (retval == -1L) {
		/*** Error occurred creating the dialog box. ***/
		DspMsgCode( hWnd, "Error Creating Window", 0, FALSE);
		return 0L;
	}
	return retval;
}


/******************************************************************************
	FUNCTION:   CenterWindow()

	PURPOSE:  	To center the given window inside the main HappyMail window.
*******************************************************************************/
CenterWindow(HWND hLilWnd, HWND hBigWnd)
{
	int		width, height, xStart, yStart, xCenter, yCenter;
	RECT	BigRect, LilRect;


	/*** Get the Main window Coordinates. ****/
	GetClientRect(hBigWnd,    &BigRect);

    /*** Get the size of the window to be centered. ***/
	GetWindowRect(hLilWnd, &LilRect);
	width   = LilRect.right  - LilRect.left;
	height  = LilRect.bottom - LilRect.top;

    /*** Find the center of the Main window. ***/
	xCenter = (BigRect.right - BigRect.left)/2;
	yCenter = (BigRect.bottom - BigRect.top)/2;

    /*** Calculate the starting position of the window to be centered.  ***/
	xStart  = BigRect.left + (xCenter - (width/2));
	yStart  = BigRect.top + (yCenter - (height/2));
	xStart  = (xStart < 0) ? 0 : xStart;
	yStart  = (yStart < 0) ? 0 : yStart;

    /*** Center the window. ***/
	MoveWindow(hLilWnd, xStart, yStart, width, height, TRUE);
	return;
}


/******************************************************************************
	FUNCTION:	DeClogDlgMsg()

	PURPOSE:  	This function will Peek into the message loop and pass along any
				pending messsages to the specified Dialog Box.
*******************************************************************************/
DeClogDlgMsg(HWND hDlg)
{
	MSG Msg;

    if (hDlg)
		while (PeekMessage(&Msg, hDlg, 0, 0, PM_REMOVE))
			IsDialogMessage(hDlg, &Msg);

	return;
}



/****************************************************************************
	FUNCTION:	EditProc()

	PURPOSE:	Processes messages for "Edit Box" child window.  Main purpose
				of this sub-classing is to convert [TAB]s to [CTRL] [TAB}s,
				[ENTER]s to [CTRL][ENTER]s, and the reverse.

	COMMENTS:	This code was (mis)appropriated from the Tech Specialist Sep. 91
****************************************************************************/

#define EM_CTRLBACK	501

long FAR PASCAL _export EditProc(HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
	FARPROC 	lpfnNewEdit;
	long 		lret;
	char 		States[256];
	unsigned int tabset[1];
	DWORD		tSize;

	switch (message) {

		case WM_RBUTTONDBLCLK:
		case WM_NCRBUTTONDBLCLK:
			/*** Add support for the right button double click. ***/
			PostMessage(GetParent(hwnd), WM_RBUTTONDBLCLK, 0, 0L);
			break;

		case WM_CREATE:
		case WM_SETFOCUS:
			/*** Preset the Tab positions in the Edit window. ***/
			tSize = GetDialogBaseUnits();
			tabset[0] = LOWORD(tSize)*4;
			SendMessage(hwnd, EM_SETTABSTOPS, 1, (LONG) tabset);


			/**** De-highlite the whole text box. ***/
			tSize = SendMessage(hwnd, EM_GETSEL, 0, 0L);
			SendMessage(hwnd, EM_SETSEL, 0, MAKELPARAM(LOWORD(tSize), LOWORD(tSize)));

			break;

		case WM_KEYDOWN:
			/*** An [ENTER] or [TAB] key is being pressed, ***/
			/*** so toggle the state of the [CTRL] key.    ***/
			if (wParam == VK_RETURN || wParam == VK_TAB) {
				GetKeyboardState((LPBYTE) &States);
				if (!(States[VK_SHIFT]&0x80))
					States[VK_CONTROL] ^= 0x80;
				SetKeyboardState((LPBYTE) &States);
			}
			break;


		case WM_DESTROY:
			/*** Edit window being destroyed so         ***/
			/*** release the memory for this procedure. ***/
			/*** Is this kosher to do here??			***/
			lpfnNewEdit = (FARPROC) GetWindowLong(hwnd, GWL_WNDPROC);
			SetWindowLong(hwnd, GWL_WNDPROC, (LONG) lpfnOldEdit);
			FreeProcInstance(lpfnNewEdit);
			break;
	} /** end switch(message) **/

	/*** Call the default Edit control window procedure. ***/
	lret = CallWindowProc(lpfnOldEdit, hwnd, message, wParam, lParam);

	switch (message) {
		/*** Toggle the state of the [CTRL] key back. ***/
		case WM_KEYDOWN:
			if (wParam == VK_TAB) {
				GetKeyboardState((LPBYTE) &States);
				if (!(States[VK_SHIFT]&0x80))
					States[VK_CONTROL] ^= 0x80;
				SetKeyboardState((LPBYTE) &States);
			}
			break;
		case WM_CHAR:
			if (wParam == VK_RETURN) {
				GetKeyboardState((LPBYTE) &States);
				States[VK_CONTROL] ^= 0x80;
				SetKeyboardState((LPBYTE) &States);
			}
			break;
   } /** end switch(message) **/

   return(lret);
}


/****************************************************************************
	FUNCTION:	ResizeWindow()

	PURPOSE:	Change the window to the size specified in the width
				parameter.  The old size is stored in the Rect parameter,
				and the old font is saved in the hFont parameter.
****************************************************************************/
ResizeWindow(HWND hEdit, int width, RECT far *Rect, HFONT far *hFont)
{

	char 	buff[256];
	HDC		hDC;
	HFONT	hSysFont;
	int		dx, dy;
	DWORD	textSize;

	*hFont = (HFONT) SendMessage(hEdit, WM_GETFONT, 0, 0L);
	hSysFont = GetStockObject(ANSI_FIXED_FONT);
	SendMessage(hEdit, WM_SETFONT, hSysFont, NULL);

	/*** Calculate the Options.linelength size for window -- then resize. ***/
	width = (width > (sizeof(buff)-1)) ? (sizeof(buff)-1) : width;
	_fmemset((void far *) &buff, 'A', width);
	buff[width] = '\0';
	hDC = GetDC(hEdit);
	SelectObject(hDC, hSysFont);
	textSize = GetTextExtent(hDC, buff,  width);
	ReleaseDC(hEdit, hDC);
	SendMessage(hEdit, EM_GETRECT, 0, (LPARAM) (RECT far *) Rect);
	dx = Rect->left + Rect->right;
	dy = Rect->top + Rect->bottom;
	Rect->right = LOWORD(textSize) + Rect->left; // + Options.linelength;
	Rect->bottom = dy;
	Rect->top = Rect->left = 0;
	SendMessage(hEdit, EM_FMTLINES, FALSE, 0L);
	SendMessage(hEdit, EM_SETRECT, 0, (LPARAM) (RECT far *) Rect);
	Rect->right = dx;
	Rect->bottom = dy;

	return;
}
