/****************************************************************************
	FILE:		status.c


	CONTENTS:	DrawBar()		draws a blank, gray, raised-edged Status Bar.
				NewWndProc()	subclassed window procedure for StatusBar's
								parent windows.
				StatBarCtrlMsg()utility for displaying Status bar help.
				StatusBarProc()	window procedure for StatusBar window


	COMMENTS: 	This file contains all the procedures used to display and manage
				the Status Bar.

				The Status Bar is used throughout HappyMail to display three
				basic types of help messages: 1) procedure related help
				messages, 2) help messages about dialog box controls, and 3)
				help messages about buttons in the MDI windows(Read and Folder).

				The Status Bar has two basic modes of operation.  First and
				simplest, the application sends to the Status Bar the message
				to be displayed.  All procedure related help uses this method.

				The second method, used by dialog boxes, is to send all
				window messages to a special function (StatBarCtrlMsg).
				This function examines the message for indications of a
				new control gaining the focus.  When this occurs the special
				function sends the Status Bar a pre-defined help message
				about the new control.

				Finally, in order to get help on the buttons in the MDI
				windows,  the WM_SETCURSOR message is intercepted.  This
				message allows the window procedure to know when the mouse
				cursor is over any part of the window.  So whenever the
				message indicates the cursor is over a button, a help
				message is sent to the Status Bar.
****************************************************************************/
#include "MyMail.h"
#include "string.h"

#define		WM_RESET	WM_USER+270

/****************************************************************************
	FUNCTION: 	DrawBar()

	PURPOSE:  	Draws a gray, raised-edge, bar and draws a text string inside 
****************************************************************************/
DrawBar(HWND hwnd, HDC hDC, char far *msgbuff)
{

	HPEN		hPen;
    HBRUSH		hBrush;
	RECT		Rect;
	DWORD		LGray, DGray, Black, White, OldColr, OldColr2;


    /*** Define some colors ***/
	LGray = RGB(192,192,192);
	Black = RGB(0,0,0);
	DGray = RGB(128,128,128);
	White = RGB(255,255,255);


    /*** Draw a solid, light gray, rectangle with a black border. ***/
	hPen	= SelectObject(hDC, CreatePen(PS_SOLID, 1, Black));
	hBrush	= SelectObject(hDC, CreateSolidBrush(LGray));
	GetClientRect(hwnd, &Rect);
	Rectangle(hDC, Rect.left, Rect.top, Rect.right+1, Rect.bottom+1);

	/*** Create part of the raised edge look with white hilite lines. ***/
	DeleteObject(SelectObject(hDC, CreatePen(PS_SOLID, 1, White)));
	MoveTo(hDC, Rect.left+1, Rect.bottom-1);
	LineTo(hDC, Rect.left+1, Rect.top+1);
	LineTo(hDC, Rect.right-1, Rect.top+1);

	MoveTo(hDC, Rect.left+3, Rect.bottom-3);
	LineTo(hDC, Rect.right-3, Rect.bottom-3);
	LineTo(hDC, Rect.right-3, Rect.top+3);

	/*** Create the rest of the raised edge with dark gray shadow lines. ***/
	DeleteObject(SelectObject(hDC, CreatePen(PS_SOLID, 1, DGray)));
	MoveTo(hDC, Rect.left+3, Rect.bottom-4);
	LineTo(hDC, Rect.left+3, Rect.top+3);
    LineTo(hDC, Rect.right-4, Rect.top+3);
    	
	MoveTo(hDC, Rect.left+1,  Rect.bottom-1);
	LineTo(hDC, Rect.right-1, Rect.bottom-1);
	LineTo(hDC, Rect.right-1, Rect.top+1);

	/*** Draw in any text ***/
	if (msgbuff[0] != '\0') {
		OldColr = SetBkColor(hDC, LGray);
		OldColr2 = SetTextColor(hDC, Black);
		TextOut(hDC, Rect.left + 10, Rect.top + 4, msgbuff, lstrlen((LPSTR) msgbuff));
		SetBkColor(hDC, OldColr);
		SetTextColor(hDC, OldColr2);
	}

    /*** Clean up our resources. ***/
	DeleteObject(SelectObject(hDC, hPen));
	DeleteObject(SelectObject(hDC, hBrush));
	return;
}



/****************************************************************************
	FUNCTION:	NewWndProc()

	PURPOSE:	Processes messages for a statusbar parent window
****************************************************************************/
long FAR PASCAL _export NewWndProc(HWND hwnd, WORD message, WORD wParam, LONG lParam)
{

	long		 	retval;
	static FARPROC	lpOldProc=NULL;
	RECT			Rect;
    int				height;			

	switch (message) {
		case WM_INITIALIZE:
            /*** Store a pointer to the old window procedure. ***/
			lpOldProc = (FARPROC) lParam;
			return (NULL);

    	case WM_SIZE:
			/*** If this window is being resized, tell the ***/
			/*** Status Bar to resize itself to match.     ***/
			if (wParam == SIZEFULLSCREEN || wParam == SIZENORMAL) {
				if (hStatus) {
					SendMessage(hStatus, message, wParam, lParam);
					height = GetSystemMetrics(SM_CYCAPTION);
					lParam = MAKELONG(LOWORD(lParam), HIWORD(lParam) - (height + 4));
					return CallWindowProc(lpOldProc, hwnd, message, wParam, lParam);
                }
            }
			break;

		case WM_PAINT:
			/*** If this window is being repainted, tell ***/
			/*** the Status Bar to repaint itself also.  ***/
            if (hStatus) {
				GetClientRect(hStatus, &Rect);
				InvalidateRect(hStatus, &Rect, TRUE);
            }
			break;


	} /** end switch(message) **/

    /*** Call the old window procedure. ***/
	if (lpOldProc)
		return CallWindowProc(lpOldProc, hwnd, message, wParam, lParam);
	else
    	return NULL;
}



/****************************************************************************
	FUNCTION:	StatBarCtrlMsg()

	PURPOSE:	Displays the correct status bar help message for the
				current control. Does so by checking the incoming window
				messages for indications of a change in focus. 

	COMMENTS:	The DlgVal parameter is a pre-defined value corresponding to
				the dialog box receiving the messages.	All help messages
				are referenced by this value.

				Example:
					DlgVal    = 10000      //Dialog Value
					CtrlID 	  = 101        //Control ID value
                    HelpMsgID = 10101      //Help message String ID value

****************************************************************************/
#pragma argsused
StatBarCtrlMsg(HWND hDlg, unsigned int DlgVal, WORD message, WORD wParam, LONG lParam)
{
	BOOL			Success;
	char 			buff[100];
	DRAWITEMSTRUCT far *ds;
    HWND			hFocusWnd;


	switch (message) {
		case WM_CTLCOLOR:
			/*** Many system defined controls send this message ***/
			/*** just prior to being drawn, allowing the        ***/
			/*** application to set the color of the control.   ***/
            hFocusWnd = GetFocus(); 
            if (hFocusWnd) {
				DlgVal += GetDlgCtrlID(hFocusWnd);
            	if (DlgVal>0) {
					LoadString(hInst, DlgVal, (LPSTR) &buff, 100);
					if (hStatus)
						SendMessage(hStatus, STB_STRING, 0, (LONG) &buff);
				}
			}
			break;

		case WM_COMMAND:
			switch (HIWORD(lParam)) {
				case BBN_SETFOCUS:
				case BBN_SETFOCUSMOUSE:
				case BN_HILITE:
				case EN_SETFOCUS:
				case CBN_SETFOCUS:
				case LBN_SETFOCUS:
				case BN_CLICKED:
					/*** All of the above messages are indicative ***/
					/*** of a control gaining the focus.          ***/
                    if (IsWindow(LOWORD(lParam))) {
						DlgVal += GetDlgCtrlID(LOWORD(lParam));
						if (DlgVal>0) {
							LoadString(hInst, DlgVal, buff, 100);
							if (hStatus)
								SendMessage(hStatus, STB_STRING, 0, (LONG) &buff);
						}
					}
            }
            break;

		case WM_DESTROY:
			if (hStatus)
				SendMessage(hStatus, STB_STRING, 0, 0L);

	}

	return;
}



/****************************************************************************
	FUNCTION: 	StatusBarProc()

	PURPOSE:  	Processes messages for "StatusBar" window

	COMMENTS:	To display a string, send a STB_STRING message to the Status
				Bar with the following parameters:

				wParam = length of string       or  zero
				(specify a length if string contains embedded null characters)

                lParam = long pointer to string  or zero
				(specify a zero to erase current string)

****************************************************************************/
long FAR PASCAL _export StatusBarProc(HWND hwnd, WORD message, WORD wParam, LONG lParam)
{

	PAINTSTRUCT	ps;
	HDC			hDC;
	HWND		hDad;
	HPEN		hPen;
	HBRUSH		hBrush;
	RECT		Rect;
	DWORD		LGray, Black, OldColr, OldColr2;
	FARPROC		lpfnOld, lpfnNew;
	static		char msgbuff[256];
	char		oldmsg[256];

	switch (message) {
		case WM_CREATE:
            /*** Set the Global pointer to the Status Bar. ***/
            hStatus = hwnd;

			/*** MsgBuff stores the currently displayed message. This ***/
			/*** is stored to avoid re-displaying the same message.   ***/
            msgbuff[0] = '\0';

			/*** Move the Status-bar to its appropriate          ***/ 
			/*** position, i.e. the bottom of the parent window. ***/
 			hDad = GetParent(hwnd);
            GetClientRect(hDad, &Rect);
			MoveWindow(hwnd, 0, (Rect.bottom - GetSystemMetrics(SM_CYCAPTION))+4, Rect.right, GetSystemMetrics(SM_CYCAPTION)+4, TRUE);

			/*** Sub-Class the parent window in order to automatically        ***/
			/*** resize the Status-bar whenever the parent window is resized. ***/
			lpfnOld = (FARPROC) GetWindowLong(hDad, GWL_WNDPROC);
			lpfnNew = MakeProcInstance((FARPROC) NewWndProc, hInst);
			SetWindowLong(hDad, GWL_WNDPROC, (LONG) lpfnNew);
			SendMessage(hDad, WM_INITIALIZE, hwnd, (LONG) lpfnOld);
			break;

		case WM_SIZE:
            /*** Parent is re-sizing.  So resize the Status Bar to match. ***/
 			hDad = GetParent(hwnd);
            GetClientRect(hDad, &Rect);
			MoveWindow(hwnd, 0, (Rect.bottom - GetSystemMetrics(SM_CYCAPTION))-4, Rect.right, GetSystemMetrics(SM_CYCAPTION)+4, TRUE);
			break;

		case STB_STRING:
			/*** This is a fake windows message used ***/
			/*** to indicate what string to display. ***/

			/*** First store the current message, then retrieve the new message. ***/
			lstrcpy((LPSTR) &oldmsg, (LPSTR) &msgbuff);
			if (wParam > 0) {
				lstrncpy(msgbuff, (LPSTR) lParam, wParam);
				msgbuff[wParam] = '\0';
			}
			else
				if (lParam != NULL)
					lstrcpy((LPSTR) &msgbuff, (LPSTR) lParam);
				else
					msgbuff[0] = '\0';
			/*** Check to make sure the new string is ***/
			/*** different from the current string.   ***/
			if (lstrcmp((LPSTR) &oldmsg, (LPSTR) &msgbuff) != 0) {
				hDC = GetDC(hwnd);
				/*** If the Status Bar needs to be repainted,  ***/
				/*** redraw the entire status bar with the     ***/
				/*** new message. Else, draw only the insides. ***/
				if (GetUpdateRect(hwnd, &Rect, FALSE) != 0) {
					DrawBar(hwnd, hDC, (LPSTR)&msgbuff);
				}
				else {
					/*** Redraw the inside of status bar ***/
					LGray = RGB(192,192,192);
					Black = RGB(0,0,0);
					hPen	= SelectObject(hDC, CreatePen(PS_SOLID, 1, LGray));
					hBrush	= SelectObject(hDC, CreateSolidBrush(LGray));
					GetClientRect(hwnd, &Rect);
					Rectangle(hDC, Rect.left + 5, Rect.top + 4, Rect.right-5, Rect.bottom-4);

					/*** Redraw the message text ***/
					OldColr = SetBkColor(hDC, LGray);
					OldColr2 = SetTextColor(hDC, Black);
					TextOut(hDC, Rect.left + 10, Rect.top + 4,(LPSTR) &msgbuff, lstrlen((LPSTR)&msgbuff));
					SetBkColor(hDC, OldColr);
					SetTextColor(hDC, OldColr2);
					DeleteObject(SelectObject(hDC, hPen));
					DeleteObject(SelectObject(hDC, hBrush));
				}
				ReleaseDC(hwnd, hDC);
			}

			break;

		case WM_PAINT:
			/*** Repaint the entire bar. ***/
			hDC		= BeginPaint(hwnd, &ps);
			DrawBar(hwnd, hDC, (LPSTR)&msgbuff);
			EndPaint(hwnd, &ps);
			break;


		default:
			/*** Pass on message if unproccessed. ***/
			return (DefWindowProc(hwnd, message, wParam, lParam));
	} /** end switch(message) **/

	return (NULL);
}

