/****************************************************************************
	FILE:		sort.c


	CONTENTS:	AddNode()	Adds a node to a binary tree
    			KillTree()	Deletes all nodes in a binary tree
                ReadTree()	Fills a list box by reading a sorted binary tree
				SortLists()	re-sorts the message lists in folder windows.


	COMMENTS:	This file conatins the functions used for sorting the messages
				contained in Folder windows.
****************************************************************************/
#include "MyMail.h"
#include "string.h"

/******************************************************************************
	FUNCTION: 	AddNode()

	PURPOSE:  	this function adds the passed node to a binary tree
				based on the key field of the record.

	NOTE:		This function is called recursively.
*******************************************************************************/
AddNode(SortNode far *innode, SortNode far *leafptr)
{

	if (lstrcmp(innode->ItemKey, leafptr->ItemKey) < 0 ) {
		if (leafptr->lptr == NULL) {
			leafptr->lptr = innode;
        }
		else
			AddNode(innode, leafptr->lptr);
    }
    else {
		if (leafptr->rptr == NULL) {
            leafptr->rptr = innode;
        }
        else
            AddNode(innode, leafptr->rptr);
    }
    return;
}


/******************************************************************************
	FUNCTION: 	KillTree()

	PURPOSE:  	this function deletes all the nodes in the tree and
				frees the memory reserved for each node.

	NOTE:		This function is called recursively.
*******************************************************************************/
KillTree(SortNode far *leaf)
{
	if (leaf->lptr != NULL)
        KillTree(leaf->lptr);

	if (leaf->rptr != NULL)
        KillTree(leaf->rptr);

	SafeFree(GLOBAL, leaf->hSelf);

	return;
}


/******************************************************************************
	FUNCTION: 	ReadTree()

	PURPOSE:  	this function reads the nodes of the binary tree in sorted
				order and adds the item field to the specified list box.

	NOTE:		This function is called recursively.
*******************************************************************************/
ReadTree(SortNode far *leaf, HWND hList, MailFolder far *CurFolder)
{
	if (leaf->lptr != NULL)
        ReadTree(leaf->lptr, hList, CurFolder);

	SendMessage(hList, LB_ADDSTRING, 0, (LONG)CurFolder->MsgCount);
	lstrncpy(CurFolder->Item[CurFolder->MsgCount].ID, leaf->ItemID, 13);
	lstrcpy(CurFolder->Item[CurFolder->MsgCount].string, leaf->ItemStr);
	CurFolder->MsgCount++;

	if (leaf->rptr != NULL)
		ReadTree(leaf->rptr, hList, CurFolder);
    return;
}


/******************************************************************************
	FUNCTION: 	SortLists()

	PURPOSE:  	To sort the messages in a Folder Window based on one of several
    			sort keys: 1) Date (default), 2)Name, or 3) Subject.
*******************************************************************************/
SortLists(char far *FoldName)
{

	GLOBALHANDLE		hTempMem, hTempFold;
	HWND 				hList;
	char 				buff[100], key[30];
	MailFolder far		*TempFolder;
	SortNode far		*TopNode, far *CurNode;
	int 				i, j, count, len, top, hilite[1], ptr, pos;


	if(Options.sortorder == BYDATE)
    	return;

	/*** Grab a handle to the folder window's listbox. ***/
	hTempFold = GetFolder2(FoldName);
	TempFolder = (MailFolder far *) GlobalLock(hTempFold);
	hList = TempFolder->hListBox;

	/*** Read the contents of the list box, storing    ***/
	/*** the items into a linked list/tree as we read. ***/
	count = SendMessage(hList, LB_GETCOUNT, 0, 0L);
	if (count == 0 || count == LB_ERR)
		return;

	TopNode = CurNode = NULL;
	for(j=0;j<count;j++) {
        /*** Create a new node and initialize the fields. ***/
		hTempMem = SafeAlloc(GLOBAL, sizeof(SortNode), hWnd);
		CurNode = (SortNode far *) GlobalLock(hTempMem);
		NullFill((LPSTR) CurNode, sizeof(SortNode));
		CurNode->hSelf = hTempMem;
		CurNode->ItemNum = j;
		ReadItem(TempFolder, j, (char far *)&CurNode->ItemStr);
		lstrncpy(CurNode->ItemID, TempFolder->Item[j].ID, 13);

		if (Options.sortorder == BYSTATUS) {
			lstrncpy(CurNode->ItemKey, (CurNode->ItemStr), 6);
			CurNode->ItemKey[6] = '\0';
			for(i=0;i<6;i++)
				CurNode->ItemKey[i] = 'A' + ('Z' - CurNode->ItemKey[i]);
		}

		/*** Skip to first tab character -- i.e. the Date ***/
		len = lstrlen(CurNode->ItemStr);
        i=0;
		while (CurNode->ItemStr[i] != '\t' && i<len) i++;
		i++;
		if (Options.sortorder == BYDATE) {
			lstrncpy(CurNode->ItemKey, (CurNode->ItemStr + i), 28);
			CurNode->ItemKey[28] = '\0';
		}

		/*** Skip to next tab character -- i.e. the Name ***/
		while (CurNode->ItemStr[i] != '\t' && i<len) i++;
		i+=7;
		if (Options.sortorder == BYNAME) {
			lstrncpy(CurNode->ItemKey, (CurNode->ItemStr + i), 28);
			CurNode->ItemKey[28] = '\0';
		}

		/*** Skip to next tab character -- i.e. the Subject ***/
		while (CurNode->ItemStr[i] != '\t' && i<len) i++;
		i++;
		if (Options.sortorder == BYSUBJECT) {
			lstrncpy(CurNode->ItemKey, (CurNode->ItemStr + i), 28);
			CurNode->ItemKey[28] = '\0';

			/*** Excise any RE:'s or FWD:'s from the message. ***/
			for(ptr=0, pos=0;ptr<28 && CurNode->ItemKey[ptr] != '\0';ptr++)
				if (CapnCmp(CurNode->ItemKey+ptr, "re:", 3))
					pos = ptr+1;
			if (pos>0 && (pos+3) < ptr)
				_fstrcpy(CurNode->ItemKey, CurNode->ItemKey+pos+3);

			for(ptr=0, pos=0;ptr<28 && CurNode->ItemKey[ptr] != '\0';ptr++)
				if (CapnCmp(CurNode->ItemKey+ptr, "fwd:", 4))
					pos = ptr+1;
			if (pos>0 && (pos+4) < ptr)
				_fstrcpy(CurNode->ItemKey, CurNode->ItemKey+pos+4);
		}

		/*** Clean up the key if necessary. ***/
		for(i=0;CurNode->ItemKey[i] != '\0' && CurNode->ItemKey[i] != '\t';i++);
		if (CurNode->ItemKey[i] == '\t')
			CurNode->ItemKey[i] = '\0';

		/*** Now insert the new node into the tree.  ***/
		if (TopNode == NULL) {
			TopNode = CurNode;
		}
        else
			AddNode(CurNode, TopNode);
		
	}


	/*** Reset the listbox. ***/
	SendMessage(hList, WM_SETREDRAW, 0, NULL);
	top = SendMessage(hList, LB_GETTOPINDEX, NULL, NULL);
	SendMessage(hList, LB_GETSELITEMS, 1, (LONG) &hilite);
	SendMessage(hList, LB_RESETCONTENT, NULL, NULL);


	/*** Add the msgs in sorted order. ***/
	TempFolder->MsgCount = 0;
	ReadTree(TopNode, hList, TempFolder);
	SendMessage(hList, WM_SETREDRAW, 1, NULL);
	SendMessage(hList, LB_SETTOPINDEX, top, NULL);
	SendMessage(hList, LB_SETSEL, TRUE, MAKELPARAM(hilite[0],0));


	/*** Destroy the tree -- release the memory. ***/
	GlobalUnlock(hTempFold);
	KillTree(TopNode);

	return;

}

