/****************************************************************************
	FILE:		read.c


	CONTENTS:   NewEditProc()		replacement window proc for Edit window.
				ReadMail()			window procedure for a Read window.
				ReformatMessage()   reformats the message in a read window
									whenever the size of the window is changed.

	COMMENTS:	This file contains the window procedure(s) used to manage
				Read windows.
****************************************************************************/
#include "MyMail.h"
#include "string.h"

BOOL ReformatMessage(char far *, char far *, Env far *, HWND );

/****************************************************************************
	FUNCTION: 	NewEditProc()

	PURPOSE:  	Processes messages for "ViewMail" -- the child window
				edit control of a Read window.  Main purpose is to add
				support for the double right mouse click.
****************************************************************************/
long FAR PASCAL _export NewEditProc(HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
	FARPROC 		lpfnNewEdit;
	GLOBALHANDLE	hLetterMem;
	Letter far *	TempLetter;
	long 			lret;

	hLetterMem = GetLetter(GetParent(hwnd));
	if (hLetterMem != NULL)
		TempLetter = (Letter far *) GlobalLock(hLetterMem);

	switch (message) {

		case WM_RBUTTONDBLCLK:
		case WM_NCRBUTTONDBLCLK:
			/*** Add support for the right button double click. ***/
			PostMessage(GetParent(hwnd), WM_RBUTTONDBLCLK, 0, 0L);
			break;

		case WM_DESTROY:
			/*** Edit window being destroyed so         ***/
			/*** release the memory for this procedure. ***/
			/*** Is this kosher to do here??			***/
			if (hLetterMem != NULL) {
				lpfnNewEdit = (FARPROC) GetWindowLong(hwnd, GWL_WNDPROC);
				SetWindowLong(hwnd, GWL_WNDPROC, (LONG) TempLetter->lpOldEditProc);
				FreeProcInstance(lpfnNewEdit);
			}
			break;


		case WM_KEYUP:
			/*** If user pressed F1. ***/
			if (wParam == VK_F1)
				PostMessage(hWnd, WM_F1, HELP_CONTENTS, 0L);

	} /** end switch(message) **/

	/*** Call the default Edit control window procedure. ***/
	if (hLetterMem != NULL) {
		lret = CallWindowProc(TempLetter->lpOldEditProc, hwnd, message, wParam, lParam);
		GlobalUnlock(hLetterMem);
	}


   return(lret);
}



/****************************************************************************
	FUNCTION:	ReadMail()

	PURPOSE:	Processes messages for a Read window

	COMMENTS:	The Read window is an MDI child window and uses the
				DefMDIChildProc() versus the standard DefWindowProc().
				For more information on the MDI interface see Petzold's
				Programming Windows.
****************************************************************************/
#define BUTTONCOUNT		9
int ReadBtns[BUTTONCOUNT] = {	IDM_ANSWER,		IDM_DELETE, 	IDM_MOVE,
								IDM_FILE, 		IDM_FORWARD,	IDM_PRINT,
								IDM_ATTACHMENTS,IDM_NEXT, 		IDM_PREV};

long FAR PASCAL _export ReadMail(HWND hChildWnd, WORD message, WORD wParam, LONG lParam)
{
	GLOBALHANDLE	hLetterMem, hTempArr, hEditDS, hTemp;
	char			MsID[14], FoldName[32], far *textstr, buff[100];
	Env				env;
	Letter far		*TempLetter;
	MailFolder far	*TempFolder;
	HWND			hwndTemp;
	RECT			rect;
	unsigned int	dx, dy;
	LOGFONT			lf;
	int 			i, NumItems, far *lPtr;
	HFONT			hFont;
	static HWND		hWndFrame, hWndClient;
	FARPROC			lpfnNewEdit;
	CREATESTRUCT far *lpcs;
	TEXTMETRIC		tm;
	HDC				hDC;
	unsigned int 	tabset[1];

	switch (message) {

		case WM_CREATE:
			/*** Retrieve the message ID numbers and folder name for the ***/
			/*** currently selected messages.  If more than one message  ***/
			/*** is selected, the first one returned will be read.       ***/
			if (hDlgSearch && bSearchRead) {
				SendMessage(hDlgSearch, SB_GETID, 0, (LPARAM)(LPSTR) &MsID);
				SendMessage(hDlgSearch, SB_GETFOLDER, 0, (LPARAM)(LPSTR) &FoldName);
			}
			else {
				NumItems = GetLetterInfo((LPSTR) &MsID, (LPSTR) &FoldName, (long) NULL, NULL);

				/*** No Messages in Folder or None Selected. ***/
				if (NumItems <= 0) {
					BWCCMessageBox(hWnd, "No Message Selected.", "WARNING!", MB_ICONSTOP | MB_OK);
					return -1;
				}
			}


			/*** Initialize the Read Window size. ***/
			GetClientRect(hChildWnd, &rect);
			dx = rect.right;
			dy = rect.right - 32;


			/*** Create a New Data Segment for the Edit window. ***/
			/*** This will maximize the edit control's available memory. ***/
			hEditDS = GlobalAlloc(GMEM_MOVEABLE | GMEM_ZEROINIT | GMEM_SHARE, 256L);
			if (hEditDS == NULL) {
				hEditDS = hInst;  // Allocation failed; use default.
			}

			/*** Now create an Edit control for viewing the message. ***/
			hwndTemp = CreateWindow("edit", "", WS_VISIBLE | WS_CHILDWINDOW | WS_VSCROLL |
				ES_NOHIDESEL | WS_BORDER  | ES_MULTILINE | ES_WANTRETURN, 0, 32, dx, dy,
				hChildWnd, 1, hEditDS, NULL);


			/*** Now limit the text to the maximum possible amount. ***/
			SendMessage(hwndTemp, EM_LIMITTEXT, 0, 0L);

			/*** Change to the user defined font. ***/
			if (GetINIFont("ReadFont", &lf))
				ChangeFont(&lf, hwndTemp);


			/*** Append a new Letter structure to the Letter Chain. ***/
			if ((hLetterMem = AddLetter()) == NULL) {
				/*** Unable to allocate the structure so destroy the window. ***/
				DestroyWindow(hChildWnd);
				return (-1);
			}


			/*** Use the folder structure to fill in some ***/
			/*** of the fields of the Letter structure.   ***/
			TempLetter = (Letter far *) 	GlobalLock(hLetterMem);
			TempLetter->lpOldEditProc = (FARPROC) GetWindowLong(hwndTemp, GWL_WNDPROC);
			TempLetter->hViewMail	= hwndTemp;
			TempLetter->hChild		= hChildWnd;
			lstrncpy(TempLetter->MsgID, (LPSTR) &MsID, 14);
			lstrcpy(TempLetter->FolderName, (LPSTR) &FoldName);
			TempLetter->hList		= NULL;
			TempLetter->hFolder		= NULL;
			if (!(hDlgSearch && bSearchRead)) {
				hTemp = GetFolder2(FoldName);
				TempFolder = (MailFolder far *) GlobalLock(hTemp);
				if (TempFolder->hListBox) {
					TempLetter->hList		= TempFolder->hListBox;
					TempLetter->hFolder		= hTemp;
				}
				GlobalUnlock(hTemp);
			}
			GlobalUnlock(hLetterMem);


			/**** Store address of old Editproc and assign new Editproc ****/
			if (TempLetter->lpOldEditProc)
				SetWindowLong(hwndTemp, GWL_WNDPROC, (LONG) MakeProcInstance((FARPROC) NewEditProc, hInst));


			/*** Put Buttons in Read window. ****/
			for(i=0;i<BUTTONCOUNT;i++) {
				CreateWindow("SuperBorBtn",	"",	WS_VISIBLE | WS_CHILDWINDOW |
					BS_DEFPUSHBUTTON | BBS_PARENTNOTIFY, i*62, 0, 62, 32, hChildWnd,
					ReadBtns[i], hInst, NULL);

			}


			/*** Set static variables for future use ***/
			hWndClient = GetParent(hChildWnd);
			hWndFrame  = GetParent(hWndClient);


			/*** Call NextLetter to fill in the Read window with the current message. ***/
			NextLetter(hChildWnd, SAME, CLOSE);
			return (NULL);


		case WM_DESTROY:
			/*** Destroy the letter structure and release its memory ***/
			/*** Then close down all buttons, edit window, etc... 	 ***/
			if (TopLetter == GetLetter(hChildWnd))
				SaveTopLetter();
			KillLetter(hChildWnd);
			break;

		case WM_KEYDOWN:
			/*** Pass on arrow keys to the Edit control. ***/
  			if (wParam >= VK_PRIOR && wParam <= VK_DOWN) {
				hLetterMem = GetLetter(hChildWnd);
				if (hLetterMem != NULL) {
					TempLetter = (Letter far *) GlobalLock(hLetterMem);
					SendMessage(TempLetter->hViewMail, WM_KEYDOWN, wParam, lParam);
					GlobalUnlock(hLetterMem);
				}
				return 0;
			}

			/*** Use tabs to change focus ***/
			if (wParam == VK_TAB)
				SetFocus(GetDlgItem(hChildWnd, ReadBtns[0]));

			break;

		case WM_KEYUP:
			/*** Add support for the delete key. ***/
			if (wParam == VK_DELETE)
				PostMessage(hWnd, WM_COMMAND, IDM_DELETE, GetLetter(hChildWnd));

			/*** If user pressed F1. ***/
			if (wParam == VK_F1) 
				PostMessage(hWnd, WM_COMMAND, HELP_INDEXES, 0L);
			break;

		case WM_PARENTNOTIFY:
			if (wParam == WM_RBUTTONDOWN){
			}
			break;

		case WM_RBUTTONDBLCLK:
		case WM_NCRBUTTONDBLCLK:
			/*** Add support for the right button double click. ***/
			PostMessage(hChildWnd, WM_CLOSE, 0, 0L);
			return 0;

		case WM_COMMAND:
			switch(HIWORD(lParam)) {
				case EN_SETFOCUS:
					/*** If the Edit control is given the focus, ***/
					/*** give it back to the Read window.        ***/
  					SetFocus(hChildWnd);
					break;

				case BBN_GOTATAB:
					/*** If a button receives a [TAB] key,  ***/
					/*** pass the focus to the next button. ***/
  					if (wParam != IDM_PREV) {
						hwndTemp = GetWindow(LOWORD(lParam), GW_HWNDNEXT);
						SetFocus(hwndTemp);
					}
					break;

				case BBN_GOTABTAB:
					/*** If a button receives a [SHIFT][TAB],   ***/
					/*** pass the focus to the previous button. ***/
					hwndTemp = GetWindow(LOWORD(lParam), GW_HWNDPREV);
					SetFocus(hwndTemp);
					break;

				case ACC_KEYPRESS:
				case BN_CLICKED:
					/*** Accelerator key was pressed or Button was clicked. ***/
					switch(wParam) {
						case IDM_NEXT:
							/*** User select Next button. ***/
  							NextLetter(hChildWnd, NEXT, CLOSE);
							return 0;
						case IDM_PREV:
							/*** User select Previous button. ***/
  							NextLetter(hChildWnd, PREV, CLOSE);
							return 0;

						case IDM_COPY:
							/*** User select Copy from Edit Menu. ***/
  							SendMessage(GetWindow(hChildWnd, GW_CHILD), WM_COPY, 0, 0L);
							return 0;

						case IDCANCEL:
							/*** User select Close from System Menu. ***/
							PostMessage(hChildWnd, WM_SYSCOMMAND, SC_CLOSE, 0L);
							return 0;

						case IDM_ANSWER:
						case IDM_DELETE:
						case IDM_MOVE:
						case IDM_FILE:
						case IDM_FORWARD:
						case IDM_PRINT:
						case IDM_ATTACHMENTS:
							/*** Pass button commands on to main window procedure.  Use ***/
							/*** the lParam value to store the Letter structure handle. ***/
  							SendMessage(hWnd, WM_COMMAND, wParam, GetLetter(hChildWnd));
							return 0;
					} /*** end switch(wParam) ***/
					break;


			}  /*** end switch (HIWORD(lParam)) ***/

			break;



		case WM_COPY:
			SendMessage(GetWindow(hChildWnd, GW_CHILD), WM_COPY, 0, 0L);
			return 0;

		case WM_MDIACTIVATE:
			/*** This Read window is being activated -- so do some initializing ***/
			if (wParam) {
				/*** Enable the Edit menu. ***/
				EnableMenuItem(hMenuEdit, IDM_COPY, MF_BYCOMMAND | MF_ENABLED);

				/*** Move highlight bar in folder to match this message. ***/
				hLetterMem = GetLetter(hChildWnd);
				if (hLetterMem != NULL) {
					TempLetter = (Letter far *) GlobalLock(hLetterMem);
					if (IsWindow(TempLetter->hList)) {
						SendMessage(TempLetter->hList, LB_SETSEL, FALSE, MAKELPARAM(-1, 0));
						i = FindID(TempLetter->hFolder, TempLetter->MsgID);
						i = i < 0 ? 0 : i;
						SendMessage(TempLetter->hList, LB_SETSEL, TRUE, MAKELPARAM(i, 0));
					}
					GlobalUnlock(hLetterMem);
				}
			}
			break;


		case WM_SIZE:
			/*** Resize the Edit control to match the new size of the Read Window. ***/
			if(wParam == SIZENORMAL || wParam == SIZEFULLSCREEN) {
				hLetterMem = GetLetter(hChildWnd);
				if (hLetterMem != NULL) {
					TempLetter = (Letter far *) GlobalLock(hLetterMem);
					/*** Check the MsgID  in case the message was ***/
					/*** deleted while window was maximized.      ***/
					if(TempLetter->MsgID[0] != NULL) {
						MoveWindow(TempLetter->hViewMail, 0, 32, (int) (lParam & (LONG) 0x0000ffff), (int) ((lParam >> 16) - 32), TRUE);
						/*** Reformat letter and load into Edit Window ***/
						ReformatMessage(TempLetter->FolderName, TempLetter->MsgID, &env, TempLetter->hViewMail);
					}

					GlobalUnlock(hLetterMem);
				 }
			}
			break;

		case WM_SETCURSOR:
			/*** The mouse cursor is being moved -- use this message to add  ***/
			/*** status bar messages when the mouse cursor is over a button. ***/
			if (GetFocus() == hChildWnd || GetParent(GetFocus()) == hChildWnd) {

				/*** Get the Control ID of the Window with the Mouse ***/
				/*** If its a Button display the button Help text,   ***/
				/*** else delete the text 							 ***/
				dy = GetDlgCtrlID(wParam);
				i=0;
				while(i<BUTTONCOUNT && (dy != ReadBtns[i])) i++;
				if (dy == ReadBtns[i]) {
					/*** Load the string resource ***/
					LoadString(hInst, dy, buff, 100);

					/*** Send the string to the status bar ***/
  					if (hStatus)
						SendMessage(hStatus, STB_STRING, 0, (LONG) &buff);
				}
				else
					/*** Cursor over nothing important -- Blank out the status bar. ***/
  					if (hStatus)
						SendMessage(hStatus, STB_STRING, 0, 0L);
			}
			break;

	} /* end switch ()  */

	return (DefMDIChildProc(hChildWnd, message, wParam, lParam));
}


/******************************************************************************
	FUNCTION: 	ReformatMessage()

	PURPOSE:  	To return a handle to a string which contains a mail message
				in a format suitable for displaying.  This function retrieves
				the message text from an existing read window.

*******************************************************************************/
BOOL ReformatMessage(char far *FoldName, char far *MsID, Env far *env, HWND hEdit)
{
	GLOBALHANDLE	hEnvelope;
	char 			far *textstr, far *startptr, far *endptr;
	int				bytecnt, end;

	/*** Format the message envelope ***/
	FormatEnvelope(FoldName, MsID, env, &hEnvelope, hEdit, TRUE);
	textstr = GlobalLock(hEnvelope);
	end = (int) lstrlen(textstr);

	/*** Add the text from the existing read window. ***/
	startptr = textstr + end;
	bytecnt = SendMessage(hEdit, WM_GETTEXT, SendMessage(hEdit, WM_GETTEXTLENGTH, 0, 0L) + 1, (LPARAM) startptr);
	startptr[bytecnt] = '\0';

	/*** Locate the end of the original envelope info, delete ***/
	/*** the info, then reassign the text to read window.     ***/
	endptr = _fstrstr(startptr, "\r\012\r\012");
	if (endptr != NULL) {
		lstrcpy(startptr, endptr + 4);
		SendMessage(hEdit, WM_SETTEXT, 0, (LPARAM) textstr);
	}

	/*** Release the memory for the new envelope. ***/
	SafeFree(GLOBAL, hEnvelope);
	VnsReleaseMailEnvelope(hVNMAS, MsID);

	return(TRUE);
}
