/****************************************************************************
	FILE:		print.c


	CONTENTS:	AbortProc()		message loop procedure to run while printing.
				PrintMessage()	utility that works with PrintText() to print
								a formatted message.
				PrintText()		utility that works with PrintMessage() to print
								a formatted message.
				PrintWindow()	utility that prints the contents of an edit
								control to the printer


	COMMENTS:	The About dialog states version, name, and programmer.
				The Logo box redisplays the opening Logo with some added
				animation and sound.
****************************************************************************/

#include "MyMail.h"
#include "commdlg.h"
#include "string.h"
#include "print.h"

/*** PrintText is declared here instead of in MyMail.h ***/
/*** because Commdlg is not included in MyMail.h.      ***/
BOOL PrintText( PRINTDLG, HWND, unsigned int, char far *);
BOOL PrintWindow( PRINTDLG, HWND );


/****************************************************************************
	FUNCTION:	AbortProc()

	PURPOSE:	Message loop used when printing so that messages
				are still passed along.

	COMMENTS:	Necessary for implementing a cancel button.
****************************************************************************/
#pragma argsused
BOOL FAR PASCAL _export AbortProc(HDC hdcPrn, short nCode)
{
	MSG Msg;

	while (PeekMessage(&Msg, NULL, 0, 0, PM_REMOVE)) {
		TranslateMessage(&Msg);
		DispatchMessage(&Msg);
	}

	return TRUE;
}


/****************************************************************************
	FUNCTION:	PrintMessage()

	PURPOSE:	Prints all the currently selected messages.

	COMMENTS:	This procedure does some simple initializations, then calls
				the common dialog Print box and finally calls PrintText()
				for the actual printing chores.
****************************************************************************/
PrintMessage(LONG lParam)
{

	HWND			hEdit;
	unsigned int 	tabset[1];
	int				NumItems;
	PRINTDLG		pd;
	char			MsID[14], FoldName[14], buff[256];
	DWORD			tSize;
	GLOBALHANDLE	hEditDS;
	RECT			Rect;
	HDC				hDC;
	HFONT			hFont, hOldFont;
	unsigned int 	ScreenAspect, PrinterAspect, ScreenRes, PrinterRes;
	LOGFONT			lfPrint;
	TEXTMETRIC  	tm;

	/***** Get the Message ID and Folder Name for the Current or Hi-Lighted Messages ***/
	NumItems = GetLetterInfo((LPSTR) &MsID,  (LPSTR) &FoldName, lParam, &hTempArr);

	/*** No Messages in Folder or None Selected ***/
	if (NumItems <= 0) {
		BWCCMessageBox(hWnd, "No Message(s) Selected.", "WARNING!", MB_ICONSTOP | MB_OK);
		return FALSE;
	}

	/*** Multiple messages selected -- have user verify if not in Expert mode. ***/
	if (!Options.expertmode && NumItems > 1) {
		if (!CallDlg(hWnd, AskChange, YesNoBox, 6L)) {
			SafeFree(GLOBAL, hTempArr);
			return FALSE;
		}
	}


	/*** Create a New Data Segment for the hidden Edit window. ***/
	/*** This will maximize the edit control's available memory. ***/
	hEditDS = GlobalAlloc(GMEM_MOVEABLE | GMEM_ZEROINIT | GMEM_SHARE, 256L);
	if (hEditDS == NULL) {
		hEditDS = hInst;  // Allocation failed; use default.
	}

	/*** Create an invisible edit control for formatting the Message. ***/
	hEdit = CreateWindow("edit", "", WS_CHILD | ES_MULTILINE, 0, 0, 200,
			200, hWnd, PB_EDIT, hEditDS, NULL);

	if (!hEdit) {
		BWCCMessageBox(hWnd, "Unable to Format Mail Message for Printing!", "WARNING!", MB_OK | MB_ICONEXCLAMATION);
		SafeFree(GLOBAL, hEditDS);
		return FALSE;
	}

	/*** Set the font, size,  and tab settings in our hidden edit control. ***/
	SendMessage(hEdit, EM_LIMITTEXT, 0, 0L);
	SendMessage(hEdit, EM_GETRECT, 0, (LPARAM) (RECT far *) &Rect);
	SendMessage(hEdit, EM_FMTLINES, FALSE, 0L);
	hDC = GetDC(hEdit);
	ScreenAspect=GetDeviceCaps(hDC, LOGPIXELSY);
	ReleaseDC(hEdit, hDC);
	Rect.bottom += Rect.top;
	Rect.top = Rect.left = 0;


	/*** Initialize the PRINTDLG structure ***/
	NullFill((LPSTR) &pd, sizeof(PRINTDLG));
	pd.lStructSize = sizeof(PRINTDLG);
	pd.hwndOwner   = hWnd;
	pd.Flags = PD_RETURNDC | PD_HIDEPRINTTOFILE | PD_NOSELECTION | PD_NOPAGENUMS;

	/*** Call the Print common dialog box. ***/
	if (PrintDlg(&pd)) {
		/*** User OK'ed the print action. ***/
		if (pd.hDC == NULL)
			BWCCMessageBox(hWnd, "Not enough memory\nto access printer.", "HappyMail Print", MB_ICONHAND);
		else {
			/*** Change the Printer's font, & the edit windows font and size. ***/
			PrinterAspect=GetDeviceCaps(pd.hDC, LOGPIXELSY);
			Rect.right = GetDeviceCaps(pd.hDC, HORZRES) - (PrinterAspect/2);
			SendMessage(hEdit, EM_SETRECT, 0, (LPARAM) (RECT far *) &Rect);
			MoveWindow(hEdit, 0, 0, Rect.right, Rect.bottom, TRUE);
			GetINIFont("PrintFont", &lfPrint);
			lfPrint.lfHeight = (signed int) ((long)((long) lfPrint.lfHeight * (long)PrinterAspect)/(long)ScreenAspect);
			hFont = CreateFontIndirect(&lfPrint);
			hOldFont = SelectObject(pd.hDC, hFont);
			SendMessage(hEdit, WM_SETFONT, hFont, NULL);
			PrintText(pd, hEdit, NumItems, (LPSTR) &FoldName);
			DeleteObject(SelectObject(pd.hDC, hOldFont));
			DeleteDC(pd.hDC);
		}
	}

	/*** Destroy the Hidden Edit window. ***/
	DestroyWindow(hEdit);
	SafeFree(GLOBAL, hEditDS);
	return TRUE;
}


/****************************************************************************
	FUNCTION:	PrintText()

	PURPOSE:	Prints all the currently selected messages
****************************************************************************/
BOOL PrintText(PRINTDLG pd, HWND hEdit, unsigned int NumItems, char far *FoldName)
{
	char 			szMsg[50], far *MsgID, far *textstr, Message[14];
	BOOL			bError = FALSE;
	FARPROC			lpfnAbortProc, lpfnPrintCancel;
	unsigned int 	i, copy;
	GLOBALHANDLE	hMessage;
	Env				env;
	DOCINFO			DI;
	HFONT			hFont, hOldFont;


	/*** Create a Bar Graf dialog box to illustrate the progress of our printing. ***/
	bUserAbort = FALSE;
	lpfnPrintCancel = MakeProcInstance(BarGraf, hInst);
	hDlgCancel = CreateDialog(hInst, MAKEINTRESOURCE(BarGrafBox), hWnd, lpfnPrintCancel);
	if (NumItems == 1)
		lstrcpy((LPSTR) &szMsg,"Now Printing 1 Message");
	else
		wsprintf((char far *) &szMsg,"Now Printing %d Messages", NumItems);
	SendMessage(hDlgCancel, WM_SETTEXT, NULL, (LONG) &szMsg);


	/*** Set the Abort procedure to run while spooling to the printer. ***/
	lpfnAbortProc = MakeProcInstance(AbortProc, hInst);
	if (SetAbortProc(pd.hDC, (ABORTPROC) lpfnAbortProc) < 0) {
		BWCCMessageBox(hWnd, "Error starting print job.", "Error", MB_ICONSTOP);
		DestroyWindow(hDlgCancel);
		FreeProcInstance(lpfnAbortProc);
		FreeProcInstance(lpfnPrintCancel);
		return TRUE;
	}


	/*** Begin Printing. ***/
	lstrcpy((LPSTR) &szMsg, "H.M. - Messages");
	bError = FALSE;
	MsgID 	= (char far *) GlobalLock(hTempArr);
	DI.lpszDocName = (LPCSTR) &szMsg;
	DI.lpszOutput  = NULL;
	DI.cbSize = sizeof(DOCINFO);
	if (StartDoc(pd.hDC, &DI) > 0) {

		/*** Skip thru each message one at a time. ***/
		for (i=0; i<NumItems && !bUserAbort && !bError; i++) {

			/*** Read, format and store message in invisible edit window ***/
			lstrncpy((LPSTR)Message, &(MsgID[i*14]), 14);
			if (FormatMessage(FoldName, (LPSTR) &Message, &env, &hMessage, hEdit) != TRUE) {
				BWCCMessageBox(hWnd, "Unable to Read a Mail Message.", "WARNING!", MB_OK | MB_ICONEXCLAMATION);
				continue;
			}
			textstr = (char far *) GlobalLock(hMessage);
			SendMessage(hEdit, WM_SETTEXT, 0, (LONG) textstr);
			SafeFree(GLOBAL, hMessage);

			/*** Loop to print multiple copies of the same message. ***/
			for (copy=0;copy < pd.nCopies && !bUserAbort && !bError; copy++) {

				/*** Add printing data to Bar Graph Box ***/
				wsprintf((char far *) &szMsg,"Message #%d : Copy #%d", i+1, copy+1);
				SendMessage(GetDlgItem(hDlgCancel, BG_TITLE), WM_SETTEXT, NULL, (LONG) &szMsg);
				SendMessage(hDlgCancel, WM_INITDIALOG, 0, 0L);

				bError = PrintWindow(pd, hEdit);
				if (bError || bUserAbort)
					break;

			}
		}
	}
	else
		bError = TRUE;

	EndDoc(pd.hDC);

	if (bError)
		/*** Error occurred when initializing printer or when sending a page break. ***/
		BWCCMessageBox(hWnd, "Unable to print. Please check printer connection.", "HappyMail", MB_OK | MB_ICONEXCLAMATION);


	if (bUserAbort)
		/*** User aborted the print operation. ***/
		BWCCMessageBox(hWnd, "Print Operation Aborted!", "HappyMail", MB_OK | MB_ICONEXCLAMATION);
	else
		DestroyWindow(hDlgCancel);


	/*** Release memory blocks, bitmaps, and device contexts. ***/
	hDlgCancel = NULL;
	SafeFree(GLOBAL, hTempArr);
	FreeProcInstance(lpfnAbortProc);
	FreeProcInstance(lpfnPrintCancel);
	return(bError || bUserAbort);
}



/****************************************************************************
	FUNCTION:	PrintWindow()

	PURPOSE:	Prints the contents of an edit control
****************************************************************************/
BOOL PrintWindow(PRINTDLG pd, HWND hEdit)
{

	BITMAP			bm;
	HBITMAP			hBitMap;
	char 			far *psBuffer, datebuff[64], pagebuff[10];
	BOOL			bError;
	RECT			rect;
	unsigned int 	tabset[10];
	short			yChar, nCharsPerLine, nLinesPerPage, nTotalLines, nLength,
					nPage, nLine, nLineNum, nTotalPages, index, yStart, xStart,
					xSize, ySize;
	TEXTMETRIC  	tm;
	GLOBALHANDLE	hMessage, hBuffer;
	struct tm 		*curtime;
	time_t			timer;
	DWORD			tSize;
	HFONT			hFont;
	unsigned long	ulPercent;

	/*** Initialize date string for later printing. ***/
	timer = time(NULL);
	strcpy((char *) datebuff, ctime((time_t *)&timer));
	delnchr(datebuff, 0, 4);
	datebuff[6] = ',';
	delnchr(datebuff, 7, 8);
	datebuff[12] = '\0';

	/*** Initialize some variables based on text      ***/
	/*** sizes for the device context of the printer. ***/
	hBitMap = LoadBitmap(hInst, (LPSTR) "PrnLogo");
	GetTextMetrics(pd.hDC, &tm);
	GetObject(hBitMap, sizeof(BITMAP), (LPSTR) &bm);
	yStart = GetDeviceCaps(pd.hDC, VERTRES) - bm.bmHeight;
	xStart = GetDeviceCaps(pd.hDC, LOGPIXELSX)/4; // Start a quarter inch in from edge
	xSize  = GetDeviceCaps(pd.hDC, HORZRES);
	ySize  = GetDeviceCaps(pd.hDC, VERTRES);
	yChar = tm.tmHeight + tm.tmExternalLeading;
	nCharsPerLine = xSize / tm.tmAveCharWidth;
	nLinesPerPage = (yStart/yChar) - 2;
	if (nLinesPerPage == 0)  nLinesPerPage = 1;
	bError = FALSE;
	hFont = SendMessage(hEdit, WM_GETFONT, 0, 0L);



	/*** Calculate the tab size based on wether ***/
	/*** or not we're using a fixed=pitch font. ***/
	if (GetTextExtent(pd.hDC, "abc", 3) == GetTextExtent(pd.hDC, "WWW", 3))
		tabset[0] = GetTextExtent(pd.hDC, "Attached", 8);
	else
		tabset[0] = GetTextExtent(pd.hDC, "Attached:", 9)+4;

	/*** Grab a buffer big enough for one line of text.  ***/
	if ((hBuffer = SafeAlloc(GLOBAL, (long) 5 * nCharsPerLine, hWnd)) == NULL) {
		BWCCMessageBox(hWnd, "Unable to Complete Print Operation", "HappyMail", MB_OK | MB_ICONEXCLAMATION);
		DeleteObject(hBitMap);
		return TRUE;
	}
	psBuffer = (char far *) GlobalLock(hBuffer);


	/*** Set the number of lines and pages.  If   ***/
	/*** text box is empty, skip to next message. ***/
	if (0 == (nTotalLines = SendMessage(hEdit, EM_GETLINECOUNT, 0, 0L))) {
		/*** Release memory blocks, and bitmaps. ***/
		BWCCMessageBox(NULL, "No Text Found in Message", "ERROR", MB_ICONHAND);
		SafeFree(GLOBAL, hBuffer);
		DeleteObject(hBitMap);
		return FALSE;
	}
	nTotalPages   = (nTotalLines + nLinesPerPage - 1)/nLinesPerPage;


	/***** Output text of message -- one line at a time. ******/
	for (nPage=0, nLineNum=0; nPage < nTotalPages && !bUserAbort; nPage++) {


		/*** Start Printing another page. ***/
		if (StartPage(pd.hDC) < 0) {
			bError = TRUE;
			break;
		}

		for (nLine=0; nLine < nLinesPerPage && nLineNum < nTotalLines; nLine++, nLineNum++) {
			/*** Check for a user abort. ***/
			if (bUserAbort)
				break;

			*(short far *) psBuffer = 5 * nCharsPerLine;
			nLength = (short) SendMessage(hEdit, EM_GETLINE, nLineNum, (LONG) psBuffer);
			if (nLength > 0)
				TabbedTextOut(pd.hDC, xStart, yChar*nLine, psBuffer,	(short) nLength,
				   1, (int far *) &tabset, xStart);

			/*** Update Bar Graf dialog box. ***/
			if (hDlgCancel) {
				ulPercent = (unsigned long)(((unsigned long)nLineNum+1L)*100L)/(unsigned long)nTotalLines;
				SendMessage(hDlgCancel, WM_NEXTSTEP, (unsigned int) ulPercent, 0L);
			}

		}

		/*** Print Page number on Bottom of Page. ***/
		if (Options.pagenum) {
			wsprintf(pagebuff, "Page %u", nPage + 1);
			tSize = GetTextExtent(pd.hDC, pagebuff,  lstrlen(pagebuff));
			TextOut(pd.hDC, xSize - LOWORD(tSize), ySize - yChar, pagebuff,
					lstrlen(pagebuff));
		}


		/*** Draw logo on bottom of page. ***/
		if (hBitMap != NULL && Options.logo)
			DrawBitMap(pd.hDC, hBitMap, -1, yStart, FALSE, FALSE);

		/*** Print Date on Bottom of Page. ***/
		if (Options.date)
			TextOut(pd.hDC, 0, ySize - yChar, datebuff, lstrlen(datebuff));

		/*** Insert page break. ***/
		if (EndPage(pd.hDC) < 0) {
			bError = TRUE;
			break;
		}

		/*** The EndPage command wipes the font out of ***/
		/*** the Printer DC, so we reselect it here.   ***/
		SelectObject(pd.hDC, hFont);
	}

	/*** Release memory blocks, and bitmaps. ***/
	SafeFree(GLOBAL, hBuffer);
	DeleteObject(hBitMap);

	return(bError);

}

