/****************************************************************************
	FILE:		options.c


	CONTENTS:	ComposeOptions()Window Procedure for ComposeOptBox Dialog Box
				DirBrowseHook()	Hook procedure for picking directories
				FolderOptions()	Window procedure for FolderOptBox dialog box.
				GeneralOptions()Window procedure for GeneralOptBox dialog box.
				InitOptions()  	Read in options from the INI File
				PrintOptions()	Window procedure for PrintOptBox dialog box.
				ReadOptions()	Window procedure for ReadOptBox dialog box.
				SaveOptions()	Saves Options structure to the INI File.

	COMMENTS:	This file contains the procedures used in changing HappyMail
				options (except for fonts -- see fonts.c).
****************************************************************************/
#include "MyMail.h"
#include "commdlg.h"
#include "string.h"
#include "stdlib.h"


/******************************************************************************
	FUNCTION:	ComposeOptions()

	PURPOSE:	Window Procedure for ComposeOptBox Dialog Box
*******************************************************************************/
BOOL FAR PASCAL _export ComposeOptions(HWND hDlg, unsigned message, WORD wParam, LONG lParam)
{
	static BOOL		newFont = FALSE;
	static LOGFONT	lf;
	static HWND		hCompose;
	char			Settings[128];
	HWND			eHnd;
	BOOL			translated;

	/*** Send Window Messages to Status Bar ***/
	StatBarCtrlMsg(hDlg, ComposeOptBox, message, wParam, lParam);

	switch (message) {
		case WM_INITDIALOG:
			/*** Center the Dialog Window in the HappyMail Main window ***/
			CenterWindow(hDlg, GetDesktopWindow());

			/*** Set the text limits and prefill the     ***/
			/*** Signature field with the old signature. ***/
			SendDlgItemMessage(hDlg, COB_SIGN, EM_LIMITTEXT, MAX_SIGN, NULL);
			if (Options.sign[0] != '\0')
				SetDlgItemText(hDlg, COB_SIGN, (LPSTR) &Options.sign);

			/*** Change the font of the edit window ***/
			SendDlgItemMessage(hDlg, COB_SIGN, WM_SETFONT, GetStockObject(SYSTEM_FIXED_FONT), NULL);

			/*** Sub-class the multi-line Edit control by assigning ***/
			/***  a new Window procedure to the Edit control.       ***/
			eHnd = NULL;
			lpfnOldEdit = NULL;
			eHnd = GetDlgItem(hDlg, COB_SIGN);
			if (eHnd) {
				lpfnOldEdit = (FARPROC) GetWindowLong(eHnd, GWL_WNDPROC);
				if (lpfnOldEdit)
					SetWindowLong(eHnd, GWL_WNDPROC, (LONG) MakeProcInstance((FARPROC) EditProc, hInst));
			}

			/*** Prefill the other fields. ***/
			CheckDlgButton(hDlg, COB_CERTIFY, Options.certify);
			CheckRadioButton(hDlg, COB_SENDDEL, COB_JUSTSAVE, (COB_SENDDEL - 1) + Options.defaultdest);
			SetDlgItemInt(hDlg, COB_LINELENGTH, Options.linelength, FALSE);
			if ( GetINIFont("ComposeFont", &lf) )
				SetDlgItemText(hDlg, COB_FONTTEXT, (LPSTR) &lf.lfFaceName);

			hCompose = (HWND) lParam;

			break;

		case WM_COMMAND:
			switch(wParam) {
				case COB_FONT:
					newFont = FontPick(hDlg, COB_FONTTEXT, &lf);
					break;
				case IDCANCEL:
					/*** User selected Cancel or Close. ***/
					EndDialog(hDlg, FALSE);
					return(TRUE);
				case IDM_OK:
					/*** User selected OK.  Retrieve the new fields. ***/
					GetDlgItemText(hDlg, COB_SIGN, (LPSTR) &Options.sign, MAX_SIGN);
					Options.certify =  IsDlgButtonChecked(hDlg, COB_CERTIFY) ? 1 : 0;
					if (IsDlgButtonChecked(hDlg, COB_SENDDEL))
						Options.defaultdest = 1;
					if (IsDlgButtonChecked(hDlg, COB_SENDSAVE))
						Options.defaultdest = 2;
					if (IsDlgButtonChecked(hDlg, COB_JUSTSAVE))
						Options.defaultdest = 3;
					Options.linelength = GetDlgItemInt(hDlg, COB_LINELENGTH, &translated, FALSE);

					if (newFont) {
						/*** Write new font to INI file ***/
						wsprintf((LPSTR) Settings, "%s,%i,%i,%i,%i", (LPSTR) lf.lfFaceName,
							(int) lf.lfItalic, (int) lf.lfWeight, lf.lfHeight, lf.lfWidth);
						WritePrivateProfileString("Fonts", "ComposeFont", Settings, INI_FILE);

						/*** Change the font in the Compose Box, if its open. ***/
						if (hCompose)
							ChangeFont(&lf, GetDlgItem(hCompose, CB_MBODY));

					}


					SaveOptions();
					EndDialog(hDlg, TRUE);
					return (TRUE);
			} /** end switch(wParam) **/
			break;
	} /** end switch(message) **/
	return (FALSE);
}


/******************************************************************************
	FUNCTION:	DirBrowseHook()

	PURPOSE:	Dialog Procedure for Common Dialog Save As Box

	COMMENTS:	This hook is to center the dialog box and force the reader to
				select a path without choosing a specific file-name. It disables
				all file specific controls and adds some logic to determine when
				the user has selected a final directory.
*******************************************************************************/
#pragma argsused
UINT CALLBACK _export DirBrowseHook(HWND hDlg, UINT message, WORD wParam, LONG lParam)
{

	char	pathbuff[256], newpath[256], buff[128];
	int		i, j;

	switch (message) {
		case WM_INITDIALOG:       	     /* message: initialize dialog box */
			/*** Stretch the window, then Center it. ***/
			CenterWindow(hDlg, GetDesktopWindow());
			EnableWindow(GetDlgItem(hDlg, FILE_NAME), FALSE);
			EnableWindow(GetDlgItem(hDlg, FILE_CBTYPE), FALSE);
			return (TRUE);

		case WM_COMMAND:
			if (wParam == IDOK) {
				/*** If user selects OK and they haven't selected a different  ***/
				/*** sub-directory to move to, add a fake filename to the FILE ***/
				/*** NAME field, causing the Common DlgProc to close & return. ***/
				SendDlgItemMessage(hDlg, FILE_PATH, WM_GETTEXT, 128, (LPARAM) &pathbuff);
				_fstrupr((LPSTR) &pathbuff);
				i = SendDlgItemMessage(hDlg, FILE_LBPATH, LB_GETCURSEL, 0, 0L);
				if (i == LB_ERR) {
					return 0;
				}
				newpath[0] = '\0';
				for(j=0; j<=i; j++) {
					SendDlgItemMessage(hDlg, FILE_LBPATH, LB_GETTEXT, j, (LPARAM) &buff);
					lstrcat((LPSTR) &newpath, (LPSTR) &buff);
					if (lstrncmp(newpath, pathbuff, lstrlen(newpath)) != 0)
						return 0;
					if ((j>0) && (j!=i)) lstrcat(newpath,  "\\");
				}
				if (lstrcmp(newpath, pathbuff) != 0)
					return 0;
				EnableWindow(GetDlgItem(hDlg, FILE_CBTYPE), TRUE);
				SetDlgItemText(hDlg, FILE_NAME, "_");

			}

	} /** end switch (message) **/

	return (FALSE);
}


/****************************************************************************
	FUNCTION:	FolderOptions()

	PURPOSE:	Window procedure for "FolderOptBox" dialog box.
****************************************************************************/
BOOL FAR PASCAL _export FolderOptions(HWND hDlg, unsigned message, WORD wParam, LONG lParam)
{
	GLOBALHANDLE	hFold, hFold2;
	MailFolder far	*CurFolder;
	int				oldsort;
	static BOOL		newFont = FALSE;
	static LOGFONT	lf;
	char			Settings[128];
	HDC				hDC;
	HFONT			hFont, hOldFont;
	TEXTMETRIC		tm;



	/*** Send Window Messages to Status Bar ***/
	StatBarCtrlMsg(hDlg, FolderOptBox, message, wParam, lParam);

	switch (message) {
		case WM_INITDIALOG:
			/*** Center Dialog Window ***/
			CenterWindow(hDlg, GetDesktopWindow());
			CheckRadioButton(hDlg, FOB_DATE, FOB_SUBJECT, (FOB_DATE - 1) + Options.sortorder);
			if ( GetINIFont("FolderFont", &lf) )
				SetDlgItemText(hDlg, FOB_FONTTEXT, (LPSTR) &lf.lfFaceName);

			return (TRUE);
		case WM_COMMAND:
			switch(wParam) {
				case FOB_FONT:
					newFont = FontPick(hDlg, FOB_FONTTEXT, &lf);
					break;

				case IDM_OK:
					/*** User selected OK.  Copy the selected option to a character ***/
					/*** string and return that value to the parent window.         ***/
					oldsort = Options.sortorder;
					if (IsDlgButtonChecked(hDlg, FOB_DATE))
						Options.sortorder = 1;
					if (IsDlgButtonChecked(hDlg, FOB_NAME))
						Options.sortorder = 2;
					if (IsDlgButtonChecked(hDlg, FOB_STATUS))
						Options.sortorder = 3;
					if (IsDlgButtonChecked(hDlg, FOB_SUBJECT))
						Options.sortorder = 4;

					/*** Check to see if parameter changed. ***/
					if (Options.sortorder != oldsort) {

						/*** Change to HourGlass Cursor before starting ***/
						SetCursor(LoadCursor(NULL, IDC_WAIT));

						/*** Resort all open folders ***/
						hFold = TopFolder;
						while(hFold != NULL) {
							CurFolder = (MailFolder far *) GlobalLock(hFold);
							SendMessage(CurFolder->hListMail, (int) WM_LBRESET, 0, 0L);
							hFold2 = CurFolder->Next;
							GlobalUnlock(hFold);
							hFold = hFold2;
						}
					}

					/** Change Font in all open Folder windows **/
					if (newFont) {
						hFold = TopFolder;
						while(hFold != NULL) {
							CurFolder = (MailFolder far *) GlobalLock(hFold);
							hFold2 = CurFolder->Next;
							ChangeFont(&lf, CurFolder->hListBox);
							bNewListFont = TRUE;

							/*** Modify Height of items in list box. ***/
							hDC = GetDC(CurFolder->hListBox);
							hFont = SendMessage(CurFolder->hListBox, WM_GETFONT, 0, 0L);
							hOldFont = SelectObject(hDC, hFont);
							GetTextMetrics(hDC, &tm);
							SendMessage(CurFolder->hListBox, LB_SETITEMHEIGHT, 0, tm.tmHeight + 1);
							SelectObject(hDC, hOldFont);
							ReleaseDC(CurFolder->hListBox, hDC);

							GlobalUnlock(hFold);
							hFold = hFold2;
						}

						/*** Write new font to INI file ***/
						wsprintf((LPSTR) Settings, "%s,%i,%i,%i,%i", (LPSTR) lf.lfFaceName,
							(int) lf.lfItalic, (int) lf.lfWeight, lf.lfHeight, lf.lfWidth);
						WritePrivateProfileString("Fonts", "FolderFont", Settings, INI_FILE);
					}

					SaveOptions();

					/*** No break intentionally. ***/

				case IDCANCEL:
					/*** User selected Close or Cancel. ***/
					EndDialog(hDlg, TRUE);
					return (TRUE);
			} /** end switch(wParam) **/
	} /** end switch(message) **/
	return (FALSE);
}


/****************************************************************************
	FUNCTION:	GeneralOptions()

	PURPOSE:	Window procedure for "GeneralOptBox" dialog box.
*****************************************************************************/
BOOL FAR PASCAL _export GeneralOptions(HWND hDlg, unsigned message, WORD wParam, LONG lParam)
{
	char			Buff[80], FileBuff[256];
	OPENFILENAME 	ofn;
	DWORD			Errval;

	/*** Send Window Messages to Status Bar ***/
	StatBarCtrlMsg(hDlg, GeneralOptBox, message, wParam, lParam);

	switch (message) {
		case WM_INITDIALOG:

			/*** Center the Dialog Window in the HappyMail Main window ***/
			CenterWindow(hDlg, GetDesktopWindow());

			/*** Set the text limits and pre-fill the pathname   ***/
			/*** field with the old path (passed in via lParam). ***/
			SendDlgItemMessage(hDlg, GOB_PATH, EM_LIMITTEXT, 256, NULL);
			if (Options.defdir[0] != '\0')
				SetDlgItemText(hDlg, GOB_PATH, (LPSTR) &Options.defdir);
			CheckDlgButton(hDlg, GOB_STDA, Options.STDA);
			CheckDlgButton(hDlg, GOB_WARN, Options.expertmode);
			CheckDlgButton(hDlg, GOB_ICON, Options.icon);
			CheckDlgButton(hDlg, GOB_USEHEADER, Options.useheader);
			CheckDlgButton(hDlg, GOB_USEPREFIX, Options.useprefix);
			CheckRadioButton(hDlg, GOB_TOSEND, GOB_TOALLCOPY, (GOB_TOSEND - 1) + Options.answermode);
			SetDlgItemText(hDlg, GOB_PREFIX, (LPSTR) &Options.prefix);
			SendDlgItemMessage(hDlg, GOB_PREFIX, EM_LIMITTEXT, 5, NULL);
			break;

		case WM_COMMAND:
			switch(wParam) {
				case GOB_BROWSE:
					/*** Initialize the OpenFileName structure ***/
					ofn.lStructSize = sizeof( OPENFILENAME );
					ofn.hwndOwner = hDlg;
					ofn.hInstance = hInst;
					ofn.lpstrFilter = NULL;
					ofn.lpstrCustomFilter = NULL;
					ofn.nMaxCustFilter = 0;
					ofn.nFilterIndex = 0;
					ofn.lpstrFile = (LPSTR) &FileBuff;      // Stores the result in this variable
					ofn.nMaxFile = sizeof( FileBuff );
					ofn.lpstrFileTitle = NULL;
					ofn.nMaxFileTitle = 0;
					ofn.lpstrInitialDir = NULL;
					ofn.lpstrTitle = "Select a path for the Default Directory";
					ofn.Flags = OFN_HIDEREADONLY | OFN_NOCHANGEDIR | OFN_ENABLEHOOK;
					ofn.nFileOffset = 0;
					ofn.nFileExtension = 0;
					ofn.lpstrDefExt = "*";
					ofn.lCustData = NULL;
					ofn.lpfnHook = (DLGHOOKPROC) MakeProcInstance((FARPROC)DirBrowseHook, hInst);
					ofn.lpTemplateName = NULL;

					/*** Open Common dialog Box for Saving Files ***/
					if(GetSaveFileName( &ofn ) != TRUE) {
						/*** Possible error detected ***/
						Errval=CommDlgExtendedError();
						if(Errval!=0) {                  // 0 value means user selected Cancel
							wsprintf((LPSTR)&Buff,"Error = %ld",Errval);
							BWCCMessageBox(hDlg, "Unable to Choose Directory", Buff, MB_ICONSTOP | MB_OK);
						}
						return (TRUE);
					}

					FileBuff[ParseName(FileBuff)] = '\0';
					SetDlgItemText(hDlg, GOB_PATH, (LPSTR) &FileBuff);
					FreeProcInstance((FARPROC)ofn.lpfnHook);
					break;

				case IDCANCEL:
					/*** User selected Cancel or Close. ***/
					EndDialog(hDlg, FALSE);
					return(TRUE);

				case IDM_OK:
					/*** User selected OK. ***/
					/*** Retrieve the new pathname. ***/
					GetDlgItemText(hDlg, GOB_PATH, (LPSTR) &Options.defdir, 256);
					if (Options.defdir[0] != '\0')
						/*** If last character isn't a backslash add one. ***/
						if (Options.defdir[lstrlen(Options.defdir)-1] != '\\')
							lstrcat(Options.defdir, "\\");

					/*** Check for a valid pathname. ***/
					if (!fexist((LPSTR) &Options.defdir)) {
						BWCCMessageBox(hDlg, "Path Does Not Exist", "WARNING!", MB_ICONSTOP | MB_OK);
						return TRUE;
					}
					Options.STDA = IsDlgButtonChecked(hDlg, GOB_STDA) ? 1 : 0;
					Options.icon = IsDlgButtonChecked(hDlg, GOB_ICON) ? 1 : 0;
					Options.expertmode = IsDlgButtonChecked(hDlg, GOB_WARN) ? 1 : 0;
					Options.useheader = IsDlgButtonChecked(hDlg, GOB_USEHEADER) ? 1 : 0;
					Options.useprefix = IsDlgButtonChecked(hDlg, GOB_USEPREFIX) ? 1 : 0;
					GetDlgItemText(hDlg, GOB_PREFIX, (LPSTR) &Options.prefix, 6);

					/*** Save the answermode. ***/
					if (IsDlgButtonChecked(hDlg, GOB_TOSEND))
						Options.answermode = 1;
					if (IsDlgButtonChecked(hDlg, GOB_TOALL))
						Options.answermode = 2;
					if (IsDlgButtonChecked(hDlg, GOB_TOSENDCOPY))
						Options.answermode = 3;
					if (IsDlgButtonChecked(hDlg, GOB_TOALLCOPY))
						Options.answermode = 4;

					SaveOptions();
					EndDialog(hDlg, TRUE);
					return (TRUE);
			} /** end switch(wParam) **/
			break;
	} /** end switch(message) **/
	return (FALSE);
}


/****************************************************************************
	FUNCTION: 	InitOptions()

	PURPOSE:  	Reads in all the options from the INI File and stores them
				in the Options structure.
****************************************************************************/
InitOptions()
{

	char			tempbuff[512];
	int				i;


	Options.certify		= (BOOL) GetPrivateProfileInt("Options", "Certify", 0, INI_FILE);
	Options.happybook	= (BOOL) GetPrivateProfileInt("Options", "HappyMail Address Book", 0, INI_FILE);
	Options.expertmode	= (BOOL) GetPrivateProfileInt("Options", "Expert", 0, INI_FILE);
	Options.STDA		= (BOOL) GetPrivateProfileInt("Options", "STDA", 1, INI_FILE);
	Options.icon		= (BOOL) GetPrivateProfileInt("Options", "Icon", 0, INI_FILE);
	Options.logo		= (BOOL) GetPrivateProfileInt("Options", "Logo", 1, INI_FILE);
	Options.date		= (BOOL) GetPrivateProfileInt("Options", "Date", 0, INI_FILE);
	Options.pagenum		= (BOOL) GetPrivateProfileInt("Options", "Page Numbers", 0, INI_FILE);
	Options.usedefault	= (BOOL) GetPrivateProfileInt("Options", "Use Default", 0, INI_FILE);
	Options.usefilters	= (BOOL) GetPrivateProfileInt("Options", "Use Filters", 0, INI_FILE);
	Options.useheader	= (BOOL) GetPrivateProfileInt("Options", "Use Quote Header", 1, INI_FILE);
	Options.useprefix	= (BOOL) GetPrivateProfileInt("Options", "Use Prefix String", 0, INI_FILE);
	Options.defaultdest	= (char) GetPrivateProfileInt("Options", "Default Destination", 1, INI_FILE);
	Options.deletemode	= (char) GetPrivateProfileInt("Options", "Delete Mode", 1, INI_FILE);
	Options.answermode	= (char) GetPrivateProfileInt("Options", "Answer Mode", 1, INI_FILE);
	Options.sortorder	= (char) GetPrivateProfileInt("Options", "Sort Order", 1, INI_FILE);
	Options.linelength	= GetPrivateProfileInt("Options", "Line Length", 78, INI_FILE);

	GetPrivateProfileString("Options", "Prefix String", "> ", (LPSTR) &Options.prefix, 6, INI_FILE);


	/*** The Options.defdir variable is initial- ***/
	/*** ized in the InitSysVars() function.     ***/


	/*** Grab the signature and change Octal 36 pairs to CR-LF ***/
	/*** pairs. Also change octal 35's to tab characters.      ***/
	tempbuff[0] = '\0';
	GetPrivateProfileString("Options", "Signature", "\0", (LPSTR) &tempbuff, MAX_SIGN, INI_FILE);
	/*** Change Octal 36 pairs to CR-LF pairs and Octal 35's to tabs ***/
	for(i=0; tempbuff[i] != '\0'; i++) {
		if (tempbuff[i] == '\036' && tempbuff[i+1] == '\036') {
			tempbuff[i] = '\r';
			tempbuff[i+1] = '\012';
		}
		if (tempbuff[i] == '\035') tempbuff[i] = '\t';
	}
	if (tempbuff[0] == '\0') {
		lstrcpy((LPSTR) &tempbuff, "\r\012");
		lstrcat((LPSTR) &tempbuff, (LPSTR) &UserName);
		for(i=0; tempbuff[i] != '@' &&  tempbuff[i] != '\0'; i++);
		tempbuff[i] = '\0';
		lstrcat((LPSTR) &tempbuff, "\r\012via HappyMail!");
	}
	lstrcpy((LPSTR) &Options.sign, (LPSTR) &tempbuff);

	return;
}



/****************************************************************************
	FUNCTION:	PrintOptions()

	PURPOSE:	Window procedure for "PrintOptBox" dialog box.
****************************************************************************/
BOOL FAR PASCAL _export PrintOptions(HWND hDlg, unsigned message, WORD wParam, LONG lParam)
{
	static BOOL     newFont;
	static LOGFONT	lf;
	char			Settings[128];

	/*** Send Window Messages to Status Bar ***/
	StatBarCtrlMsg(hDlg, PrintOptBox, message, wParam, lParam);

	switch (message) {
		case WM_INITDIALOG:
			/*** Center Dialog Window ***/
			CenterWindow(hDlg, GetDesktopWindow());

			/*** Prefill data fields. ***/
			CheckDlgButton(hDlg, POB_LOGO, Options.logo);
			CheckDlgButton(hDlg, POB_DATE, Options.date);
			CheckDlgButton(hDlg, POB_PAGENUM, Options.pagenum);

			if ( GetINIFont("PrintFont", &lf) )
				SetDlgItemText(hDlg, POB_FONTTEXT, (LPSTR) &lf.lfFaceName);

			return (TRUE);
		case WM_COMMAND:
			switch(wParam) {
				case POB_FONT:
					newFont = FontPick(hDlg, POB_FONTTEXT, &lf);
					break;

				case IDM_OK:
					Options.logo	= IsDlgButtonChecked(hDlg, POB_LOGO) ? 1 : 0;
					Options.date	= IsDlgButtonChecked(hDlg, POB_DATE) ? 1 : 0;
					Options.pagenum	= IsDlgButtonChecked(hDlg, POB_PAGENUM) ? 1 : 0;

					if (newFont) {
						/*** Write new font to INI file ***/
						wsprintf((LPSTR) Settings, "%s,%i,%i,%i,%i", (LPSTR) lf.lfFaceName,
							(int) lf.lfItalic, (int) lf.lfWeight, lf.lfHeight, lf.lfWidth);
						WritePrivateProfileString("Fonts", "PrintFont", Settings, INI_FILE);
					}

					SaveOptions();
					/*** No break intentionally. ***/

				case IDCANCEL:
					/*** User selected Close or Cancel. ***/
					EndDialog(hDlg, (wParam == IDM_OK) ? TRUE : FALSE);
					return (TRUE);
			} /** end switch(wParam) **/
	} /** end switch(message) **/
	return (FALSE);
}


/****************************************************************************
	FUNCTION:	ReadOptions()

	PURPOSE:	Window procedure for "ReadOptBox" dialog box.
*****************************************************************************/
BOOL FAR PASCAL _export ReadOptions(HWND hDlg, unsigned message, WORD wParam, LONG lParam)
{

	char			Settings[128];
	static LOGFONT	lf;
	static BOOL		newFont;
	GLOBALHANDLE 	hTempMem, hTemp;
	Letter 			far *TempLetter;

	/*** Send Window Messages to Status Bar ***/
	StatBarCtrlMsg(hDlg, ReadOptBox, message, wParam, lParam);

	switch (message) {
		case WM_INITDIALOG:
			/*** Center Dialog Window ***/
			CenterWindow(hDlg, GetDesktopWindow());

			/*** Place a check mark by the current delete mode option. ***/
			CheckRadioButton(hDlg, ROB_CLOSE, ROB_PREV, (ROB_CLOSE - 1) + Options.deletemode);

			if ( GetINIFont("ReadFont", &lf) )
				SetDlgItemText(hDlg, ROB_FONTTEXT, (LPSTR) &lf.lfFaceName);

			return (TRUE);

		case WM_COMMAND:
			switch(wParam) {
				case ROB_FONT:
					newFont = FontPick(hDlg, ROB_FONTTEXT, &lf);
					break;


				case IDM_OK:
					/*** User selected OK.  Copy the selected option to a character ***/
					/*** string and return that value to the parent window.         ***/
					if (IsDlgButtonChecked(hDlg, ROB_CLOSE))
						Options.deletemode = 1;
					if (IsDlgButtonChecked(hDlg, ROB_NEXT))
						Options.deletemode = 2;
					if (IsDlgButtonChecked(hDlg, ROB_PREV))
						Options.deletemode = 3;

					if (newFont) {
						/** Change Font in all open Letter windows **/
						hTempMem = TopLetter;
						while(hTempMem != NULL) {
							TempLetter = (Letter far *) GlobalLock(hTempMem);
							hTemp = TempLetter->Next;
							ChangeFont(&lf, TempLetter->hViewMail);
							GlobalUnlock(hTempMem);
							hTempMem = hTemp;
						}

						/*** Write new font to INI file ***/
						wsprintf((LPSTR) Settings, "%s,%i,%i,%i,%i", (LPSTR) lf.lfFaceName,
							(int) lf.lfItalic, (int) lf.lfWeight, lf.lfHeight, lf.lfWidth);
						WritePrivateProfileString("Fonts", "ReadFont", Settings, INI_FILE);
					}

					SaveOptions();

					/*** No break intentionally. ***/

				case IDCANCEL:
					/*** User selected Close or Cancel. ***/
					EndDialog(hDlg, (wParam == IDM_OK) ? TRUE : FALSE);
					return (TRUE);
			} /** end switch(wParam) **/
	} /** end switch(message) **/
	return (FALSE);
}



/****************************************************************************
	FUNCTION: 	SaveOptions()

	PURPOSE:  	Saves all the options in the Options structure to the INI
				File.
****************************************************************************/
SaveOptions()
{

	char			Signature[512];
	int				i;

	WritePrivateProfileInt("Options", "Certify", Options.certify, INI_FILE);
	WritePrivateProfileInt("Options", "HappyMail Address Book", Options.happybook, INI_FILE);
	WritePrivateProfileInt("Options", "Expert", Options.expertmode, INI_FILE);
	WritePrivateProfileInt("Options", "STDA", Options.STDA, INI_FILE);
	WritePrivateProfileInt("Options", "Icon", Options.icon, INI_FILE);
	WritePrivateProfileInt("Options", "Logo", Options.logo, INI_FILE);
	WritePrivateProfileInt("Options", "Date", Options.date, INI_FILE);
	WritePrivateProfileInt("Options", "Page Numbers", Options.pagenum, INI_FILE);
	WritePrivateProfileInt("Options", "Default Destination", Options.defaultdest, INI_FILE);
	WritePrivateProfileInt("Options", "Delete Mode", Options.deletemode, INI_FILE);
	WritePrivateProfileInt("Options", "Use Default", Options.usedefault, INI_FILE);
	WritePrivateProfileInt("Options", "Use Filters", Options.usefilters, INI_FILE);
	WritePrivateProfileInt("Options", "Answer Mode", Options.answermode, INI_FILE);
	WritePrivateProfileInt("Options", "Line Length", Options.linelength, INI_FILE);
	WritePrivateProfileInt("Options", "Sort Order", Options.sortorder, INI_FILE);
	WritePrivateProfileInt("Options", "Use Quote Header", Options.useheader, INI_FILE);
	WritePrivateProfileInt("Options", "Use Prefix String", Options.useprefix, INI_FILE);
	WritePrivateProfileString("Options", "Prefix String", (LPSTR) &Options.prefix, INI_FILE);
	WritePrivateProfileString("Options", "Save Directory", (LPSTR) &Options.defdir, INI_FILE);

	/*** Since carriage returns, linefeeds and tabs confuse the INI file,  ***/
	/*** they must be modified. So... Change CR-LF pairs to Octal 36       ***/
	/*** pairs and convert tabs to Octal 35's -- before saving to INI file.***/
	lstrcpy(Signature, (LPSTR) &Options.sign);
	for(i=0; Signature[i] != '\0'; i++) {
		if (Signature[i] == '\r' && Signature[i+1] == '\012') {
			Signature[i] = '\036';
			Signature[i+1] = '\036';
		}
		if (Signature[i] == '\t') Signature[i] = '\035';
	}

	/*** Store the modified signature string to the INI file. ***/
	WritePrivateProfileString("Options", "Signature", (LPSTR) &Signature, INI_FILE);

	/*** Flush INI buffer ***/
	WritePrivateProfileString(NULL, NULL, NULL, INI_FILE);

	return;
}








