/****************************************************************************
	FILE:		list.c


	CONTENTS:   ListMail()	window procedure for Folder windows.
				ListProc()	subclassed window procedure for a Folder
							window's list box.
				ReadBlock() reads in a block of message structure
				ReadID()    returns the msg ID of an item in a Folder window
				ReadItem()  returns the text string for an item in a Folder window


	COMMENTS:	This file contains the window procedures used to manage
				Folder windows.
****************************************************************************/
#include "MyMail.h"
#include "string.h"

/****************************************************************************
	FUNCTION:	ListMail()

	PURPOSE:	Processes messages for a Folder window

	COMMENTS:	The Folder window is an MDI child window and uses the
				DefMDIChildProc() versus the standard DefWindowProc().
				For more information on the MDI interface see Petzold's
				Programming Windows.

				This procedure utilizes a fake window message (WM_LBRESET).
				Its purpose is to wipe out the contents of the Folders
				list box and refill it by reading in the messages from the
				Banyan folder. This is necessary to refresh a Folder's
				list after receiving new mail or after a move or delete
				action has occurred.
****************************************************************************/

/*** This array determines which buttons are used in a Folder window ***/
/*** and what order they will appear in.  Future revisions might use ***/
/*** this as a means of allowing the user to modify his button bars. ***/
#define BUTTONCOUNT		11
#define GETMSGAMOUNT	30
int	buttons[BUTTONCOUNT] = { 	IDM_READ, 	IDM_COMPOSE, 	IDM_ANSWER,		IDM_DELETE,
								IDM_MOVE, 	IDM_SEND, 	 	IDM_FILE, 		IDM_FORWARD,
								IDM_PRINT,  IDM_MANAGE,		IDM_ATTACHMENTS  };

BOOL bAttach = FALSE;

long FAR PASCAL _export ListMail(HWND hChildWnd, WORD message, WORD wParam, LONG lParam)
{
	char 				buff[100];
	MsgInfo 	 		OneMsg;
	int 				y, i, top, hilite[1];
	unsigned short	 	offset=0, retcount=0, foldercount, msgcount, count, x;
	static unsigned int tabset[7];
	MailFolder far 		*NewFolder;
	HWND				hButton, hListBox;
	GLOBALHANDLE		hTempMem;
	FARPROC     		lpfnOldList, lpfnNewList;
	static HWND			hWndFrame, hWndClient;
	LOGFONT				lf;
	HFONT				hFont, hOldFont;
	LPDRAWITEMSTRUCT	lpDIS;
	LPMEASUREITEMSTRUCT	lpMIS;
	RECT				Rect;
	HDC					hDC;
	TEXTMETRIC			tm;
	CallStatus			uStat;
	LPSTR				SampleText= "\tUNREAD\tMar 30\tFrom:\tWWWWWWWWWWWWWWW";
	WINDOWPLACEMENT 	wp;


	switch (message) {

		case WM_CREATE:
			/*** Create a Folder structure and add it to the linked list of Folders ***/
			SendMessage(hChildWnd, WM_GETTEXT, 14, (LONG) &buff);
			if ((hTempMem = AddFolder((char far *) &buff)) == NULL) {
				/*** Error occurred creating new Folder structure ***/
				DspMsgCode( hWnd, "Error creating Folder window : ", 3, FALSE);
				SendMessage(GetParent(hChildWnd), WM_MDIDESTROY, hChildWnd, NULL);
				return (FALSE);
			}


			/*** Now, create the list box ***/
			hListBox = CreateWindow("listbox", NULL, WS_VISIBLE | WS_VSCROLL | LBS_USETABSTOPS |
				LBS_NOTIFY | LBS_NOINTEGRALHEIGHT |	WS_CHILDWINDOW  | LBS_EXTENDEDSEL | WS_BORDER |
				LBS_OWNERDRAWFIXED | WS_TABSTOP , 0, 32, 629, (GetFontHeight(hChildWnd) * 12) - 32,
				hChildWnd, LISTBOX_ID, hInst, NULL);


			/*** List box could not be created -- do an error return ***/
			if (!hListBox) {
				DspMsgCode( hWnd, "Error creating Folder window : ", 1, FALSE);
				SendMessage(GetParent(hChildWnd), WM_MDIDESTROY, hChildWnd, NULL);
				return (FALSE);
			}

			/*** Change font if necessary  ***/
			if (GetINIFont("FolderFont", &lf))
				ChangeFont(&lf, hListBox);

			/*** Set the Height of a line in the List box. ***/
			hDC = GetDC(hListBox);
			hOldFont = hFont = NULL;
			if (hDC) {
				if ((hFont = SendMessage(hListBox, WM_GETFONT, 0, 0L)) == NULL)
					hOldFont = SelectObject(hDC, hFont);
				GetTextMetrics(hDC, &tm);
				SendMessage(hListBox, LB_SETITEMHEIGHT, 0, tm.tmHeight + 1);
				if (hOldFont)
					SelectObject(hDC, hOldFont);
				ReleaseDC(hListBox, hDC);
			}


			/*** Add Buttons to the List Box -- Buttons are superclassed Borland  ***/
			/*** Buttons.  They've been modified to handle cursor keys like tabs. ***/
			for(i=0;i<BUTTONCOUNT;i++) {
				hButton = CreateWindow("SuperBorBtn", "", WS_VISIBLE | WS_CHILD |
					BS_DEFPUSHBUTTON | BBS_PARENTNOTIFY, i * 62, 0, 62, 32, hChildWnd,
					buttons[i],	hInst, NULL);

				/*** Couldn't create a button -- do an error return ***/
				if (!hButton) {
					DspMsgCode( hWnd, "Error creating Folder window : ", 2, FALSE);
					SendMessage(GetParent(hChildWnd), WM_MDIDESTROY, hChildWnd, NULL);
					return (FALSE);
				}

			}

			/*** Store address of old listbox window procedure ***/
			/*** and create a new list box window procedure.   ***/
			lpfnOldList = (FARPROC) GetWindowLong(hListBox, GWL_WNDPROC);
			lpfnNewList = MakeProcInstance((FARPROC) ListProc, hInst);

			/*** Initialize the fields of the new Folder structure ***/
			NewFolder = (MailFolder far *) GlobalLock(hTempMem);
			NewFolder->lpOldListProc = lpfnOldList;
			NewFolder->hListBox  	 = hListBox;
			NewFolder->hListMail 	 = hChildWnd;

			/*** Assign the new list box window procedure. ***/
			SetWindowLong(hListBox, GWL_WNDPROC, (LONG) lpfnNewList);

			/*** Set handle variables for future use ***/
			hWndClient = GetParent(hChildWnd);
			hWndFrame  = GetParent(hWndClient);


			/*** If this is the first folder gray the close option. ***/
			if (hTempMem == TopFolder)
				EnableMenuItem(GetSystemMenu(NewFolder->hListMail, FALSE), SC_CLOSE, MF_BYCOMMAND | MF_GRAYED);

			/*** Unlock the Folder structure ***/
			GlobalUnlock(hTempMem);

			PostMessage(hChildWnd, WM_LBRESET, 0, 0L);

			break;


		case WM_LBRESET:
			/*** Fake window message for refilling the  ***/
			/*** list box with the Folder's messages.   ***/
			/*** Empty the list box, and save the index ***/
			/*** of the selected message. 				***/
			top = hilite[0] = 0;
			hTempMem = GetFolder(hChildWnd);

			NewFolder = (MailFolder far *) GlobalLock(hTempMem);
			SendMessage(NewFolder->hListBox, WM_SETREDRAW, 0, NULL);
			top = SendMessage(NewFolder->hListBox, LB_GETTOPINDEX, NULL, NULL);
			SendMessage(NewFolder->hListBox, LB_GETSELITEMS, 1, (LONG) &hilite);
			SendMessage(NewFolder->hListBox, LB_RESETCONTENT, NULL, NULL);

			/*** Wipe the messages out of the folder structure memory. ***/
			if (wParam == 0)
				for(i=0;i<MaxMsgs;i++) NewFolder->Item[i].flags = 0x00;


			/*** Update the status Bar.    ***/
			if (hStatus)
				SendMessage(hStatus, STB_STRING, 0, (LPARAM) "Loading Messages ...");

			offset = 0;
			uStat = VnsListMailMsgs(hVNMAS, UserName, NewFolder->Name,(MsgInfo WFAR *)&OneMsg, 0, 1, &retcount,&msgcount, CREATEDATE);
			if (Recover(uStat))
				VnsListMailMsgs(hVNMAS, UserName, NewFolder->Name,(MsgInfo WFAR *)&OneMsg, 0, 1, &retcount,&msgcount, CREATEDATE);
			NewFolder->MsgCount = msgcount;

			for(i=0;i<msgcount && i<MaxMsgs;i++)
				SendMessage(NewFolder->hListBox, LB_ADDSTRING, 0, (long) i);

			if ((offset==0) && (top==0) && (hilite[0]==0)) {
				SendMessage(NewFolder->hListBox, LB_SETTOPINDEX, 0, 0L);
				SendMessage(NewFolder->hListBox, LB_SETSEL, TRUE, 0L);
			}


			/*** Select a message from the listbox. ***/
			if (SendMessage(NewFolder->hListBox, LB_GETSELCOUNT, 0, 0L)==0) {
				SendMessage(NewFolder->hListBox, LB_SETTOPINDEX, top, NULL);
				SendMessage(NewFolder->hListBox, LB_SETSEL, TRUE, MAKELPARAM(hilite[0],0));
			}

			/*** Reset the redraw flag and repaint the list box ***/
			SendMessage(NewFolder->hListBox, WM_SETREDRAW, 1, NULL);
			InvalidateRect(NewFolder->hListBox, NULL, FALSE);


			/*** Reset the title of the list box with the new message count ***/
			WriteTitleBar(NewFolder);

			/*** Sort by the current sort order. ***/
			/*** Update the status Bar.    ***/
			if (hStatus)
				SendMessage(hStatus, STB_STRING, 0, (LONG) "Sorting Messages ...");
			if (Options.sortorder != BYDATE)
				SortLists(NewFolder->Name);

			/*** Unlock the Folder structure ***/
			GlobalUnlock(hTempMem);

			/*** Blank out the Status bar. ***/
			if (hStatus)
				SendMessage(hStatus, STB_STRING, 0, 0L);

			return (TRUE);


		case WM_DRAWITEM:
			{
				auto int		iDcOld, iCount;
				auto HDC		hDC;
				auto RECT		rc;
				auto char		szText[256];
				auto DWORD		oldTextColor, oldBkgdColor;


				/*** Grab a Folder pointer. ***/
				hTempMem = GetFolder(hChildWnd);
				NewFolder = (MailFolder far *) GlobalLock(hTempMem);


				lpDIS	= (DRAWITEMSTRUCT far *) lParam;
				hDC		= lpDIS->hDC;
				rc		= lpDIS->rcItem;
				iDcOld 	= SaveDC(hDC);
				szText[0] = '\t';

				/*** If the font was changed recently reset the tabs and height of the list box. ***/
				if (bNewListFont) {
					tabset[0] = 25;
					tabset[1] = GetTabbedTextExtent(hDC, SampleText,  7, 1, (int far *) &tabset)+25;
					tabset[2] = GetTabbedTextExtent(hDC, SampleText, 14, 2, (int far *) &tabset)+25;
					tabset[3] = GetTabbedTextExtent(hDC, SampleText, 20, 3, (int far *) &tabset)+5;
					tabset[4] = GetTabbedTextExtent(hDC, SampleText, 34, 4, (int far *) &tabset);
					tabset[5] = GetTabbedTextExtent(hDC, SampleText, 36, 4, (int far *) &tabset)+20;
					tabset[6] = 1200;
					bNewListFont = FALSE;
				}

				if (lpDIS->itemAction & ODA_DRAWENTIRE) {

					/*** Gray the Foreground color if necessary. ***/
					if ((lpDIS->itemState & ODS_DISABLED) ||
						(lpDIS->itemState & ODS_GRAYED))
						SetTextColor(hDC, GetSysColor(COLOR_GRAYTEXT));


					if (lpDIS->itemState & ODS_SELECTED) {
						oldTextColor = SetTextColor(hDC, GetSysColor(COLOR_HIGHLIGHTTEXT));
						oldBkgdColor = SetBkColor(hDC, GetSysColor(COLOR_HIGHLIGHT));
					}

					ReadItem(NewFolder, lpDIS->itemID, (char far *)&szText + 1);
					iCount = lstrlen((LPSTR) &szText);
					TabbedTextOut(hDC, rc.left + 1, rc.top + 1, szText, iCount, 7, (int far *) &tabset, 0);
					if (NewFolder->Item[lpDIS->itemID].flags & ATTACHED)
						DrawBitMap(hDC, hBMPaperClip, rc.left, rc.top+2, TRUE, FALSE);

					if (lpDIS->itemState & ODS_SELECTED) {
						SetTextColor(hDC, oldTextColor);
						SetBkColor(hDC, oldBkgdColor);
					}

				}
				else {

					if (lpDIS->itemAction & ODA_SELECT) {
						if (lpDIS->itemState & ODS_SELECTED) {
							oldTextColor = SetTextColor(hDC, GetSysColor(COLOR_HIGHLIGHTTEXT));
							oldBkgdColor = SetBkColor(hDC, GetSysColor(COLOR_HIGHLIGHT));
						}

						ReadItem(NewFolder, lpDIS->itemID, (char far *) &szText + 1);
						iCount = lstrlen((LPSTR) &szText);
						TabbedTextOut(hDC, rc.left + 1, rc.top + 1, szText, iCount, 7, (int far *) &tabset, 0);
						if (NewFolder->Item[lpDIS->itemID].flags & ATTACHED)
							DrawBitMap(hDC, hBMPaperClip, rc.left, rc.top+2, TRUE, FALSE);

						if (lpDIS->itemState & ODS_SELECTED) {
							SetTextColor(hDC, oldTextColor);
							SetBkColor(hDC, oldBkgdColor);
						}
					}

				}
				RestoreDC(hDC, iDcOld);
			}

			GlobalUnlock(hTempMem);

			return TRUE;


		case WM_MEASUREITEM:
			lpMIS = (MEASUREITEMSTRUCT far *) lParam;
			lpMIS->itemWidth = 200;
			lpMIS->itemHeight = 30;
			return TRUE;


		case WM_DESTROY:
			/*** Release the Folder structure and its memory. ***/
			KillFolder(hChildWnd);
			break;


		case WM_RBUTTONDBLCLK:
		case WM_NCRBUTTONDBLCLK:
			/*** Add support for the right button double click. ***/
			if (TopFolder != NULL) {
				NewFolder = (MailFolder far *) GlobalLock(TopFolder);
				if (NewFolder->Next != NULL)
					PostMessage(hChildWnd, WM_SYSCOMMAND, SC_CLOSE, 0L);
				GlobalUnlock(TopFolder);
			}
			return 0;


		case WM_COMMAND:
			switch (HIWORD(lParam)) {

				case LBN_DBLCLK:
					/*** The user double-clicked the mouse on a message in  ***/
					/*** the list box -- so initiate a READ command, or the ***/
					/*** ATTACH command if they clicked on the paperclip.   ***/
					if (bAttach) {
						bAttach = FALSE;
						hTempMem = GetFolder(hChildWnd);
						NewFolder = (MailFolder far *) GlobalLock(hTempMem);
						SendMessage(NewFolder->hListBox, LB_GETSELITEMS, 1, (LONG) &hilite);
						ReadItem(NewFolder, hilite[0], (LPSTR) &buff);
						if (NewFolder->Item[hilite[0]].flags & ATTACHED) {
							GlobalUnlock(hTempMem);
							SendMessage(hWnd, WM_COMMAND, IDM_ATTACHMENTS, NULL);
							break;
						}
						GlobalUnlock(hTempMem);
					}
					SendMessage(hWnd, WM_COMMAND, IDM_READ, NULL);
					break;


				case BBN_GOTATAB:
					/*** User pressed [TAB] so go to the next control. ***/
					if (wParam != IDM_MANAGE)
						SetFocus(GetWindow(LOWORD(lParam), GW_HWNDNEXT));
					break;

				case BBN_GOTABTAB:
					/*** User pressed [SHIFT] [TAB] so go to the previous control. ***/
					SetFocus(GetWindow(LOWORD(lParam), GW_HWNDPREV));
					break;

				case ACC_KEYPRESS:
					/*** Accelerator Key was pressed. ***/
				case BN_CLICKED:
					/*** Button was selected. ***/
					switch(wParam) {
						case IDCANCEL:
							/*** Cancel was selected ***/
							if (TopFolder != NULL) {
								NewFolder = (MailFolder far *) GlobalLock(TopFolder);
								if (NewFolder->Next != NULL)
									PostMessage(hChildWnd, WM_SYSCOMMAND, SC_CLOSE, 0L);
								GlobalUnlock(TopFolder);
							}
							return 0;

						case IDM_READ:
						case IDM_COMPOSE:
						case IDM_ANSWER:
						case IDM_DELETE:
						case IDM_MOVE:
						case IDM_SEND:
						case IDM_FILE:
						case IDM_FORWARD:
						case IDM_PRINT:
						case IDM_MANAGE:
						case IDM_ATTACHMENTS:
							/*** Pass all button and accelerator key messages on  ***/
							/*** to the main Mail window. The code to handle all  ***/
							/*** these messages already exists in that procedure. ***/
							SendMessage(hWnd, WM_COMMAND, wParam, NULL);
							return 0;

				}  /** end switch **/
				break;
			} /*** end switch(HIWORD(lParam)) ***/

			break;


		case WM_SIZE:
			/*** Folder is being resized -- List box must also be resized to match ***/
			if(wParam == SIZENORMAL || wParam == SIZEFULLSCREEN) {
				/*** Grab pointer to Folder structure ***/
				hTempMem = GetFolder(hChildWnd);
				if (hTempMem != NULL) {
					NewFolder = (MailFolder far *) GlobalLock(hTempMem);
					/*** Stretch the list box to match the folder size. ***/
					MoveWindow(NewFolder->hListBox, 0, 32, (int) LOWORD(lParam), (int) (HIWORD(lParam) - 32), FALSE);
					/*** Release Folder structure pointer ***/
					GlobalUnlock(hTempMem);
				}
			}
			break;

		case WM_MDIACTIVATE:
			/*** This Folder is being activated -- so do some initializing ***/
			/*** No break intentional. 									   ***/
			if (wParam)
				EnableMenuItem(hMenuEdit, IDM_COPY, MF_BYCOMMAND | MF_GRAYED);

		case WM_SETFOCUS:
			/*** This code is necessary to give the focus to the list ***/
			/*** box when HM is restored (from a Minimized position). ***/
			if (wParam) {
				/*** Give the focus to the listbox ***/
				hTempMem = GetFolder(hChildWnd);
				if (hTempMem != NULL) {
					NewFolder = (MailFolder far *) GlobalLock(hTempMem);
					SetFocus(NewFolder->hListBox);
					GlobalUnlock(hTempMem);
				}
			}
			break;


		case WM_SETCURSOR:
			/*** The mouse cursor is being moved -- use this message to add  ***/
			/*** status bar messages when the mouse cursor is over a button. ***/
			if (GetParent(GetFocus()) == hChildWnd) {

				/*** Get the Control ID of the Window with the Mouse ***/
				/*** If its a Button display the button Help text,   ***/
				/*** else delete the text.                           ***/
				y = GetDlgCtrlID(wParam);
				i=0;
				while(i<BUTTONCOUNT && (y != buttons[i])) i++;
				if (y == buttons[i]) {
					/*** Load the string resource ***/
					LoadString(hInst, y, buff, 100);
					/*** Send the string to the status bar ***/
					if (hStatus)
						SendMessage(hStatus, STB_STRING, 0, (LONG) &buff);
				}
				else {
					if (y == LISTBOX_ID) {
						/*** Mouse is over the listbox -- so check if there are ***/
						/*** more than 200 messages and if so send the warning. ***/
						hTempMem = GetFolder(hChildWnd);
						if (hTempMem != NULL) {
							NewFolder = (MailFolder far *) GlobalLock(hTempMem);
							msgcount = NewFolder->MsgCount;
							GlobalUnlock(hTempMem);
							if (msgcount > MaxMsgs) {
								wsprintf(buff, "WARNING! Folder contains more than %u messages.  Only most recent %u shown.", MaxMsgs, MaxMsgs);
								if (hStatus)
									SendMessage(hStatus, STB_STRING, 0, (LONG) &buff);
								break;
							}
						}
					}
					/*** If no other message is sent - Erase the current Status bar message. ***/
					if (hStatus)
						SendMessage(hStatus, STB_STRING, 0, 0L);
				}
			}
			break;

	} /* end switch ()  */

	return (DefMDIChildProc(hChildWnd, message, wParam, lParam));
}



/****************************************************************************
	FUNCTION: 	ListProc()

	PURPOSE:  	Processes messages for "ListMail" -- the child window
				listbox of a Folder window.
****************************************************************************/

long FAR PASCAL _export ListProc(HWND hwnd, WORD message, WORD wParam, LONG lParam)
{


	GLOBALHANDLE 	hTempFold;
	MailFolder far 	*OpenFolder;
	long		 	retval;


	/*** Grab a pointer to the folder structure ***/
	/*** which corresponds to this listbox.     ***/
	hTempFold = GetFolder(GetParent(hwnd));
	if (hTempFold != NULL)
		OpenFolder = (MailFolder far *) GlobalLock(hTempFold);

	switch (message) {

		case WM_RBUTTONDBLCLK:
		case WM_NCRBUTTONDBLCLK:
			/*** Add support for the right button double click. ***/
			PostMessage(GetParent(hwnd), WM_RBUTTONDBLCLK, 0, 0L);
			break;

		case WM_LBUTTONDBLCLK:
			/*** Add support for the right button double click. ***/
			if (LOWORD(lParam) < 14) bAttach = TRUE;
			break;

		case WM_CHAR:
			/*** If the user presses the [ENTER] key or the [SPACE-BAR] ***/
			/*** send a READ command to the main window procedure.      ***/
			if (wParam == VK_RETURN)
				if (IsIconic(GetParent(hwnd)))
					ShowWindow(GetParent(hwnd), SW_RESTORE);
				else
					PostMessage(hWnd, WM_COMMAND, IDM_READ, NULL);

			if (wParam == VK_SPACE)
				PostMessage(hWnd,WM_COMMAND,IDM_READ,NULL);


			/*** If the user presses the [TAB] key,  set the ***/
			/*** focus to the first button of the Folder window. ***/
			if (wParam == VK_TAB)
				SetFocus(GetDlgItem(GetParent(hwnd), buttons[0]));

			break;


		case WM_KEYUP:
			/*** If the user presses the [DEL] key, send a    ***/
			/*** DELETE command to the main window procedure. ***/
			if (wParam == VK_DELETE)
				if (!IsIconic(GetParent(hwnd)))
					PostMessage(hWnd, WM_COMMAND, IDM_DELETE, NULL);

			/*** If user pressed F1. ***/
			if (wParam == VK_F1)
				PostMessage(hWnd, WM_COMMAND, HELP_INDEXES, 0L);

			break;

	} /** end switch(message) **/


	/*** Pass all messages to  the original list box window procedure ***/
	if (hTempFold != NULL) {
		retval = CallWindowProc(OpenFolder->lpOldListProc, hwnd, message, wParam, lParam);
		GlobalUnlock(hTempFold);
	}
	else
		retval = NULL;

	return retval;
}




/******************************************************************************
	FUNCTION:	ReadID()

	PURPOSE:  	This function will return an items Message ID.  If the item is
				not	in memory, another block of messages is read in first.
*******************************************************************************/
BOOL ReadID(MailFolder far *CurFolder, int index, char far *buff)
{

	if (!(CurFolder->Item[index].flags & INMEMORY) || (lstrlen(CurFolder->Item[index].ID)==0))
		if (!ReadBlock(CurFolder, index))
			return FALSE;
	lstrcpy(buff, (char far *) &(CurFolder->Item[index].ID));
	return TRUE;

}

/******************************************************************************
	FUNCTION:	ReadItem()

	PURPOSE:  	This function will return an items string.  If the item is not
				in memory, another block of messages is read in first.
*******************************************************************************/
BOOL ReadItem(MailFolder far *CurFolder, int index, char far *buff)
{

	if (!(CurFolder->Item[index].flags & INMEMORY) || (lstrlen(CurFolder->Item[index].string)==0))
		if (!ReadBlock(CurFolder, index))
			return FALSE;
	lstrcpy(buff, (char far *) &(CurFolder->Item[index].string));
	return TRUE;

}



/******************************************************************************
	FUNCTION:	ReadBlock()

	PURPOSE:  	This function will read a block of messages from the Mail
				service	and store them in memory using the old
				VnsListMailMsgs function.
*******************************************************************************/
BOOL ReadBlock(MailFolder far *CurFolder, int index)
{

	GLOBALHANDLE	hMsgBlock;
	MsgInfo WFAR 	*Msg, far *CurMsg;
	CallStatus 		uStat;
	unsigned short	retcount=0, msgcount, x, getcount, bpcount;
	char			stat[16], buff[128];
	Env				env;


	/*** Allocate enough space for a block of Msg structures. ***/
	if ((hMsgBlock = SafeAlloc(GLOBAL, GETMSGAMOUNT*sizeof(MsgInfo), hWnd)) == NULL) {
		return (FALSE);
	}
	Msg = (MsgInfo WFAR *) GlobalLock(hMsgBlock);


	/*** Update the status Bar.    ***/
	if (hStatus)
		SendMessage(hStatus, STB_STRING, 0, (LONG) "Reading Messages ...");


	/*** Modify index value based on direction to read block in. ***/
	getcount = GETMSGAMOUNT;
	msgcount = (CurFolder->MsgCount > MaxMsgs) ? MaxMsgs : CurFolder->MsgCount;
	if (((index==0) && (CurFolder->Item[index+1].flags & INMEMORY)) ||
		((index==MaxMsgs) && (CurFolder->Item[index-1].flags & INMEMORY)) ||
		((CurFolder->Item[index+1].flags & INMEMORY) && (CurFolder->Item[index-1].flags & INMEMORY))) {
		getcount = 1;
	}
	if ((index + getcount) > msgcount)
		index = (msgcount - getcount);
	else if (CurFolder->Item[index+1].flags & INMEMORY)
		index -= (getcount-1);
	if (index < 0) index = 0;


	uStat=VnsListMailMsgs(hVNMAS, UserName, CurFolder->Name,(MsgInfo WFAR *)Msg, index, getcount, &retcount, &msgcount, CREATEDATE);
	if (Recover(uStat))
		uStat=VnsListMailMsgs(hVNMAS, UserName, CurFolder->Name,(MsgInfo WFAR *)Msg, index, getcount, &retcount, &msgcount, CREATEDATE);

	if(uStat) {
		SafeFree(GLOBAL, hMsgBlock);
		return (FALSE);
	}

	/*** Add each of the messages to the folder structure as a formatted  ***/
	/*** string. The message ID is added to the end of the string ***/
	/*** and is placed in a hidden column, off the screen.        ***/
	for(x=0;x<retcount;x++, index++){
        CurMsg = &(Msg[x]);
		switch(CurMsg->status){
			case MA_SENT:
				lstrcpy((LPSTR) &stat,"SENT  ");
				break;
			case MA_UNSENT:
				lstrcpy((LPSTR) &stat,"UNSENT");
				break;
			case MA_READ:
				lstrcpy((LPSTR) &stat,"READ  ");
				break;
			case MA_UNREAD:
				lstrcpy((LPSTR) &stat,"UNREAD");
				break;
		} /* switch (Msg[x]->status) */

		CurMsg->name[5] = '\t';

		wsprintf((LPSTR)&(CurFolder->Item[index].string), "%s\t%s\t%s\t%s\t\t", (LPSTR) stat, (LPSTR) CurMsg->date, (LPSTR) CurMsg->name, (LPSTR) CurMsg->subj);
		lstrncpy(CurFolder->Item[index].ID, (LPSTR) CurMsg->id, 13);
		CurFolder->Item[index].flags = INMEMORY;


		/*** Retrieve info on the attached files ***/
		bpcount = 0;
		uStat=VnsGetMailEnvelope(hVNMAS, (LPSTR) &UserName, CurFolder->Name, (LPSTR) CurMsg->id, (Env far *) &env);
		if (Recover(uStat))
			uStat=VnsGetMailEnvelope(hVNMAS, (LPSTR) &UserName, CurFolder->Name, (LPSTR) CurMsg->id, (Env far *) &env);
		if (!uStat) {
			uStat=VnsGetMailBodyPartCount(hVNMAS, CurMsg->id, &bpcount);
			if (!uStat && bpcount>1)
				CurFolder->Item[index].flags |= ATTACHED;
			VnsReleaseMailEnvelope(hVNMAS, (LPSTR) CurMsg->id);
		}



	} /* for (x=0;...) */

	SafeFree(GLOBAL, hMsgBlock);

	/*** Update the status Bar.    ***/
	if (hStatus)
		SendMessage(hStatus, STB_STRING, 0, 0L);

	return TRUE;
}


