/****************************************************************************
	FILE:		library.c


	CONTENTS:	CapCmp()		compares two strings, case-insensitive.
				CapnCmp()		compares a portion of two strings,
								case-insensitive.
				CheckNames()	verifies a list of names as legal Banyan
								recipients.
				DeleteTempFiles()deletes HappyMail created temp files.
				delnchr()	  	deletes characters in a string
				DrawBitMap()	draws a bitmap to the specified location.
				DspMsgCode()	displays error messages.
				fexist()		checks for the existence of a file.
				fsize()			returns the size of a file.
				get_datestr()	creates a date string from a Banyan date.
				getrev()		gets the current Vines revision number.
				NullFill()		fills a string with nulls.
				Recover()		attempts to restart a lost mail connection
				SafeAlloc()		allocates memory.
				SafeFree()		releases and deallocates memory.
				setcwd()		sets the current woirking directory.
				StartMail()		begins a session with the users Mail Service.
				StartSTDA()		begins a session with STDA.



	COMMENTS:	This file contains general utility funtctions which are used
				throughout HappyMail code. Utility functions which are
				specific to a certain task may be contained in one of the
				following task specific libraries: foldrlib, ini_lib, lettrlib,
                superlib and winlib. 

****************************************************************************/
#include "MyMail.h"
#include "stdlib.h"
#include <sys\stat.h>
#include "string.h"
#include "io.h"
#include "dos.h"

#define INCL_STDA_UI
#define INCL_STDA
#include "vnsstda.h"
#undef  INCL_STDA_UI
#undef  INCL_STDA

#define INCL_SOCK
#include "vnsapi.h"
#undef  INCL_SOCK

#define UPPER(x)	(x<123 && x>96 ? x-32:x)


/******************************************************************************
	FUNCTION:   CapCmp()

	PURPOSE:  	To compare two strings case insensitive.

	COMMENTS:	Converts both strings to uppercase as it compares.
*******************************************************************************/
BOOL CapCmp(char far *str1, char far *str2)
{
	int i, len;

	len = lstrlen(str1);
	if (len != lstrlen(str2))
		return FALSE;
	for(i=0; i<len; i++) if (UPPER(str1[i]) != UPPER(str2[i])) return FALSE;
	return TRUE;

}

/******************************************************************************
	FUNCTION:   CapnCmp()

	PURPOSE:  	To compare two strings case insensitive.

    COMMENTS:	Converts both strings to uppercase as it compares.
*******************************************************************************/
BOOL CapnCmp(char far *str1, char far *str2, int spn)
{
	int i, len;

	len = lstrlen(str1);
	if (len < spn)
		return FALSE;
	for(i=0; i<len && i<spn; i++) if (UPPER(str1[i]) != UPPER(str2[i])) return FALSE;
	return TRUE;

}

/******************************************************************************
	FUNCTION:	CheckNames()

	PURPOSE:	Verify a list of names as legal Banyan recipients.

	COMMENTS:	If a name is not found in the address book and cannot be found
				in STDA, this function display a warning and asks if the user
				wishes to change his list.  If so, then the focus is returned
				to the proper control and the invalid name is highlighted.


*******************************************************************************/
extern Cap2			UserCaps;
unsigned short 	STDA_Array[]={STDA_USERS, STDA_NICKNAMES,
				STDA_LISTS, STDA_FILES, STDA_PRINTERS, STDA_SERVICES};

CheckNames(char far *input, HWND ControlID, int Level, int far *length)
{
	CallStatus 		uStat;
	char 			tempname[512], name[65], group[35], org[35], copy[512];
	char 			far	*tempptr, far *oldptr, far *strptr;
	int 			offset, i, ErrorLevel, len, cnt;
	unsigned short 	valid;
	HVNSD			hVNSD;
	BOOL			InAdBook;
	GLOBALHANDLE	hNames;


	/*** Preset the length and check for unmatched brackets. ***/
	if (Level == 0) {
		*length = 0;
		for (i=0, cnt=0; i<lstrlen(input); i++) {
			if (input[i] == '[') cnt++;
			if (input[i] == ']') cnt--;
		}
		if (cnt != 0) {
			BWCCMessageBox(GetParent(ControlID), "\n\nMissing bracket in address.", "Address Error", MB_ICONSTOP);
			SetFocus(ControlID);
			return ADD_ERR_INVALID;
		}
	}

	/*** If too many levels of recursion, return an Error. ***/
	if (Level > 6) return ADD_ERR_LEVEL;


	/*** Initialize temporary pointers ***/
	oldptr = input;
	tempptr = input;
	NullFill((LPSTR) &tempname, 512);


	/*** Grab one name from the list at a time ***/
	ErrorLevel = 0;
	while (VnsParseMailRecipient(tempptr, (LPSTR) &tempname, (char far * far *) &tempptr) != MS_NONAMEADDR) {
		if (tempname[0] != '\0') {

			/*** First check if name is in an addressbook ***/
			InAdBook = InAddressBook((LPSTR) &tempname,(LPSTR) &copy, TRUE);

			if (InAdBook) {
				ErrorLevel = CheckNames((LPSTR)&copy, ControlID, Level + 1, length);
				if (ErrorLevel > 0) break;
			}

			if (!InAdBook) {

				/*** Increment length parameter. ***/
				*length += lstrlen(tempname)+1;

				/*** Name is not in Address book so we have to keep checking. ***/
				/*** Anything inside brackets in a Banyan address is ignored. ***/
				/*** So start by stripping off anything after a left bracket. ***/
				for(offset = 0; tempname[offset] != '\0' && tempname[offset] != '[' && offset < 512; offset++);
				tempname[offset] = '\0';

				/*** Strip any trailing blanks off name ***/
				for(offset=lstrlen((LPSTR) &tempname)-1;tempname[offset] == 32 && offset>0;offset--)tempname[offset]='\0';

				/*** Parse Streettalk name into three component parts. ***/
				lstrcpy((LPSTR) &copy, (LPSTR) &tempname);
				lstrcpy((LPSTR) &name, 	_fstrtok((LPSTR) &copy, "@"));
				strptr = (LPSTR) _fstrtok(NULL, "@");
				group[0] 	= '\0';
				org[0]		= '\0';
				if (strptr != NULL) {
					lstrcpy((LPSTR) &group, strptr);
					strptr = (LPSTR) _fstrtok(NULL, "@");
					if (strptr != NULL)
						lstrcpy((LPSTR) &org, strptr);
				}


				/*** Check for Wildcard violations. ***/
				if (((UserCaps.sendtplate == GRP_ORG_RESTRICT) && (_fstrchr((LPSTR) &group, '*') != NULL)) ||
				   ((UserCaps.sendtplate != NO_RESTRICT) && (_fstrchr((LPSTR) &org, '*') != NULL))) {
					ErrorLevel = ADD_ERR_WILDCARD;
					break;
				}


				/*** If first character is an asterik strip   ***/
				/*** it off and check for a valid group name. ***/
				/*** ELSE... call VnsFormatStName in case the ***/
				/*** name is a nickname.                      ***/
				if (tempname[0] == '*') {
					for(offset=0; tempname[offset] != '@' && offset<lstrlen((LPSTR) &tempname); offset++);
					lstrcpy((LPSTR) &copy, (LPSTR) &(tempname[offset+1]));
					uStat = VnsFormatStGroup((LPSTR) &copy);
					if (Recover(uStat))
						uStat = VnsFormatStGroup((LPSTR) &copy);
				}
				else {
					uStat = VnsFormatStName((LPSTR) &tempname);
					if (Recover(uStat))
						uStat = VnsFormatStName((LPSTR) &tempname);
				}


				if (uStat) {
					/*** The name failed one of the above tests so crank    ***/
					/*** up STDA and check for the names existence in STDA. ***/
					valid = 0;
					if ( hSTDA ) {
						for (i=0; i<6 && valid == 0; i++)
							VnsVerifyStdaName(hSTDA, (LPSTR) &tempname, STDA_Array[i], &valid, (LPSTR) &copy);

						/*** If name is found AND its an inclusion, check the inclusion list. ***/
						if ((valid) && (copy[0] != '\0')) {
							/*** Decrement length parameter. ***/
							*length -= lstrlen(tempname)-1;
							ErrorLevel = CheckNames((LPSTR)&copy, ControlID, Level + 1, length);
							if (ErrorLevel > 0) break;
						}
					}
					if (!valid) {
						/*** Finally, check if name is a Servername by ***/
						/*** converting it to a Mail Service ST name.  ***/
						if (_fstrstr(tempname, "@") == NULL) {
							lstrcpy((LPSTR) &copy, "MS@");
							lstrcat((LPSTR) &copy, (LPSTR) &tempname);
							lstrcat((LPSTR) &copy, "@Servers");
							ErrorLevel = CheckNames((LPSTR)&copy, ControlID, Level + 1, length);
							if (ErrorLevel > 0) break;
						}
						else {
							/*** Name is invalid -- so ask user if they would like to modify the name ***/
							if (CallDlg(GetParent(ControlID), AskChange, YesNoBox, 1L)) {
								/*** User wishes to change the name, so set errorlevel ***/
								ErrorLevel = ADD_ERR_INVALID;
								break;
							}
						}
					}
				}
			}
		}
		if ((*length) > MAXNAMESLEN) {
			ErrorLevel = ADD_ERR_LENGTH;
			break;
		}


		/*** Reset buffer and continue to next name ***/
		NullFill((LPSTR) &tempname, 512);

		oldptr = tempptr;
	}


	if (Level == 0) {
		if (ErrorLevel == ADD_ERR_WILDCARD) {
			BWCCMessageBox(GetParent(ControlID), "\nYou are restricted from\n\nsending wildcard messages.", "Address Error", MB_ICONSTOP);
			while (*oldptr == ' ') oldptr++;
			offset = (int) (FP_OFF(oldptr) - FP_OFF(input));
			SendMessage(ControlID, EM_SETSEL, NULL, MAKELONG(offset + lstrlen((LPSTR) &tempname), offset));
			SetFocus(ControlID);
		}

		if (ErrorLevel == ADD_ERR_LEVEL) {
			BWCCMessageBox(GetParent(ControlID), "\nError interpreting addresses!\n\n Check for looping address book entries.", "WARNING!", MB_ICONSTOP | MB_OK);
			while (*oldptr == ' ') oldptr++;
			offset = (int) (FP_OFF(oldptr) - FP_OFF(input));
			SendMessage(ControlID, EM_SETSEL, NULL, MAKELONG(offset + lstrlen((LPSTR) &tempname), offset));
			SetFocus(ControlID);
		}

		if (ErrorLevel == ADD_ERR_INVALID) {
			/*** User wishes to change the name, so highlight it and ***/
			/*** reset the focus to the proper control, then return. ***/
			while (*oldptr == ' ') oldptr++;
			offset = (int) (FP_OFF(oldptr) - FP_OFF(input));
			SendMessage(ControlID, EM_SETSEL, NULL, MAKELONG(offset + lstrlen((LPSTR) &tempname), offset));
			SetFocus(ControlID);
		}


		if (ErrorLevel == ADD_ERR_LENGTH) {
			BWCCMessageBox(GetParent(ControlID), "\nAddress string too long!\n\nMax size is 1000 characters.", "WARNING!", MB_ICONSTOP | MB_OK);
			while (*oldptr == ' ') oldptr++;
			offset = (int) (FP_OFF(oldptr) - FP_OFF(input));
			SendMessage(ControlID, EM_SETSEL, NULL, MAKELONG(offset + lstrlen((LPSTR) &tempname), offset));
			SetFocus(ControlID);
		}


	}

	return(ErrorLevel);
}


/******************************************************************************
	FUNCTION:	DeleteTempFiles()

	PURPOSE:  	Deletes all HappyMail temp files which are not in use.
*******************************************************************************/
DeleteTempFiles(void)
{

	char 			SearchName[256], PathBuff[256], FileName[256];
	struct find_t 	ffiles;
	int				done, fhandle;

	if (hStatus)
		SendMessage(hStatus, STB_STRING, 0, (LONG)"Deleting Temporary Files...");

	GetTempFileName(0, "HMA", 0, (LPSTR) &PathBuff);
	PathBuff[ParseName((LPSTR) &PathBuff)] = '\0';
	lstrcpy((LPSTR) &SearchName, (LPSTR) &PathBuff);
	lstrcat((LPSTR) &SearchName, "~HMA*.*");
	done = _dos_findfirst((char *)SearchName, _A_NORMAL, (struct find_t *)&ffiles);
	while (!done) {
		lstrcpy((LPSTR) &FileName, (LPSTR) &PathBuff);
		lstrcat((LPSTR) &FileName, (LPSTR) &ffiles.name);
		if ((fhandle = _lopen((LPSTR) &FileName, WRITE | OF_SHARE_EXCLUSIVE)) != HFILE_ERROR) {
			_lclose(fhandle);
			unlink((char *) FileName);
		}
		done = _dos_findnext((struct find_t *)&ffiles);
	}


	if (hStatus)
		SendMessage(hStatus, STB_STRING, 0, 0L);

	return;

}


/******************************************************************************
	FUNCTION:	delnchr()

	PURPOSE:  	deletes n characters in a string starting at index position
*******************************************************************************/
delnchr(char far *str, int index, int n)
{
	while(str[index] != '\0') {
		str[index] = str[(index + n)];
		index++;
	}
	return;
}



/******************************************************************************
	FUNCTION:   DrawBitMap()


	PURPOSE:  	To draw a bitmap onto a device context (screen, printer, etc..).

	COMMENTS:	This function also will center a bitmap vertically and/or
				horizontally.  Additionally, it can write a time stamp string
				over top of the bitmap.  This feature is used to time stamp the
				Logo screen for more accurate version update information.  This
				routines main function (displaying the bitmap) was derived from
                the Petzold book, Programming Windows.
*******************************************************************************/

DrawBitMap(HDC hdc, HBITMAP hBitMap, short xStart, short yStart, BOOL Screen, BOOL TimeStamp)
{
	BITMAP 	bm;
	HBITMAP	OldBm;
	HDC		hdcMem;
	DWORD	dwStrSize, OldColor;
	POINT	ptSize, ptOrg;
	RECT	rect;
	char	output[40];


	/*** Create a temporary Device Context and load the bitamp into it. ***/
	hdcMem = CreateCompatibleDC(hdc);
	if (hdcMem == NULL)
		return;
	OldBm = SelectObject(hdcMem, hBitMap);
	SetMapMode(hdcMem, GetMapMode(hdc));

	/*** Determine the dimensions of the Bitmap. ***/
	GetObject(hBitMap, sizeof(BITMAP), (LPSTR) &bm);
	ptSize.x = bm.bmWidth;
	ptSize.y = bm.bmHeight;


	/*** Center the Bitmap ***/
	if (Screen) {
		/*** Get Screen Coordinates if needed ****/
		GetClientRect(hWnd, &rect);
		if (xStart == -1)
			xStart = ((rect.right-rect.left)/2) - (ptSize.x/2);
		if (yStart == -1)
			yStart = ((rect.bottom-rect.top)/2) - (ptSize.y/2);
	}
	else {
		if (xStart == -1)
			xStart = (GetDeviceCaps(hdc, HORZRES)/2) - (ptSize.x/2);
		if (yStart == -1)
			yStart = (GetDeviceCaps(hdc, VERTRES)/2) - (ptSize.y/2);
	}


    /*** Convert device coordinates to logical coordinates ***/
	DPtoLP(hdc, &ptSize, 1);
	ptOrg.x = 0;
	ptOrg.y = 0;
	DPtoLP(hdcMem, &ptOrg, 1);

	/*** Copy the bitmap to the destination DC from our temporary DC. ***/
	BitBlt(hdc, xStart, yStart, ptSize.x, ptSize.y, hdcMem, ptOrg.x, ptOrg.y, SRCCOPY);

	/*** Remove our bitmap and release the temporary DC ***/
	SelectObject(hdcMem, OldBm);
	DeleteDC(hdcMem);

    /*** Put on the optional Version and time stamp string ***/
	if (TimeStamp) {

		/*** The __DATE__ variable is a compiler created variable that      ***/
		/*** contains the current date as a string.  By using this variable ***/
		/*** the date is whenever this particular file is recompiled.       ***/
        OldColor = SetBkColor(hdc, RGB(192,192,192));
		lstrcpy((LPSTR) &output, VerString);
		lstrcat((LPSTR) &output, "  --  ");
		lstrcat((LPSTR) &output, __DATE__);
		dwStrSize = GetTextExtent(hdc, output, lstrlen((LPSTR) &output));
		ptOrg.x = xStart + (ptSize.x/2) - (LOWORD(dwStrSize)/2);
		ptOrg.y = yStart + ptSize.y - (8 + (HIWORD(dwStrSize)));
		TextOut(hdc,  ptOrg.x, ptOrg.y, output, lstrlen((LPSTR) &output));

		SetBkColor(hdc, OldColor);
	}

	return;
}


/******************************************************************************
	FUNCTION:	DspMsgCode()

	PURPOSE:	To Display Error Messages and send the Quit Message if requested
*******************************************************************************/
void DspMsgCode( HWND hWnd, char far *pBuf, unsigned long uStat, int postit )
{
	char pMyBuf[80];
	char pStat[20];

	/*** Copy the output string to a temp buffer ***/
	lstrcpy((LPSTR) &pMyBuf, pBuf );

	/*** If an error code is included, convert it   ***/
	/*** to a string and add it to the temp buffer. ***/
	if( uStat ) {
		wsprintf((LPSTR) &pStat, "%u", (long) uStat);
		lstrcat((LPSTR) &pMyBuf, (LPSTR) &pStat );
	}

	/*** Display the error message ***/
	BWCCMessageBox( hWnd, (LPSTR) &pMyBuf, "Status Message", MB_OK );

	if (postit)
		PostQuitMessage(0);
}


/******************************************************************************
	FUNCTION:	fexist()

	PURPOSE:  	checks for the existence of a file (or path).
*******************************************************************************/
BOOL fexist(char far *fname)
{
	char	filename[256];

	lstrcpy((LPSTR) &filename, fname);
	/*** If last character is a backslash add a ".". ***/
	if (filename[lstrlen(filename)-1] == '\\')
		lstrcat(filename, ".");

	return(access((char *)filename, 0x00)==0);
 }



/******************************************************************************
	FUNCTION:   fsize()

	PURPOSE:  	returns the size of a file.
*******************************************************************************/
long fsize(int filehandle)
{
	struct stat statbuf;

	if (fstat(filehandle, (struct stat *) &statbuf) == 0)
		return (statbuf.st_size);
	return(0L);

}


/******************************************************************************
	FUNCTION: 	get_datestr()

	PURPOSE:  	To return a string formatted date based on today's date.
*******************************************************************************/
get_datestr(char far *datestr)
{
	struct tm  *tblock;
	short       tz_off;
	char   		tz[4], temp[16], *pDate;
	time_t 		timer, *tpTemp;

	/*** Get the current time/date from Vines ***/
	VnsGetTime((unsigned long far *)&timer, (short far *)&tz_off, (char far *)&tz);


	/*** Convert the time to a time_t structure -- the time is stored as     ***/
	/*** number of seconds elapsed sing 1/1/70.  The time_t structure breaks ***/
	/*** the time down into days, months, years, hours, minutes, etc...      ***/
	tblock = localtime((time_t *)&timer);

	/*** Add the Name of the day to the output string ***/
	switch((int)tblock->tm_wday) {
		case 0:
			lstrcpy(datestr, "Sunday,  ");
			break;
		case 1:
			lstrcpy(datestr, "Monday,  ");
			break;
		case 2:
			lstrcpy(datestr, "Tuesday,  ");
			break;
		case 3:
			lstrcpy(datestr, "Wednesday,  ");
			break;
		case 4:
			lstrcpy(datestr, "Thursday,  ");
			break;
		case 5:
			lstrcpy(datestr, "Friday,  ");
			break;
		case 6:
			lstrcpy(datestr, "Saturday,  ");
			break;
	} /** end switch (tblock->wday) **/

	/*** Add the month name to the output string ***/
	switch(tblock->tm_mon) {
		case 0:
			lstrcat(datestr, "January ");
			break;
		case 1:
			lstrcat(datestr, "February ");
			break;
		case 2:
			lstrcat(datestr, "March ");
			break;
		case 3:
			lstrcat(datestr, "April ");
			break;
		case 4:
			lstrcat(datestr, "May ");
			break;
		case 5:
			lstrcat(datestr, "June ");
			break;
		case 6:
			lstrcat(datestr, "July ");
			break;
		case 7:
			lstrcat(datestr, "August ");
			break;
		case 8:
			lstrcat(datestr, "September ");
			break;
		case 9:
			lstrcat(datestr, "October ");
			break;
		case 10:
			lstrcat(datestr, "November ");
			break;
		case 11:
			lstrcat(datestr, "December ");
			break;
	}  /** end switch **/

	/*** Add the day of the month and year to the output string ***/
	/*** Add the year to the output string ***/
	wsprintf((LPSTR) &temp, "%i,  %i  ", (int) tblock->tm_mday, (int) tblock->tm_year + 1900);
	lstrcat(datestr, (LPSTR) &temp);

	/*** Add the time to the output string ***/
	if (tblock->tm_hour < 10)
		lstrcat(datestr, "0");
	wsprintf((LPSTR) &temp, "%i:", (int) tblock->tm_hour);
	lstrcat(datestr, (LPSTR) &temp);

	if (tblock->tm_min < 10)
		lstrcat(datestr, "0");
	wsprintf((LPSTR) &temp, "%i:", (int) tblock->tm_min);
	lstrcat(datestr, (LPSTR) &temp);

	if (tblock->tm_sec < 10)
		lstrcat(datestr, "0");
	wsprintf((LPSTR) &temp, "%i  ", (int) tblock->tm_sec);
	lstrcat(datestr, (LPSTR) &temp);

	/*** Add the time zone to the output string ***/
	lstrcat(datestr, (tblock->tm_isdst) ? tzname[1] : tzname[0]);

	return(1);
}


/******************************************************************************
	FUNCTION:	getrev()

	PURPOSE:	This funcion gets the current vines revision from the users Z:
				drive.
*******************************************************************************/
unsigned short getrev()
{
	IPCPORT			svrport;
	char 			name[64], revstring[16];
	unsigned long 	rev, serial;
	CallStatus		uStat=0;
	unsigned short	len, revlen;

	uStat = VnsGetDriveZPort((LPIPCPORT) &svrport);
	uStat = VnsCreateLocalWkp(0x03, (LPIPCPORT) &svrport);
	MAXBODY = (unsigned long) 6144L;
	MAXMSGBODY = (unsigned long) 6144L;

	if (!uStat)
		uStat = VnsGetSvrLRev((LPIPCPORT) &svrport, (LPSTR) &name,
				(unsigned short far *) &len, &rev, (LPSTR) &revstring,
				(unsigned short far *) &revlen, &serial);

	if (uStat) {
		return((unsigned short)41100);
	}
	else {
		if (rev >= (unsigned short) 50005)
			MAXBODY = (unsigned long) 61440L;
			MAXMSGBODY = (unsigned long) 58000L;
		return((unsigned short) rev);
	}

}


/******************************************************************************
	FUNCTION:	NullFill()

	PURPOSE:	To fill a structure or string with NULLs
*******************************************************************************/
void NullFill(char far *ptr, unsigned int size)
{
	for(;size>0;size--) ptr[size-1] = '\0';
}


/******************************************************************************
	FUNCTION:	Recover()

	PURPOSE:	Check error code for a recoverable error.
*******************************************************************************/
BOOL Recover(CallStatus uStat)
{
	CallStatus		error;
	BOOL			started;
	int 			i;
	unsigned short	srev;

	switch (uStat) {
		case 1911:
			BWCCMessageBox(hWnd, "Current STDA Session lost.\nAttempting to restart session.", "HappyMail", MB_ICONHAND);
			return (StartSTDA());
		case 3017:
		case 3019:
		case 3020:
		case 3027:
			BWCCMessageBox(hWnd, "Current Mail Session lost.\nAttempting to restart session.", "HappyMail", MB_ICONHAND);
			return(StartMail(&hVNMAS));
		default:
			break;
	}
	return FALSE;

}


/******************************************************************************
	FUNCTION:   SafeAlloc()

	PURPOSE:	Allocate a block of memory.  If the memory cannot be allocated,
				a warning message is issued.
*******************************************************************************/
#define NUMSELECTORS 128
HANDLE SafeAlloc( BOOL Global, unsigned long BlockSize, HWND hWND )
{
	DWORD   hGDMem;
	UINT	selectors[NUMSELECTORS], i;
	HANDLE 	uHnd;

	/*** Try to allocate the memory ***/
	if (Global)
		uHnd = GlobalAlloc(GHND, (unsigned long) BlockSize);
	else
		uHnd = LocalAlloc(GHND, BlockSize);

	/*** Issue warning if return handle is NULL ***/
	if (uHnd == NULL)
		BWCCMessageBox(hWND, "Unable to allocate enough memory!", "WARNING!", MB_ICONSTOP | MB_OK);

	return uHnd;
}



/******************************************************************************
	FUNCTION:   SafeFree( BOOL, HANDLE)

	PURPOSE:  	Deallocate a block of memory.  First call unlock until all
				locks have been removed, then deallocate the memory.
*******************************************************************************/
SafeFree(BOOL Global, HANDLE uHnd)
{
	if (Global) {
		while(GlobalUnlock(uHnd) != 0);
		GlobalFree(uHnd);
	}
	else {
		while(LocalUnlock(uHnd) != 0);
		LocalFree(uHnd);
	}
	return;
}


/******************************************************************************
	FUNCTION:	setcwd()

	PURPOSE:	This funcion sets the current working directory (and drive) to
				the specified path.
*******************************************************************************/
#include "dir.h"
#include "direct.h"
setcwd(char far *path)
{
	char newpath[256];

	/*** Change to the new directory ***/
	lstrcpy((LPSTR) &newpath, path);
	if ((newpath[lstrlen((LPSTR) &newpath)-1] == '\\') && (newpath[lstrlen((LPSTR) &newpath)-2] != ':'))
		newpath[lstrlen((LPSTR) &newpath)-1] = '\0';
	chdir((char *) newpath);

	/*** Change to the new drive ***/
	while (path[0] == ' ') path++;
	_chdrive(UPPER(path[0]) - 64);
	return;
}


/******************************************************************************
	FUNCTION: 	StartSTDA()

	PURPOSE:  	To return a Valid STDA Session Handle.
*******************************************************************************/
HVNSD	StartSTDA(void)
{

	HVNSD       	hVNSD;
	CallStatus		error;
	int 			i, x;
	char 			defgroup[STNAMELEN], SetSTDA[]={"!SETSTDA"}, quote,
					ServiceName[STNAMELEN], ServerName[STNAMELEN], buff[MAXASSOC];
	IPCPORT			ipcPort;
	BOOL			bNoAlt, bUsePort;

	unsigned int 	uSock, uCID;

	/*** Get the STDA Service Name -- Give it five tries ***/
	bNoAlt = bUsePort = FALSE;
	_fmemset(ipcPort, 0, sizeof(IPCPORT));
	ServiceName[0] = '\0';
	for (i=0, error=TRUE; error && i<5; i++)
		error = VnsGetStProfileEntry((LPSTR) &UserName, (LPSTR) &SetSTDA, (LPSTR) &buff, (LPSTR) &defgroup);
	if(error) {
		if (error != 1810)
			DspMsgCode(hWnd, "Error reading user profile : ", error, FALSE);
	}
	else {
		/*** Check for the /OFF parameters. ***/
		_fstrupr((LPSTR) &buff);
		if (_fstrstr((LPSTR) &buff, "/OFF")) {
			BWCCMessageBox(hWnd, "Access to STDA denied.", "HappyMail", MB_ICONSTOP);
			return 0;
		}

		/*** Set the bNoAlt parameter. ***/
		if (_fstrstr((LPSTR) &buff, "/NOALT")) bNoAlt = TRUE;

		/*** Parse the ServerName from the ProfileString. ***/
		for(x=0, i=0;buff[x];x++) {
			if(buff[x] =='/')
				break;
			if(buff[x]!='\"' || buff[x]!='\'')
				ServerName[i++]=buff[x];
		}
		ServerName[i]='\0';

		/*** Remove trailing blanks. ***/
		for(i--;ServerName[i]==' ';i--)
			ServerName[i]='\0';

		/*** Use the Open Socket call to get the right ServiceName. ***/
		if ((error = VnsOpenStdaSocket(ipcPort, 1, ServerName, ServiceName, 25, 2, &uSock, &uCID))==0) {
			VnsCloseSocket(uSock);
			/*** Grab the specified STDA RPC port. ***/
			VnsGetRpcPort(ServiceName, ipcPort, ServiceName);
		}
		if (ServiceName[0] != '\0')
			bUsePort = TRUE;
	}


	/*** Start STDA Session -- Give 3 tries before giving up ***/
	hVNSD = (HVNSD) 0;
	error = TRUE;
	if (bUsePort)
		error = VnsStartStdaSession(ipcPort, &hVNSD );
	if ( error && !bNoAlt)
		error = VnsStartStdaSession(NULL, &hVNSD );
	if ( error ) {
		DspMsgCode(hWnd, "Unable to Start an STDA Session: ", error, FALSE);
		return 0;
	}
	return hVNSD;

}

/******************************************************************************
	FUNCTION: 	StartMail()

	PURPOSE:  	To return a Valid Mail Session Handle.
*******************************************************************************/
BOOL StartMail(HVNMA WFAR *hMail)
{
	CallStatus		error;
	unsigned short  srev;
	int 			i, x;
	char 			defgroup[STNAMELEN], SetMail[]={"SETMAIL"}, MSName[STNAMELEN];
	char			buff[MAXASSOC];


	/*** Get the Mail Service Name -- Give it five tries ***/
	for (i=0, error=TRUE; error && i<10; i++)
		error = VnsGetStProfileEntry((LPSTR) &UserName, (LPSTR) &SetMail, (LPSTR) &buff, (LPSTR) &defgroup);
	if(error) {
		DspMsgCode(hWnd, "Unable to read user profile : ", error, FALSE);
		return FALSE;
	}

	/*** Strip quotes from the mail service name. ***/
	for(i=0,x=0;buff[x];x++)
		if(buff[x]!='\"' && buff[x]!='\'')
			MSName[i++]=buff[x];
	MSName[i]='\0';

	/*** Start the Mail Session -- Give it five tries. ***/
	do {
		for(i=0, error = TRUE; error && i<5; i++)
			error = VnsStartMailSession((LPSTR) &UserName, (LPSTR) &MSName, &srev, hMail);
		if(error) {
			if (CallDlg(hWnd, AskChange, YesNoBox, 12L)==FALSE) {
				return FALSE;
			}
		}
	} while (error);

	VnsRev = srev;
	MaxMsgs = (VnsRev < 5) ? 200 : 800;

	return TRUE;
}



