/****************************************************************************
	FILE:		lettrlib.c


	CONTENTS:   AddLetter()		adds a Letter to the linked list.
				AddTabs()		utility for inserting tabs into text
				ChangeUnread()	utility for changing a mail message's status.
				FormatBody()	returns a text hadle for a formatted body.
				FormatEnvelope()returns a text handle for a formatted envelope.
				FormatMessage()	returns a text handle for a formatted message.
				FormatText()	adds carriage returns and tabs to text.
				GetLetter()		returns a Letter handle given a HWND.
				GetLetterByFoldName()	returns a Letter handle given the folder name.
				GetLetterByID()	returns a Letter handle given a message ID.
				GetLetterInfo()	returns message IDs and folder name of
								selection messages.
				KillLetter()	deletes a Letter from the linked list.
				NextLetter()	advances Letter and Read window to next
								or previous message.
				ReadRecipients()puts message recipients into a string.


	COMMENTS:	This file contains all the functions used to manipulate
				the linked list of Letter structures and their corresponding
				Read windows.  The functions for formatting messages for
                display 
****************************************************************************/
#include "MyMail.h"
#include "string.h"


/******************************************************************************
	FUNCTION: 	AddLetter()

	PURPOSE:  	To return a pointer to a new Letter structure.
*******************************************************************************/
GLOBALHANDLE AddLetter()
{
	GLOBALHANDLE    hMem, hTempMem, hTempMem2;
	Letter far *TempLetter, far *TempLetter2;

	/*** Grab some memory for the new Letter structure and get a pointer to it ***/
	if ((hMem = SafeAlloc(GLOBAL, (long) sizeof(Letter), hWnd)) == NULL) {
		return (NULL);
	}
	TempLetter = (Letter far *) GlobalLock(hMem);

	/*** Check to see if this is the first Letter, ***/
	/*** and if it is make it the TopLetter        ***/
	if (TopLetter == NULL) {
		TopLetter = hMem;
		TempLetter->Next = NULL;
		TempLetter->Prev = NULL;
		GlobalUnlock(hMem);
		return(hMem);
	}

	/*** Skip thru the linked list of letters till we get to the end. ***/
	TempLetter2 = (Letter far *) GlobalLock(TopLetter);
	hTempMem  = TopLetter;
	while(TempLetter2->Next !=  NULL) {
		hTempMem2 = TempLetter2->Next;
		GlobalUnlock(hTempMem);
		hTempMem = hTempMem2;
		TempLetter2 = (Letter far *) GlobalLock(hTempMem);
	}

	/*** Add the new Letter to the end of the linked list. ***/
	TempLetter2->Next  	 = hMem;
	TempLetter->Next     = NULL;
	TempLetter->Prev 	 = hTempMem;
	GlobalUnlock(hTempMem);
	GlobalUnlock(hMem);

	/*** Return with the handle to our newly created Letter. ***/
	return (hMem);
}


/******************************************************************************
	FUNCTION: 	AddTabs()

	PURPOSE:  	adds tabs to a string till string length >= len

	COMMENTS:	This function is used in formatting mail messages.  It enables
				us to line up items in the envelope portion	of the mail
				message by inserting the proper number of tabs.  This number
				will vary according to the current font.  Also, since I'm
				unable to accurately calculate the number of tabs, I determine
				the right number by adding tabs to the string until the string
				length matches a predetermined value (len).
*******************************************************************************/
AddTabs(char far *buff, unsigned int len, HWND hEdit, int far *tabset)
{
	HDC				hdc;
	HFONT			hFont;

	/*** Get the device context and font of the window ***/
	/*** which will be used to display the string.     ***/
	hdc = GetDC(hEdit);
	hFont = SendMessage(hEdit, WM_GETFONT, 0, 0L);

	/*** Make sure font is selected into the Window ***/
	if (hFont != NULL)
		hFont = SelectObject(hdc, hFont);

	/*** Add tabs until string is long enough ***/
	while ((unsigned int) LOWORD(GetTabbedTextExtent(hdc, buff, lstrlen(buff), 1, tabset)) < len)
		lstrcat(buff, "\t");

	/*** Release the Device context ***/
	ReleaseDC(hEdit, hdc);

	return;
}


/******************************************************************************
	FUNCTION: 	ChangeUnread()

	PURPOSE:  	Changes the status of a messsage in an
				open folder from UNREAD to READ.
*******************************************************************************/
ChangeUnread(char far *FoldName, char far *MsgID)
{
	int				index, i;
	char			buff[100];
	MailFolder far 	*TempFolder;
	GLOBALHANDLE	hFold;

	/*** Get a pointer to the Folder window structure ***/
	hFold = GetFolder2(FoldName);
	if (hFold == NULL)
		return;
	TempFolder = (MailFolder far *) GlobalLock(hFold);
	if (TempFolder->hListBox == NULL) {
		GlobalUnlock(hFold);
		return;
	}


	/*** Find the index to the proper message in the folder ***/
	i = FindIdInMem(hFold, MsgID);

	if (i != -1) {
		/*** Get the Message string from the folder window ***/
		ReadItem(TempFolder, i, (char far *) &buff);

		/*** If the status is UNREAD delete the old message ***/
		/*** and insert a message with the status of READ.  ***/
		if (lstrncmp(buff, "UNREAD", 6) == 0) {
			SendMessage(TempFolder->hListBox, LB_DELETESTRING, i, 0L);
			delnchr(TempFolder->Item[i].string, 0, 2);
			SendMessage(TempFolder->hListBox, LB_INSERTSTRING, i, (LONG) index);
		}

		/*** Resort the List box, if necessary. ***/
		if (Options.sortorder == BYSTATUS) SortLists(FoldName);


	}

	/*** Release the pointer to the Folder structure. ***/
	GlobalUnlock(hFold);


	return;
}


/******************************************************************************
	FUNCTION: 	FormatBody()

	PURPOSE:  	To return a handle to a string which contains the body of a mail
				message in a format suitable for printing, filing to disk, or
				displaying.
*******************************************************************************/
#define fmode O_WRONLY|O_BINARY|O_CREAT|O_TRUNC|O_APPEND
BOOL FormatBody(char far *FoldName, char far *MsID, GLOBALHANDLE hMessage, unsigned long MAXSIZE)
{
	GLOBALHANDLE	hTempMem;
	char 			far *textstr, far *finalbuff, far *tempbuffptr, buff[1025],
					far *startBody;
	int 			i, curlen, j;
	unsigned short 	offset, retcount=0;
	CallStatus 		uStat = 0;
	BodyPart 		Body[11];


	/***** Grab enough memory for a temporary buffer ****/
	if ((hTempMem = SafeAlloc(GLOBAL, 1600L, hWnd)) == NULL) {
		return (FALSE);
	}
	finalbuff = GlobalLock(hTempMem);

	/*** The message body is seperated into 1K chunks.  So we have to    ***/
	/*** read in one chunk at a time and add it to our text string.      ***/
	/*** The chunks will be read into a temporary buffer (tempbuffptr)   ***/
	/*** then LF characters will be replaced with CR-LF pairs as we      ***/
	/*** copy the chunk to a new buffer (finalbuff).  The last character ***/
	/*** will be stored just prior to the start of our temp buffer, in   ***/
	/*** case a CR-LF pair is split into two different chunks.           ***/

	/*** Initialize some variables. ***/
	textstr = GlobalLock(hMessage);
	startBody = textstr + lstrlen(textstr);
	tempbuffptr = (char far *) (buff + 1);
	buff[0] = '\0';
	offset=0;


	/*** Retrieve Body Part info to determine message format (ie CR-LF or just LF) ***/
	NullFill((char far *) &Body, sizeof(BodyPart) * 11);
	for(i=0, uStat=TRUE; i<3 && uStat; i++) {
		uStat=VnsGetMailBodyPartList(hVNMAS, MsID, Body, (unsigned short far *)&retcount);
		if (Recover(uStat))
			uStat=VnsGetMailBodyPartList(hVNMAS, MsID, Body, (unsigned short far *)&retcount);
	}


	/*** Grab the first chunk of the mail message and get the total number of chunks ***/
	for(i=0, uStat=TRUE; i<3 && uStat; i++) {
		uStat=VnsGetMailBodyPart(hVNMAS, (LPSTR) &UserName, FoldName, MsID, MAINBODY, tempbuffptr, offset, (unsigned short far *)&retcount);
		if (Recover(uStat))
			uStat=VnsGetMailBodyPart(hVNMAS, (LPSTR) &UserName, FoldName, MsID, MAINBODY, tempbuffptr, offset, (unsigned short far *)&retcount);
	}
	offset++;
	if (uStat) {
		/*** Error occurrred in reading message body ***/
		SafeFree(GLOBAL, hTempMem);
		return (FALSE);
	}


	/*** Cycle through once for each message body chunk ***/
	while(retcount > 0){

		switch(Body[MAINBODY].contentType) {
			case FREETXT2:
				/*** Remove single LF characters. They ***/
				/*** represent soft carriage returns.  ***/
				finalbuff[0] = '\0';
				for(i=0, j=0; i<retcount && i<1024; i++, j++) {
					if ((tempbuffptr[i] == 0x0A) && (tempbuffptr[i-1] != 0x0D)) {
						if (tempbuffptr[i-1] == 0x0A) {
							finalbuff[j++] = 0x0D;
							finalbuff[j++] = 0x0A;
							finalbuff[j++] = 0x0D;
							finalbuff[j] = 0x0A;
						}
						else
							j--;
					}
					else
						finalbuff[j] = tempbuffptr[i];
				}
				finalbuff[j] = '\0';

				/*** Copy this chunk out to the Global text string ***/
				lstrncat(textstr, finalbuff, j+1);

				/*** The next statement is necessary if the CR and LF are ***/
				/*** contained in seperate chunks.  That is, if the first ***/
				/*** character of the next cchunk is a LF, then we simply ***/
				/*** look at this character to determine if it was a CR.  ***/
				buff[0] = finalbuff[j];
				break;

			case FREETXT1:
				/*** Spin through the message body replacing LF's with ***/
				/*** CR-LF pairs, and storing the output in finalbuff. ***/
				finalbuff[0] = '\0';
				for(i=0, j=0; i<retcount && i<1024; i++, j++) {
					/*** convert LF to CR-LF ***/
					if (tempbuffptr[i] == 0x0A && tempbuffptr[i-1] != 0x0D){
						finalbuff[j++] = '\r';
						finalbuff[j] = '\012';
					}
					else
						finalbuff[j] = tempbuffptr[i];
				}
				finalbuff[j] = '\0';

				/*** Copy this chunk out to the Global text string ***/
				lstrncat(textstr, finalbuff, j+1);

				/*** The next statement is necessary if the CR and LF are ***/
				/*** contained in seperate chunks.  That is, if the first ***/
				/*** character of the next cchunk is a LF, then we simply ***/
				/*** look at this character to determine if it was a CR.  ***/
				buff[0] = finalbuff[j];
				break;

			default:
				lstrncat(textstr, tempbuffptr, retcount);
				break;
		} /*** End switch ***/




		/*** Grab the next chunk of message body ***/
		for(i=0, uStat=TRUE; i<3 && uStat; i++) {
			uStat = VnsGetMailBodyPart(hVNMAS, (LPSTR) &UserName, FoldName, MsID, MAINBODY, tempbuffptr, offset, (unsigned short far *) &retcount);
			if (Recover(uStat))
				uStat = VnsGetMailBodyPart(hVNMAS, (LPSTR) &UserName, FoldName, MsID, MAINBODY, tempbuffptr, offset, (unsigned short far *) &retcount);
		}
		offset++;

		if (uStat) {
			/*** Error occurrred in reading message body ***/
			SafeFree(GLOBAL, hTempMem);
			return (FALSE);
		}

		/*** If the message is bigger than 62K (approx.) then stop 		***/
		/*** importing the message.  This is necessary because the 		***/
		/*** message is always placed in an edit control to manipulate, ***/
		/*** and standard edit controls cannot handle text > 64K. 		***/
		if (offset >= (unsigned) (MAXSIZE/1024L)){
			break;
		}

	}


	/*** Add a NULL to the end of our message text (just in case) ***/
	lstrcat(textstr, "\0");

	/*** Convert text to PC-Multinational. ***/
	OemToAnsi(startBody, startBody);

	/*** Unlock our message text pointer, (but don't free it) ***/
	GlobalUnlock(hMessage);

	/*** Free up the memory for our temp buffer. ***/
	SafeFree(GLOBAL, hTempMem);
	return(TRUE);
}



/******************************************************************************
	FUNCTION: 	FormatEnvelope()

	PURPOSE:  	To return a handle to a string which contains a mail message's envelope
				in a format suitable for printing, filing to disk, or displaying.

	COMMENTS:	The longform variable is used to differentiate between normal
				messages and the abbreviated format message which is used when
				forwarding a message.  In the abbreviated format certain fields
				are not included (Subject, Certify, etc... ).
*******************************************************************************/
FormatEnvelope(char far *FoldName, char far *MsID, Env far *env, GLOBALHANDLE far *hMessage, HWND hEdit, BOOL longForm)
{
	GLOBALHANDLE	hTempMem;
	char 			far *textstr, far *buf2;
	char 			date_str[50];
	int 			i;
	unsigned short 	retcount=0;
	unsigned long 	bsize;
	CallStatus 		uStat = 0;
	BodyPart 		Body[11];
	HWND			hNewEdit;
	RECT			Rect;
	HDC				hDC;
	unsigned int	Headerlen, Linelen;
	HFONT			hFont;
	unsigned int	tabset[1];


	/*** Fill the envelope structure with Mail Message Info ***/
	NullFill((char far *) env, sizeof(Env));
	for(i=0, uStat=TRUE; i<3 && uStat; i++) {
		uStat=VnsGetMailEnvelope(hVNMAS, (LPSTR) &UserName, FoldName, MsID, env);
		if (Recover(uStat))
			uStat=VnsGetMailEnvelope(hVNMAS, (LPSTR) &UserName, FoldName, MsID, env);
	}
	if (uStat) return (FALSE);

	/*** About to read an envelope, so change ***/
	/*** the status to READ in the folder.    ***/
	ChangeUnread(FoldName, MsID);


	/*** Grab enough memory for the message and get a pointer to it. ***/
	/*** Even though I'm only storing the envelope here, This string ***/
	/*** is sometimes used to store the whole message so we'll grab  ***/
	/*** plenty of memory to start with.                             ***/
	bsize = 64000L;
	if ((*hMessage = SafeAlloc(GLOBAL, (unsigned long) bsize, hWnd)) == NULL) {
		VnsReleaseMailEnvelope(hVNMAS, MsID);
		return (FALSE);
	}
	textstr = GlobalLock(*hMessage);
	NullFill(textstr, (unsigned int)bsize);


	/*** Grab enough memory for a temporary buffer and get a pointer to it. ***/
	if ((hTempMem = SafeAlloc(GLOBAL, 1600L, hWnd)) == NULL) {
		VnsReleaseMailEnvelope(hVNMAS, MsID);
		SafeFree(GLOBAL, *hMessage);
		return (FALSE);
	}
	buf2 = GlobalLock(hTempMem);


	/*** Get the Device context and font of our edit window ***/
	hDC = GetDC(hEdit);
	hFont = SendMessage(hEdit, WM_GETFONT, 0, 0L);
	if (hFont != NULL)
		hFont = SelectObject(hDC, hFont);

	/*** Calculate the tab size based on wether ***/
	/*** or not we're using a fixed=pitch font. ***/
	if (GetTextExtent(hDC, "abc", 3) == GetTextExtent(hDC, "WWW", 3))
		tabset[0] = GetTextExtent(hDC, "Attached", 8);
	else
		tabset[0] = GetTextExtent(hDC, "Attached:", 9)+4;

	/*** Find the length of our longest field title with one tab appended ***/
	if(lstrlen(env->forwarder) > 0)
		Headerlen = LOWORD(GetTabbedTextExtent(hDC, "Forwarded By:\t", 14, 1, (LPINT) &tabset));
	else
		Headerlen = LOWORD(GetTabbedTextExtent(hDC, "Attached:\t", 10, 1, (LPINT) &tabset));
	ReleaseDC(hEdit, hDC);


	/*** Create a temporary Edit Window for formatting envelope fields.       ***/
	/*** The width of the new window is set to the width of our real edit     ***/
	/*** window minus the field title width established above.  The font      ***/
	/*** of the temporary window is set to match the font of our destination. ***/
	SendMessage(hEdit, EM_GETRECT, 0, (LPARAM) &Rect);
	Linelen = (Rect.right - Headerlen);
	hNewEdit = CreateWindow("edit", "", GetWindowLong(hEdit, GWL_STYLE)&WS_INVISIBLE, 0, 0,
				Linelen, Rect.bottom, hWnd, PB_EDIT, hInst, NULL);
	SendMessage(hNewEdit, WM_SETFONT, SendMessage(hEdit, WM_GETFONT, 0, 0L), TRUE);


	/*** Now create the envelope text by concatenating  ***/
	/*** all the various message parts.                  ***/

	/*** Get the Mail To: Recipients ***/
	ReadRecipients(hNewEdit, RECIPIENTTO, textstr, MsID, Headerlen, (LPINT) &tabset);
	ReadRecipients(hNewEdit, RECIPIENTCC, textstr, MsID, Headerlen, (LPINT) &tabset);
	ReadRecipients(hNewEdit, RECIPIENTBCC, textstr, MsID, Headerlen, (LPINT) &tabset);


	/*** Add the From field ***/
	FormatText(hNewEdit, "From:", env->from, textstr, Headerlen, (LPINT) &tabset);


	if (longForm) {
		/*** Add the Certified field ***/
		lstrcpy(buf2, "Certify:");
		AddTabs(buf2, Headerlen, hNewEdit, (LPINT) &tabset);
		lstrcat(textstr, buf2);
		if(env->certified==NULL || env->certified==NOMESCONFIRM)
			lstrcat(textstr, "N\r\012");
		else
			lstrcat(textstr, "Y\r\012");

		/*** Add the Subject field ***/
		if(env->subject!=NULL && lstrlen(env->subject) > 0)
			FormatText(hNewEdit, "Subject:", env->subject, textstr, Headerlen, (LPINT) &tabset);
		else
			FormatText(hNewEdit, "Subject:", "...no subject...", textstr, Headerlen, (LPINT) &tabset);
	}


	/*** Add the Date field ***/
	VnsGetMailMsgTimeStr(hVNMAS, (LPSTR) &UserName, MsID, (LPSTR) &date_str);
	FormatText(hNewEdit, "Date:", (LPSTR) &date_str, textstr, Headerlen, (LPINT) &tabset);


	/*** Retrieve info on the attached files ***/
	NullFill((char far *) &Body, sizeof(BodyPart) * 11);
	for(i=0, uStat=TRUE; i<3 && uStat; i++) {
		uStat=VnsGetMailBodyPartList(hVNMAS, MsID, Body, &retcount);
		if (Recover(uStat))
			uStat=VnsGetMailBodyPartList(hVNMAS, MsID, Body, &retcount);
	}


	/*** Error detected trying to read attachment info ***/
	if(uStat){
		VnsReleaseMailEnvelope(hVNMAS, MsID);
		SafeFree(GLOBAL, hTempMem);
		SafeFree(GLOBAL, *hMessage);
		return (FALSE);
	}/* if */


	/**** Add the Attached Files Names ****/
	buf2[0] = '\0';
	for (i=1;i<=10 && Body[i].present == PRESENT;i++) {
		lstrcat(buf2, Body[i].label);
		if (i<10 && Body[i+1].present == PRESENT) lstrcat(buf2, ", ");
	}
	if (i==1) lstrcat(buf2, "None");
	FormatText(hNewEdit, "Attached:", buf2, textstr, Headerlen, (LPINT) &tabset);


	/**** Add the Name of the forwarder ****/
	if(lstrlen(env->forwarder) > 0)
		FormatText(hNewEdit, "Forwarded By:\t", env->forwarder, textstr, Headerlen, (LPINT) &tabset);

	lstrcat(textstr, "\r\012");

	/*** Done with the temp window so.. destroy it. ***/
	DestroyWindow(hNewEdit);

	/*** Unlock Envelop Text pointer, but do not destroy ***/
	GlobalUnlock(*hMessage);

	/*** Free memory for the temporary buffer ***/
	SafeFree(GLOBAL, hTempMem);

	return (TRUE);
}


/******************************************************************************
	FUNCTION: 	FormatMessage()

	PURPOSE:  	To return a handle to a string which contains a mail message
				in a format suitable for printing, filing to disk, or displaying.
*******************************************************************************/
#define fmode O_WRONLY|O_BINARY|O_CREAT|O_TRUNC|O_APPEND
BOOL FormatMessage(char far *FoldName, char far *MsID, Env far *env, GLOBALHANDLE far *hMessage, HWND hEdit)
{

	/*** Format the message envelope ***/
	if (!FormatEnvelope(FoldName, MsID, env, hMessage, hEdit, TRUE))
		return FALSE;
	if (!FormatBody(FoldName, MsID, *hMessage, MAXBODY)) {
		SafeFree(GLOBAL, *hMessage);
		VnsReleaseMailEnvelope(hVNMAS, MsID);
		return FALSE;
	}
	VnsReleaseMailEnvelope(hVNMAS, MsID);
	return(TRUE);
}


/******************************************************************************
	FUNCTION: 	FormatText()

	PURPOSE:  	To format a string of text by inserting CR-LF pairs at
				the end of a line with length Headerlen.

	COMMENTS:	The output from this function will be a field title (e.g.
				"To:") followed by tabs and the field information.  If the
				field info flows onto the next line, a CR-LF is added to the
				end of the first line and tabs are inserted on the next line
				before the text.  In this manner all the field information
				will be tabbed over, and will appear lined up on the display
				device.


	PARAMETERS:	hNewEdit:	an edit window which is pre-sized to the proper
							width for the field information,
				pre:		the field title,
				intext:		a string containing the field information,
				textstr:	append output string to this string pointer
				Headerlen:	length of space to place before the field
							information
*******************************************************************************/
FormatText(HWND hNewEdit, char far *pre, char far *intext, char far *textstr, unsigned int Headerlen, int far *tabset)
{
	int			j, index, retcount, far *nptr;
	char 		buff[1024], bstr[512];

	/*** Add the field information text to the temporary edit window ***/
	SendMessage(hNewEdit, WM_SETTEXT, 0, (LONG) intext);

	/*** Copy the field title to a temporary buffer ***/
	lstrcpy((LPSTR)&bstr, pre);
	buff[0] = '\0';

	/*** Get the number of lines in our edit ***/
	/*** window and spin through each line   ***/
	j = SendMessage(hNewEdit, EM_GETLINECOUNT, 0, 0L);
	index = 0;
	while (index < j) {
		/*** Add tabs to our temp buffer until it is ***/
		/*** long enough to append it to our output  ***/
		AddTabs((LPSTR)&bstr, Headerlen, hNewEdit, tabset);
		lstrcat(textstr, (LPSTR) &bstr);

		/*** Grab the next line from the edit window and add a null to the end ***/
		nptr = (int far *) &buff;
		*nptr = 500;
		retcount = SendMessage(hNewEdit, EM_GETLINE, index++, (LONG) &buff);
		buff[retcount] = '\0';

		/*** Append this line and a CR-LF pair to our output ***/
		lstrcat(textstr, (LPSTR) &buff);
		lstrcat(textstr, "\r\012");

		/*** Reset the temporary buffer ***/
		bstr[0] = '\0';
	}
	return;
}


/******************************************************************************
	FUNCTION: 	GetLetter()

	PURPOSE:  	To return a pointer to the Letter structure with
				the matching Read window handle.
*******************************************************************************/
GLOBALHANDLE GetLetter(HWND hLetter)
{
	GLOBALHANDLE hTempMem, hTempMem2;
	Letter far *TempLetter;

	/*** Start with first Letter in the linked list and compare all    ***/
	/*** Letters till we find one with a matching child window handle. ***/
	hTempMem = TopLetter;
	while(hTempMem != NULL) {
		TempLetter = (Letter far *) GlobalLock(hTempMem);
		if (TempLetter->hChild == hLetter) {
			/*** Found a match ***/
			GlobalUnlock(hTempMem);
			return (hTempMem);
		}
		else {
			hTempMem2 = TempLetter->Next;
			GlobalUnlock(hTempMem);
			hTempMem = hTempMem2;
		}
	}
	/*** No match found ***/
	return (NULL);
}




/******************************************************************************
	FUNCTION: 	GetLetterByFoldName()

	PURPOSE:  	To return a pointer to the first Letter structure with
				the matching folder name.
*******************************************************************************/
GLOBALHANDLE GetLetterByFoldName(char far *FoldName)
{
	GLOBALHANDLE hTempMem, hTempMem2;
	Letter far *TempLetter;

	/*** Start with first Letter in the linked list and compare   ***/
	/*** all Letters till we find one with a matching folder name. ***/
	hTempMem = TopLetter;
	while(hTempMem != NULL) {
		TempLetter = (Letter far *) GlobalLock(hTempMem);
		if (lstrcmp(FoldName, TempLetter->FolderName) == 0) {
			/*** Found a match ***/
			GlobalUnlock(hTempMem);
			return (hTempMem);
		}
		else {
			hTempMem2 = TempLetter->Next;
			GlobalUnlock(hTempMem);
			hTempMem = hTempMem2;
		}
	}
	/*** No match found ***/
	return (NULL);
}


/******************************************************************************
	FUNCTION: 	GetLetterByID()

	PURPOSE:  	To return a pointer to the Letter structure with
				the matching Msg ID.
*******************************************************************************/
GLOBALHANDLE GetLetterByID(char far *ID)
{
	GLOBALHANDLE hTempMem, hTempMem2;
	Letter far *TempLetter;

	/*** Start with first Letter in the linked list and compare   ***/
	/*** all Letters till we find one with a matching folder name. ***/
	hTempMem = TopLetter;
	while(hTempMem != NULL) {
		TempLetter = (Letter far *) GlobalLock(hTempMem);
		if (lstrcmp(ID, TempLetter->MsgID) == 0) {
			/*** Found a match ***/
			GlobalUnlock(hTempMem);
			return (hTempMem);
		}
		else {
			hTempMem2 = TempLetter->Next;
			GlobalUnlock(hTempMem);
			hTempMem = hTempMem2;
		}
	}
	/*** No match found ***/
	return (NULL);
}


/******************************************************************************
	FUNCTION:   GetLetterInfo()

	PURPOSE:  	To return the Message ID and folder name of the current mail
				message and an array of message IDs for	all selected messages.

	PARAMETERS:	MsgID:		stores the first selected Message ID on output.
				FoldName:	stores Folder Name of selected Message on output.

				Optional Parameters:	(Set to NULL if not using)
				lParam 		stores Letter structure handle on input.
				hIDArray	stores ID array of all selected messages on output.
				first		stores index of first selected on output.
*******************************************************************************/
int GetLetterInfo(char far *MsgID, char far *FoldName, long lParam, GLOBALHANDLE far *hIDArray)
{
	GLOBALHANDLE	hArrMem, hFolder, hLetter;
	HWND 			hTempWnd;
	MailFolder far 	*TempFolder;
	Letter far		*TempLetter;
	WORD			index;
	char far 		*IDArray, buff[20];
	int				numitems, far *IndexArray, i;

	/*** When GetLetterInfo is called from the read Window, lParam is  ***/
	/*** used to identify which message we want to return information  ***/
	/*** on, and only the first element of the IDArray is filled. When ***/
	/*** this function is called from a Folder window, however, we     ***/
	/*** assume the programmer wants IDArray filled with all selected  ***/
	/*** messages of that folder.                                      ***/

	numitems = 1;
	/*** Called from a folder window ***/
	if (lParam == NULL) {
		/*** Get pointer to Top Folder currently on the screen. ***/
		/*** Actually the Top Child Window of the MDI CLient.   ***/
		hTempWnd = GetWindow(GetWindow(hWnd, GW_CHILD), GW_CHILD);
		hFolder = GetFolder(hTempWnd);
		if (hFolder == 0) {
			return -1;
		}

		TempFolder = (MailFolder far *) GlobalLock(hFolder);

		/*** If folder is empty -- issue a warning and return ***/
		if (TempFolder->MsgCount <= 0) {
			BWCCMessageBox(hTempWnd, "Folder is Empty!", "WARNING!", MB_ICONSTOP | MB_OK);
			GlobalUnlock(hFolder);
			return 0;
		}

		/*** Get the number of selected items ***/
		numitems =  SendMessage(TempFolder->hListBox, LB_GETSELCOUNT, 0, 0L);
		if (numitems > 0) {

			/*** Grab enough memory to store the indices of the selected items ***/
			if ((hArrMem = SafeAlloc(GLOBAL, (long) (2*numitems), hWnd)) == NULL) {
				/*** Couldn't allocate the memory so pretend nothing was selected ***/
				index = 0;
				numitems = 0;
			}
			else {
                /*** Get a pointer to our indices array and fill it with indices ***/
				IndexArray = (int far *) GlobalLock(hArrMem);
				numitems   = SendMessage(TempFolder->hListBox, LB_GETSELITEMS, numitems, (LONG) IndexArray);
				if (hIDArray != NULL) {
                    /*** Allocate memory for the message ID array ***/
					if ((*hIDArray = SafeAlloc(GLOBAL, (long)(14*numitems), hWnd)) == NULL) {
						/*** Couldn't allocate the memory so pretend nothing was selected ***/
						numitems = 0;
						index = 0;
					}
					else {
						/*** Get the message ID's for the selected ***/
						/*** items and store them in the ID array. ***/
						IDArray = GlobalLock(*hIDArray);
						for (i=0;i<numitems;i++)
							ReadID(TempFolder, IndexArray[i], (char far *)&(IDArray[i*14]));
						GlobalUnlock(*hIDArray);
					}
				}

                /*** Store selected message index and free indices array ***/
				index = IndexArray[0];
				SafeFree(GLOBAL, hArrMem);
			}
		}
		else
			index = 0;

		/*** Store the ID of the first selected message to MsgID   ***/
		/*** and store the name of the current folder to FoldName. ***/
		ReadID(TempFolder, index, MsgID);
		lstrcpy((LPSTR)&buff, TempFolder->Name);
		lstrcpy(FoldName, (LPSTR) &buff);

		/*** Unlock Folder. ***/
		GlobalUnlock(hFolder);
	}
	else {
		/*** Called from a read window -- so don't ***/
		/*** mess with all the selected messages.  ***/

		/*** Use lParam to get the handle of the appropriate Letter ***/
		hLetter = (GLOBALHANDLE) lParam;
		TempLetter = (Letter far *) GlobalLock(hLetter);

		/*** Copy the message ID and folder name from the Letter structure. ***/
		lstrncpy(MsgID, TempLetter->MsgID, 14);
		lstrcpy(FoldName, TempLetter->FolderName);


		/*** Store the message ID to the first element of the array (also optional) ***/
		if (hIDArray != NULL) {
			if ((*hIDArray = SafeAlloc(GLOBAL, 14L, hWnd)) == NULL) {
				numitems = 0;
			}
			else {
				IDArray = GlobalLock(*hIDArray);
				lstrncpy(IDArray, MsgID, 14);
				GlobalUnlock(*hIDArray);
			}
		}

		/*** Unlock Letter. ***/
		GlobalUnlock(hLetter);
	}
	return numitems;

}


/******************************************************************************
	FUNCTION: 	KillLetter()

	PURPOSE:  	To delete a Letter structure, release resources associated
				with the letter, and reset the pointers.
*******************************************************************************/
void KillLetter(HWND hLetter)
{
	GLOBALHANDLE 	hTempMem, hTempMem2, hEditDS;
	Letter 			far *TempLetter, far *TempLetter2;
	HFONT			hFont;


	/*** Find the right letter to kill ***/
	hTempMem = GetLetter(hLetter);
	if (hTempMem == NULL)
		return;
	TempLetter = (Letter far *) GlobalLock(hTempMem);

	/**** Delete the Font for the Letter and destroy the edit box ***/
	hFont = SendMessage(TempLetter->hViewMail, WM_GETFONT, 0, 0L);
	if (hFont != NULL) {
		SendMessage(TempLetter->hViewMail, WM_SETFONT, 0, 0L);
		DeleteObject(hFont);
	}
	hEditDS = GetWindowWord(TempLetter->hViewMail, GWW_HINSTANCE);
	DestroyWindow(TempLetter->hViewMail);
	SafeFree(GLOBAL, hEditDS);


	/*** Take Letter out of linked list and  rewrite the pointers ***/
	if (hTempMem == TopLetter) {
		TopLetter = TempLetter->Next;
	}
	else {
		/*** Letter is in middle of list ***/
		TempLetter2 = (Letter far *) GlobalLock(TempLetter->Prev);
		TempLetter2->Next = TempLetter->Next;
		GlobalUnlock(TempLetter->Prev);
		if (TempLetter->Next != NULL) {
			TempLetter2 = (Letter far *) GlobalLock(TempLetter->Next);
			TempLetter2->Prev = TempLetter->Prev;
			GlobalUnlock(TempLetter->Next);
		}
	}

	/*** Free the letter structure memory ***/
	SafeFree(GLOBAL, hTempMem);
	return;
}


/******************************************************************************
	FUNCTION: 	NextLetter()

	PURPOSE:  	To read the next letter, the previous letter, or the re-read
				the same letter into a letter structure and its associated
				Read window.

	PARAMETERS:	hLetter:	window handle of the Read window to update
				Next:		which letter to read in (Next, Previous, or Same)
				Close:		whether to Close the Read window when attempting
							to read beyond the end of the list
*******************************************************************************/
BOOL NextLetter(HWND hLetter, BOOL Next, BOOL Close)
{
	GLOBALHANDLE 	hTextMem, hFolderMem, hLetterMem;
	Letter far 		*TempLetter;
	MailFolder far	*TempFolder;
	char far		*textstr, MsID[14], buff[256];
	WORD far		*TempMs;
	int				LetterCount, i, start, LetterPtr;
	Env         	env;
	unsigned int	lSize, lSize2;

	HANDLE			HTEMP;


	/*** Get a pointer to the associated letter structure ***/
	hLetterMem = GetLetter(hLetter);
	if (hLetterMem == NULL) {
		return FALSE;
	}
	TempLetter = (Letter far *) GlobalLock(hLetterMem);


	LetterCount = 0;
	if (IsWindow(TempLetter->hList)) {
		/*** Find the number of messages in the folder. ***/
		LetterCount = SendMessage(TempLetter->hList, LB_GETCOUNT, 0, 0L);

		/*** If Folder is empty, close the Read Window ***/
		if (LetterCount == 0) {
			GlobalUnlock(hLetterMem);
			SendMessage(GetParent(hLetter), WM_MDIDESTROY, hLetter, NULL);
			return FALSE;
		}


		/*** Set LetterPtr to the index of the current message.    ***/
		LetterPtr   = FindID(TempLetter->hFolder, TempLetter->MsgID);
	}
	else if (Next != SAME) {
		GlobalUnlock(hLetterMem);
		SendMessage(GetParent(hLetter), WM_MDIDESTROY, hLetter, NULL);
		return FALSE;
	}


	/*** Next or Previous Letter is to be read in... So advance    ***/
	/*** our LetterPtr to point to the next message to be read in. ***/
	if (Next != SAME) {
		if ((LetterPtr<1 && (Next == NEXT)) || ((LetterPtr+1) >= LetterCount && (Next == PREV))) {
			/*** Attempting to read beyond end of list (or before beginning of list)     ***/
			/*** So if Close was specified in the parameters then close the Read window. ***/
			if (Close || LetterCount == 0) {
				GlobalUnlock(hLetterMem);
				SendMessage(GetParent(hLetter), WM_MDIDESTROY, hLetter, NULL);
				return FALSE;
			}
		}
		else {
			/*** Somewhere in the middle of the list... ***/
			/*** so simply advance the Letter pointer.  ***/
			if (Next == NEXT)
				LetterPtr--;
			else
				LetterPtr++;
		}
	}


	if (IsWindow(TempLetter->hList)) {
		/*** Grab a pointer to the Folder associated with the current message ***/
		hFolderMem = GetFolder2(TempLetter->FolderName);
		TempFolder = (MailFolder far *) GlobalLock(hFolderMem);

		/*** Make sure LetterPtr is not pointing beyond the ends of the list ***/
		if (LetterPtr > TempFolder->MsgCount)
			LetterPtr = TempFolder->MsgCount;
		if (LetterPtr > MaxMsgs)
			LetterPtr = MaxMsgs;
		if (LetterPtr < 0)
			LetterPtr = 0;

		/*** Retrieve the Message ID for the new message and release the Folder pointer ***/
		ReadID(TempFolder, LetterPtr, MsID);
		GlobalUnlock(hFolderMem);
	}
	else {
		lstrncpy((LPSTR) &MsID, TempLetter->MsgID, 14);
	}


	/*** Read and format the new message into a memory handle ***/
	if (FormatMessage(TempLetter->FolderName, (LPSTR) &MsID, &env, &hTextMem, TempLetter->hViewMail) != TRUE) {
		DspMsgCode(hLetter, "Unable to Display Mail Message ", 0, FALSE );
	} /* if */
	else {
		/** Copy current letter ID to Letter Structure ***/
		lstrncpy(TempLetter->MsgID, (LPSTR) &MsID, 14);

		/*** Put the new message text into the Read window ***/
		textstr = (char far *) GlobalLock(hTextMem);
		if ((unsigned)lstrlen(textstr) > (unsigned)MAXBODY) {
			BWCCMessageBox(hWnd, "Message too long\n\nfor HappyMail!\n\nText will be truncated.", "WARNING!", MB_ICONSTOP | MB_OK);
			textstr[MAXBODY] = '\0';
		}
		SetWindowText(TempLetter->hViewMail, (LPSTR) textstr);
		SafeFree(GLOBAL, hTextMem);


		/**** Display Subject at top of Read Window ****/
		if(env.subject!=NULL && lstrlen((LPSTR) &env.subject) > 0)
			SendMessage(hLetter, WM_SETTEXT, NULL, (LONG) (LONG) env.subject);
		else
			SendMessage(hLetter, WM_SETTEXT, NULL, (LONG) (LONG) "....no subject....");


	}

	/*** Move the List box ptr to match the new messsage ***/
	if (IsWindow(TempLetter->hList)) {
		PostMessage(TempLetter->hList, LB_SETSEL, FALSE, MAKELPARAM(-1, 0));
		PostMessage(TempLetter->hList, LB_SETSEL,  (WPARAM)(BOOL)TRUE, MAKELPARAM(LetterPtr, 0));
	}

	/*** Release our pointer to the Letter structure ***/
	GlobalUnlock(hLetterMem);
	return TRUE;
}




/******************************************************************************
	FUNCTION: 	ReadRecipients();

	PURPOSE:  	Reads in all recipients from a given class (TO:, CC:, or BCC:)
				and stores them to a string.  This function also uses FormatText
				to format the output string to match the format used in the
				envelope.

	PARAMETERS: hNewEdit:	a temporary edit window for formatting the output
							text (used in call to FormatText) [Optional]
				rclass:		which recipient class to read (TO:, CC: or BCC:)
				textstr:	destination string pointer (append output to end)
				MsID:		message ID
				HeaderLen:	length of space to place in front of the field
							information (used in call to FormatText)

*******************************************************************************/
ReadRecipients(HWND hNewEdit, char rclass, char far *textstr, char far *MsID, unsigned int Headerlen, int far *tabset)
{
	int			offset, i;
	char 		buff[2048], bstr[512];

	/*** Initialize variables ***/
	offset = 0;
	buff[0] = '\0';

	/*** Read in one recipient at a time and append them to our temp buffer ***/
	do {
		/*** Read a recipient into bstr ***/
		if (VnsGetMailRecipient(hVNMAS, MsID, rclass, bstr, offset++) != 0) {
			return FALSE;
		}

		if (lstrlen((LPSTR) &bstr) > 0){
			/*** Append a comma if this is not the first name ***/
			if (lstrlen((LPSTR) &buff) > 0)
				lstrcat((LPSTR) &buff, ", ");
			/*** Append the new name ***/
			lstrcat((LPSTR) &buff, (LPSTR) &bstr);
		}
	} while (bstr[0] != '\0');

	/*** Check for Empty List of Names ***/
	if (lstrlen((LPSTR) &buff) == 0 && (rclass != RECIPIENTTO))
		return FALSE;

	/*** If the optional hNewEdit parameter == NULL, append the list of names ***/
	/*** directly to the output string without any additional formatting.     ***/
	if (hNewEdit == NULL)
		lstrcat(textstr, (char far *) &buff);
	else {
		/*** Create a Title string based on the recipient class ***/
		if (rclass== RECIPIENTTO)
			lstrcpy((LPSTR) &bstr, "To:");
		else if (rclass== RECIPIENTCC)
			lstrcpy((LPSTR) &bstr, "Cc:");
		else if (rclass== RECIPIENTBCC)
			lstrcpy((LPSTR) &bstr, "Bcc:");

		/*** Format the output and add it to the output string ***/
		FormatText(hNewEdit, bstr, buff, textstr, Headerlen, tabset);
	}


	return TRUE;

}

