/****************************************************************************
	FILE:		launch.c


	CONTENTS:   ManageAttach()		window procedure for Manage Attachments
									dialog box.
				SaveAttachments()	determines file name and saves attachment
				SaveAttachment()	saves an attachment to specified filename

	COMMENTS:	This file contains all the procedure(s) used in launching
				files attached to Banyan mail messages.
****************************************************************************/
#include "MyMail.h"
#include "string.h"
#include "commdlg.h"
#include "shellapi.h"
#include "dos.h"
#include "dir.h"
#include "stdio.h"

#define MAKEINT(low, high)  ((UINT)(((BYTE)(low)) | (((WORD) ((BYTE)(high))) << 8)))

BOOL SaveAttachment(HWND, Attach_Rec far *, BodyPart far *, char far *);


/******************************************************************************
	FUNCTION: 	ManageAttach()

	PURPOSE:  	Dialog Box procedure for the Manage Attachments Window.
*******************************************************************************/
BOOL FAR PASCAL _export ManageAttach(HWND hDlg, unsigned message, WORD wParam, LONG lParam)
{
	char 					Buff[144], FileName[144], TempName[144], AttachName[144],
							directory[144], testchar;
	int 					RetCount, fhandle, i, j;
	Env						env;
	CallStatus 				uStat = 0;
	unsigned short			bytes, Errval, Last;
	GLOBALHANDLE			hBody;
	static BodyPart 		Body[11];
	static char 			MsID[14], FoldName[32];
	static unsigned short 	Partnum;
	static Boolean			filing, endproc;
	static Attach_Rec		aRec;
	FARPROC					lpfnBarGraf;
	BOOL					UseDef;


	/*** Send Window Messages to Status Bar ***/
	StatBarCtrlMsg(hDlg, AttachBox, message, wParam, lParam);

	switch (message) {
		case WM_INITDIALOG:        /* message: initialize dialog box */

			/*** Center the Dialog Window in the HappyMail Main window ***/
			CenterWindow(hDlg, GetDesktopWindow());

			/***** Get the Message ID and Folder Name for the Current Message ***/
			GetLetterInfo((LPSTR) &MsID,  (LPSTR) &FoldName, lParam, NULL);
			lstrcpy(aRec.FoldName, FoldName);
			lstrcpy(aRec.MsID, MsID);

			/**** Get the envelope and Body Part List ***/
			uStat = VnsGetMailEnvelope(hVNMAS, UserName, FoldName, MsID, &env);
			if (Recover(uStat))
				uStat = VnsGetMailEnvelope(hVNMAS, UserName, FoldName, MsID, &env);
			if (uStat){
				DspMsgCode( hWnd, "Error Opening Mail : ", uStat, FALSE );
				EndDialog(hDlg, TRUE);
				return (TRUE);
			}

			/*** Change Status on old message (UNREAD to READ) ***/
			ChangeUnread(FoldName, MsID);

			NullFill((char far *) &Body, sizeof(BodyPart) * 11);
			uStat = VnsGetMailBodyPartList(hVNMAS, MsID, (BodyPart WFAR *) Body, (unsigned short far *) &RetCount);
			if (Recover(uStat))
				uStat = VnsGetMailBodyPartList(hVNMAS, MsID, (BodyPart WFAR *) Body, (unsigned short far *) &RetCount);
			VnsReleaseMailEnvelope(hVNMAS, MsID);
			if (uStat){
				DspMsgCode( hWnd, "Error Opening Attachments : ", uStat, FALSE );
				EndDialog(hDlg, TRUE);
				return (TRUE);
			}


			/*** Add the attachment names to the Listbox ***/
			Partnum = 1;
			while (Body[MAINBODY+Partnum].present == PRESENT) {
				for(i=lstrlen(Body[MAINBODY+Partnum].label); i>0; i--) {
					testchar = Body[MAINBODY+Partnum].label[i];
					if ((testchar == '\\') || (testchar == ':')) {
						i++;
						break;
					}
				}
				SendDlgItemMessage(hDlg, ATB_FILENAMES, LB_ADDSTRING, NULL, (LONG)  Body[MAINBODY+Partnum].label + i);
				Partnum++;
			}
			SendDlgItemMessage(hDlg, ATB_FILENAMES, LB_SETCURSEL, 0, NULL);

			/*** If no attachments are present display error and return ***/
			if (Partnum == 1) {
				BWCCMessageBox(hDlg, "No Attachments for This Message.", "WARNING!", MB_ICONSTOP | MB_OK);
				EndDialog(hDlg, TRUE);
				return (TRUE);
			}


			/*** Initialize the BarGraf. ***/
			BarGraf(hDlg, WM_NEXTSTEP, 0, 0L);
			filing = FALSE;
			endproc = FALSE;
			bUserAbort = FALSE;

			/*** Initialize the Checkboxes. ***/
			CheckDlgButton(hDlg, ATB_USEDEF, Options.usedefault);
			CheckDlgButton(hDlg, ATB_FILTER, Options.usefilters);

			return (TRUE);


		case WM_SYSCOMMAND:
			/*** User selected Close from the Sytem Menu. ***/
			if ((wParam & 0xFFF0) == SC_CLOSE) {
				EndDialog(hDlg, TRUE);
				return (TRUE);
			}
			break;

		case WM_COMMAND:
			switch (wParam) {
				case ATB_FILENAMES:
					/*** If user double-clicks an attachment   ***/
					/*** name, execute the Launch button code. ***/
					if (lParam >> 16 != LBN_DBLCLK)
						return(TRUE);

				case IDM_LAUNCH:
					/*** Update Status bar and Window Title. ***/
					if (hStatus)
						SendMessage(hStatus, STB_STRING, 0, (LONG)"Reading Attachment File...");
					SendMessage(hDlg, WM_SETTEXT, 0, (LPARAM) "Loading Attachment");

					/*** Change to HourGlass Cursor before starting ***/
					SetCursor(LoadCursor(NULL, IDC_WAIT));

					/*** Get the Currently Selected attachment from the ***/
					/*** combo-box, store the attachment to DOS temp    ***/
					/*** file, then launch the application.             ***/
					Partnum = SendDlgItemMessage(hDlg, ATB_FILENAMES, LB_GETCURSEL, 0, 0L);
					Partnum++;
					aRec.Partnum = Partnum;
					RetCount = 0;

					/*** Open a temporary file. ***/
					GetTempFileName(0, "HMA", 0, TempName);

					/*** Create a variable containing the attachment's directory. ***/
					lstrcpy(directory, TempName);
					for(i=lstrlen((LPSTR) &directory); (directory[i] != '\\') && (directory[i] != ':');i--);
					directory[i+1] = '\0';

					/*** Read in the real name of the attachment and grab the extension ***/
					/*** To check if we have an association before saving the file,     ***/
					SendDlgItemMessage(hDlg, ATB_FILENAMES, LB_GETTEXT, Partnum-1, (LPARAM) AttachName );
					for(i=0,Buff[0] = '\0'; AttachName[i] != '\0' && AttachName[i] != '.'; i++);
					if (AttachName[i] == '.')
						lstrcpy((LPSTR) &Buff, (LPSTR) &AttachName + i + 1);
					Errval = GetProfileString("Extensions", Buff, "\0", (LPSTR) &Buff, 144);
					if (Errval <= 0) {
						lstrcpy(Buff, "Error Occured.\n\nNo association exists\nfor this file.");
						BWCCMessageBox(hDlg, Buff, "HappyMail", MB_ICONHAND);
					}
					else {
						/*** Save the attachment file to disk. ***/
						filing = TRUE;
						EnableWindow(GetDlgItem(hDlg, IDABORT), TRUE);
						SaveAttachment(hDlg, &aRec, (BodyPart far *) &Body, TempName);
						EnableWindow(GetDlgItem(hDlg, IDABORT), FALSE);
						filing = FALSE;

						/*** Grab the extension from the real file      ***/
						/*** name, and append it to our Temp file name. ***/
						for(i=lstrlen(AttachName)-1; i>0 && (AttachName[i] != '.'); i--);
						if (i>0) {
							lstrcpy(FileName, TempName);
							for(j=lstrlen(TempName); j>0 && (TempName[j] != '.'); j--);
							if (j>0) FileName[j] = '\0';
							lstrcat(FileName, AttachName + i);
							/*** Rename our tempfile with the proper extension. ***/
							rename((char *)TempName, (char *)FileName);
						}


						/*** Update Status bar. ***/
						if (hStatus)
							SendMessage(hStatus, STB_STRING, 0, (LONG)"Looking for File Association...");

						/*** Try to Execute the File or its association. ***/
						Errval = ShellExecute(hDlg, NULL, FileName, NULL, directory, SW_SHOW);
						if (Errval < 33) {
							switch (Errval) {
								case 0:
								case 8:
									lstrcpy(Buff, "Error Occured.\n\nInsufficient memory to\nlaunch application.");
									break;
								case 2:
								case 3:
									wsprintf(Buff, "Error Occured.\n\n%s not found.", (LPSTR) &FileName);
									break;
								case 10:
								case 15:
									lstrcpy(Buff, "Error Occured.\n\nIncorrect Windows version.");
									break;
								case 20:
									lstrcpy(Buff, "Error Occured.\n\nMissing or Invalid DLL.");
									break;
								case (unsigned int) 31:
									lstrcpy(Buff, "Error Occured.\n\nNo association exists\nfor this file.");
									break;
								default:
									wsprintf(Buff, "Error #%u occurred\nwhen attempting to launch\n%s.", Errval, AttachName);
							}
							BWCCMessageBox(hDlg, Buff, "HappyMail", MB_ICONHAND);
						}


					}

					/*** Reset Title bar of Window. ***/
					SendMessage(hDlg, WM_SETTEXT, 0, (LPARAM) "Attachments");

					/*** Update Status bar. ***/
					if (hStatus)
						SendMessage(hStatus, STB_STRING, 0, 0L);

					return (TRUE);


				case IDM_FILE2:
				case IDM_FILEALL:
					/*** Place the index of the attachment to save in Partnum.     ***/
					/*** Use the Last variable to indicate saving all attachments. ***/
					if (wParam == IDM_FILEALL) {
						/*** If user selected Save All Attachments AND ***/
						/*** their are NO attachments, then exit.      ***/
						Last =  SendDlgItemMessage(hDlg, ATB_FILENAMES, LB_GETCOUNT, 0, 0L);
						if (Last < 1) // If no attachments  -- skip
							break;
						Partnum = 1;
					}
					else {
						Partnum = SendDlgItemMessage(hDlg, ATB_FILENAMES, LB_GETCURSEL, 0, 0L);
						Partnum++;
						Last = Partnum;
					}

					/*** Get the Default Directory setting. ***/
					aRec.UseDef =  IsDlgButtonChecked(hDlg, ATB_USEDEF);

					/*** Save the current working directory (cwd) -- then ***/
					/*** reset the cwd to the default directory (DEF_DIR) ***/
					getcwd((char *) &directory, 128);
					setcwd(Options.defdir);

					filing = TRUE;
					EnableWindow(GetDlgItem(hDlg, IDABORT), TRUE);
					for (i=Partnum; i<=Last; i++) {
						aRec.Partnum = i;
						SaveAttachments(hDlg, &aRec, (BodyPart far *) &Body);
					}
					EnableWindow(GetDlgItem(hDlg, IDABORT), FALSE);
					filing = FALSE;

					/*** Reset the current working directory ***/
					setcwd(directory);

					/*** Check to see if user cancel-ed during the filing. ***/
					if (endproc)
						PostMessage(hDlg, WM_COMMAND, IDCANCEL, 0L);

					/*** Change Window Title Bar. ***/
					SendMessage(hDlg, WM_SETTEXT, 0, (LPARAM) "Filing Complete");

					return (TRUE);


				case IDM_VIEW:
					aRec.Partnum = SendDlgItemMessage(hDlg, ATB_FILENAMES, LB_GETCURSEL, 0, 0L);
					aRec.Partnum++;

					/*** Get the Default Directory setting. ***/
					aRec.UseFilters =  IsDlgButtonChecked(hDlg, ATB_FILTER);

					/*** Pick a temporary file name. ***/
					GetTempFileName(0, "HMAIL", 0, FileName);
					lstrcpy(aRec.FileName, FileName);

					/*** Save the attachment to a temporary file. ***/
					filing = TRUE;
					EnableWindow(GetDlgItem(hDlg, IDABORT), TRUE);
					SaveAttachment(hDlg, &aRec, (BodyPart far *) &Body, FileName);
					EnableWindow(GetDlgItem(hDlg, IDABORT), FALSE);
					filing = FALSE;

					/*** Check to see if user cancel-ed during the filing. ***/
					if (endproc)
						PostMessage(hDlg, WM_COMMAND, IDCANCEL, 0L);
					if (bUserAbort)
						break;

					/*** Reset Title bar of Window. ***/
					SendMessage(hDlg, WM_SETTEXT, 0, (LPARAM) "Attachments");

					/*** Call up the View Attachment dialog box. ***/
					CallDlg(hDlg, ViewFile, ViewBox, (LPARAM) &aRec);

					/*** Delete the temporary file. ***/
					unlink(aRec.FileName);

					return 0;


				case IDABORT:
					/*** Abort a scan   ***/
					if (filing) {
						SendMessage(hDlg, WM_SETTEXT, 0, (LPARAM) "File Save Aborted");
						bUserAbort = TRUE;
						break;
					}
					return 0;

				case IDCANCEL:
					/*** First abort the scan   ***/
					if (filing) {
						SendMessage(hDlg, WM_SETTEXT, 0, (LPARAM) "File Save Aborted");
						bUserAbort = TRUE;
						endproc = TRUE;
						break;
					}


					/*** Store the status of the two checkboxes. ***/
					Options.usedefault =  IsDlgButtonChecked(hDlg, ATB_USEDEF) ? 1 : 0;
					Options.usefilters =  IsDlgButtonChecked(hDlg, ATB_FILTER) ? 1 : 0;


					/*** User select Exit button. ***/
					EndDialog(hDlg, TRUE);
					return (TRUE);


			} /** end switch(wParam) **/
			break;
	} /* end switch (message) */
	return (FALSE);

}



/******************************************************************************
	FUNCTION:	SaveAttachments()

	PURPOSE:	Determines the new file name for an attachment and the saves it
				by calling the SaveAttachment function.
*******************************************************************************/
BOOL SaveAttachments(HWND hDlg, Attach_Rec far *aRec, BodyPart far *BP)
{
	char			Buff[80], FileBuff[256], OldName[155], testchar,
					szFilter[] = "All Files (*.*)\0*.*\0";
	OPENFILENAME 	ofn;
	DWORD			Errval;
	FARPROC			lpfnBarGraf;
	int				i;
	BodyPart far 	*Body;
	BOOL			UseDef;



	/*** Copy the name of the file being saved to FileBuff ***/
	Body = &BP[aRec->Partnum];
	lstrcpy((LPSTR) &FileBuff, (LPSTR) Body->label) ;
	for(i=lstrlen(FileBuff); i>0; i--) {
		testchar = FileBuff[i];
		if ((testchar == '\\') || (testchar == ':')) {
			i++;
			break;
		}
	}
	lstrcpy(OldName, FileBuff + i);


	/*** If we're using the defaults, concatenate the file name to the  ***/
	/*** default directory, else, ask the user for a filename and path. ***/
	UseDef = aRec->UseDef;
	if (UseDef) {
		lstrcpy((LPSTR) &FileBuff, Options.defdir);
		lstrcat((LPSTR) &FileBuff, (LPSTR) &OldName);

		/*** Check for previous file existence ***/
		if (fexist(FileBuff)) {
			wsprintf((LPSTR)&Buff,"%s\n\nThis file already exists.\n\nReplace existing file?", (LPSTR) &FileBuff);
			i = BWCCMessageBox(hDlg, (LPSTR) &Buff, "HappyMail", MB_YESNOCANCEL);
			if ( i == IDNO) UseDef = FALSE;
			if ( i == IDCANCEL) return(TRUE);
		}

	}

	if (!UseDef) {
		/*** Get a Filename using the the OpenFile dialog box ***/
		lstrcpy((LPSTR) &FileBuff, (LPSTR) &OldName);

		/*** Initialize the OpenFileName structure ***/
		ofn.lStructSize = sizeof( OPENFILENAME );
		ofn.hwndOwner = hDlg;
		ofn.hInstance = hInst;
		ofn.lpstrFilter = szFilter;
		ofn.lpstrCustomFilter = NULL;
		ofn.nMaxCustFilter = 0;
		ofn.nFilterIndex = 1;
		ofn.lpstrFile = (LPSTR) &FileBuff;      // Stores the result in this variable
		ofn.nMaxFile = sizeof( FileBuff );
		ofn.lpstrFileTitle = NULL;
		ofn.nMaxFileTitle = 0;
		ofn.lpstrInitialDir = NULL;
		ofn.lpstrTitle = "Select a File Name for the Attachment"; // Title for dialog
		ofn.Flags = OFN_PATHMUSTEXIST | OFN_HIDEREADONLY | OFN_OVERWRITEPROMPT | OFN_ENABLEHOOK;
		ofn.nFileOffset = 0;
		ofn.nFileExtension = 0;
		ofn.lpstrDefExt = "*";
		ofn.lCustData = NULL;
		ofn.lpfnHook = (DLGHOOKPROC) MakeProcInstance((FARPROC)SaveHook, hInst);
		ofn.lpTemplateName = NULL;

		/*** Open Common dialog Box for Saving Files ***/
		if(GetSaveFileName( &ofn ) != TRUE) {
			/*** Possible error detected ***/
			Errval=CommDlgExtendedError();
			if(Errval!=0) {                  // 0 value means user selected Cancel
				wsprintf((LPSTR)&Buff,"Error = %ld",Errval);
				BWCCMessageBox(hDlg, "Unable to Choose Files", Buff, MB_ICONSTOP | MB_OK);
			}
			FreeProcInstance((FARPROC)ofn.lpfnHook);
			return (TRUE);
		}

		/*** Free up the procedure memory. ***/
		FreeProcInstance((FARPROC)ofn.lpfnHook);
	}

	SaveAttachment(hDlg, aRec, BP, FileBuff);

	return(TRUE);

}



/******************************************************************************
	FUNCTION:	SaveAttachment()

	PURPOSE:	Saves an attachment file to the specified filename and path.
*******************************************************************************/
BOOL SaveAttachment(HWND hDlg, Attach_Rec far *aRec, BodyPart far *BP, char far *FileName)
{
	char			Buff[1024], FileBuff[144];
	int 			fhandle;
	unsigned short	bytes, offset, size;
	unsigned int 	date, time, attr;
	OPENFILENAME 	ofn;
	DWORD			Errval;
	unsigned long	fileSize, fileTime, tbytes;
	BodyPart far 	*Body;
	CallStatus		uStat;

	/*** Initialize the Body pointer. ***/
	Body = &BP[aRec->Partnum];

	/***  Copy FileName to FileBuff for use in non-far C calls. ***/
	lstrcpy((LPSTR) &FileBuff, FileName);

	/**** Open/Create the Specified File ***/
	if ( (fhandle = (_lcreat(FileBuff, 0)) ) == HFILE_ERROR){
		BWCCMessageBox(hDlg, "Unable to Open File -- Invalid File Name.", "WARNING!", MB_ICONSTOP | MB_OK);
		return (FALSE);
	}

	/*** Initialize the Bar Graph abort flag. ***/
	bUserAbort = FALSE;


	/*** Update Window Title bar. ***/
	wsprintf((LPSTR)&Buff, "Saving %s", (LPSTR) (FileName + ParseName(FileName)));
	SendMessage(hDlg, WM_SETTEXT, 0, (LPARAM) &Buff);


	/*** Grab the file size from the attributes in the BodyPart structure    ***/
	/*** Banyan Mail does not implement this feature.  So messages from a    ***/
	/*** Banyan mail client have no file size available and I arbitrarly     ***/
	/*** assign a file size of 1 Meg for displaying progress in the BarGraf. ***/
	/*** As of 5.5 the Mail API does not support the VnsGetMailbodyPartSize. ***/
	fileSize = MAKELONG(MAKEINT(Body->contentAttributes[3], Body->contentAttributes[2]),
				MAKEINT(Body->contentAttributes[1], Body->contentAttributes[0]));
	offset = size = 0;
	tbytes = 0L;
	if (fileSize != 0L) {
		size = ((unsigned short) ((fileSize/1024L) + 1L));
		/*** Initialize the BarGraf. ***/
		BarGraf(hDlg, WM_NEXTSTEP, 0, 0L);
	}
	else
		BarGraf(hDlg, WM_NEXTSTEP, 0, 1L);



	/*** Read in the body parts and write them to the output file ***/
	NullFill((LPSTR) &Buff, 1024);
	do {
		uStat = VnsGetMailBodyPart(hVNMAS, UserName, aRec->FoldName, aRec->MsID, aRec->Partnum, (LPSTR) Buff, offset, (unsigned short WFAR *) &bytes);
		if (uStat == 0) {
			if (bytes > 0)
				_lwrite(fhandle, (void huge *) &Buff, (unsigned int) bytes);
			offset+=1;
		}
		else {
			wsprintf((LPSTR)&Buff,"Error Reading Attachment = %ld", uStat);
			BWCCMessageBox(hDlg, Buff, "Error", MB_ICONSTOP | MB_OK);
			bytes = 0;
		}

		/*** Send a percentage to the BarGraf dialog ***/
		if (size != 0) {
			BarGraf(hDlg, WM_NEXTSTEP, (WPARAM) ((offset*100L)/(long) size), 0L);
		}
		else {
			tbytes += (long) bytes;
			BarGraf(hDlg, WM_NEXTSTEP, 0, tbytes);
		}
		DeClogDlgMsg(hDlg);

		/*** if cancel was selected break out of loop ***/
		if (bUserAbort)
			break;
	} while(bytes!=0);

	/*** Send a percentage to the Bargraf dialog. ***/
	if (!bUserAbort)
		BarGraf(hDlg, WM_NEXTSTEP, 100, 0L);
	DeClogDlgMsg(hDlg);


	/*** Set File Date and Attributes if they were passed ***/
	date = MAKEINT(Body->contentAttributes[5], Body->contentAttributes[4]);
	time = MAKEINT(Body->contentAttributes[7], Body->contentAttributes[6]);
	attr = MAKEINT(Body->contentAttributes[9], Body->contentAttributes[8]);
	if (date) {
		_dos_setftime(fhandle, date, time);
		_dos_setfileattr((char *)FileBuff, attr);
	}

	/*** Close the output file ***/
	_lclose(fhandle);


	if (bUserAbort) {
		/*** User pressed cancel during save, so ***/
		/**** Unlink (delete) the output file.   ***/
		unlink((char *)FileBuff);
		BWCCMessageBox(hWnd, "File Save Aborted!", "HappyMail", MB_OK | MB_ICONEXCLAMATION);
		return FALSE;

	}

	return(TRUE);
}
