/****************************************************************************
	FILE:		ini_lib.c


	CONTENTS:	GetINIFont()	returns a font from the INI file
				GetINILetter()	returns the Read Window size from INI file.
				GetINIParam()	returns a parameter string from INI file.
				InitMain()		restores the main screen size.
				InitScreen()	restores the screen setup.
				InitSysVars()	initializes system variables.
				SaveScreen()	saves screen parameters to INI file.
				SaveTopLetter() saves size and position of topmost read window
				WritePrivateProfileInt() writes an integer value to an INI file


	COMMENTS:	This file contains all the functions that read, write or
				manipulate the users INI files.
****************************************************************************/
#include "mymail.h"
#include "string.h"
#include "stdlib.h"
#include "io.h"

int GetINIParam(char far *, char far *, char far *);


/******************************************************************************
	FUNCTION:   GetINIFont()

	PURPOSE:  	To read a font Parameter string from the users INI file.
*******************************************************************************/
BOOL GetINIFont(char far *Entry, LOGFONT far *lf)
{
	char  	INIString[512], Face[100];
	int		height, width, weight;
	BYTE	italic;

	/*** Get the Parameter string from the INI file ***/
	NullFill((LPSTR) lf, sizeof(LOGFONT));
	if (GetINIParam("Fonts", Entry, (LPSTR) &INIString) != 0) {
		/*** Convert the string to font characteristics ***/
		strtok((char *) INIString, ",");
		italic = (BYTE)	atoi(strtok(NULL, ","));
		weight = atoi(strtok(NULL, ","));
		height = atoi(strtok(NULL, ","));
		width  = atoi(strtok(NULL, ","));
		lstrcpy((LPSTR) &Face, (LPSTR) &INIString);

		/*** Initialize the LOGFONT structure that was ***/
		/*** passed in, based on font characteristics. ***/
		lf->lfItalic = italic;
		lf->lfWeight = weight;
		lf->lfHeight = height;
		lf->lfWidth  = width;
		lstrcpy(lf->lfFaceName, (LPSTR) &Face);
		return(TRUE);
	}
	/*** No font defined in INI file, so make up some values. ***/
	lf->lfItalic = 0;
	lf->lfWeight = 400;
	lf->lfHeight = -16;
	lf->lfWidth  = 0;
	lf->lfPitchAndFamily = FIXED_PITCH | FF_ROMAN;
	return FALSE;

}


/******************************************************************************
	FUNCTION:   GetINILetter()

	PURPOSE:  	To read the letter Parameter string from the users INI file.
*******************************************************************************/
BOOL GetINILetter(int far *dx, int far *dy, int far *xStart, int far *yStart)
{
	BOOL	retval;
	char  	INIString[512];

	/*** Set some default values in case we can't read them from the INI file. ***/
	retval = TRUE;
	*xStart = CW_USEDEFAULT;
	*yStart = CW_USEDEFAULT;
	*dx = CW_USEDEFAULT;
	*dy = CW_USEDEFAULT;

	/*** Get the Parameter string from the INI file ***/
	if (GetINIParam("Windows", "Letter", (char far *) &INIString) != 0) {
		/*** Set the two dimensional values based on the INI string ***/
		*dx = atoi(strtok((char *) INIString, ","));
		*dy = atoi(strtok(NULL, ","));
		*xStart = atoi(strtok(NULL, ","));
		*yStart = atoi(strtok(NULL, ","));
		retval = FALSE;
	}
	if (*xStart < 0 || (*yStart < 0)) {
		*xStart = CW_USEDEFAULT;
		*yStart = CW_USEDEFAULT;
		*dx = CW_USEDEFAULT;
		*dy = CW_USEDEFAULT;
	}
	return retval;
}


/******************************************************************************
	FUNCTION:   GetINIParam()

	PURPOSE:  	To return a Parameter string from the users INI file.
*******************************************************************************/
int GetINIParam(char far *Section, char far *Entry, char far *OutString)
{
	return(GetPrivateProfileString(Section, Entry, "", (LPSTR) OutString, 512, INI_FILE));
}


/******************************************************************************
	FUNCTION:   InitMain()

	PURPOSE:  	To restore main window from the users INI file.
*******************************************************************************/
InitMain(int nCmdShow)
{

	char  			Params[512];
	int				minx, miny;
	WINDOWPLACEMENT	wp;


	/*** Check to see if INI file is right version ***/
	if (GetINIParam("Version", "HappyMail", (LPSTR)&Params) != 0) {
		/*** Read in the main window paramaters ***/
		if (GetINIParam("Windows", "Main", (LPSTR)&Params) != 0) {
			/*** Initialize the WINOWPLACEMENTS structure ***/
			GetWindowPlacement(hWnd, &wp);
			wp.length = sizeof(wp);
			wp.flags					= atoi(strtok((char *)Params, ","));
			wp.showCmd					= atoi(strtok(NULL, ","));
			minx						= atoi(strtok(NULL, ","));
			miny						= atoi(strtok(NULL, ","));
			wp.ptMaxPosition.x			= atoi(strtok(NULL, ","));
			wp.ptMaxPosition.y			= atoi(strtok(NULL, ","));
			wp.rcNormalPosition.left	= atoi(strtok(NULL, ","));
			wp.rcNormalPosition.top		= atoi(strtok(NULL, ","));
			wp.rcNormalPosition.right	= atoi(strtok(NULL, ","));
			wp.rcNormalPosition.bottom	= atoi(strtok(NULL, ","));
			if (Options.icon) {
				wp.ptMinPosition.x = minx;
				wp.ptMinPosition.y = miny;
			}



			/*** Use the show style (NORMAL, MIN, MAX, etc) from the command line. ***/
			wp.showCmd = nCmdShow;
			SetWindowPlacement(hWnd, &wp);
			return NULL;

		}
	}
	ShowWindow(hWnd, nCmdShow);
	return NULL;
}


/******************************************************************************
	FUNCTION:   InitScreen()

	PURPOSE:  	To restore screen, i.e. folders, letters, etc..,
				from Parameter strings in the users INI file.
*******************************************************************************/
InitScreen()
{

	MailFolder far	*TempFolder;
	GLOBALHANDLE 	hTemp;
	char  			Params[512], FolderName[15], *fnptr;
	HWND			hNewWnd;
	BOOL			ScreenEmpty;
	RECT			Rect;
	WINDOWPLACEMENT	wp;


	/*** First re-create the Open Folders ***/
	ScreenEmpty = TRUE;
	/*** Check to see if INI file is right version ***/
	if (GetINIParam("Version", "HappyMail", (LPSTR)&Params) != 0) {
		/*** Grab the Folders parameter string ***/
		if (GetINIParam("Windows", "Folders", (LPSTR)&Params) != 0) {
			/*** Spin thru the string of folders, opening each one as we go ***/
			fnptr = strtok((char *)Params, ",");
			while (strcmp(fnptr, END_TOKEN) != 0) {

				/*** Check for a missing end token or missing folder name.***/
				if ((fnptr == NULL) || (strlen(fnptr) == 0)) {
					SendMessage(hWnd, WM_COMMAND, IDM_LIST, (LONG) "General");
					return FALSE;
				}

				strcpy((char *)FolderName, fnptr);
				wp.length = sizeof(wp);
				wp.flags					= atoi(strtok(NULL, ","));
				wp.showCmd					= atoi(strtok(NULL, ","));
				wp.ptMinPosition.x			= atoi(strtok(NULL, ","));
				wp.ptMinPosition.y			= atoi(strtok(NULL, ","));
				wp.ptMaxPosition.x			= atoi(strtok(NULL, ","));
				wp.ptMaxPosition.y			= atoi(strtok(NULL, ","));
				wp.rcNormalPosition.left	= atoi(strtok(NULL, ","));
				wp.rcNormalPosition.top		= atoi(strtok(NULL, ","));
				wp.rcNormalPosition.right	= atoi(strtok(NULL, ","));
				fnptr						= strtok(NULL, ",");
				wp.rcNormalPosition.bottom	= atoi(fnptr);

				/*** Try to open the folder ***/
				if (!SendMessage(hWnd, WM_COMMAND, IDM_LIST, (LONG) FolderName))
					break;

				/*** If successful in opening folder, resize it ***/
				hTemp = GetFolder2(FolderName);
				if (hTemp) {
					TempFolder 	= (MailFolder far *) GlobalLock(hTemp);
					hNewWnd 	= TempFolder->hListMail;
					SetWindowPlacement(hNewWnd, &wp);
					GlobalUnlock(hTemp);
					ScreenEmpty = FALSE;
				}

				/*** Skip to next folder in string ***/
				fnptr += strlen(fnptr) + 1;
				fnptr = strtok(fnptr, ",");
			}
		}

	}

	/*** Create the List Box for the General Folder if no other folders created ***/
	if (ScreenEmpty)
		SendMessage(hWnd, WM_COMMAND, IDM_LIST, (LONG) "General");


	return FALSE;

}


/******************************************************************************
	FUNCTION:	InitSysVars()

	PURPOSE:	This funcion initializes the paths and file names of the INI file,
				and the default file save directory.  It uses the paths specified
				in HMCONFIG.INI. This allows the network admins to specify the
				default for these files and paths.  The default directory path
				can be overridden by the users default directory.
*******************************************************************************/
extern char **_argv;
InitSysVars()
{
	char		FileBuff[256], PathBuff[256], string[128], ExePath[256];
	int			i;
	HFILE		hFile=0;
	OFSTRUCT    of;

	/*** Grab the command line string, which contains  ***/
	/*** the path to the HMAIL.EXE file.  This is      ***/
	/*** where the HMCONFIG.INI file should be stored. ***/
	strcpy((char *)ExePath, _argv[0]);
	for(i=lstrlen((LPSTR) &ExePath); (ExePath[i] != '\\') && (ExePath[i] != ':');i--);
	ExePath[i+1] = '\0';
	lstrcpy((LPSTR) &FileBuff, (LPSTR) &ExePath);
	lstrcat((LPSTR) &FileBuff, "HMCONFIG.INI");

	/*** Setup the INI_FILE Variable ***/
	GetPrivateProfileString("Options", "INI File Directory", ExePath, PathBuff, 256, FileBuff);
	if (PathBuff[lstrlen((LPSTR) &PathBuff)-1] != '\\' && PathBuff[lstrlen((LPSTR) &PathBuff)-1] != ':' && PathBuff[0] != '\0')
		lstrcat((LPSTR) &PathBuff, "\\");
	lstrcpy((LPSTR) &string, (LPSTR) &PathBuff);
	lstrcat((LPSTR) &PathBuff, "HAPEMAIL.INI");
	lstrcpy(INI_FILE, (LPSTR) &PathBuff);

	/*** Check INI file for a valid file name and valid access rights. ***/
	if (!fexist(INI_FILE))
		if ((hFile = OpenFile(INI_FILE, &of, OF_CREATE)) != HFILE_ERROR)
			_lclose(hFile);
	if (access(INI_FILE, 0x06)) {
		GetWindowsDirectory(PathBuff, 128);
		lstrcat((LPSTR) &PathBuff, "\\HAPEMAIL.INI");
		lstrcpy(INI_FILE, (LPSTR) &PathBuff);
	}



	/*** Setup the DEF_DIR Variable ***/
	GetPrivateProfileString("Options", "Default Save Directory", "C:\\MAIL", PathBuff, 256, FileBuff);
	if (PathBuff[lstrlen((LPSTR) &PathBuff)-1] != '\\' && PathBuff[lstrlen((LPSTR) &PathBuff)-1] != ':')
		lstrcat((LPSTR) &PathBuff, "\\");
	lstrcpy(Options.defdir, (LPSTR) &PathBuff);

	/*** Get the users default directory if it exists. ***/
	GetPrivateProfileString("Options", "Save Directory", Options.defdir, PathBuff, 256, INI_FILE);
	if (PathBuff[lstrlen((LPSTR) &PathBuff)-1] != '\\' && PathBuff[lstrlen((LPSTR) &PathBuff)-1] != ':')
		lstrcat((LPSTR) &PathBuff, "\\");
	lstrcpy(Options.defdir, (LPSTR) &PathBuff);


	return;
}


/******************************************************************************
	FUNCTION:   SaveScreen()

	PURPOSE:  	To save screen Parameters to the users INI file.
*******************************************************************************/
SaveScreen()
{

	MailFolder far	*TempFolder;
	GLOBALHANDLE	hTempMem, hTemp;
	char  			INIString[512], Folders[512], NumBuff[33], *fnptr;
	POINT			pt[2];
	WINDOWPLACEMENT	wp;

	/** Save Main Window dimensions **/
	wp.length = sizeof(wp);
	GetWindowPlacement(hWnd, &wp);
	pt[0].x = wp.ptMinPosition.x;
	pt[0].y = wp.ptMinPosition.y;
	if (IsIconic(hWnd))
		GetWindowRect(hWnd, (RECT far *)&pt);
	wsprintf((LPSTR) &INIString, "%i,%i,%i,%i,%i,%i,%i,%i,%i,%i", wp.flags,
			wp.showCmd, pt[0].x, pt[0].y, wp.ptMaxPosition.x, wp.ptMaxPosition.y,
			wp.rcNormalPosition.left, wp.rcNormalPosition.top,
			wp.rcNormalPosition.right, wp.rcNormalPosition.bottom);
	WritePrivateProfileString("Windows", "Main", INIString, INI_FILE);


	/** Save Params for all open folders **/
	hTempMem = TopFolder;
	INIString[0] = '\0';
	while(hTempMem != NULL) {
		TempFolder = (MailFolder far *) GlobalLock(hTempMem);
		hTemp = TempFolder->Next;

		/*** Get WindowPlacement structure of open folders ****/
		wp.length = sizeof(wp);
		GetWindowPlacement(TempFolder->hListMail, &wp);
		wsprintf((LPSTR) &Folders, "%s,%i,%i,%i,%i,%i,%i,%i,%i,%i,%i,", TempFolder->Name,
			wp.flags, wp.showCmd, wp.ptMinPosition.x, wp.ptMinPosition.y, wp.ptMaxPosition.x,
			wp.ptMaxPosition.y,	wp.rcNormalPosition.left, wp.rcNormalPosition.top,
			wp.rcNormalPosition.right, wp.rcNormalPosition.bottom);
		lstrcat((LPSTR) &INIString, (LPSTR) &Folders);
		GlobalUnlock(hTempMem);
		hTempMem = hTemp;
	}
	lstrcat((LPSTR) &INIString, END_TOKEN);

	/*** Write out the folders parameter string ***/
	WritePrivateProfileString("Windows", "Folders", (LPSTR) &INIString, (LPSTR) &INI_FILE);


	/** Save Top Letter Dimensions **/
	SaveTopLetter();

	/*** Save Version Info to INI file ***/
	WritePrivateProfileString("Version", "HappyMail", VerString, INI_FILE);

	return NULL;
}



/******************************************************************************
	FUNCTION:   SaveTopLetter()

	PURPOSE:  	To save the Top letter dimensions to the users INI file.
*******************************************************************************/
SaveTopLetter()
{
	char 			INIString[64];
	RECT			LilRect;
	int				dx, dy;
	Letter far		*TempLetter;
	WINDOWPLACEMENT	wp;

	/** Save Top Letter Dimensions **/
	INIString[0] = '\0';
	if (TopLetter) {
		TempLetter = (Letter far *) GlobalLock(TopLetter);
		if (!IsIconic(TempLetter->hChild)) {
			wp.length = sizeof(wp);
			GetWindowPlacement(TempLetter->hChild, &wp);
			GetWindowRect(TempLetter->hChild,  (RECT far *)&LilRect);
			dx	= LilRect.right  - LilRect.left;
			dy  = LilRect.bottom - LilRect.top;
			wsprintf((LPSTR) &INIString, "%i,%i,%i,%i", dx, dy, wp.rcNormalPosition.left, wp.rcNormalPosition.top);
			WritePrivateProfileString("Windows", "Letter", INIString, INI_FILE);
		}
		GlobalUnlock(TopLetter);
	}
	return NULL;
}



/******************************************************************************
	FUNCTION:   WritePrivateProfileInt()

	PURPOSE:  	To save an integer to a private profile file (INI file).
*******************************************************************************/
WritePrivateProfileInt(char far *section, char far *entry, int num, char far *file)
{
	char numbuff[32];

	wsprintf(numbuff, "%i", num);
	WritePrivateProfileString(section, entry, numbuff, file);
	return;
}


