/****************************************************************************
	PROGRAMMER:	David C. Ritchie
                United States Treasury Department
                Financial Management Service
				Network Management Division


	PROGRAM:	Happy Mail! (written in Borland C/C++)


	DATE:		April 30, 1993


	PURPOSE:	A Windows based Banyan Mail client for Vines version 4.11
				and greater.  This program duplicates all the functionality
				of the Banyan DOS based mail client. In addition, it adds
				several new features such as deleting, moving, and printing
				multiple messages.

	FILES:
	(code)      about.c					address.c
				attach.c				bargraf.c
				file.c  				folders.c
				foldrlib.c				font.c
				hmail.c					import.c
				ini_lib.c				launch.c
				lettrlib.c				library.c
				list.c    				message.c
				move.c    				options.c
				print.c   				read.c
				search.c				sort.c
				status.c				superlib.c
				view.c    				winlib.c
				write.c                 writelib.c

	(headers)	defines.h				mymail.h

	(module
	definition)	mail.def

	(resource)	mail.rc

	(libraries)	bwcc.lib				cwm.lib
				vnsapi.lib				vnsmail.lib

****************************************************************************/
/****************************************************************************
	FILE:		hmail.c


	CONTENTS:   WinMain()			initializations and main message loop.
				InitApplication()	initializes system, registers classes,
									and creates main window.
				FrameWndProc()		main window procedure for HappyMail.


	COMMENTS:	This file contains all the files directly associated with
				the main message loop and the main window procedure.  It also
				contains the variable declarations and initializations
				for all global variables.
****************************************************************************/
#include "MyMail.h"
#include "stdlib.h"
#include "stdio.h"
#include "string.h"
#include "mmsystem.h"


#define INCL_STDA
#include "vnsstda.h"
#undef  INCL_STDA

HVNMA      	hVNMAS = (HVNMA)0;   /*** handle to the Banyan Mail system        ***/
HVNSD		hSTDA = (HVNSD)0;	 /*** Global STDA Session handle.             ***/

HANDLE     	hInst;               /*** handle to the current instance          ***/

HWND       	hWnd,                /*** handle to the main HappyMail window     ***/
			hDlgCancel,          /*** handle used for non-modal dialog boxes  ***/
			hDlgSearch,          /*** handle used for non-modal search box    ***/
			hStatus = NULL;      /*** handle for the Status Bar window        ***/


GLOBALHANDLE 	TopLetter,       /*** handle to the top letter in the linked list of Letters ***/
				TopFolder,       /*** handle to the top folder in the linked list of Folders ***/
				hTempArr;        /*** handle to a temporary array of message ID's            ***/

char 		UserName[65],        /*** the users StreetTalk name               ***/
			INI_FILE[256];       /*** the path/filename of the users INI file ***/

short      	HelpID,              /*** flag indicating current help topic      ***/
			VnsRev,				 /*** flag indicating Mail Service Revision.  ***/
			MaxMsgs;			 /*** Maximum messages displayed in a folder. ***/

unsigned long 	MAXBODY,		 /*** Max size for reading/viewing a msg body.***/
				MAXMSGBODY;		 /*** Max size for composing a message body.  ***/

FARPROC		lpfnOldEdit,         /*** old Multi-line Edit window procedure    ***/
			lpfnOldBBProc,       /*** old Borland Button window procedure     ***/
			lpFilterFunc,        /*** old message filter function             ***/
			lpFiltProc;          /*** new message filter function             ***/

HANDLE		hAccel;              /*** current Accelerator handle              ***/

HMENU      	hMenuMail,           /*** Main window menu handle                 ***/
			hMenuWrite,		 	 /*** Write window menu handle				  ***/
			hMenuEdit,			 /*** Edit sub-menu handle in Main menu       ***/
			hMenuWindow;		 /*** Window sub-menu handle in Main menu     ***/

BOOL		bUserAbort,          /*** flag used to cancel a non-modal dialog  ***/
			bSpellMode,			 /*** flag used to dis-allow spell checking   ***/
			bNewListFont = TRUE,
			bNewMail = FALSE,	 /*** flag indicating New Mail has arrived    ***/
			bTimer = FALSE,    	 /*** flag indicating a Timer is installed    ***/
			bSearchRead = FALSE;

HBITMAP		hBMPaperClip;

HICON		hIconMail, hIconNew;

Cap2		UserCaps;

OptionRec	Options;

unsigned int VM_LOGOUT;

extern DeleteMsgs(LONG lParam);
extern MultiSave(long lParam);
extern char **_argv;

/****************************************************************************
	FUNCTION:	WinMain()

	PURPOSE:	To initialize the system and act as the message handler.
				Also checks for correct version of Vines and Windows.
****************************************************************************/
#pragma argsused
int PASCAL WinMain(HANDLE hInstance, HANDLE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
	MSG 			msg;
	unsigned short 	AnyChange;
	GLOBALHANDLE	hFolder;
	MailFolder far	*TempFolder;
	HWND			hWndFrame, hWndClient, hTmpWnd;
	DWORD			winver;
	HDC				hDC;
	RECT			Rect;
	HANDLE			hLibrary;

	/*** First check the Windows version ***/
	winver = GetVersion();
	if ((LOBYTE(LOWORD(winver))<3) || (LOBYTE(LOWORD(winver))>=3 && HIBYTE(LOWORD(winver))<10)) {
		MessageBox(NULL, "HappyMail! requires Windows\n\n      version 3.1 or higher", "ERROR", MB_ICONHAND);
		VnsDone();
		return(FALSE);
	}

	/*** Now check the Vines Rev ***/
	if (getrev() < (unsigned short) 41100) {
		MessageBox(NULL, "HappyMail! requires Banyan\n\nVines version 4.11 or higher", "ERROR", MB_ICONHAND);
		VnsDone();
		return(FALSE);
	}


	/*** Happy Mail! does not allow multiple instances due to problems    ***/
	/*** with coordinating Folder updates.  If a user tries to create a   ***/
	/*** new instance, the previous instance is located and re-displayed. ***/
	if (!hPrevInstance) {
		/*** No other instances of app running. ***/
		/*** So perform system initializations. ***/
		hLibrary = LoadLibrary("WINCTL.DLL");
		if (!InitApplication(hInstance, nCmdShow, &hWndFrame, &hWndClient)) {
			/*** Release Resources before exiting. ***/
			if (hMenuMail) {
				DestroyMenu(hMenuMail);
				DestroyMenu(hMenuWrite);
			}
			UnregisterClass("HMailWClass", hInst);
			UnregisterClass("SuperBorBtn", hInst);
			VnsDone();
			return (FALSE);
		}
	}
	else {
		/*** Application already running.   ***/
		/*** Redisplay current application. ***/
		hTmpWnd = FindWindow("HMailWClass", NULL);
		SetWindowPos(hTmpWnd, HWND_TOP, 0, 0, 0, 0, SWP_SHOWWINDOW | SWP_NOSIZE | SWP_NOMOVE);
		ShowWindow(hTmpWnd, SW_RESTORE);
		VnsDone();
		return(FALSE);
	}


	/* Acquire and dispatch messages until a WM_QUIT message is received. */
	VnsIsAnyMailChanges(hVNMAS, UserName, (unsigned short WFAR *)&AnyChange);
	while (GetMessage(&msg, NULL, NULL, NULL)) {
		if (!hDlgSearch || !IsDialogMessage(hDlgSearch, &msg)) {
			if (!hDlgCancel || !IsDialogMessage(hDlgCancel, &msg)) {
				if (!TranslateMDISysAccel(hWndClient, &msg) &&
					!TranslateAccelerator(hWndFrame, hAccel, &msg)) {
					TranslateMessage(&msg);    /* Translates virtual key codes       */
					DispatchMessage(&msg);     /* Dispatches message to window       */

					/*** If new mail has arrived, reset the General folder listboxes ***/
					AnyChange = FALSE;
					VnsIsAnyMailChanges(hVNMAS, UserName, (unsigned short WFAR *)&AnyChange);
					if (AnyChange) {
						if (IsIconic(hWnd)) {
							bNewMail = TRUE;
							GetClientRect(hWnd, &Rect);
							hDC = GetDC(hWnd);
							DefWindowProc(hWnd, WM_ICONERASEBKGND, (WORD) hDC, 0L);
							DrawIcon(hDC, Rect.left, Rect.top, hIconNew);
							ReleaseDC(hWnd, hDC);
							if (bTimer)
								bTimer = !(KillTimer(hWnd, 1));
						}

						hFolder = GetFolder2("General");
						if (hFolder != NULL) {
							TempFolder = (MailFolder far *) GlobalLock(hFolder);
							/*** First remove any duplicate LBRESET messages. ***/
							while(PeekMessage(&msg, TempFolder->hListMail, WM_LBRESET, WM_LBRESET, PM_REMOVE)!=0);
							PostMessage(TempFolder->hListMail, (int) WM_LBRESET, (int) NULL, (long) NULL);
							GlobalUnlock(hFolder);
						}
					}
				}
			}
		}
	}

	/* Unregister the Main Window Class */
	UnregisterClass("HMailWClass", hInst);
	UnregisterClass("SuperBorBtn", hInst);

	/*** Release the Icon and bitmap Resources. ***/
	DestroyIcon(hIconMail);
	DestroyIcon(hIconNew);
	DeleteObject(hBMPaperClip);

	/*** Kill the timer if its still running. ***/
	if (bTimer)
		KillTimer(hWnd, 1);

	/*** Release the text search resources if its still in use. ***/
	if (hDlgSearch)
		SendMessage(hDlgSearch, WM_COMMAND, IDCANCEL, 0L);


	/*** Disconnect from the mail, STDA, and StreetTalk service. ***/
	if( hVNMAS ) VnsEndMailSession( hVNMAS );
	if ( hSTDA ) VnsEndStdaSession( hSTDA );

	/*** VnsDone release resources which VnsAPI.dll allocates. ***/
	VnsDone();

	/*** Free the WinCTL Library. ***/
	FreeLibrary(hLibrary);

	return (msg.wParam);       /* Returns the value from PostQuitMessage */
}


/****************************************************************************

	FUNCTION:	InitApplication()

	PURPOSE:	Initializes window data and registers window class

	COMMENTS:

****************************************************************************/
BOOL InitApplication(HANDLE hInstance, int nCmdShow, HANDLE far *hFrame, HANDLE far *hClient)
{
	WNDCLASS  		wc;
	BOOL 			registered;
	CallStatus		uStat;
	HDC             hDC;
	int 			i;
	char			buff[MAXASSOC];
	HBITMAP			hBitMap;
	RECT			Rect;

	/*** Set the global variable UserName. ***/
	uStat=VnsGetUserName((LPSTR) &UserName);
	if(uStat) {
		if (uStat == 51)
			BWCCMessageBox(NULL, "   No Network Support Found!\n  Exit Windows and Run Setup.\n                   -or-\nSee your Network Administrator.", "ERROR", MB_ICONHAND);
		else
			DspMsgCode(hWnd, "Unable to access User Name: ", uStat, FALSE);
		return FALSE;
	}

	/*** Initialize system variables from the global INI file. ***/
	InitSysVars();
	InitOptions();

	/*** This call sets environment variables, on the users computer, ***/
	/*** that are necessary for time-related function calls.          ***/
	time(NULL);

	/*** Fill in the window class structure with parameters    ***/
	/*** that describe the Main window and register the class. ***/
	wc.style         = CS_HREDRAW | CS_VREDRAW | CS_DBLCLKS;
	wc.lpfnWndProc   = (WNDPROC) FrameWndProc;
	wc.cbClsExtra    = 0;
	wc.cbWndExtra    = 0;
	wc.hInstance     = hInstance;
	wc.hIcon         = NULL;
	wc.hCursor       = LoadCursor(hInstance, "mail");
	wc.hbrBackground = GetStockObject(WHITE_BRUSH);
	wc.lpszMenuName  = "MDIMailMenu";
	wc.lpszClassName = "HMailWClass";
	registered = RegisterClass(&wc);

	/*** Now describe the Read window and register its class. ***/
	wc.lpfnWndProc   = (WNDPROC) ReadMail;
	wc.hIcon         = LoadIcon(hInstance, "zGlasses");
	wc.lpszMenuName  = NULL;
	wc.lpszClassName = "ReadWClass";
	registered |= RegisterClass(&wc);

	/*** Now describe the Folder window and register its class. ***/
	wc.lpfnWndProc   = (WNDPROC) ListMail;
	wc.hIcon         = LoadIcon(hInstance, "zListIcon");
	wc.lpszClassName = "ListWClass";
	registered |= RegisterClass(&wc);


	/*** Now describe the Status Bar window and register its class. ***/
	wc.lpfnWndProc   = (WNDPROC) StatusBarProc;
	wc.hIcon         = NULL;
	wc.hCursor       = LoadCursor(NULL, IDC_ARROW);
	wc.lpszMenuName  = NULL;
	wc.lpszClassName = "StatusBar";
	registered |= RegisterClass(&wc);

	/*** If any of the registers failed, return with an error flag ***/
	if (!registered) return (registered);


	/*** Save the instance handle in a static variable. ***/
	hInst = hInstance;


	/*** Load the menu and get handles to the sub-menus ***/
	hMenuMail  			= LoadMenu(hInst, "MDIMailMenu");
	hMenuWrite 			= LoadMenu(hInst, "MDIWriteMenu");
	hMenuEdit	  		= GetSubMenu(hMenuMail, EDIT_MAIL_POS);
	hMenuWindow	  		= GetSubMenu(hMenuMail, WIN_MAIL_POS);


	/*** Create the main window for this application. ***/
	if ((hWnd = CreateWindow("HMailWClass", "Happy Mail", WS_OVERLAPPEDWINDOW |
		MDIS_ALLCHILDSTYLES | WS_CLIPCHILDREN, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT,
		CW_USEDEFAULT, NULL, NULL, hInstance, NULL)) == NULL)
			return (FALSE);

	*hFrame = hWnd;
	*hClient = GetWindow(hWnd, GW_CHILD);


	/*** Initialize the main window's size and position ***/
	InitMain(nCmdShow);

	/**** Create the new superclass for Borland style buttons. ***/
	BB_InitSuperClass(hInst, hWnd);


	/*** Paint the Logo on the screen if no command line switch /q or /Q ***/
	SendMessage(*hClient, WM_SETFONT, GetStockObject(SYSTEM_FIXED_FONT), NULL);
	strcpy((char *) buff, _argv[1]);
	if (!(buff[0] == '/' && ((buff[1] == 'q') || (buff[1]== 'Q')))) {
		hDC = GetDC(*hClient);
		hBitMap = LoadBitmap(hInst, (LPSTR) "ScrnLogo");
		if (hBitMap != NULL)
			DrawBitMap(hDC, hBitMap, -1, -1, TRUE, TRUE);
		else
			DspMsgCode(hWnd, "Missing Bitmap", 0, FALSE);
		ReleaseDC(*hClient, hDC);
		DeleteObject(hBitMap);
	}


	/*** Initialize the Global Letter Chain and Folder Chain handles. ***/
	TopLetter = NULL;
	TopFolder = NULL;


	/*** Attempt to Start the STDA, and Mail Sessions. ***/
	if (Options.STDA)
		hSTDA = StartSTDA();
	else
		hSTDA = 0;
	if (!StartMail(&hVNMAS))
		return FALSE;


	/*** Retrieve the user capabilities for examining wildcard sends. ***/
	UserCaps.sendtplate = GRP_ORG_RESTRICT;
	uStat = TRUE;
	i = 0;
	while (uStat && i++ < 5)
		uStat = VnsGetUserMailInfo2(hVNMAS, (LPSTR) &UserName, &UserCaps);


	/*** Open folders and adjust their size and posiotion. ***/
	InitScreen();


	/*** Install message hook for function key support ***/
	/*** in dialog boxes (compose and forward).        ***/
	lpFiltProc = MakeProcInstance(FilterFunc, hInst);
	if (lpFiltProc)
		lpFilterFunc = SetWindowsHook(WH_MSGFILTER, lpFiltProc);
	else
		lpFilterFunc = NULL;


	/*** Load Accelerator ***/
	hAccel = LoadAccelerators(hInst, "mail");

	/*** Clear the HappyMail Logo from screen. ***/
	GetClientRect(*hClient, &Rect);
	InvalidateRect(*hClient, &Rect, TRUE);


	/*** Load the Icons and bitmaps ***/
	hIconMail = LoadIcon(hInstance, "mail");
	hIconNew = LoadIcon(hInstance, "pending");
	hBMPaperClip = LoadBitmap(hInstance, "PaperClip");

	/*** Register the Login / Logout Messages for Vines. ***/
	VM_LOGOUT = RegisterWindowMessage("VM_LOGINLOGOUT");


	/*** Check for new mail and set the icon flag. ***/
	i = 0;
	uStat = VnsCountNewMailMsgs(hVNMAS, UserName, (unsigned short far *) &i);
	if (uStat==0 && i !=0)
		bNewMail=TRUE;



	return (TRUE);               /* Returns the value from PostQuitMessage */
}



/****************************************************************************
	FUNCTION:	FrameWndProc()

	PURPOSE:	Window procedure for the main HappyMail window.

	COMMENTS:	The HappyMail window is an MDI frame window and uses the
				DefFameProc() as the default message handler. For more
				information on the MDI interface see Charles Petzold's
				Programming Windows.

				In an MDI interface, menu messages and accelerator key
				messages are passed to the child windows** directly.
				In order to centralize the processing, I have implemented
				code that passes menu, accelerator key, and button messages
				back to the main window procedure.  Thus, most of the code
				for calling the various functions of HappyMail is contained
				in this procedure.

				The Compose window, being a modal dialog box, cannot pass
				messages back to the main window procedure and still remain
				active.  Therefore, much of the code found here, for calling
				HappyMail functions, is duplicated in the window procedure
				for the Compose dialog box.

				** Happymail's Child windows are the Read and Folder windows.

****************************************************************************/
long FAR PASCAL _export FrameWndProc(HWND hWnd, unsigned message, WORD wParam, LONG lParam)
{
	GLOBALHANDLE		hTempArr, hArrMem, hTempMem, hTemp;
	Env         		env;
	HWND 				hLetter, hFolder, hWndChild;
	char 				MessID[14], FoldName[32], far *MsgID, buff[100];
	MailFolder far		*TempFolder;
	Letter far			*TempLetter;
	int 				i, index, count, NumItems, far *IndexArray, dx, dy,
						xStart, yStart, border;
	long				inc;
	WORD				CtrlID;
	CallStatus 			uStat = 0;
	CLIENTCREATESTRUCT	clientcreate;
	MDICREATESTRUCT		mdicreate;
	static HWND			hWndClient;
	WNDCLASS			wc;
	FARPROC				lpProc, lpfnBarGraf;
	BOOL				Zoomed;
	RECT				Rect;
	WINDOWPLACEMENT		wp;


	/*** Check for the Windows/Vines Logout Message. ***/
	if (message == VM_LOGOUT && (wParam == 0x0000)) {
		DestroyWindow(hWnd);
		return 0;
	}

	switch (message) {

		case WM_CREATE:
			/*** Change to HourGlass Cursor before starting ***/
			SetCursor(LoadCursor(NULL, IDC_WAIT));

			/*** Create an MDI client window ***/
			clientcreate.hWindowMenu  = hMenuWindow;
			clientcreate.idFirstChild = 9000;
			hWndClient = CreateWindow("MDICLIENT", NULL, WS_CHILD |
						 WS_CLIPCHILDREN | WS_VISIBLE, 0, 0, 0, 0,
						 hWnd, 1, hInst, (LPSTR) &clientcreate);

			/*** Create the status bar for the main Window ***/
			hStatus = CreateWindow("StatusBar", "", WS_CLIPCHILDREN | WS_CHILD |
						 WS_VISIBLE, 0, 0, 200, 200,
						 hWnd, 1, hInst, NULL);

			SendMessage(hWndClient, WM_MDISETMENU, 0, MAKELONG(hMenuMail, hMenuWindow));

			HelpID = 1;
			return 0;

		case WM_F1:
			/**** Provide Context-sensitive help for menus and dialog boxes ***/
			if (wParam == MSGF_DIALOGBOX) {
				CtrlID = GetWindowWord(GetFocus(), GWW_ID);
				if (CtrlID == IDM_OK || CtrlID == IDCANCEL || CtrlID == IDOK)
					WinHelp(hWnd, "HMHELP.HLP", HELP_CONTENTS, 0L);
				else
					WinHelp(hWnd, "HMHELP.HLP", HELP_CONTEXT, (long) CtrlID);

			}

			if (wParam == MSGF_MENU) {
				WinHelp(hWnd, "HMHELP.HLP", HELP_CONTEXT, (long) HelpID);
			}

			return 0;

		case WM_PAINT:
			{
				PAINTSTRUCT ps;
				if (IsIconic(hWnd)) {
					BeginPaint(hWnd, &ps);
					DefFrameProc(hWnd, hWndClient, WM_ICONERASEBKGND, ps.hdc, 0L);
					if (bNewMail)
						DrawIcon(ps.hdc, 0, 0, hIconNew);
					else
						DrawIcon(ps.hdc, 0, 0, hIconMail);
					EndPaint(hWnd, &ps);
					return TRUE;
				}
			}
			return (DefFrameProc(hWnd, hWndClient, message, wParam, lParam));

		case WM_ERASEBKGND:
			if (IsIconic(hWnd))
				return TRUE;
			return (DefFrameProc(hWnd, hWndClient, message, wParam, lParam));


		case WM_QUERYDRAGICON:
			if (bNewMail)
				return ((LONG) (WORD) hIconNew);
			else
				return ((LONG) (WORD) hIconMail);

		case WM_SIZE:
			bNewMail = FALSE;
			if (wParam == SIZE_MINIMIZED) {
				if (!bTimer)
					SetTimer(hWnd, 1, 15000, NULL);
			}
			else {
				if (bTimer)
					KillTimer(hWnd, 1);
			}
			return (DefFrameProc(hWnd, hWndClient, message, wParam, lParam));

		case WM_QUERYENDSESSION:
			/*** Save folder and main window sizes and positions. ***/
			SaveScreen();
			SaveOptions();

			/*** Delete all HappyMail created Temporary files. ***/
			DeleteTempFiles();
			return (DefFrameProc(hWnd, hWndClient, message, wParam, lParam));


		case WM_COMMAND:
			/*** If command is from the system menu translate ***/
			/*** into a WM_SYSCOMMAND message. Currently,     ***/
			/*** only the 'switch to' command is enabled.     ***/
			if((wParam & 0xFFF0) == SC_TASKLIST)
				return (DefFrameProc(hWnd, hWndClient, WM_SYSCOMMAND, wParam, lParam));

			/*** If command is from an accelerator-key, delete       ***/
			/*** lParam.  This is necessary since lParam is used to  ***/
			/*** pass the handle of a Read window to some functions. ***/
			if (HIWORD(lParam) == 1) lParam = 0L;

			/*** If Menu is activated while a Read window has ***/
			/*** focus, change lParam to current Letter ptr   ***/
			hWndChild = LOWORD(SendMessage(hWndClient, WM_MDIGETACTIVE, 0, 0L));
			if (IsWindow(hWndChild)) {
				GetClassName(hWndChild, (LPSTR) &buff, 100);
				if (lstrcmp((LPSTR) &buff, "ReadWClass") == 0)
					lParam = GetLetter(hWndChild);
			}

			/*** If user double-clicked in the Folder's   ***/
			/*** list box initiate a Read message command.***/
			if ((lParam >> 16) == LBN_DBLCLK) {
				wParam = IDM_READ;
				return (DefFrameProc(hWnd, hWndClient, message, wParam, lParam));
			}


			switch(wParam){
				case IDM_ABOUT:
					/*** User selected About from the Help menu. ***/
					CallDlg(hWnd, About, AboutBox, NULL);
					return 0;

				case HELP_HELP:
					/*** User selected Help on Help from the Help menu. ***/
					if(WinHelp(hWnd, NULL, HELP_HELPONHELP, NULL) == 0)
						MessageBox(hWnd, "Error Starting Help", "HappyMail", MB_ICONSTOP);
					return 0;

				case HELP_INDEXES:
				case HELP_MENUS:
				case HELP_GLOSSARY:
				case HELP_PROCEDURES:
				case HELP_FIELDS:
				case HELP_ERRORS:
					/*** User selected one of the above help topics from  ***/
					/*** the Help menu. All these topics are contained in ***/
					/*** the HMHELP.HLP file.  This file should be placed ***/
					/*** in the same directory as the HappyMail exe file. ***/
					if (WinHelp(hWnd, "HMHELP.HLP", HELP_CONTEXT, (long) wParam) == 0)
						MessageBox(hWnd, "Error Starting Help", "HappyMail", MB_ICONSTOP);
					return 0;

				case IDM_LOGO:
					/*** User selected Logo from the Help menu. ***/
					CallDlg(hWnd, Logo, SplashBox, NULL);
					return 0;

				case IDM_LIST:
					/*** User is opening a folder. ***/
					/*** Update the status Bar.    ***/
					if (hStatus)
						SendMessage(hStatus, STB_STRING, 0, (LONG) "Opening Folder ...");

					/*** Change to HourGlass Cursor before starting ***/
					SetCursor(LoadCursor(NULL, IDC_WAIT));

					/*** Store the zoomed status of the Top MDI child window, ***/
					/*** so we can copy that status to the new MDI child.  ***/
					Zoomed = FALSE;
					xStart = CW_USEDEFAULT;
					yStart = CW_USEDEFAULT;
					dx = 629;
					dy = (GetFontHeight(hWnd)*12) + 32;
					if (TopFolder != NULL) {
						hWndChild = LOWORD(SendMessage(hWndClient, WM_MDIGETACTIVE, 0, 0L));
						if (IsZoomed(hWndChild))
							Zoomed = TRUE;
					}

					if (TopFolder != NULL) {
						hTempMem = TopFolder;
						while(hTempMem != NULL) {
							TempFolder = (MailFolder far *) GlobalLock(hTempMem);
							hTemp = TempFolder->Next;
							if (hTemp == NULL) {
								if (!IsIconic(TempFolder->hListMail)) {
									border = GetSystemMetrics(SM_CYBORDER) + GetSystemMetrics(SM_CYCAPTION);
									GetWindowRect(TempFolder->hListMail, &Rect);
									ScreenToClient(hWnd, (LPPOINT) &Rect);
									xStart = Rect.left + border;
									yStart = Rect.top + border;
								}
								GlobalUnlock(hTempMem);
								break;
							}
							GlobalUnlock(hTempMem);
							hTempMem = hTemp;
						}
					}

					/*** Create an MDI child window i.e. a Folder window. ***/
					mdicreate.szClass = "ListWClass";
					mdicreate.szTitle = (LPSTR) lParam;
					mdicreate.hOwner  = hInst;
					mdicreate.x       = xStart;
					mdicreate.y       = yStart;
					mdicreate.cx      = dx;
					mdicreate.cy      = dy;
					mdicreate.style   = DS_SETFONT;
					/*** If the top MDI window is maximized, you MUST use ***/
					/*** the default position to keep from encountering   ***/
					/*** the double controls on the menu bar.			  ***/
					if (Zoomed) {
						mdicreate.x = CW_USEDEFAULT;
						mdicreate.y = CW_USEDEFAULT;
						mdicreate.cx = CW_USEDEFAULT;
						mdicreate.cy = CW_USEDEFAULT;
						mdicreate.style |= WS_MAXIMIZE;
					}
					mdicreate.lParam  = NULL;
					hWndChild = SendMessage(hWndClient, WM_MDICREATE, 0, (LONG) &mdicreate);


					/*** If window could not be created, display error message. ***/
					if (!hWndChild) {
						DspMsgCode( hWnd, "Error creating Folder window", 0, FALSE);
					}
					else if (Zoomed) {
						GetWindowPlacement(hWndChild, &wp);
						wp.rcNormalPosition.left	= xStart;
						wp.rcNormalPosition.top		= yStart;
						wp.rcNormalPosition.right	= xStart + dx;
						wp.rcNormalPosition.bottom	= yStart + dy;
						SetWindowPlacement(hWndChild, &wp);
					}


					/*** Blank out the Status bar. ***/
					if (hStatus)
						SendMessage(hStatus, STB_STRING, 0, 0L);

					return hWndChild;


				case IDM_READ:
					/*** User is opening a Read window. ***/
					/*** Update the status Bar message. ***/
					if (hStatus)
						SendMessage(hStatus, STB_STRING, 0, (LONG) "Opening Mail Message ...");

					/*** Change to HourGlass Cursor before starting ***/
					SetCursor(LoadCursor(NULL, IDC_WAIT));


					/*** If the status of the first message is UNSENT,   ***/
					/*** open an Edit(Compose) window instead of a read. ***/
					NumItems = GetLetterInfo((LPSTR) &MessID,  (LPSTR) &FoldName, lParam, NULL);
					if (NumItems==1) {
						uStat=VnsGetMailEnvelope(hVNMAS, (LPSTR) &UserName, (LPSTR) &FoldName, (LPSTR) &MessID, (Env far *) &env);
						if (Recover(uStat))
							uStat=VnsGetMailEnvelope(hVNMAS, (LPSTR) &UserName, (LPSTR) &FoldName, (LPSTR) &MessID, (Env far *) &env);
						if (uStat){
							DspMsgCode( hWnd, "Error Reading Mail: ", uStat, FALSE );
							/*** Blank out the Status bar. ***/
							if (hStatus) SendMessage(hStatus, STB_STRING, 0, 0L);
							return 0;
						}
						VnsReleaseMailEnvelope(hVNMAS, (LPSTR) &MessID);
						if (env.status == MA_UNSENT) {
							/*** Blank out the Status bar. ***/
							if (hStatus) SendMessage(hStatus, STB_STRING, 0, 0L);
							PostMessage(hWnd, WM_COMMAND, IDM_EDIT, NULL);
							return 0;
						}
					}

					/*** No Messages in Folder or None Selected ***/
					if (NumItems <= 0) {
						BWCCMessageBox(hWnd, "No Message Selected.", "WARNING!", MB_ICONSTOP | MB_OK);
						return 0;
					}

					/*** Store the zoomed status of the Top MDI child window, ***/
					/*** so we can copy that status to the new MDI child.  ***/
					hWndChild = LOWORD(SendMessage(hWndClient, WM_MDIGETACTIVE, 0, 0L));
					Zoomed = FALSE;
					if (IsZoomed(hWndChild))
						Zoomed = TRUE;

					/*** Determine the size and position of the read window. ***/
					GetINILetter((LPINT)&dx, (LPINT)&dy, (LPINT)&xStart, (LPINT)&yStart);
					if (TopLetter != NULL) {
						hTempMem = TopLetter;
						while(hTempMem != NULL) {
							TempLetter = (Letter far *) GlobalLock(hTempMem);
							hTemp = TempLetter->Next;
							if (hTemp == NULL) {
								if (!IsIconic(TempLetter->hChild)) {
									border = GetSystemMetrics(SM_CYBORDER) + GetSystemMetrics(SM_CYCAPTION);
									GetWindowRect(TempLetter->hChild, &Rect);
									ScreenToClient(hWnd, (LPPOINT) &Rect);
									xStart = Rect.left + border;
									yStart = Rect.top + border;
								}
								GlobalUnlock(hTempMem);
								break;
							}
							GlobalUnlock(hTempMem);
							hTempMem = hTemp;
						}
					}

					/*** Create an MDI child window i.e. a Read window. ***/
					mdicreate.szClass = "ReadWClass";
					mdicreate.szTitle = "Reading Mail";
					mdicreate.hOwner  = hInst;
					mdicreate.x       = xStart;
					mdicreate.y       = yStart;
					mdicreate.cx      = dx;
					mdicreate.cy      = dy;
					mdicreate.style = 0;
					/*** If the top MDI window is maximized, you MUST use ***/
					/*** the default position to keep from encountering   ***/
					/*** the double controls on the menu bar.			  ***/
					if (Zoomed) {
						mdicreate.x = CW_USEDEFAULT;
						mdicreate.y = CW_USEDEFAULT;
						mdicreate.cx = CW_USEDEFAULT;
						mdicreate.cy = CW_USEDEFAULT;
						mdicreate.style |= WS_MAXIMIZE;
					}
					mdicreate.lParam  = NULL;
					hWndChild = SendMessage(hWndClient, WM_MDICREATE, 0, (LONG) &mdicreate);

					/*** If window could not be created, display error message. ***/
					if (!hWndChild) {
						DspMsgCode( hWnd, "Error creating Read window", 0, FALSE);
					}
					else if (Zoomed) {
						GetWindowPlacement(hWndChild, &wp);
						wp.rcNormalPosition.left	= xStart;
						wp.rcNormalPosition.top		= yStart;
						wp.rcNormalPosition.right	= xStart + dx;
						wp.rcNormalPosition.bottom	= yStart + dy;
						SetWindowPlacement(hWndChild, &wp);
					}

					/*** Blank out the Status bar. ***/
					if (hStatus)
						SendMessage(hStatus, STB_STRING, 0, 0L);
					return hWndChild;


				case IDM_SEARCH:
					/*** User selected Text Search from the Utility menu. ***/
					/*** Update the status Bar.    ***/
					if (hStatus)
						SendMessage(hStatus, STB_STRING, 0, (LONG) "Initializing Search Engine -- please wait...");

					lpProc = MakeProcInstance(SearchProc, hInst);
					hDlgCancel = CreateDialog(hInst, MAKEINTRESOURCE(SearchBox), hWnd, lpProc);

					/*** Blank out the Status bar. ***/
					if (hStatus)
						SendMessage(hStatus, STB_STRING, 0, 0L);
					return 0;


				case IDM_ANSWER:
					switch (Options.answermode) {
						case 2: wParam = IDM_ANSALL; break;
						case 3: wParam = IDM_QUOTE; break;
						case 4: wParam = IDM_QUOTEALL; break;
					}
				case IDM_ANSALL:
				case IDM_QUOTE:
				case IDM_QUOTEALL:
				case IDM_EDIT:
				case IDM_FORWARD:
					/*** User is attempting to answer or edit a message. ***/
					/*** Change to HourGlass Cursor before starting ***/
					SetCursor(LoadCursor(NULL, IDC_WAIT));

					/*** Check to make sure user has not selected multiple messages. ***/
					NumItems = GetLetterInfo((LPSTR) &MessID,  (LPSTR) &FoldName, lParam, NULL);
					if (NumItems > 1) {
						if (wParam == IDM_EDIT)
							BWCCMessageBox(hWnd, "Cannot Edit Multiple Messages.", "WARNING!", MB_ICONSTOP | MB_OK);
						else if (wParam == IDM_FORWARD)
							BWCCMessageBox(hWnd, "Cannot Forward Multiple Messages.", "WARNING!", MB_ICONSTOP | MB_OK);
						else
							BWCCMessageBox(hWnd, "Cannot Reply to Multiple Messages.", "WARNING!", MB_ICONSTOP | MB_OK);
						return 0;
					}

					/*** No Messages in Folder or None Selected ***/
					if (NumItems <= 0) {
						BWCCMessageBox(hWnd, "No Message Selected.", "WARNING!", MB_ICONSTOP | MB_OK);
						return 0;
					}


					/*** If user is attempting to edit, make sure message status is UNSENT. ***/
					if (wParam == IDM_EDIT) {
						uStat=VnsGetMailEnvelope(hVNMAS, UserName, FoldName, MessID, &env);
						if (Recover(uStat))
							uStat=VnsGetMailEnvelope(hVNMAS, UserName, FoldName, MessID, &env);
						if (uStat){
							DspMsgCode( hWnd, "Error Editing Mail: ", uStat, FALSE );
							return 0;
						}
						VnsReleaseMailEnvelope(hVNMAS, MessID);
						if (env.status != MA_UNSENT) {
							BWCCMessageBox(hWnd, "Only UNSENT Messages May Be Edited.", "WARNING!", MB_ICONSTOP | MB_OK);
							return 0;
						}
					}

					/*** No break on purpose. Answer, edit and compose all   ***/
					/*** use the same dialog box and window procedure.       ***/
					/*** Parameters are passed to the dialog box to indicate ***/
					/*** which action to take and which message is selected. ***/

				case IDM_COMPOSE:

					/*** User is Composing a new mail message. ***/
					CallDlg(hWnd, Compose, ComposeBox, MAKELONG(wParam, (int) lParam));
					return 0;


				case IDM_DELETE:
					/*** User is deleting message(s). ***/
					/*** Change to HourGlass Cursor before starting. ***/
					SetCursor(LoadCursor(NULL, IDC_WAIT));

					/*** First Find Message ID and Folder name. ****/
					NumItems = GetLetterInfo((LPSTR) &MessID,  (LPSTR) &FoldName, lParam, &hTempArr);

					/*** No Messages in Folder or None Selected. ***/
					if (NumItems <= 0) {
						BWCCMessageBox(hWnd, "No Message(s) Selected.", "WARNING!", MB_ICONSTOP | MB_OK);
						return 0;
					}

					/*** Verify a multiple delete unless user is in expert mode ***/
					if (!Options.expertmode && NumItems > 1)
						if (CallDlg(hWnd, AskChange, YesNoBox, 4L)==FALSE) {
							SafeFree(GLOBAL, hTempArr);
							return 0;
						}

					/*** Get a pointer to the array of selected message IDs. ***/
					MsgID = (char far *) GlobalLock(hTempArr);

					/*** Update the status Bar. ***/
					if (hStatus)
						SendMessage(hStatus, STB_STRING, 0, (LONG) "Deleting Mail Message(s) ...");

					/*** Create and initialize the Bar Graph ***/
					/*** dialog box for displaying progress. ***/
					bUserAbort = FALSE;
					if (NumItems > 4) {
						lpfnBarGraf = MakeProcInstance(BarGraf, hInst);
						hDlgCancel = CreateDialog(hInst, MAKEINTRESOURCE(BarGrafBox), hWnd, lpfnBarGraf);
						wsprintf(buff, "Deleting %u Messages", NumItems);
						SendMessage(hDlgCancel, WM_SETTEXT, NULL, (LONG) buff);
						wsprintf(buff, "From: %s Folder", (char far *) FoldName);
						SendDlgItemMessage(hDlgCancel, BG_TITLE, WM_SETTEXT, NULL, (LONG) buff);
					}

					if (hDlgCancel)	DeClogDlgMsg(hDlgCancel);

					/*** Now move the letters to the Banyan Wastebasket    ***/
					/*** folder.  As each letter is being deleted check    ***/
					/*** for UNREAD message(s) and verify before deleting. ***/
					for (i=0; i<NumItems && !bUserAbort; i++) {

						if (!Options.expertmode) {
							/*** Grab the message envelope (to check its status) ***/
							uStat=VnsGetMailEnvelope(hVNMAS, UserName, FoldName, &(MsgID[i*14]), &env);
							if (Recover(uStat))
								uStat=VnsGetMailEnvelope(hVNMAS, UserName, FoldName, &(MsgID[i*14]), &env);
							if (uStat){
								/*** Couldn't get the envelope so display an error ***/
								/*** and remove the ID from the message ID array.  ***/
								DspMsgCode( hWnd, "Error Deleting Mail: ", uStat, FALSE );
								lstrncpy(&(MsgID[i*14]), &(MsgID[(i+1)*14]), 14*(NumItems-(i+1)));
								NumItems--;
								i--;
								continue;
							}
							VnsReleaseMailEnvelope(hVNMAS, &(MsgID[i*14]));

							/*** If message is unread, verify the delete. ***/
							if (env.status == MA_UNREAD)
								if (CallDlg(hWnd, AskChange, YesNoBox, 11L)==FALSE) {
									/*** Delete characters from MsgID array and reduce numitems by one ***/
									lstrncpy(&(MsgID[i*14]), &(MsgID[(i+1)*14]), 14*(NumItems-(i+1)));
									NumItems--;
									i--;
									continue;
								}
						}

						/*** Delete the mail message. ***/
						VnsDeleteMailMsg(hVNMAS, UserName, FoldName, &(MsgID[i*14]));

						/*** Send a percentage to the BarGraf dialog ***/
						if (hDlgCancel) {
							inc = ((long)i * 100L)/(long)NumItems;
							SendMessage(hDlgCancel, WM_NEXTSTEP, (WPARAM) (unsigned int) inc, 0L);
							DeClogDlgMsg(hDlgCancel);
						}

					}
					if (hDlgCancel)
						SendMessage(hDlgCancel, WM_NEXTSTEP, 100, 0L);

					/*** Now fix up the list boxes to show the      ***/
					/*** deleted messages in their correct folders. ***/
					ResetLists((LPSTR) &FoldName, "Wastebasket", i, MsgID);

					/*** Deallocate the message ID array. ***/
					SafeFree(GLOBAL, hTempArr);

					/*** Free the memory reserved for the dialog box procedure ***/
					if (NumItems > 4) {
						if (!bUserAbort)
							/*** Close Bar graph dialog box ***/
							DestroyWindow(hDlgCancel);
						else
							/*** User pressed cancel so issue warning. ***/
							BWCCMessageBox(hWnd, "Message Delete Aborted!", "HappyMail", MB_OK | MB_ICONEXCLAMATION);

						hDlgCancel = NULL;
						FreeProcInstance(lpfnBarGraf);
					}


					/*** Update Status Bar ***/
					if (hStatus)
						SendMessage(hStatus, STB_STRING, 0, 0L);

					return 0;

				case IDM_SEND:
					/*** User is sending a message. ***/
					/*** Change to HourGlass Cursor before starting ***/
					SetCursor(LoadCursor(NULL, IDC_WAIT));

					/*** First find message ID and folder name. ****/
					NumItems = GetLetterInfo((LPSTR) &MessID,  (LPSTR) &FoldName, lParam, NULL);

					/*** Display error if multiple messages are selected. ***/
					if (NumItems > 1) {
						BWCCMessageBox(hWnd, "Cannot Send \n\nMultiple Messages.", "WARNING!", MB_ICONSTOP | MB_OK);
						return 0;
					}

					/*** No Messages in Folder or None Selected. ***/
					if (NumItems <= 0) {
						BWCCMessageBox(hWnd, "No Message Selected.", "WARNING!", MB_ICONSTOP | MB_OK);
						return 0;
					}

					/*** Grab the envelope in order to check the status. ***/
					uStat=VnsGetMailEnvelope(hVNMAS, UserName, FoldName, MessID, &env);
					if (Recover(uStat))
						uStat=VnsGetMailEnvelope(hVNMAS, UserName, FoldName, MessID, &env);
					if (uStat){
						DspMsgCode( hWnd, "Error Sending Mail: ", uStat, FALSE );
						return 0;
					}
					/*** Release the envelope memory. ***/
	  				VnsReleaseMailEnvelope(hVNMAS, MessID);

					if (env.status == MA_UNSENT) {
						/*** Update the status Bar. ***/
						if (hStatus)
							SendMessage(hStatus, STB_STRING, 0, (LONG) "Sending Mail Message ...");

						/*** Attempt to send the message ***/
						if (VnsSendMailMsg(hVNMAS, UserName, FoldName, MessID) != 0) {
							/*** Send failed ***/
							DspMsgCode( hWnd, "Error Sending Mail", 0, FALSE);
						}
						else {

							/*** Change UNSENT to SENT in Open ListBox ***/
							if (lParam) {
								hTempMem = (GLOBALHANDLE) lParam;
								TempLetter = (Letter far *) GlobalLock(hTempMem);
								i = FindID(TempLetter->hFolder, (LPSTR) &MessID);
								GlobalUnlock(hTempMem);
							}

							hTempMem = GetFolder2(FoldName);
							TempFolder = (MailFolder far *) GlobalLock(hTempMem);

							if (!lParam)
								i = FindID(hTempMem, (LPSTR) &MessID);

							/*** Change the Status from UNSENT to SENT. ***/
							SendMessage(TempFolder->hListBox, LB_DELETESTRING, i, 0L);
							delnchr(TempFolder->Item[i].string, 0, 2);
							SendMessage(TempFolder->hListBox, LB_INSERTSTRING, i, (LONG) index);
							GlobalUnlock(hTempMem);

							/*** Resort the List box, if necessary. ***/
							if (Options.sortorder == BYSTATUS) SortLists(FoldName);

						}

						/*** Update the status Bar ***/
						if (hStatus)
							SendMessage(hStatus, STB_STRING, 0, NULL);

					}
					else
						BWCCMessageBox(hWnd, "Only UNSENT messages\n\n     may be Sent.", "WARNING!", MB_ICONSTOP | MB_OK);

					return 0;


				case IDM_MANAGE:
					/*** User is Managing folders. ***/
					CallDlg(hWnd, Manage, FolderBox, NULL);
					return 0;


				case IDM_FILE:
					/*** User is trying to file messages and/or attachments. ***/

					/*** First, see if one or more messages have been selected. ***/
					NumItems = GetLetterInfo((LPSTR) &MessID,  (LPSTR) &FoldName, lParam, NULL);
					if (NumItems > 0) {
						/*** Begin a multiple message save ***/
						MultiSave(lParam);
					}
					else {
					/*** No Messages in Folder or None Selected ***/
						BWCCMessageBox(hWnd, "No Message Selected.", "WARNING!", MB_ICONSTOP | MB_OK);
					}
					return 0;

				case IDM_MOVE:
					/*** User is moving message(s). ***/
					CallDlg(hWnd, MoveMail, MoveBox, lParam);
					return 0;

				case IDM_PRINT:
					/*** User is printing message(s). ***/

					/*** Change to HourGlass Cursor before starting ***/
					SetCursor(LoadCursor(NULL, IDC_WAIT));

					PrintMessage(lParam);
					return 0;

				case IDM_TILE:
					/*** User selected Tile from the Windows menu. ***/
					SendMessage(hWndClient, WM_MDITILE, MDITILE_HORIZONTAL, 0L);
					return 0;

				case IDM_CASCADE:
					/*** User selected Cascade from the Windows menu. ***/
					SendMessage(hWndClient, WM_MDICASCADE, 0, 0L);
					return 0;

				case IDM_ARRANGE:
					/*** User selected Arrange Icons from the Windows menu. ***/
					SendMessage(hWndClient, WM_MDIICONARRANGE, 0, 0L);
					return 0;

				case IDM_ATTACHMENTS:
					/*** User is managing attachments. ***/

					/*** Check for mulitple selected messages. ***/
					NumItems = GetLetterInfo((LPSTR) &MessID,  (LPSTR) &FoldName, lParam, NULL);
					if (NumItems > 1) {
						BWCCMessageBox(hWnd, "Cannot manage attachments\nfrom multiple messages.", "WARNING!", MB_ICONSTOP | MB_OK);
						return 0;
					}

					/*** No Messages in Folder or None Selected ***/
					if (NumItems <= 0) {
						BWCCMessageBox(hWnd, "No Message Selected.", "WARNING!", MB_ICONSTOP | MB_OK);
						return 0;
					}

					/*** Call up the Manage Attachments dialog box. ***/
					CallDlg(hWnd, ManageAttach, AttachBox, lParam);
					return 0;

				case IDM_COMPOSEOPT:
					/*** Change to HourGlass Cursor before starting. ***/
					SetCursor(LoadCursor(NULL, IDC_WAIT));
					CallDlg(hWnd, ComposeOptions, ComposeOptBox, 0L);
					return 0;
				case IDM_FOLDEROPT:
					/*** Change to HourGlass Cursor before starting. ***/
					SetCursor(LoadCursor(NULL, IDC_WAIT));
					CallDlg(hWnd, FolderOptions, FolderOptBox, 0L);
					return 0;
				case IDM_GENERALOPT:
					/*** Change to HourGlass Cursor before starting. ***/
					SetCursor(LoadCursor(NULL, IDC_WAIT));
					CallDlg(hWnd, GeneralOptions, GeneralOptBox, 0L);
					return 0;
				case IDM_PRINTOPT:
					/*** Change to HourGlass Cursor before starting. ***/
					SetCursor(LoadCursor(NULL, IDC_WAIT));
					CallDlg(hWnd, PrintOptions, PrintOptBox, 0L);
					return 0;
				case IDM_READOPT:
					/*** Change to HourGlass Cursor before starting. ***/
					SetCursor(LoadCursor(NULL, IDC_WAIT));
					CallDlg(hWnd, ReadOptions, ReadOptBox, 0L);
					return 0;

				case IDM_EXIT:
					/*** User selected Exit from the Mail menu. ***/
					DestroyWindow(hWnd);
					return 0;

				default:
					/*** If the command message was unrecognized, pass a ***/
					/*** copy of the message to the active child window. ***/
					hWndChild = LOWORD(SendMessage(hWndClient, WM_MDIGETACTIVE, 0, 0L));
					if (IsWindow(hWndChild))
						SendMessage(hWndChild, WM_COMMAND, wParam, lParam);
					return (DefFrameProc(hWnd, hWndClient, message, wParam, lParam));

			}  /* end switch (wParam)  */
			break;

		case WM_MENUSELECT:
			/*** Use the current menu item's control ID number to load the  ***/
			/*** correct status bar message string.   If the menu item      ***/
			/*** is a pop-up, there is no control ID number, so the text    ***/
			/*** of the menu item is used to determine which string to use. ***/
			if ((LOWORD(lParam)&MF_SYSMENU) || (LOWORD(lParam)&MF_SEPARATOR))
				break;

			HelpID = HELP_INDEXES;
			buff[0] = '\0';
			if (!(LOWORD(lParam) & MF_POPUP)) {
				/*** Standard menu item -- so load the string ***/
				/*** which corresponds to the menu item ID.   ***/
				LoadString(hInst, wParam, (LPSTR) &buff, 100);
				HelpID = wParam;
			}
			else {
				/*** Grab the text of the NEXT menu item and ***/
				/*** examine the first and 2nd characters to ***/
				/*** determine which message string to load. ***/
				GetMenuString(wParam, 0, buff, 100, MF_BYPOSITION);
				switch (buff[1]) {
					case 'C':
						if (buff[3] == 'p') {
							LoadString(hInst, STR_EDIT, (LPSTR) &buff, 100);
							HelpID = STR_EDIT;
						}
						if (buff[3] == 's') {
							LoadString(hInst, STR_WINDOW, (LPSTR) &buff, 100);
							HelpID = STR_WINDOW;
						}
						if (buff[3] == 'm') {
							LoadString(hInst, STR_OPTIONS, (LPSTR) &buff, 100);
							HelpID = STR_OPTIONS;
						}
						break;
					case 'M':
						LoadString(hInst, STR_UTILITY, (LPSTR) &buff, 100);
						HelpID = STR_UTILITY;
						break;
					case 'R':
					case 'E':
						LoadString(hInst, STR_MAIL, (LPSTR) &buff, 100);
						HelpID = STR_MAIL;
						break;
					case 'I':
						LoadString(hInst, STR_HELP, (LPSTR) &buff, 100);
						HelpID = HELP_INDEXES;
						break;
					case 'o':
						LoadString(hInst, STR_ANSWER, (LPSTR) &buff, 100);
						HelpID = STR_ANSWER;
						break;
					default:
						buff[0] = '\0';
						HelpID = HELP_INDEXES;
						break;

				}
			}
				/*** Display the new Status bar message. ***/
			if (hStatus)
				SendMessage(hStatus, STB_STRING, 0, (LONG) &buff);
			break;

		case WM_DESTROY:
			/*** Destroy any Help Windows. ***/
			WinHelp(hWnd, NULL, HELP_QUIT, NULL);

			/*** Save folder and main window sizes and positions. ***/
			SaveScreen();
			SaveOptions();

			/*** Release Icon and Cursor Resources. ***/
			GetClassInfo(hInst, "HMailWClass", (WNDCLASS FAR *) &wc);
			DestroyCursor(wc.hCursor);

			GetClassInfo(hInst, "ReadWClass", (WNDCLASS FAR *) &wc);
			DestroyIcon(wc.hIcon);

			GetClassInfo(hInst, "ListWClass", (WNDCLASS FAR *) &wc);
			DestroyIcon(wc.hIcon);

			/*** Destroy the Status Bar and Client Window ***/
			if (hStatus) DestroyWindow(hStatus);
			DestroyWindow(hWndClient);
			hWndClient = NULL;

			/*** Release Menu Resources. ***/
			DestroyMenu(hMenuMail);
			DestroyMenu(hMenuWrite);


			/**** Release Class Structure Memory. ***/
			UnregisterClass("ListWClass", hInst);
			UnregisterClass("ReadWClass", hInst);
			UnregisterClass("StatusBar", hInst);

			/*** Unhook Message Filter and release procedure instance. ***/
			UnhookWindowsHook(WH_MSGFILTER, lpFiltProc);
			FreeProcInstance(lpFiltProc);

			/*** Delete all HappyMail created Temporary files. ***/
			DeleteTempFiles();

			PostQuitMessage(0);
			return 0;

		default:
			return (DefFrameProc(hWnd, hWndClient, message, wParam, lParam));
	}  /* end switch message */

	return 0;
}


