/****************************************************************************
	FILE:		foldrlib.c


	CONTENTS:   AddFolder()		adds a MailFolder to the linked list.
				FindID()		returns the list box index of a message.
				GetFolder()		returns a MailFolder handle given a HWND.
				GetFolder2()	returns a MailFolder handle given a name.
				KillFolder()	deletes a MailFolder from the linked list
								given a HWND
				KillFolder2()	deletes a MailFolder from the linked list
								given a folder name
				ResetLists()	rebuilds the message lists in folder windows.
				WriteTitleBar() formats and writes the caption on a folder


	COMMENTS:	This file conatins the functions used in manipulating
				Folder windows and their underlying structures.
****************************************************************************/
#include "MyMail.h"
#include "string.h"



/******************************************************************************
	FUNCTION: 	AddFolder( )

	PURPOSE:  	To return a pointer to a new Folder structure.
*******************************************************************************/
GLOBALHANDLE AddFolder(char far *FoldName)
{
	GLOBALHANDLE 	hGlobalMem, hTempMem, hTempMem2;
	MailFolder far 	*TempFolder, far *TempFolder2;
    int 			i;

    /*** Allocate the memory for the new MailFolder structure ***/
	if ((hGlobalMem = SafeAlloc(GLOBAL, (long) sizeof(MailFolder), hWnd)) == NULL) {
		return (NULL);
	}

    /*** Initialize the name and next pointer fields of the new structure ***/
	TempFolder = (MailFolder far *) GlobalLock(hGlobalMem);
	lstrcpy((char far *)TempFolder->Name, FoldName);
	TempFolder->Next     = NULL;

	/*** Initailize Item array. ***/
	for(i=0;i<MaxMsgs;i++) {
		TempFolder->Item[i].flags = 0x00;
	}

    /*** Check to see if this is the first in the list and make it the Top pointer ***/
	if (TopFolder == NULL) {
		TopFolder = hGlobalMem;
		TempFolder->Prev = NULL;
		GlobalUnlock(hGlobalMem);
		return(hGlobalMem);
	}

    /*** Find the end of the linked list ***/
	TempFolder2 = (MailFolder far *) GlobalLock(TopFolder);
	hTempMem  = TopFolder;
	while(TempFolder2->Next !=  NULL) {
		hTempMem2 = TempFolder2->Next;
		GlobalUnlock(hTempMem);
		hTempMem = hTempMem2;
		TempFolder2 = (MailFolder far *) GlobalLock(hTempMem);
	}

    /*** Add the new structure to the last structure in the list ***/
	TempFolder2->Next  	 = hGlobalMem;
	TempFolder->Prev 	 = hTempMem;

	/*** Restore the Close Menu on the previous Folder. ***/
	EnableMenuItem(GetSystemMenu(TempFolder2->hListMail, FALSE), SC_CLOSE, MF_BYCOMMAND | MF_ENABLED);

	GlobalUnlock(hTempMem);
	GlobalUnlock(hGlobalMem);

	return (hGlobalMem);
}





/******************************************************************************
	FUNCTION: 	FindID()

	PURPOSE:  	To return the List Box Index with the matching LetterID.
*******************************************************************************/
int FindID(GLOBALHANDLE hFold, char far *MsgID)
{
	MailFolder far 	*TempFolder;
	int				index, MaxMsg;
	char			buff[256];
	MSG 			msg;


	/*** First check in memory for the ID. ***/
	if ((index = FindIdInMem(hFold, MsgID))!= -1)
		return index;

	/*** Get the pointer to the list box ***/
	TempFolder = (MailFolder far *) GlobalLock(hFold);
	MaxMsg = SendMessage(TempFolder->hListBox, LB_GETCOUNT, 0, 0L);

	/*** Didn't find a matching ID in memory, ***/
	/*** so now spin thru again loading in    ***/
	/*** IDs that are not already in memory.  ***/
	for (index = 0; (index<MaxMsg) && (index<TempFolder->MsgCount); index++) {
		if (!(TempFolder->Item[index].flags & INMEMORY))
			if (!ReadID(TempFolder, index, (LPSTR) &buff))
				return -1;

		/*** If the ID's match return the index number ***/
		if (lstrncmp(MsgID, (char far *) &(TempFolder->Item[index].ID), 13) == 0) {
			GlobalUnlock(hFold);
			return(index);
		}
	}

	GlobalUnlock(hFold);
	return (-1);
}



/******************************************************************************
	FUNCTION: 	FindIdInMem()

	PURPOSE:  	To return the List Box Index with the matching LetterID if
				its already in memory.
*******************************************************************************/
int FindIdInMem(GLOBALHANDLE hFold, char far *MsgID)
{
	MailFolder far 	*TempFolder;
	int				index, MaxMsg;
	char			buff[256];
	MSG 			msg;

	/*** Get the pointer to the list box ***/
	TempFolder = (MailFolder far *) GlobalLock(hFold);
	MaxMsg = SendMessage(TempFolder->hListBox, LB_GETCOUNT, 0, 0L);

	/*** Look at messages in memory for a matching Message ID. ***/
	for (index = 0; (index<MaxMsg) && (index<TempFolder->MsgCount); index++) {
		if (!(TempFolder->Item[index].flags & INMEMORY))
			continue;
		/*** If the ID's match return the index number ***/
		if (lstrncmp(MsgID, (char far *) &(TempFolder->Item[index].ID), 13) == 0) {
			GlobalUnlock(hFold);
			return(index);
		}
	}

	GlobalUnlock(hFold);
	return (-1);
}



/******************************************************************************
	FUNCTION: 	GetFolder()

	PURPOSE:  	To return a pointer to the Folder structure with
				the matching Folder window handle.
*******************************************************************************/
GLOBALHANDLE GetFolder(HWND hFolder)
{
	GLOBALHANDLE 	hTempMem, hTempMem2;
	MailFolder far 	*TempFolder;

	/*** Spin thru all the MailFolder structures and compare     ***/
    /*** the hListMail field with the handle that was passed in. ***/
	hTempMem = TopFolder;
	while(hTempMem != NULL) {
		TempFolder = (MailFolder far *) GlobalLock(hTempMem);
		if (TempFolder->hListMail == hFolder) {
			GlobalUnlock(hTempMem);
			return (hTempMem);
		}
		else {
			hTempMem2 = TempFolder->Next;
			GlobalUnlock(hTempMem);
			hTempMem = hTempMem2;
		}
	}
    /*** If no match was found return a NULL. ***/
	return (NULL);
}



/******************************************************************************
	FUNCTION: 	GetFolder2()

	PURPOSE:  	To return a pointer to the Folder structure with
				the matching folder Name.
*******************************************************************************/
GLOBALHANDLE GetFolder2(char far *FoldName)
{
	GLOBALHANDLE	hTempMem, hTempMem2;
	MailFolder far 	*TempFolder;

	/*** Spin thru all the MailFolder structures and compare ***/
    /*** the folder name with the string that was passed in. ***/
	hTempMem = TopFolder;
	while(hTempMem != NULL) {
		TempFolder = (MailFolder far *) GlobalLock(hTempMem);
		if (lstrcmp(TempFolder->Name, FoldName) == 0) {
			GlobalUnlock(hTempMem);
			return (hTempMem);
		}
		else {
			hTempMem2 = TempFolder->Next;
			GlobalUnlock(hTempMem);
			hTempMem = hTempMem2;
		}
	}
	/*** If no match was found return a NULL. ***/
	return (NULL);
}


/******************************************************************************
	FUNCTION: 	KillFolder()

	PURPOSE:  	To delete a folder structure, close the windows associated
				with the folder, and reset the pointers.
*******************************************************************************/
void KillFolder(HWND hFolder)
{
	GLOBALHANDLE 	hTempMem, hTempMem2;
	MailFolder far	*TempFolder, far *TempFolder2;
	HFONT			hFont;

	/*** Get a pointer to the MailFolder structure. ***/
	hTempMem = GetFolder(hFolder);
	if (hTempMem == NULL)
		return;
	TempFolder = (MailFolder far *) GlobalLock(hTempMem);

	/*** Most folder windows have had their fonts changed.     ***/
	/*** So, we need to destroy the font resource to free up   ***/
	/*** the memory.  If the font hasn't been changed, Windows ***/
	/*** will ignore the Delete Object command.                ***/
	if (TempFolder->hListBox) {
		hFont = SendMessage(TempFolder->hListBox, WM_GETFONT, 0, 0L);
		if (hFont != NULL) {
			SendMessage(TempFolder->hListBox, WM_SETFONT, 0, 0L);
			DeleteObject(hFont);
		}

		/*** Destroy the list box ***/
		DestroyWindow(TempFolder->hListBox);
	}

	/*** Take Folder out of linked list, rewriting the ptrs ***/
	/*** and release the Memory assigned to the Folder      ***/
	if (hTempMem == TopFolder) {
		TopFolder = TempFolder->Next;
	}
	else {
		TempFolder2 = (MailFolder far *) GlobalLock(TempFolder->Prev);
		TempFolder2->Next = TempFolder->Next;
		GlobalUnlock(TempFolder->Prev);
		if (TempFolder->Next != NULL) {
			TempFolder2 = (MailFolder far *) GlobalLock(TempFolder->Next);
			TempFolder2->Prev = TempFolder->Prev;
			GlobalUnlock(TempFolder->Next);
		}
	}
	SafeFree(GLOBAL, hTempMem);

	/*** Check for last open folder -- change menu. ***/
	if (TopFolder) {
		TempFolder = (MailFolder far *) GlobalLock(TopFolder);
		if (TempFolder->Next == NULL)
			EnableMenuItem(GetSystemMenu(TempFolder->hListMail, FALSE), SC_CLOSE, MF_BYCOMMAND | MF_GRAYED);
		GlobalUnlock(TopFolder);
	}


	return;
}


/******************************************************************************
	FUNCTION: 	KillFolder2()

	PURPOSE:  	To delete a folder structure, close the windows associated
				with the folder, and reset the pointers.
*******************************************************************************/
void KillFolder2(char far *FoldName)
{
	GLOBALHANDLE 	hTempMem, hTempMem2;
	MailFolder far	*TempFolder, far *TempFolder2;
	HFONT			hFont;

	/*** Get a pointer to the MailFolder structure. ***/
	hTempMem = GetFolder2(FoldName);
	if (hTempMem == NULL)
		return;
	TempFolder = (MailFolder far *) GlobalLock(hTempMem);

	/*** Most folder windows have had their fonts changed.     ***/
	/*** So, we need to destroy the font resource to free up   ***/
	/*** the memory.  If the font hasn't been changed, Windows ***/
	/*** will ignore the Delete Object command.                ***/
	if (TempFolder->hListBox) {
		hFont = SendMessage(TempFolder->hListBox, WM_GETFONT, 0, 0L);
		if (hFont != NULL) {
			SendMessage(TempFolder->hListBox, WM_SETFONT, 0, 0L);
			DeleteObject(hFont);
		}

		/*** Destroy the list box ***/
		DestroyWindow(TempFolder->hListBox);
	}

	/*** Take Folder out of linked list, rewriting the ptrs ***/
	/*** and release the Memory assigned to the Folder      ***/
	if (hTempMem == TopFolder) {
		TopFolder = TempFolder->Next;
	}
	else {
		TempFolder2 = (MailFolder far *) GlobalLock(TempFolder->Prev);
		TempFolder2->Next = TempFolder->Next;
		GlobalUnlock(TempFolder->Prev);
		if (TempFolder->Next != NULL) {
			TempFolder2 = (MailFolder far *) GlobalLock(TempFolder->Next);
			TempFolder2->Prev = TempFolder->Prev;
			GlobalUnlock(TempFolder->Next);
		}
	}
	SafeFree(GLOBAL, hTempMem);

	/*** Check for last open folder -- change menu. ***/
	if (TopFolder) {
		TempFolder = (MailFolder far *) GlobalLock(TopFolder);
		if (TempFolder->Next == NULL)
			EnableMenuItem(GetSystemMenu(TempFolder->hListMail, FALSE), SC_CLOSE, MF_BYCOMMAND | MF_GRAYED);
		GlobalUnlock(TopFolder);
	}


	return;
}




/******************************************************************************
	FUNCTION: 	ResetLists()

	PURPOSE:  	To fix-up Folders and Letters after a Move or Delete.  Adds or
				removes lines from the Folder windows' list boxes and fixes the
				message count in their titles.  For Letter windows, if the
				message being displayed is one that was moved or deleted, the
				next available message is placed in the Letter and it is redis-
				played OR the Letter is closed (depending on the delete mode).
*******************************************************************************/
ResetLists(char far *FoldName, char far *FoldName2, int NumItems, char far *MsgID)
{

	GLOBALHANDLE		hArrMem, hTempFold, hTempMem, hTemp2;
	HWND 				hLetter;
	char 				buff[80], DelArr[1000];
	MailFolder far		*TempFolder;
	Letter far 			*TempLetter;
	int 				i, j, count, action, index;
	BOOL				FolderClosed;


	/*** Reset Folder2's List Box by using my fake window    ***/
	/*** message WM_LBRESET.  This will cause the Folder's   ***/
	/*** window procedure to re-read the folder from Banyan. ***/
	if (lstrcmp(FoldName, FoldName2)!= 0) {
		hTempFold = GetFolder2(FoldName2);
		if (hTempFold != NULL) {
			TempFolder = (MailFolder far *) GlobalLock(hTempFold);
			SendMessage(TempFolder->hListMail, (int) WM_LBRESET, (int) NULL, (long) NULL);
			GlobalUnlock(hTempFold);
		}
	}


	/*** Get a pointer to the first Folder. ***/
	FolderClosed = FALSE;
	hTempFold = GetFolder2(FoldName);
	if (hTempFold == NULL)
		FolderClosed = TRUE;
	else
		TempFolder = (MailFolder far *) GlobalLock(hTempFold);

	/*** Determine the delete mode action for future reference ***/
	action = Options.deletemode;

	/*** Fill an array with the deleted item indices 			 ***/
	/*** i.e. DelArr[0] = 1 means the first item will be deleted ***/
	if (!FolderClosed) {
		NullFill((LPSTR) &DelArr, MaxMsgs);
		for(i=0;i<NumItems;i++) DelArr[FindID(hTempFold, &(MsgID[i*14]))] = 1;
	}


	/*** Scan thru open letters -- if one contains a deleted ID ***/
	/*** execute the delete mode action i.e. scan for the next  ***/
	/*** undeleted letter in the DelArr OR close the letter. 	***/
	hTempMem = TopLetter;
	while(hTempMem != NULL) {
		TempLetter = (Letter far *) GlobalLock(hTempMem);
		hLetter = TempLetter->hChild;
		hTemp2 = TempLetter->Next;
		if (lstrcmp(FoldName, TempLetter->FolderName) == 0) {

			/*** Check for an id that was moved or deleted and doesn't have an open folder.***/
			for(j=0;j<NumItems;j++) {
				if (lstrncmp(TempLetter->MsgID, (LPSTR) &(MsgID[j*14]), 13)==0) {
					if (FolderClosed || (TempFolder->MsgCount - NumItems <= 0)) {
						/*** Folder is empty so close this open letter window. ***/
						TempLetter->MsgID[0] = NULL;
						GlobalUnlock(hTempMem);
						hTempMem = hTemp2;
						SendMessage(GetParent(hLetter), WM_MDIDESTROY, hLetter, NULL);
						continue;
					}

					/*** Check to see if this Letter window's message has      ***/
					/*** been moved or deleted. If so, open the next available ***/
					/*** letter or close the box depending on the delete mode. ***/
					index =  FindID(hTempFold, TempLetter->MsgID);
					if (DelArr[index]) {
						/*** Check for Delete Mode Action i.e. Prev or Next or close ***/
						switch (action) {
							case 1:
								i=-1; // Set i to an invalid value forcing the letter to close
								break;

							case 2:
								for(i=index; DelArr[i] && i>=0; i--);
								break;

							case 3:
								for(i=index; DelArr[i] && i<MaxMsgs && i<TempFolder->MsgCount; i++);
								break;
						}

						/*** if the next available message index is above 200 ***/
						/*** or less than 0, then close the letter window.    ***/
						if (i>=0 && i<MaxMsgs && i<TempFolder->MsgCount)
							ReadID(TempFolder, i, TempLetter->MsgID);
						else {
							TempLetter->MsgID[0] = NULL;
							GlobalUnlock(hTempMem);
							hTempMem = hTemp2;
							SendMessage(GetParent(hLetter), WM_MDIDESTROY, hLetter, NULL);
							break;
						}
					}
				}
			}
		}
		if (hTempMem)
			GlobalUnlock(hTempMem);
		hTempMem = hTemp2;

	}

	/*** Reset the contents of the first folder's listbox. ***/
	if (!FolderClosed)
		SendMessage(TempFolder->hListMail, WM_LBRESET, 0, 0L);

	/*** Redisplay all open letters, forcing the display of any new messages. ***/
	hTempMem = TopLetter;
	while(hTempMem != NULL) {
		TempLetter = (Letter far *) GlobalLock(hTempMem);
		hLetter = TempLetter->hChild;
		NextLetter(hLetter, SAME, OPEN);
		hTemp2 = TempLetter->Next;
		GlobalUnlock(hTempMem);
		hTempMem = hTemp2;
	}


	if (!FolderClosed)
		GlobalUnlock(hTempFold);
	return 0;

}


/******************************************************************************
	FUNCTION: 	WriteTitleBar()

	PURPOSE:  	To write the title bar of a folder window
*******************************************************************************/
WriteTitleBar(MailFolder far *CurFolder)
{

	char buff[100];

	switch (Options.sortorder) {
    	case BYDATE:
			wsprintf(buff,"%s : %d Msgs : Sorted By Date", (char far *) CurFolder->Name, (unsigned int) CurFolder->MsgCount);
    		break;
		case BYNAME:
			wsprintf(buff,"%s : %d Msgs : Sorted By Name", (char far *) CurFolder->Name, (unsigned int) CurFolder->MsgCount);
			break;
		case BYSTATUS:
			wsprintf(buff,"%s : %d Msgs : Sorted By Status", (char far *) CurFolder->Name, (unsigned int) CurFolder->MsgCount);
    		break;
		case BYSUBJECT:
			wsprintf(buff,"%s : %d Msgs : Sorted By Subject", (char far *) CurFolder->Name, (unsigned int) CurFolder->MsgCount);
			break;
    }
	SendMessage(CurFolder->hListMail, WM_SETTEXT, NULL, (LONG) buff);
    return;

}
