/****************************************************************************
	FILE:		folders.c


	CONTENTS:	GetName()		window procedure for Rename Folder dialog box.
				GetNewName()	window procedure for Create Folder dialog box.
				Manage()		window procedure for Manage Folders dialog box.

	COMMENTS:	This file contains all the procedures used with the Manage
				Folder dialog box.
****************************************************************************/

#include "MyMail.h"
#include "string.h"

/******************************************************************************
	FUNCTION:	GetName()

	PURPOSE:  	Dialog Procedure for NameBox Dialog Box

	COMMENTS:	This dialog box procedure is used for getting a new name
				for an existing	folder.  The only difference between this
				procedure and GetNewName is that this one is passed the
                old name in lParam and it copies that name to the window title.
*******************************************************************************/
BOOL FAR PASCAL _export GetName(HWND hDlg, unsigned message, WORD wParam, LONG lParam)
{

	char strbuff[50];
    static char far *lpFoldName;

	/*** Send Window Messages to Status Bar ***/
	StatBarCtrlMsg(hDlg, NameBox, message, wParam, lParam);

	switch (message) {
		case WM_INITDIALOG:
			/*** Center the Dialog Window in the HappyMail Main window ***/
			CenterWindow(hDlg, GetDesktopWindow());

			/*** Get the current folder name and copy ***/
            /*** it to the title bar of the window.   ***/
			lpFoldName = (char far *) lParam;
			lstrcpy((LPSTR) &strbuff, "Renaming the Folder : ");
			lstrcat((LPSTR) &strbuff, (LPSTR) lpFoldName);
			SendMessage(hDlg, WM_SETTEXT, NULL, (LPARAM) &strbuff);

            /*** Set the text limits and initialize the focus ***/
			SendDlgItemMessage(hDlg, NB_NAME, EM_LIMITTEXT, 13, NULL);
			SetFocus(GetDlgItem(hDlg, NB_NAME));
			break;

		case WM_COMMAND:        
			switch(wParam) {
				case IDCANCEL: 
                    /*** Cancel was selected ***/
					EndDialog(hDlg, FALSE);
					return(TRUE);
				case IDM_OK:
				    /*** OK box selected -- copy new name over top of old name ***/
					SendDlgItemMessage(hDlg, NB_NAME, WM_GETTEXT, 14, (LONG) lpFoldName);
					EndDialog(hDlg, TRUE);
					return (TRUE);
			} /** end switch(wParam) **/
			break;
	} /** end switch(message) **/
	return (FALSE);
}


/******************************************************************************
	FUNCTION: 	GetNewName()

	PURPOSE:  	Dialog Procedure for NameBox Dialog Box

	COMMENTS:	This dialog box procedure is used for getting a name for a 
				new folder.
*******************************************************************************/
BOOL FAR PASCAL _export GetNewName(HWND hDlg, unsigned message, WORD wParam, LONG lParam)
{

	char strbuff[50];
    static char far *lpFoldName;

	/*** Send Window Messages to Status Bar ***/
	StatBarCtrlMsg(hDlg, NameBox, message, wParam, lParam);

	switch (message) {
		case WM_INITDIALOG:
    		/*** Center the Dialog Window in the HappyMail Main window ***/
			CenterWindow(hDlg, GetDesktopWindow());

            /*** Store the name pointer for output ***/
			lpFoldName = (char far *) lParam;

            /*** Initialize controls and title bar ***/
			SendDlgItemMessage(hDlg, NB_NAME, EM_LIMITTEXT, 13, NULL);
			if (lpFoldName[0] == '\0')
				SendMessage(hDlg, WM_SETTEXT, NULL, (LONG) "Creating a New Folder");
			SetFocus(GetDlgItem(hDlg, NB_NAME));
			break;

		case WM_COMMAND:                
			switch(wParam) {
				case IDCANCEL:
                	/*** Cancel was selected ***/
					EndDialog(hDlg, FALSE);
					return(TRUE);
				case IDM_OK:
					/*** OK box selected -- copy new name to pointer ***/
					SendDlgItemMessage(hDlg, NB_NAME, WM_GETTEXT, 14, (LONG) lpFoldName);
					EndDialog(hDlg, TRUE);
					return (TRUE);
			} /** end switch(wParam) **/
			break;
	} /** end switch(message) **/
	return (FALSE);
}


/****************************************************************************

	FUNCTION: 	Manage()

	PURPOSE:  	Processes messages for "FolderBox" dialog box

	COMMENTS:	The folderbox window allows the user to create, rename,
				delete, and open folders.	
****************************************************************************/

BOOL FAR PASCAL _export Manage(HWND hDlg, unsigned message, WORD wParam, LONG lParam)
{

	CallStatus			uStat;
	Letter far			*TempLetter;
	MailFolder far 		*TempFolder;
	char 				buff[32], FoldName[20], tempStr[32];
	unsigned short 		offset, retcount=0, foldercount, i, tabset[1];
	short				index;
	HWND				hListMail, hMessage;
	GLOBALHANDLE		hFolder, hLetter;
	FolderInfo			Folder[MAXFOLDERS];
	BOOL				Empty;

	/*** Send Window Messages to Status Bar ***/
	StatBarCtrlMsg(hDlg, FolderBox, message, wParam, lParam);

	switch (message) {
		case WM_INITDIALOG:        /* message: initialize dialog box */
			/*** Center the Dialog Window in the HappyMail Main window ***/
			CenterWindow(hDlg, GetDesktopWindow());

			/**** Change the default pushbutton and set text limits ****/
			SendMessage(hDlg, DM_SETDEFID, NULL, NULL);
			SendDlgItemMessage(hDlg, FB_FOLDERS, LB_RESETCONTENT, NULL, NULL);

			/*** Set the Tab Stops for the List Box ***/
			tabset[0] = 104;
			SendDlgItemMessage(hDlg, FB_FOLDERS, LB_SETTABSTOPS, 1, (long) &tabset);

			/**** Get the Mail folder names and place in the list box ****/
			offset = uStat = 0;
			retcount = 1;
			while (uStat == 0 && retcount>0) {
				uStat=VnsListMailFolders(hVNMAS, UserName, Folder, offset, MAXFOLDERS, &retcount, &foldercount);
				if (Recover(uStat))
					uStat=VnsListMailFolders(hVNMAS, UserName, Folder, offset, MAXFOLDERS, &retcount, &foldercount);

				offset+=retcount;
				for(i=0; i<retcount && uStat == 0; i++) {
					wsprintf((char far *) &buff, "%s\t%i", (char far *) &(Folder[i].name), (int) Folder[i].num);
					SendDlgItemMessage(hDlg, FB_FOLDERS, LB_ADDSTRING, 0, (LONG) &buff);
				}
			}
			if(uStat) {
				DspMsgCode(hWnd, "Error Reading Folder List : ", uStat, FALSE);
			}

			SendDlgItemMessage(hDlg, FB_FOLDERS, LB_SETCURSEL, 0, NULL);
			break;


		case WM_COMMAND:
			/*** User double clicked in the folder names     ***/
			/*** list box, so open the folder they selected. ***/
			if (lParam >> 16 == LBN_DBLCLK)
				PostMessage(hDlg, WM_COMMAND, FB_OPEN, NULL);

			switch(wParam) {
				case FB_DELETE:
					/**** Get the name of the Folder to delete ****/
					index = (WORD) SendDlgItemMessage(hDlg, FB_FOLDERS, LB_GETCURSEL, 0, 0L);
					if (index == LB_ERR) {
						BWCCMessageBox(hDlg, "No Folder is Selected.", "WARNING!", MB_ICONSTOP | MB_OK);
						break;
					}        	
					SendDlgItemMessage(hDlg, FB_FOLDERS, LB_GETTEXT, index, (LONG) buff);

					/*** Check if Folder is empty ***/
                    Empty = FALSE;
					if (buff[lstrlen((LPSTR) &buff)-1] == '0' && buff[lstrlen((LPSTR) &buff)-2] == '\t')
                    	Empty = TRUE;

					/*** Trim the End of the string off -- leaving foldername ***/
					for(i=0;i<lstrlen((LPSTR) &buff);i++) if (buff[i] == '\t') buff[i] = '\0';

                    /*** If folder is general or wastebasket, give the warning message ***/
					if (lstrcmp((LPSTR) &buff, "General") == 0 || lstrcmp((LPSTR) &buff, "Wastebasket") ==0) {
						BWCCMessageBox(hDlg, "This Folder Cannot Be Deleted.", "WARNING!", MB_ICONSTOP | MB_OK);
						break;
					}

					/*** Ask User if he/she is sure ***/
					if (Empty || Options.expertmode || CallDlg(hDlg, AskChange, YesNoBox, 3L)) {
						VnsDeleteMailFolder(hVNMAS, UserName, (char far *)buff);

						/**** If Folder is on screen delete it ****/
						hFolder = GetFolder2(buff);
						if (hFolder != NULL) {
							TempFolder = (MailFolder far *) GlobalLock(hFolder);
							hListMail = TempFolder->hListMail;
							GlobalUnlock(hFolder);
							SendMessage(hListMail, WM_SYSCOMMAND, SC_CLOSE, NULL);
						}

						/**** Close Mail Messages from this folder that are on screen ****/
						hLetter = GetLetterByFoldName((LPSTR) &buff);
						while (hLetter != NULL) {
							TempLetter = (Letter far *) GlobalLock(hLetter);
							hMessage = TempLetter->hChild;
							GlobalUnlock(hLetter);
							SendMessage(hMessage, WM_SYSCOMMAND, SC_CLOSE, NULL);
							hLetter = GetLetterByFoldName((LPSTR) &buff);
						}


						/**** Reset the folder name List Box ***/
						SendDlgItemMessage(hDlg, FB_FOLDERS, LB_DELETESTRING, index, 0L);					}
					break;

				case FB_CREATE:
					/*** Get a new name from the user ***/
					FoldName[0] = '\0';
					if (CallDlg(hDlg, GetNewName, NameBox, (long) &FoldName)) {

						/**** Make sure name is not already in use ***/
						index = (WORD) SendDlgItemMessage(hDlg, FB_FOLDERS, LB_FINDSTRING, -1, (LONG) &FoldName);
						if (index != LB_ERR) {
							SendDlgItemMessage(hDlg, FB_FOLDERS, LB_GETTEXT, index, (LONG) &buff);
							for(i=0;i<lstrlen((LPSTR) &buff);i++) if (buff[i] == '\t') buff[i] = '\0';
							if (lstrcmp((LPSTR) &buff, (LPSTR) &FoldName) == 0) {
								BWCCMessageBox(hDlg, "Folder name already exists.", "WARNING!", MB_ICONSTOP | MB_OK);
								break;
                            }
						}

						/*** Create the new folder ***/
						uStat = VnsCreateMailFolder(hVNMAS, UserName, FoldName);
						if (Recover(uStat))
							uStat = VnsCreateMailFolder(hVNMAS, UserName, FoldName);

						if (uStat == 0) {
							/**** Addd the new folder name to the folder name List Box ***/
							wsprintf((char far *) &buff, "%s\t%i", (char far *) &FoldName, (int) 0);
							SendDlgItemMessage(hDlg, FB_FOLDERS, LB_ADDSTRING, 0, (LONG) &buff);
						}
						else {
							/*** Error creating folder, so display appropriate message. ***/
							if (uStat == MS_BADNAME)
								BWCCMessageBox(hDlg, "Name contains illegal character(s).\rSuch as a '/'.", "WARNING!", MB_ICONSTOP | MB_OK);
							else if (uStat == MS_FOLDEREXISTS)
								BWCCMessageBox(hDlg, "Folder name already exists.", "WARNING!", MB_ICONSTOP | MB_OK);
							else 
								BWCCMessageBox(hDlg, "Error Creating Folder.", "WARNING!", MB_ICONSTOP | MB_OK);
                        }
					}
					break;

				case FB_RENAME:
					index = (WORD) SendDlgItemMessage(hDlg, FB_FOLDERS, LB_GETCURSEL, 0, 0L);
					if (index == LB_ERR) {
						BWCCMessageBox(hDlg, "No Folder is Selected.", "WARNING!", MB_ICONSTOP | MB_OK);
						break;
					}        	
					SendDlgItemMessage(hDlg, FB_FOLDERS, LB_GETTEXT, index, (LONG) &FoldName);

					/*** Trim the End of the string off -- leaving foldername ***/
					for(i=0;i<lstrlen((LPSTR) &FoldName);i++) if (FoldName[i] == '\t') FoldName[i] = '\0';
					lstrcpy((LPSTR) &tempStr, (LPSTR) &FoldName + i);

					/*** Make sure user is not renmaing the wastebasket or the general folder ***/
					if (lstrcmp((LPSTR) &FoldName, "General") == 0 || lstrcmp((LPSTR) &FoldName, "Wastebasket") == 0) {
						BWCCMessageBox(hDlg, "This Folder Cannot Be Renamed.", "WARNING!", MB_ICONSTOP | MB_OK);
						break;
					}

                    /*** Get a new name from the user ***/
					lstrcpy((LPSTR) &buff, (LPSTR) &FoldName);
					if (CallDlg(hDlg, GetName, NameBox, (long) &buff)) {

						/*** Rename the folder ***/
						uStat = VnsRenameMailFolder(hVNMAS, UserName, FoldName, buff);
						if (Recover(uStat))
							uStat = VnsRenameMailFolder(hVNMAS, UserName, FoldName, buff);

						if (uStat == 0) {
							/**** If Folder is on screen change name ****/
							hFolder = GetFolder2(FoldName);
							if (hFolder != NULL) {
								TempFolder = (MailFolder far *) GlobalLock(hFolder);
								lstrcpy(TempFolder->Name, (char far *) &buff);
								WriteTitleBar(TempFolder);
								GlobalUnlock(hFolder);
							}

							/**** Change FolderName in open Mail Messages ****/
							hLetter = GetLetterByFoldName((LPSTR) &FoldName);
							while (hLetter != NULL) {
								TempLetter = (Letter far *) GlobalLock(hLetter);
								lstrcpy(TempLetter->FolderName, (LPSTR) &buff);
								GlobalUnlock(hLetter);
								hLetter = GetLetterByFoldName((LPSTR) &FoldName);
							}

							/**** Change folder name in the folder name List Box ***/
							SendDlgItemMessage(hDlg, FB_FOLDERS, LB_DELETESTRING, index, 0L);
							wsprintf((char far *) &buff, "%s\t%s", (char far *)  &buff, (char far *) &tempStr);
							SendDlgItemMessage(hDlg, FB_FOLDERS, LB_ADDSTRING, 0, (LONG) &buff);

						}
						else {
							/*** Error creating folder so issue a warning message ***/
							if (uStat == MS_BADNAME)
								BWCCMessageBox(hDlg, "Name contains illegal character(s).\rSuch as a '/'.", "WARNING!", MB_ICONSTOP | MB_OK);
							else if (uStat == MS_FOLDEREXISTS)
								BWCCMessageBox(hDlg, "Folder name already exists.", "WARNING!", MB_ICONSTOP | MB_OK);
							else
								BWCCMessageBox(hDlg, "Error Renaming Folder.", "WARNING!", MB_ICONSTOP | MB_OK);
						}
					}
					break;

				case FB_OPEN:
					/*** Get the name of the folder to open from the list box ***/
					index = (WORD) SendDlgItemMessage(hDlg, FB_FOLDERS, LB_GETCURSEL, 0, 0L);
					if (index == LB_ERR) {
						BWCCMessageBox(hDlg, "No Folder is Selected.", "WARNING!", MB_ICONSTOP | MB_OK);
						break;
					}
					SendDlgItemMessage(hDlg, FB_FOLDERS, LB_GETTEXT, index, (LONG) &buff);
					for(i=0;i<lstrlen((LPSTR) &buff);i++) if (buff[i] == '\t') buff[i] = '\0';

					/*** If folder is not open yet, open it ***/
					if (GetFolder2(buff) == NULL) {
						hListMail = SendMessage(hWnd, WM_COMMAND, IDM_LIST, (LONG) &buff);

						/* If window could not be created, return "failure" */
						if (!hListMail)
							DspMsgCode( hWnd, "Error Creating Folder Window", 0, FALSE);
					}
					else {
                        /*** Folder is already open so issue warning. ***/
						BWCCMessageBox(hDlg, "Folder is Already Open.", "WARNING!", MB_ICONSTOP | MB_OK);
						break;
					}
					break;
				case IDCANCEL:
					EndDialog(hDlg, TRUE);
			}
	} /* end switch (message) */
	return (FALSE);               /* Didn't process a message    */

}
