/****************************************************************************
	FILE:		bargraf.c


	CONTENTS: 	BarGraf() - window procedure for the Bar Graph dialog box.

****************************************************************************/
#include "mymail.h"
#include "string.h"

DrawGrafBar(HWND, int, WPARAM, unsigned long);

/******************************************************************************
	FUNCTION: 	BarGraf()

	PURPOSE:  	Dialog Box procedure for BarGrafBox Window (Displays a Bar Graf).

	COMMENTS:	The Bar Graph dialog box is used to indicate progress in file
				saves and file prints.  When initialized, it displays a raised
				-edge rectangular gray box.  It draws a button-like progress
				bar over top the box, whenever a WM_NEXTSTEP message is received.
				To use the bargraf box, the programmer sends a WM_NEXTSTEP
				message with the percentage (0-100) to diplay as the wParam
				parameter.  The movement of the bar is totally dictated by the
				frequency with which the programmer sends updated percentages.
*******************************************************************************/
BOOL FAR PASCAL _export BarGraf(HWND hDlg, unsigned message, WORD wParam, LONG lParam)
{
	static int		oldPos;
	static long		oldBytes;
	RECT			Rect;
	HWND			hCancel;

	/*** Send Window Messages to Status Bar ***/
	StatBarCtrlMsg(hDlg, BarGrafBox, message, wParam, lParam);

	switch (message) {
		case WM_INITDIALOG:
			/*** Center the Dialog Window in the HappyMail Main window ***/
			CenterWindow(hDlg, GetDesktopWindow());
			/*** Disable the close option on the system menu ***/
			EnableMenuItem (GetSystemMenu(hDlg, FALSE), SC_CLOSE, MF_GRAYED);
			oldPos = 0;
			oldBytes = 0L;
			return TRUE;

		case WM_NEXTSTEP:
			if ((oldPos > wParam) && (lParam == 0L)) oldPos = 0;
			DrawGrafBar(GetDlgItem(hDlg, BG_BAR), oldPos, wParam,  (unsigned long) lParam);
			oldPos = wParam;
			oldBytes = lParam;
			return TRUE;


		case WM_PAINT:
			DrawGrafBar(GetDlgItem(hDlg, BG_BAR), 0, oldPos,  (unsigned long) oldBytes);
			break;


		case WM_COMMAND:
			/*** User clicked the cancel button -- so set the abort flag and abort ***/
			if (wParam == IDCANCEL)
				bUserAbort = TRUE;
			EnableWindow(GetParent(hDlg), TRUE);
			DestroyWindow(hDlg);
			hDlgCancel = 0;
			return FALSE;


	}
	return FALSE;
}



/******************************************************************************
    FUNCTION:   DrawBarGraf()

    PURPOSE:    Do the actual drawing of the progress bar.
*****************************************************************************/
DrawGrafBar(HWND hBar, int oldPos, WPARAM wParam,  unsigned long lParam)
{
	HDC				hDC;
	HBRUSH			hOldBrush;
	HPEN			hOldPen;
	unsigned int 	thick=4, i, left, right;
	RECT			Rect;
	char			buff[20];
	DWORD			LGray, DGray, White, OldColor, Black, dwStrSize;


	/*** Initialize the colors ***/
	LGray = RGB(192,192,192);
	DGray = RGB(128,128,128);
	White = RGB(255,255,255);
	Black = RGB(  0,  0,  0);

	/*** Make sure the percentage value is between 0 and 100 ***/
	wParam = (wParam > 100) ? 100 : wParam;
	wParam = ((int)wParam < 0) ? 0 : wParam;

	/*** Get the size and position of the rectangle we are drawing in ***/
	GetClientRect(hBar, (RECT far *) &Rect);

	/*** Modify the values to account for the raised look of the bar ***/
	Rect.left	+= thick;
	Rect.top	+= thick;
	Rect.bottom -= thick;
	Rect.right	-= thick;

	/*** Establish some default values. ***/
	hDC	= GetDC(hBar);
	if (hDC == NULL)
		return FALSE;
	hOldPen 	= SelectObject(hDC, CreatePen(PS_SOLID, 1, LGray));
	hOldBrush  	= SelectObject(hDC, CreateSolidBrush(LGray));


	/*** lParam is used to pass in the number of bytes. ***/
	if (wParam == 0 && (lParam != 0L)) {
		if (lParam == 1L)
			DrawBar(hBar, hDC, "\0");
		wsprintf(buff, "%lu bytes", (unsigned long) lParam);
		dwStrSize = GetTextExtent(hDC, buff, lstrlen(buff));
		OldColor = SetBkColor(hDC, LGray);
		TextOut(hDC,  Rect.left + (((Rect.right - Rect.left)/2) - (LOWORD(dwStrSize)/2)),
		  Rect.top + (((Rect.bottom - Rect.top)/2) - (HIWORD(dwStrSize)/2)), buff, lstrlen(buff));
		SetBkColor(hDC, OldColor);
		DeleteObject(SelectObject(hDC, hOldPen));
		DeleteObject(SelectObject(hDC, hOldBrush));
		ReleaseDC(hBar, hDC);
		return TRUE;
	}

	left	= (int) ((((long)Rect.right)*((long)oldPos))/100L);
	right	= (int) ((((long)Rect.right)*((long)wParam))/100L);

	if (oldPos == 0)
		/*** Sink Background on first time through ***/
		DrawBar(hBar, hDC, "\0");
	else
		/*** Erase bar from a previous draw ***/
		Rectangle(hDC, (left<thick) ? thick : left, Rect.top, ((right<thick) ? thick : right)+1, Rect.bottom+1);

	/*** Print the percentage in the center of bar. ***/
	OldColor = SetBkColor(hDC, LGray);
	wsprintf(buff, "%u%%", (unsigned int) wParam);
	if (wParam == 100) wsprintf(buff, "   Complete   ");

	dwStrSize = GetTextExtent(hDC, buff, lstrlen(buff));
	TextOut(hDC,  Rect.left + (((Rect.right - Rect.left)/2) - (LOWORD(dwStrSize)/2)),
				  Rect.top + (((Rect.bottom - Rect.top)/2) - (HIWORD(dwStrSize)/2)), buff, lstrlen(buff));

	SetBkColor(hDC, OldColor);

	DeleteObject(SelectObject(hDC, CreatePen(PS_SOLID, 1, White)));


	/*** Draw the rasied-edge of the bar -- if the bar is not longer ***/
	/*** than the the raised edge then special drawing is needed     ***/
	if (left > thick) {
		/*** Draw the white lines of the edge ***/
		for(i=1; i<thick; i++) {
			MoveTo(hDC, left, Rect.top - i);
			LineTo(hDC, right + i, Rect.top - i);
		}

		/*** Draw the dark gray lines of the edge ***/
		DeleteObject(SelectObject(hDC, CreatePen(PS_SOLID, 1, DGray)));
		for(i=1; i<thick; i++) {
			MoveTo(hDC, right + i, Rect.top - i);
			LineTo(hDC, right + i, Rect.bottom + i);
			LineTo(hDC, left, Rect.bottom + i);
		}

	}
	else {
		/*** Draw the white lines of the edge ***/
		for(i=1; i<thick; i++) {
			MoveTo(hDC, Rect.left - i, Rect.bottom + i - 1);
			LineTo(hDC, Rect.left - i, Rect.top - i);
			LineTo(hDC, right + i, Rect.top - i);
		}

		/*** Draw the dark gray lines of the edge ***/
		DeleteObject(SelectObject(hDC, CreatePen(PS_SOLID, 1, DGray)));
		for(i=1; i<thick; i++) {
			MoveTo(hDC, right + i, Rect.top - i);
			LineTo(hDC, right + i, Rect.bottom + i);
			LineTo(hDC, Rect.left - i, Rect.bottom + i);
		}
	}

	/*** Draw a black line at the edge of the bar. ***/
	DeleteObject(SelectObject(hDC, CreatePen(PS_SOLID, 1, Black)));
	MoveTo(hDC, right + i, Rect.top-thick);
	LineTo(hDC, right + i, Rect.bottom+thick);


	/*** Clean up Objects and DC -- good memory hygiene ya know ***/
	DeleteObject(SelectObject(hDC, hOldPen));
	DeleteObject(SelectObject(hDC, hOldBrush));
	ReleaseDC(hBar, hDC);

	return TRUE;
}
