/****************************************************************************
	FILE:		attach.c


	CONTENTS:	FileOpen()		retrieves attachment file names using a
								common dialog box.
				HookProc()		hook procedure used with the GetOpenFileName
								common dialog box.
****************************************************************************/

#include "MyMail.h"
#include "string.h"
#include "commdlg.h"
#include "dir.h"


/******************************************************************************
	FUNCTION: 	FileOpen(HWND)

	PURPOSE:	Procedure to call the common dialog file-open box: used for
				selecting file attachments.

	COMMENTS:	FileOpen uses the comman dialog box for opening files, which
				is supplied by Microsoft.  The dialog box has been slightly
				modified to contain an extra list box and two extra buttons.
				These added controls are for displaying, adding, and deleting
				files from a selected file names list.  The box uses a hook
				procedure (HookProc) to capture messages before going to the
				standard procedure for thhis dialog box.   
*******************************************************************************/
void FileOpen( HWND hDlg )
{
	OPENFILENAME ofn;
	DWORD Errval;	// Error value
	char buf[15];	// Error buffer
	char szFilter[] = "All Files (*.*)\0*.*\0Text Files (*.txt)\0*.txt\0";
	char szName[500], curdir[128];

	/*** Save the current working directory (cwd) -- then ***/
	/*** reset the cwd to the default directory (DEF_DIR) ***/
	getcwd((char *) &curdir, 128);
	setcwd(Options.defdir);

    /*** Initialize the OpenFileName structure ***/
	lstrcpy((LPSTR) &szName, "*.*");
	ofn.lStructSize = sizeof( OPENFILENAME );
	ofn.hwndOwner = hDlg;				// An invalid hWnd causes non-modality
	ofn.hInstance = hInst;
	ofn.lpstrFilter = (LPSTR)szFilter;
	ofn.lpstrCustomFilter = NULL;
	ofn.nMaxCustFilter = 0;
	ofn.nFilterIndex = 1;
	ofn.lpstrFile = (LPSTR)szName;		// Stores the result in this variable
	ofn.nMaxFile = sizeof( szName );
	ofn.lpstrFileTitle = NULL;
	ofn.nMaxFileTitle = 0;
	ofn.lpstrInitialDir = Options.defdir;
	ofn.lpstrTitle = "Select Files for Attachment";	// Title for dialog
	ofn.Flags = OFN_FILEMUSTEXIST | OFN_ENABLEHOOK | OFN_HIDEREADONLY | OFN_ENABLETEMPLATE;
	ofn.nFileOffset = 0;
	ofn.nFileExtension = 0;
	ofn.lpstrDefExt = "*";
	ofn.lCustData = NULL;
	ofn.lpfnHook = (DLGHOOKPROC) MakeProcInstance((FARPROC)HookProc, hInst);
	ofn.lpTemplateName = MAKEINTRESOURCE(FileOpenBox);

    /*** Call the common dialog box ***/
	if(GetOpenFileName( &ofn ) != TRUE)
	{
        /*** Possible error occured ***/
		Errval=CommDlgExtendedError();
		if(Errval!=0)					// 0 value means user selected Cancel
		{
			wsprintf((char far *)&buf,"Error = %ld",Errval);
			BWCCMessageBox(hDlg, "Unable to Choose Files", buf, MB_ICONSTOP | MB_OK);
		}

	}

    /*** Reset the current working directory ***/
	setcwd(curdir);
	FreeProcInstance((FARPROC)ofn.lpfnHook);

}

/******************************************************************************
	FUNCTION: 	HookProc(HWND, unsigned, WORD, LONG)

	PURPOSE:  	Dialog Procedure for GetOpenFileName Dialog Box. Returning FALSE
				tells the dialog to pass the message on to the default window
				procedure.  This proc is designed to handle the added controls,
				the Select Files list box, the Add button, and the remove button. 
*******************************************************************************/
UINT CALLBACK _export HookProc(HWND hDlg, UINT message, WORD wParam, LONG lParam)
{

	int 				FileCount, RetCount, i, len;
	char 				StrBuff[1600], FileBuff[20], far *textstr;
	unsigned int 		FileIndex[10];
	GLOBALHANDLE 		hGlobalMem;
	MEASUREITEMSTRUCT	far *mf;
	static int			MaxFiles;


	/*** Send Window Messages to Status Bar ***/
	StatBarCtrlMsg(hDlg, FileOpenBox, message, wParam, lParam);


	switch (message) {
		case WM_INITDIALOG:       	     /* message: initialize dialog box */
			/*** Center the Dialog Window in the HappyMail Main window ***/
			CenterWindow(hDlg, GetDesktopWindow());

			/*** Determine the max number of files that can be selected. ***/
			MaxFiles = 10;
			GetDlgItemText(GetParent(hDlg), CB_ATTACH, (LPSTR) &StrBuff, 1600);
			if (StrBuff[0] != '\0') {
				for(i=0,MaxFiles--;i<lstrlen(StrBuff);i++)
					if (StrBuff[i]==',') MaxFiles--;
			}

			return (TRUE);

		case WM_COMMAND:
			switch (wParam) {

				case FILE_IDOK:                 /*** User selected OK ***/

					/*** Grab enough memory to store the selected ***/
					/*** names in a string and lock it in place.  ***/
					if ((hGlobalMem = SafeAlloc(GLOBAL, 2000L, hDlg)) == NULL) {
						break;
					}
					textstr = GlobalLock(hGlobalMem);
					NullFill(textstr, 2000);
					textstr[0] = '\0';

					/*** Store File Names from Selected Files Box into Attach Files Field. If no ***/
					/*** names are in the Selected Files Box, add the current name to it first. ***/
					GetDlgItemText(GetParent(hDlg), CB_ATTACH, textstr, 2000);
					if (textstr[0] != '\0')
						lstrcat(textstr, (LPSTR) ", ");
					FileCount = 0;
					FileCount = SendDlgItemMessage(hDlg, FILE_LBSEL, LB_GETCOUNT, NULL, NULL);
					if (FileCount == 0) {
						SendMessage(hDlg, WM_COMMAND, FILE_ADD, 0L);
						FileCount++;
					}
					for(FileCount--; FileCount>=0; ) {
						NullFill((LPSTR) &StrBuff, 200);
						SendDlgItemMessage(hDlg, FILE_LBSEL, LB_GETTEXT, FileCount, (LONG) StrBuff);
						lstrcat(textstr, StrBuff);
						if (--FileCount >= 0)
							lstrcat(textstr, (LPSTR) ", ");
					}
					SendDlgItemMessage(GetParent(hDlg), CB_ATTACH, WM_SETTEXT, NULL, (LONG) textstr);

					/*** Release and Free memory ***/
					SafeFree(GLOBAL, hGlobalMem);

					return FALSE;

				case FILE_LBNAME:
					/*** If user double-clicks in the file names ***/
					/*** list box execute the FILE_ADD command   ***/
					if (lParam >> 16 != LBN_DBLCLK)
						break;

				case FILE_ADD:
					NullFill((char far *) &FileIndex, 20);
					FileCount = 0;
					/*** Check the number of items in the Selected File Names ***/
					/*** list box. If it's less than MaxFiles, then its OK to ***/
					/*** add the current name from File Name list box.        ***/
					FileCount = SendDlgItemMessage(hDlg, FILE_LBSEL, LB_GETCOUNT, NULL, NULL);
					if (FileCount < MaxFiles && FileCount != LB_ERR) {
						/*** Get the index of the current item ***/
						RetCount = SendDlgItemMessage(hDlg, FILE_LBNAME, LB_GETCURSEL, 0, 0L);
						if (RetCount != LB_ERR) {

							/*** Concatenate the path and file name to strbuff ***/
							NullFill((char far *)&FileBuff, 15);
							GetDlgItemText(hDlg, FILE_NAME, (LPSTR)&FileBuff, 20);
							RetCount = SendDlgItemMessage(hDlg, FILE_LBPATH, LB_GETCURSEL, 0, 0L);
							if (RetCount != LB_ERR) {
								for (i=0, len = 0; i<=RetCount; i++) {
									SendDlgItemMessage(hDlg, FILE_LBPATH, LB_GETTEXT, (WPARAM) i, (LPARAM) &StrBuff + len);
									if (i>0) lstrcat((LPSTR)&StrBuff, "\\");
									len = lstrlen((LPSTR) &StrBuff);
								}
								/*** Convert path to lowercase. ***/
								for (i=0; i<len; i++)
									if ((StrBuff[i]>64) && (StrBuff[i]<91)) StrBuff[i] += 32;
							}
							else
								GetDlgItemText(hDlg, FILE_PATH, (LPSTR)&StrBuff, 200);
							if (StrBuff[lstrlen((LPSTR) &StrBuff)-1] != '\\')
								lstrcat((LPSTR) &StrBuff, "\\");
							lstrcat((LPSTR) &StrBuff, (LPSTR) &FileBuff);

							/*** Store file name in Selected Files list box ***/
							SendMessage(GetDlgItem(hDlg, FILE_LBSEL), LB_ADDSTRING, NULL, (LONG) StrBuff);
						}
					}
					return(TRUE);

				case FILE_REMOVE:
					/*** Remove all highlighted files from the Selected Files list box ***/
					NullFill((char far *)&FileIndex, 20);
					FileCount = 0;
					FileCount = SendMessage(GetDlgItem(hDlg, FILE_LBSEL), LB_GETCOUNT, NULL, NULL);
					if (FileCount > 0 && FileCount != LB_ERR) {
						/*** Get the index to one highlighted file name, delete ***/
						/*** it, then repeat till all files are deleted.        ***/
						while(SendMessage(GetDlgItem(hDlg, FILE_LBSEL), LB_GETSELITEMS, 1, (LONG) &FileIndex) > 0)
							SendMessage(GetDlgItem(hDlg, FILE_LBSEL), LB_DELETESTRING, FileIndex[0], NULL);
					}

					return (TRUE);

			} /** end switch (wParam) **/
			break;
	} /** end switch (message) **/

	return (FALSE);
}

