/****************************************************************************
    FILE:       address.c

    CONTENTS:   AddressBk()     window procedure for Address Book dialog box.

                EditAdBk()      window procedure for Edit Address Book dialog
                                box.

				InAddressBook() checks for the presence of a name in the
								users address book.

				ReadAdBook()    Read the contents of an address book into the
								AdBook structure and fill in the label list box.

				WriteAdBook()   Write the contents of the AdBook structure to
								the INI file or the Associated record.
****************************************************************************/
#include "MyMail.h"
#include "string.h"
#define INCL_STDA_UI
#define INCL_STDA
#include "vnsstda.h"
#undef  INCL_STDA_UI  
#undef  INCL_STDA

AddressBook  AdBook;

BOOL ReadAdBook( HWND );
BOOL WriteAdBook( HWND );
#define INBANYAN        1
#define INHAPPY         2

/******************************************************************************
	FUNCTION: AddressBk()

	PURPOSE:  Dialog Procedure for AddressBox Dialog Box
*******************************************************************************/
BOOL FAR PASCAL _export AddressBk(HWND hDlg, unsigned message, WORD wParam, LONG lParam)
{

	int                             i, j, x, index;
	char                            assoc_rec[1500], oname[1025], name[16];
	static BOOL                     saved;
	static char far         *result;
	STRecord                        rec;


	/*** Send Window Messages to Status Bar ***/
	StatBarCtrlMsg(hDlg, AddressBox, message, wParam, lParam);

	switch (message) {
		case WM_INITDIALOG:                  /* message: initialize dialog box */
			/*** Center the Dialog Window in the HappyMail Main window ***/
			CenterWindow(hDlg, GetDesktopWindow());


			/*** Check the current Address Book option. ***/
			ReadAdBook(hDlg);
			SendDlgItemMessage(hDlg, (Options.happybook) ? ADB_HAPPY : ADB_BANYAN, BM_SETCHECK, 1, 0L);

			/*** Static variable used to determine if changes were made to the     ***/
			/*** address book, necessitating a Save before closing the dialog box. ***/
			saved    = FALSE;

			/*** Store the character pointer passed in through lParam  --  ***/
            /*** this is where we will store any labels that are selected. ***/
			result   = (char far *) lParam;
			return (TRUE);

		case WM_COMMAND:                     /* message: received a command */
			switch(wParam) {
				case IDM_OK:                   /* "OK" button selected  */

					/*** copy the selected labels to the character string that ***/
					/*** was initially passed to the dialog box as lParam      ***/ 
					NullFill(result, 512);
					index = 0;
					index = SendDlgItemMessage(hDlg, ADB_SELECTBOX, LB_GETCOUNT, NULL, NULL);
					for(i=0; i<index; i++) {
						NullFill((LPSTR) &name, 16);
						SendDlgItemMessage(hDlg, ADB_SELECTBOX, LB_GETTEXT, i, (LONG) &name);
						lstrcat(result,(LPSTR) &name);
						if (i+1 != index) {
							lstrcat(result, (LPSTR) "\r\012");
						}
					}

				case IDCANCEL:   
					if (saved) {
						/** Update Associated Record 20 (i.e. the address book) **/

                        /*** If Cancel was pressed -- ask before saving ***/
						if (wParam == IDCANCEL) {
							if (!CallDlg(hDlg, (FARPROC) AskChange, YesNoBox, 8L)) {
								/*** Free memory before exiting ***/
								for (i=0;i<AdBook.num_entries;i++)
									SafeFree(GLOBAL, AdBook.hData[i]);
								EndDialog(hDlg, 1L);   /* then Exit the dialog box   */
								return (TRUE);
							}
						}                       
                        WriteAdBook(hDlg);

					}

					/*** Free memory ***/
					for (i=0;i<AdBook.num_entries;i++)
						SafeFree(GLOBAL, AdBook.hData[i]);

					EndDialog(hDlg, (wParam == IDM_OK) ? 0L : 1L);
					return (TRUE);

				case ADB_LABELBOX:
					/*** Execute the Select button if user    ***/
					/*** double-clicks in the label list box. ***/
					if (lParam >> 16 != LBN_DBLCLK)
						break;
					/*** No break intentional. ***/

				case ADB_SELECT:
					/*** Add the current label to the selected list box unless already there.***/
					i = SendDlgItemMessage(hDlg, ADB_LABELBOX, LB_GETCURSEL, NULL, NULL);
					if (i!=LB_ERR) {
						SendDlgItemMessage(hDlg, ADB_LABELBOX, LB_GETTEXT, i, (LONG) &oname);
						index = SendDlgItemMessage(hDlg, ADB_SELECTBOX, LB_FINDSTRINGEXACT, -1, (LONG) &oname);
						if (index == LB_ERR)
							SendDlgItemMessage(hDlg, ADB_SELECTBOX, LB_ADDSTRING, NULL, (LONG) &oname);
                    }
					return(TRUE);


				case ADB_SELECTBOX:
					/*** Execute the Select button if user    ***/
					/*** double-clicks in the select list box. ***/
					if (lParam >> 16 != LBN_DBLCLK)
						break;
					/*** No break intentional. ***/

				case ADB_REMOVE:
					/*** Remove the selected labelfrom its list box ***/
					i = SendDlgItemMessage(hDlg, ADB_SELECTBOX, LB_GETCURSEL, NULL, NULL);
					if (i!=LB_ERR)
						SendDlgItemMessage(hDlg, ADB_SELECTBOX, LB_DELETESTRING, i, 0L);
					return(TRUE);


				case ADB_BANYAN:
				case ADB_HAPPY:
					/*** User is switching Address Books. ***/

					/*** If selecting the same address book -- skip this code. ***/
					if (!((wParam==ADB_HAPPY && !Options.happybook) || (wParam==ADB_BANYAN && Options.happybook)))
						break;

					if (saved) {
						if (CallDlg(hDlg, (FARPROC) AskChange, YesNoBox, 8L)) {
							WriteAdBook(hDlg);
						}
					}
					saved = FALSE;

					Options.happybook = !(Options.happybook);

					/*** Free memory used in previous Adbook ***/
					for (i=0;i<AdBook.num_entries;i++)
						SafeFree(GLOBAL, AdBook.hData[i]);

					/*** Swap Address Books. ***/
					ReadAdBook(hDlg);
					break;


				case ADB_DELETE:
					/*** Delete an entry by removing it from the label list ***/
					/*** box and removing it from the data structure AdBook ***/
					i = SendDlgItemMessage(hDlg, ADB_LABELBOX, LB_GETCURSEL, NULL, NULL);
					if (i>=0) {
						SendDlgItemMessage(hDlg, ADB_LABELBOX, LB_DELETESTRING, i, 0L);

						/*** Correct the length field (note: 4 = 2 Nulls and 2 chars indicating position)***/
						AdBook.length -= ( lstrlen(AdBook.pData[i]) + strlen((char *)&AdBook.labels[i]) + 4);

						/*** Free the old handle ***/
						SafeFree(GLOBAL, AdBook.hData[i]);

						/*** Delete the label and data from the adbook.data field ***/
						for (; i < AdBook.num_entries; i++) {
							AdBook.hData[i] = AdBook.hData[i+1];
							AdBook.pData[i] = AdBook.pData[i+1];
							lstrcpy((char far *)&AdBook.labels[i], (char far *)&AdBook.labels[i+1]);
						}

						/*** Correct the number of entries field ***/
						AdBook.num_entries -= 1;
						saved = TRUE;
					}
					return (TRUE);

				case ADB_ADD:
				case ADB_EDIT:
					/*** Add or Edit was selected so call ***/
					/*** the Edit Address Book dialog box ***/
					if (wParam == ADB_EDIT) {
						/*** Check to see if an item is selected before continuing ***/
						i = SendDlgItemMessage(hDlg, ADB_LABELBOX, LB_GETCURSEL, NULL, NULL);
						if (i < 0)
							return TRUE;
					}
					else
						i = ADD;

					if (!CallDlg(hDlg, (FARPROC) EditAdBk, EditAB_Box, (LONG) i))
						saved = TRUE;
					return (TRUE);

				case LBN_SELCHANGE:
					if (GetFocus() == GetDlgItem(hDlg, ADB_LABELBOX))
						PostMessage(hDlg, WM_COMMAND, ADB_SELECT, 0);
					else if (GetFocus() == GetDlgItem(hDlg, ADB_SELECTBOX))
						PostMessage(hDlg, WM_COMMAND, ADB_REMOVE, 0);
					break;


			} /** end switch(wParam) **/

	} /** end switch(message) **/
	return (FALSE);
}


/******************************************************************************
	FUNCTION: EditAdBk()

	PURPOSE:  Dialog Procedure for EditAdBookBox Dialog Box
*******************************************************************************/
BOOL FAR PASCAL _export EditAdBk(HWND hDlg, unsigned message, WORD wParam, LONG lParam)
{
	int             i, j, confirmed, index;
	static int      action, limit;
	char            oname[1025], names[1025];
	HVNSD           hVNSD;
	BOOL            bInBook;

	/*** Send Window Messages to Status Bar ***/
	StatBarCtrlMsg(hDlg, EditAB_Box, message, wParam, lParam);

	switch (message) {
		case WM_INITDIALOG:                  /* message: initialize dialog box */
			/*** Center the Dialog Window in the HappyMail Main window ***/
			CenterWindow(hDlg, GetDesktopWindow());

			/*** Check to see if we're Adding or Editing -- then fill ***/
			/*** in the label and name fields and set the text limits ***/
			action = lParam;
			if (lParam < 0) {
				SetDlgItemText(hDlg, ADB_NEWLABEL,  "Untitled");
				SetDlgItemText(hDlg, ADB_NEWNAMES,  "");
				SendDlgItemMessage(hDlg, ADB_NEWLABEL, EM_LIMITTEXT, 15, NULL);
				limit = 1005 - AdBook.length;
			}
			else {
				SetDlgItemText(hDlg, ADB_NEWLABEL, (LPSTR) &(AdBook.labels[lParam]));
				SetDlgItemText(hDlg, ADB_NEWNAMES, AdBook.pData[lParam]);
				SendDlgItemMessage(hDlg, ADB_NEWLABEL, EM_LIMITTEXT, 15, NULL);
				limit = (1009 - AdBook.length) + lstrlen(AdBook.pData[lParam]) + lstrlen((LPSTR)&AdBook.labels[lParam]);

			}
			if (Options.happybook) limit = 500;
			limit = (limit > 500) ? 500 : limit;
			SendDlgItemMessage(hDlg, ADB_NEWNAMES, EM_LIMITTEXT, limit, NULL);
			return (TRUE);

		case WM_COMMAND:
			switch(wParam) {
				case IDM_OK:                   /* "OK" box selected   */
					/*** Check for valid names -- before continuing ***/
					NullFill((LPSTR) &oname, 1024);
					GetDlgItemText(hDlg, ADB_NEWNAMES, (LPSTR) &oname, MAXNAMESLEN);
					if (oname[0] != '\0') {
						if (CheckNames((LPSTR) &oname, GetDlgItem(hDlg, ADB_NEWNAMES), 0, (LPINT) &index) != 0)
							return(TRUE);
					}

					/*** Check for duplicate labels before continuing. ***/
					GetDlgItemText(hDlg, ADB_NEWLABEL, (LPSTR) &oname, 15);
					bInBook = InAddressBook((LPSTR) &oname, NULL, FALSE);
					index = SendDlgItemMessage(GetParent(hDlg), ADB_LABELBOX, LB_FINDSTRINGEXACT, -1, (LONG) &oname);
					if (action == ADD) {
						if ((bInBook==INBANYAN && Options.happybook) || (bInBook==INHAPPY && !Options.happybook) || (index != LB_ERR)) {
							wsprintf((LPSTR)&names, "\"%s\" already exists in\n\none of your Address Books.\n\nPlease choose another label.", (LPSTR)&oname);
							BWCCMessageBox(hDlg, (LPSTR) &names, "HappyMail!", MB_ICONASTERISK);
							return(TRUE);
						}
					}
					else {
						if ((bInBook==INBANYAN && Options.happybook) || (bInBook==INHAPPY && !Options.happybook) || (index != action && index != LB_ERR)) {
							wsprintf((LPSTR)&names, "\"%s\" already exists in\n\none of your Address Books.\n\nPlease choose another label.", (LPSTR)&oname);
							BWCCMessageBox(hDlg, (LPSTR) &names, "HappyMail!", MB_ICONASTERISK);
							return(TRUE);
						}
					}


					/*** Modify the AdBook structure to contain the new label and names ***/
					if (action == ADD) {
						/*** Grab the new label and add it to the label ***/
						/*** list box of the Address Book dialog box    ***/
						GetDlgItemText(hDlg, ADB_NEWLABEL, (LPSTR) &oname, 15);
						i = SendDlgItemMessage(GetParent(hDlg), ADB_LABELBOX, LB_ADDSTRING, 0, (LONG) &oname);

						/*** Add the label and data to the AdBook.data field ***/
						for (j=AdBook.num_entries; j>i; j--) {
							AdBook.hData[j] = AdBook.hData[j-1];
							AdBook.pData[j] = AdBook.pData[j-1];
							lstrcpy((LPSTR)&AdBook.labels[j], (LPSTR)&AdBook.labels[j-1]);
						}
						if ((AdBook.hData[j] = SafeAlloc(GLOBAL, 512L, hWnd)) == NULL) {
							EndDialog(hDlg, 1L);
							return (TRUE);
						}
						AdBook.pData[j] = GlobalLock(AdBook.hData[j]);
						GetDlgItemText(hDlg, ADB_NEWNAMES, (LPSTR) AdBook.pData[j], 500);
						lstrcpy((LPSTR)&AdBook.labels[j], (LPSTR) &oname);

						/*** Correct the number of entries field ***/
						AdBook.num_entries += 1;

						/*** Correct the length field (note: 4 = 2 Nulls and 2 chars indicating position)***/
						AdBook.length += lstrlen(AdBook.pData[i]) + lstrlen((LPSTR)&AdBook.labels[i]) + 4;

					}

					else {
						/*** Replace the old label in the label list box       ***/
						/*** of the Address Book dialog box with the new label ***/
						GetDlgItemText(hDlg, ADB_NEWLABEL, (LPSTR) &oname, 15);
						SendDlgItemMessage(GetParent(hDlg), ADB_LABELBOX, LB_DELETESTRING, action, NULL);
						i = SendDlgItemMessage(GetParent(hDlg), ADB_LABELBOX, LB_ADDSTRING, 0, (LPARAM) &oname);

						/*** Correct the length field ***/
						AdBook.length -= (lstrlen(AdBook.pData[action]) + lstrlen((LPSTR)&AdBook.labels[action]));

						/*** Free the old handle ***/
						SafeFree(GLOBAL, AdBook.hData[action]);

						/*** Add the label and data to the AdBook.data field ***/
						if (action < i) {
							for (j=action; j<i; j++) {
								AdBook.hData[j] = AdBook.hData[j+1];
								AdBook.pData[j] = AdBook.pData[j+1];
								lstrcpy((LPSTR)&AdBook.labels[j], (LPSTR)&AdBook.labels[j+1]);
							}
						}
						else if (action > i) {
							for (j=action; j>i; j--) {
								AdBook.hData[j] = AdBook.hData[j-1];
								AdBook.pData[j] = AdBook.pData[j-1];
								lstrcpy((LPSTR)&AdBook.labels[j], (LPSTR)&AdBook.labels[j-1]);
							}
						}
						if ((AdBook.hData[i] = SafeAlloc(GLOBAL, 512L, hWnd)) == NULL) {
							EndDialog(hDlg, 1L);
							return (TRUE);
						}
						AdBook.pData[i] = GlobalLock(AdBook.hData[i]);
						GetDlgItemText(hDlg, ADB_NEWNAMES, (LPSTR) AdBook.pData[i], 500);
						lstrcpy((LPSTR)&AdBook.labels[i], (LPSTR) &oname);

						/*** Correct the length field ***/
						AdBook.length += lstrlen(AdBook.pData[i]) + strlen((char *)&AdBook.labels[i]);

					}

				case IDCANCEL:
					/*** Return 0 if OK was selected ***/
					EndDialog(hDlg, (wParam == IDM_OK) ? 0L : 1L);
					return (TRUE);

				case IDM_NAMES:
					/*** Popup a Vines STDA dialog box to select names from ***/
					NullFill((LPSTR) &oname, 1024);
					confirmed = FALSE;
					if ( hSTDA ) {
						VnsDlgStdaMultiSelect(hSTDA, hDlg, STDATYPE_USER | STDATYPE_LIST | STDATYPE_NICKNAME, "Send Mail To", "Selected Recipients:", (BOOL far *) &confirmed, (LPSTR) &oname, MAXNAMESLEN);
					}
					else {
						/*** Should disable the names button if no STDA or provide a StreetTalk box. ***/
						BWCCMessageBox(hDlg, "No STDA Service is available.", "HappyMail", MB_ICONHAND);
						return TRUE;
					}

					/*** User exited STDA box by pressing OK, So.. add the ***/
					/*** names from the STDA box to the Names edit control ***/
					if (confirmed) {
						/*** Get a pointer to the Edit window text ***/
						SendDlgItemMessage(hDlg, ADB_NEWNAMES, WM_GETTEXT, MAXNAMESLEN, (long) (LPSTR)&names);

						/*** Copy the text before the selected text ***/
						if (lstrlen((LPSTR)&names) > 0)
							lstrcat((LPSTR)&names, "\r\012");
						lstrcat((LPSTR)&names, (LPSTR)&oname);

						/*** Check the name length before adding. ***/
						if ((unsigned int)lstrlen((LPSTR)&names) > (unsigned int) limit) {
							if (limit == 500)
								lstrcpy((LPSTR)&oname,"\nToo many names selected.\n\nMaximum size is 500 characters.");
							else
								wsprintf((LPSTR)&oname,"You have exceeded the size\n        of your Address Book.\n\nOnly %u characters remain.", (unsigned int) limit);
							BWCCMessageBox(hDlg, (LPSTR) &oname, "HappyMail!", MB_ICONASTERISK);

						}
						else
							SetDlgItemText(hDlg, ADB_NEWNAMES, (LPSTR) &names);
					}
					return TRUE;


			} /** end switch (wParam) **/
	} /** end switch (message) **/

	return (FALSE);
}



/******************************************************************************
	FUNCTION:       InAddressBook()

	PURPOSE:        Check for a label in the addressbook. If the copy parameter
				is TRUE, then copy the names associated with the label to the
				output parameter.
*******************************************************************************/
BOOL InAddressBook(char far *input, char far *output, BOOL copy)
{

	int                     len, i, j, x, index;
	char            assoc_rec[1025], oname[1025], far *seq;
	STRecord        rec;
	CallStatus      uStat;



	len = 1024;
	NullFill((LPSTR) &assoc_rec, 1024);

	/*** Read in the address book -- stored in an Associated StreetTalk record ***/
	uStat = VnsGetStAssocRecord(Add_Book, (char far *) &UserName, USER, (STRecord far *) &rec, (char far *) &oname, (char far *) &assoc_rec, len);
	if (Recover(uStat))
		uStat = VnsGetStAssocRecord(Add_Book, (char far *) &UserName, USER, (STRecord far *) &rec, (char far *) &oname, (char far *) &assoc_rec, len);

	if ((uStat != 1013) && (uStat != 0)) {
		DspMsgCode( hWnd, "Error Reading Banyan Address Book : ", uStat, FALSE);
		return (FALSE);
	}
	seq = (char far *) &assoc_rec;

	/*** Spin through the labels to find one that matches the input parameter ***/
	while (!(seq[0] == 0 && seq[1] == 0) && !(CapCmp(seq+2, input)))
		seq += lstrlen(seq+2) + 3;
	if (!(seq[0] == 0 && seq[1] == 0)) {
		/*** Found a matching label -- So copy it if the copy param was true ***/
		if (copy) {
			/*** Copy the data associated with the current label ***/
			index = (seq[0] * 256) + (unsigned char) seq[1];
			lstrcpy(output, (char far *) &(assoc_rec[index]));
		}
		return INBANYAN;
	}


	/*** Not in Banyan Address Book so look in HappyMail Book. ***/
	len = GetPrivateProfileString("Addresses", input, "", (LPSTR) &oname, 512, INI_FILE);
	if (len != 0) {
		if (copy) {
			for(i=0; oname[i] != '\0'; i++) {
				if (oname[i] == '\036' && oname[i+1] == '\036') {
					oname[i] = '\r';
					oname[i+1] = '\012';
				}
			}
			lstrcpy(output, (char far *) &oname);
		}
		return INHAPPY;
	}


	return FALSE;

}


/******************************************************************************
	FUNCTION:       ReadAdBook()

	PURPOSE:        Reads either the Banyan or HappyMail address book
				and stores the contents to the AdBook structure and
				stores the labels in the appropriate list box.
*******************************************************************************/
BOOL ReadAdBook(HWND hDlg)
{

	int                             i, j, x;
	char                            buff[1600], far *seq, name[1024];
	CallStatus                      uStat;
	STRecord                        rec;

	/*** Update the status Bar.    ***/
	if (hStatus)
		SendMessage(hStatus, STB_STRING, 0, (LONG) "Reading Address Book...");

	SendDlgItemMessage(hDlg, ADB_LABELBOX, LB_RESETCONTENT, 0, NULL);

	if (!Options.happybook) {
		/*** Read in the address book -- stored in an Associated StreetTalk record ***/
		NullFill((LPSTR) &buff, 1024);
		uStat = VnsGetStAssocRecord(Add_Book, (char far *) UserName, USER, (STRecord far *) &rec, (char far *) &name, (char far *) &buff, 1024);
		if (Recover(uStat))
			uStat = VnsGetStAssocRecord(Add_Book, (char far *) UserName, USER, (STRecord far *) &rec, (char far *) &name, (char far *) &buff, 1024);

		if (uStat != 0 && uStat != 1013) {
			DspMsgCode( hWnd, "Error Reading Banyan Address Book : ", uStat, FALSE);
			return (FALSE);
		}

		/*** Spin thru record and store labels into list box. ***/
		/*** Store labels and names into Adbook Structure.    ***/
		seq = (char far *) &buff;

		for(j=0; !(seq[0] == 0 && seq[1] == 0); seq += lstrlen(seq) + 1, j++) {
			seq += 2;
			/*** Send the label string to the label List box ***/
			SendDlgItemMessage(hDlg, ADB_LABELBOX, LB_ADDSTRING, 0, (LONG) seq);
		}
	}
	else {
		GetPrivateProfileString("Addresses", NULL, "", (LPSTR) &buff, 1600, INI_FILE);
		for(seq=buff, j=0; *seq != '\0' && j<100; seq += lstrlen(seq) + 1, j++) {
			/*** Send the label string to the Label List box ***/
			SendDlgItemMessage(hDlg, ADB_LABELBOX, LB_ADDSTRING, 0, (LONG) &seq[0]);
		}
	}
	AdBook.num_entries = j;
	AdBook.length = 0;


	for (j=0; j<AdBook.num_entries; j++) {

		SendDlgItemMessage(hDlg, ADB_LABELBOX, LB_GETTEXT, j, (LONG) &name);

		/*** Copy the label to the data structure AdBook ***/
		lstrcpy((char far *) &AdBook.labels[j], (LPSTR) &name);

		/*** Allocate space for the names attached to the current label ***/
		/*** and store the handle in the structure.                     ***/
		if ((AdBook.hData[j] = SafeAlloc(GLOBAL, 512L, hWnd)) == NULL) {
			return (FALSE);
		}

		/*** Grab a pointer to the names and store it also ***/
		AdBook.pData[j] = (char far *) GlobalLock(AdBook.hData[j]);

		InAddressBook((LPSTR) &name, AdBook.pData[j], TRUE);

		AdBook.length += lstrlen((LPSTR) &name) + lstrlen(AdBook.pData[j]) + 4;
	}

	if (j>0)
		SendDlgItemMessage(hDlg, ADB_LABELBOX, LB_SETCURSEL, 0, NULL);

	return TRUE;


}

/******************************************************************************
	FUNCTION:       WriteAdBook()

	PURPOSE:        Writes the contents of the AdBook structure to
				either the Banyan or HappyMail address book
*******************************************************************************/
BOOL WriteAdBook(HWND hDlg)
{

	int                             i, j, x, index;
	char                            buff[1600];
	CallStatus                      uStat;
	VNS_ST_SESS                     stSess;

	/*** Update the status Bar.    ***/
	if (hStatus)
		SendMessage(hStatus, STB_STRING, 0, (LONG) "Saving Address Book...");

	if (!Options.happybook) {

		/*** Create a new AddressBook from the structure -- AdBook ***/
		/*** Write the labels to a string ***/
		NullFill((LPSTR) &buff, 1025);
		for (i=0, index=2; i<AdBook.num_entries; i++) {
			j=0;
			while (AdBook.labels[i][j] != '\0' && j<16)
				buff[index++] = AdBook.labels[i][j++];
			buff[index++] = '\0';
			buff[index++] = '\0';
			buff[index++] = '\0';

		}

		/*** Add the data and insert the pointers to the data ***/
		for (i=0, x=0; i<AdBook.num_entries; i++) {
			buff[x++] = HIBYTE(index);
			buff[x++] = LOBYTE(index);
			while(buff[x++] != '\0' && x < 1024);
			j=0;
			while (AdBook.pData[i][j] != '\0' && j<500)
				buff[index++] = AdBook.pData[i][j++];
			buff[index++] = '\0';
		}

        /*** Store the new address book to StreetTalk Associated Record ***/
		if (index > 1024) {
			BWCCMessageBox(hDlg, "Address Book is too large.\nMaximum size is 1024 characters.", "HappyMail!", MB_ICONASTERISK);
			return FALSE;
		}
		else {
			uStat = VnsSetStAssocRecord(Add_Book, (char far *) UserName, USER, (char far *) &buff, index);
			if (Recover(uStat))
				uStat = VnsSetStAssocRecord(Add_Book, (char far *) UserName, USER, (char far *) &buff, index);
			if (uStat == 1013) {
				uStat = VnsAddStAssocRecord(Add_Book, (char far *) UserName, USER, (char far *) &buff, index);
			}
			if (uStat != 0) {
				DspMsgCode( hDlg, "Error Storing Banayan Address Book : ", uStat, FALSE);
				return FALSE;
			}
		}
	}
	else {
		/*** Erase old HappyMail AddressBook. ***/
		WritePrivateProfileString("Addresses", NULL, NULL, INI_FILE);

		/*** Write the HappyMail AdBook to the INI file ***/
		/*** Since carriage returns, and linefeeds confuse the INI ***/
		/*** file, they must be modified. So... Change CR-LF       ***/
		/*** pairs to Octal 36 pairs -- before saving to INI file. ***/
		for (i=0; i<AdBook.num_entries; i++) {
			for(j=0; AdBook.pData[i][j] != '\0'; j++) {
				if (AdBook.pData[i][j] == '\r' && AdBook.pData[i][j+1] == '\012') {
					AdBook.pData[i][j] = '\036';
					AdBook.pData[i][j+1] = '\036';
                }
			}
			WritePrivateProfileString("Addresses", AdBook.labels[i], AdBook.pData[i], INI_FILE);
        }

        /*** Flush the INI file to disk. ***/
		WritePrivateProfileString(NULL, NULL, NULL, INI_FILE);
	}
	
	return TRUE;

}
