/*
   Doom Editor Utility, by Brendon Wyber and Raphal Quinet.
   
   If you use any part of this code in one of your programs,
   please make it clear that you borrowed it from here...
   
   NAMES.C - Object name and type routines.
   */

/* the includes */
#include "deu.h"
#include <assert.h>

sector_class *Sector_classes = (sector_class *)NULL;
ld_class *Linedef_classes = (ld_class *)NULL;

typedef struct {
	BCINT type;
	ld_type *linedef;
} ld_arr_type;

ld_arr_type *ld_array;
int nLinedefTypes;

/*
   get the name of an object type
   */

char *GetObjectTypeName( BCINT objtype)
{
    switch (objtype) {
		
    case OBJ_THINGS:
		return "Thing";
    case OBJ_LINEDEFS:
		return "LineDef";
    case OBJ_SIDEDEFS:
		return "SideDef";
    case OBJ_VERTEXES:
		return "Vertex";
    case OBJ_SEGS:
		return "Segment";
    case OBJ_SSECTORS:
		return "SSector";
    case OBJ_NODES:
		return "Node";
    case OBJ_SECTORS:
		return "Sector";
    case OBJ_REJECT:
		return "Reject";
    case OBJ_BLOCKMAP:
		return "Blockmap";
    }
    return "< Bug! >";
}



/*
   what are we editing?
   */

char *GetEditModeName( BCINT objtype)
{
    switch (objtype) {
		
    case OBJ_THINGS:
		return "Things";
    case OBJ_LINEDEFS:
    case OBJ_SIDEDEFS:
		return "LineDefs & SideDefs";
    case OBJ_VERTEXES:
		return "Vertices";
    case OBJ_SEGS:
		return "Segments";
    case OBJ_SSECTORS:
		return "Seg-Sectors";
    case OBJ_NODES:
		return "Nodes";
    case OBJ_SECTORS:
		return "Sectors";
    }
    return "< Bug! >";
}


void index_ld_types()
{
	ld_class *c;
	ld_type *t;
	int n = 0;
	ld_arr_type *p;
	
	for(c = Linedef_classes; c; c = c->next)
		for(t = c->types; t; t = t->next)
			n++;
	/* count linedef types */
	
	nLinedefTypes = n;
	p = ld_array = (ld_arr_type *)malloc(n * sizeof(ld_arr_type));
	assert(ld_array);
	
	for(c = Linedef_classes; c; c = c->next)
		for(t = c->types; t; t = t->next) {
			p->type = t->type;
			p->linedef = t;
			p++;
		}
	
	qsort(ld_array, n, sizeof(ld_arr_type), BCINT_lessp);
}

/*
   get a short (2 + 13 char.) description of the type of a linedef
   */

char *GetLineDefTypeName( BCINT type)
{
	int lowbound = 0, highbound = nLinedefTypes - 1, middle;
	BCINT t;
	
	do {
		middle = (lowbound + highbound) / 2;
		
		t = ld_array[middle].type;
		if(t == type)
			return ld_array[middle].linedef->shortname;
		
		if(t < type)
			lowbound = middle + 1;
		else
			highbound = middle - 1;
	} while(lowbound <= highbound);
	
	return("Unknown LineDef");
}



/*
   get a long description of the type of a linedef
   */

char *GetLineDefTypeLongName( BCINT type)
{
	int lowbound = 0, highbound = nLinedefTypes - 1, middle;
	BCINT t;
	
	do {
		middle = (lowbound + highbound) / 2;
		
		t = ld_array[middle].type;
		if(t == type)
			return ld_array[middle].linedef->longname;
		
		if(t < type)
			lowbound = middle + 1;
		else
			highbound = middle - 1;
	} while(lowbound <= highbound);
	
	return("Unknown LineDef");
}


/*
   get a short description of the flags of a linedef
   */

char *GetLineDefFlagsName( BCINT flags)
{
    static char temp[ 22];
    
    if (flags & 0x0100)
		strcpy( temp, "Ma"); /* Already on the map */
    else
		strcpy( temp, "-");
    if (flags & 0x80)
		strcat( temp, "In"); /* Invisible on the map */
    else
		strcat( temp, "-");
    if (flags & 0x40)
		strcat( temp, "So"); /* Blocks sound */
    else
		strcat( temp, "-");
    if (flags & 0x20)
		strcat( temp, "Se"); /* Secret (normal on the map) */
    else
		strcat( temp, "-");
    if (flags & 0x10)
		strcat( temp, "Lo"); /* Lower texture offset changed */
    else
		strcat( temp, "-");
    if (flags & 0x08)
		strcat( temp, "Up"); /* Upper texture offset changed */
    else
		strcat( temp, "-");
    if (flags & 0x04)
		strcat( temp, "2S"); /* Two-sided */
    else
		strcat( temp, "-");
    if (flags & 0x02)
		strcat( temp, "Mo"); /* Monsters can't cross this line */
    else
		strcat( temp, "-");
    if (flags & 0x01)
		strcat( temp, "Im"); /* Impassible */
    else
		strcat( temp, "-");
	/*
    if (strlen( temp) > 13) {
		temp[13] = '|';
		temp[14] = '\0';
    } */
    return temp;
}



/*
   get a long description of one linedef flag
   */

char *GetLineDefFlagsLongName( BCINT flags)
{
    if (flags & 0x0100)
		return "Already On The Map At Startup";
    if (flags & 0x80)
		return "Invisible On The Map";
    if (flags & 0x40)
		return "Blocks Sound";
    if (flags & 0x20)
		return "Secret (Normal On Map)";
    if (flags & 0x10)
		return "Lower Texture Is \"Unpegged\"";
    if (flags & 0x08)
		return "Upper Texture Is \"Unpegged\"";
    if (flags & 0x04)
		return "Two-Sided (May Be Transparent)";
    if (flags & 0x02)
		return "Monsters Cannot Cross This Line";
    if (flags & 0x01)
		return "Impassible";
    return "UNKNOWN";
}



/*
   get a short (19 char.) description of the type of a sector
   */

char *GetSectorTypeName( BCINT type)
{
	sector_class *c;
	sector_type *t;
	
	for(c = Sector_classes; c; c = c->next)
		for(t = c->types; t; t = t->next)
			if(type == t->type)
				return t->shortname;
	
    return "DO NOT USE!";
}



/*
   get a long description of the type of a sector
   */

char *GetSectorTypeLongName( BCINT type)
{
	sector_class *c;
	sector_type *t;
	
	for(c = Sector_classes; c; c = c->next)
		for(t = c->types; t; t = t->next)
			if(type == t->type)
				return t->longname;
	
    return "DO NOT USE!";
}

/* end of file */

