/* tail and extended head
 * !(c) 1992 D. Champion
 * Tail operates identically to the Unix tail.  Head operates identically
 * to the Unix head, but has the same options as tail.  This is the
 * extended feature of head; you can head relative to the beginning or end,
 * rather than just the beginning.  Unix tail's -r option is not yet
 * implemented, but we're still ahead of every other Amiga head/tail
 * I've seen.  Head and tail read any number of files in one command; if
 * more than one fail is headed or tailed, the output is prefixed with the
 * file name.  If no files are given, input is from standard input.
 *
 * Head and tail are the same program; to distinguish one function from
 * another, the filenames must end in "head" or "tail" as called from DOS.
 * The point here is code efficiency; if you use AmigaDOS 2.0's makelink
 * facility, you can cut down on total program size by 1/3 or more.  If you
 * don't have DOS 2.0, either deal with a slightly larger program sitting
 * on your disk twice, or go buy it.
 *
 * Usage: head [+|-num[l|b|c]] [file1 [file2 [...] ] ] : '-' is head-relative
 *        tail [+|-num[l|b|c]] [file1 [file2 [...] ] ] : '-' is tail-relative
 */

/* Changes by Dan:
 * Put in includes and prototypes to compile properly with SAS/C.
 * Got rid of case sensitive check of program name.
 * Can cope with longer lines now (1024 characters instead of 256).
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

int parse_switch(char *,char *,int,char *);
void do_seg(FILE *,char,int,char,char *);
void report(char *,char *);
void strop(char *);
void chrop(int);
void clean(int);

#define shift		ac--; av++
#define LINELEN		1025
#define BLOCKSIZE	512
#define HEAD		0
#define TAIL		1
#define HEADREL		HEAD
#define TAILREL		TAIL
#define CHARS		0
#define BLOCKS		1
#define LINES		2

char	version[]	= "$VER: Unified Head/Tail 1.1 (12-Jul-95) - (C) D. Champion/Dan Cannon <dan@blender.demon.co.uk>";
char	relflags[]	= "-+";
	/* standard offset table for each of 12 functions */
signed int	offstab[2][3][2]= {0, 1, 0, 1, 0, -1, 1, 0, 1, 0, 1, -2};

char	*myname;
char	buf[80];
char	mode	= HEAD;
int	numfiles;

main(ac, av)
int	ac;
char	**av;
{
int	offset	= 10;
FILE	*fp;
char	*names[]= {"head","tail"};
char	unit	= LINES;
char	rel;

	numfiles = ac - 1;
	if ( !ac ) exit(0);
	myname = av[0];
	if ( !stricmp(&myname[strlen(myname)-4], names[HEAD]) ) 		mode = HEAD;
	else if ( !stricmp(&myname[strlen(myname)-4], names[TAIL]) )	mode = TAIL;
	else report("unknown operation; assuming", names[mode]);

	rel = mode;
	shift;
	while ( ac ) {
		if ( (av[0][0] == '-') || (av[0][0] == '+') ) offset = parse_switch(av[0], &unit, offset, &rel);
		else {
			if ( offset < 0 ) report("bad range spec; skipping file", av[0]);
			else if ( !(fp = fopen(av[0], "r")) ) report("can't open", av[0]);
			else {
				if ( numfiles > 1 ) report("file:", av[0]);
				do_seg(fp, unit, offset, rel, av[0]);
				fclose(fp);
			}
		}
		shift;
	}
	if ( numfiles == 0 ) do_seg(stdin, unit, offset, rel, NULL);
	clean(0);
}

int parse_switch(char *sw, char *unit, int offset, char *rel)
{
char	*p;

	numfiles--;

	if ( sw[0] == relflags[mode] ) *rel = HEADREL;
	if ( sw[0] == relflags[mode^1] ) *rel = TAILREL;

	p=&sw[1];
	while ( isdigit(*p) ) p++;
	switch ( *p ) {
	case	'c':
		*unit=CHARS;
		break;
	case	'b':
		*unit=BLOCKS;
		break;
	case	'l':
		*unit=LINES;
		break;
	case	'\0':
		break;
	default:
		report("undefined unit", p);
		return(-1);
	}
	*p='\0';
	return( atoi(sw+1) );
}

void do_seg(FILE *fp, char unit, int offset, char rel, char *fn)
{
register char	line[LINELEN];
register int	c;
register int	numlines	= 0;
register int	numchars	= 0;


	/* offsets to offset */
	if ( unit == BLOCKS ) offset *= BLOCKSIZE;
	offset += offstab[mode][unit][rel];

	if ( rel == TAILREL ) {
		while ( (c=fgetc(fp)) != EOF ) {
			numchars++;
			if ( c == '\n' ) numlines++;
		}
		fseek(fp, -1, SEEK_END);
		if ( fgetc(fp) != '\n' ) numlines++;
		rewind(fp);
	}

	c = mode - 1;			/* always recycle */
	if ( unit == LINES ) {
		if ( rel == TAILREL) offset = numlines - offset - 1;
		while ( --offset > c ) strop(fgets(line, LINELEN, fp));
	}
	else if ( (unit == BLOCKS) || (unit == CHARS) ) {
		if ( rel == TAILREL) offset = numchars - offset + 1;
		while ( --offset > c ) chrop(fgetc(fp));
	}

	if ( mode == TAIL ) while ( (fgets(line, LINELEN, fp)) ) fputs(line, stdout);

	return;
}

void report(char *str, char *mod)
{
	fprintf(stderr, "\n%s: %s %s\n", myname, str, mod);
}

void strop(char *str)
{
	if ( mode == HEAD ) fputs(str, stdout);
	return;
}

void chrop(int c)
{
	if ( mode == HEAD ) fputc(c, stdout);
	return;
}

void clean(int code)
{
	exit(code);
}
