/*****************************************************************************
*                                                                            *
*  GhostLib Assistant by GhostWare                                           *
*  Copyright (c) 1994  -  all rights reserved                                *
*                                                                            *
*  File : GHOST.S                                                            *
*                                                                            *
*  Description :                                                             *
*  Main macro for GhostLib Assistant                                         *
*                                                                            *
*  Date/time : 08-06-94 21:41                                                *
*                                                                            *
*****************************************************************************/
macro_file GHOST;

#include GHOST.GWH                     // GhostLib Assistant header file

//--------------------------------------------------------------------------//
// Initialization Macro                                                     //
//--------------------------------------------------------------------------//
macro GWInit trans2 {
	str cMessage, cCfgFile = 'GWCFG'+user_id; // Config File name
	int nHandle;

	rm(cCfgFile+'^'+cCfgFile);           // GhostLib Assistant configuration

	// If the minus (-) key is pressed the ExpandMinus macro is run
  macro_to_key( <-> , 'GW_EXPND^ExpandMinus', edit);

	// Set a screensaver event; every 5 seconds the event is triggered
	nHandle = global_int('!GWSaverHandle'); // Get last screen saver handle
	if ( nHandle ) {
		kill_timer_event(nHandle);         // Kill old event first, if available
	}
	nHandle = set_timer_event('GWScreenSaver /T='+str(global_int('!GWSaverDelay')), 90, EDIT);
	set_global_int('!GWSaverHandle', nHandle); // Save the handle

	cMessage = global_str('!GWNameAssistantKey')+' = '+GW_FULLNAME+' menu';
	message_row = 2;                     // Display message on row 2
	rm('Center_Message /M='+cMessage+' /COLOR=79') ;   // White on Red
	message_row = 1;                     // Reset message row to 1
};

//--------------------------------------------------------------------------//
// Center a make_message and display it in a certain color                  //
//--------------------------------------------------------------------------//
macro Center_Message trans2 {
	int nOldColor = Message_Color, nNewColor, iTemp;
	str cMessage ;
	cMessage = parse_str('/M=', mparm_str) ;
	nNewColor = parse_int('/COLOR=', mparm_str) ;
	if (nNewColor > 0){
	Message_Color = nNewColor ;
	};
	// Calculate the number of leading blanks in order to center the message
	iTemp = (message_length-length(cMessage))/2 ;
	cMessage = copy('                           ', 1, iTemp )+cMessage;
	make_message(cMessage);
	Message_Color = nOldColor;           // Reset to default colors
};

//--------------------------------------------------------------------------//
// GWMessageBox - Display a message in a centered box                       //
//--------------------------------------------------------------------------//
macro GWMessageBox trans2 {
	int  nRow, nCol, nWidth ;
	str cMessage = Parse_Str('/M=',MParm_Str),      // Message to be displayed
		cShortMessage = Parse_Str('/SM=',MParm_Str),  // Short message
		cTitle = Parse_Str('/T=',MParm_Str);          // Title

	rm('Center_Message /M='+cShortMessage+' /COLOR=79') ;
	nWidth = Length(cMessage) + 4;
	if(  nWidth < 30  ) {
		nWidth = 30;                       // Minimum width for 'press a key...'
	};
	nCol = (screen_width/2)-((Length(cMessage)/2)+2);
	nRow = (screen_length/2)-3;
	put_box(nCol, nRow, nCol+nWidth+3, nRow+5, 0, m_h_color, cTitle, True);
	write(cMessage, nCol+3, nRow+2, 0, m_h_color);
	write(' Press a key to continue ', (screen_width/2)-12, nRow+4, 0, m_h_color);
	rm('ErrorTune');
	read_key;
	kill_box;
};

//--------------------------------------------------------------------------//
// GWTempBox - Display a message in a box and remove it after x seconds     //
//--------------------------------------------------------------------------//
macro GWTempBox trans2 {
	int  nRow, nCol, nWidth ;
	str cMessage = Parse_Str('/M=',MParm_Str), // Message to be displayed
		cTitle = Parse_Str('/T=',MParm_Str);     // Title
	int nDelay = Parse_Int('/D=',MParm_Str);   // Number of milliseconds delay

	nWidth = Length(cMessage) + 4;
	nCol = (screen_width/2)-((Length(cMessage)/2)+2);
	nRow = (screen_length/2)-3;
	put_box(nCol, nRow, nCol+nWidth+3, nRow+5, 0, m_h_color, cTitle, True);
	write(cMessage, nCol+3, nRow+2, 0, m_h_color);
	delay(nDelay);                       // Wait a number of milliseconds
	kill_box;
};

//--------------------------------------------------------------------------//
// ErrorTune - Play an error tune                                           //
//--------------------------------------------------------------------------//
macro ErrorTune trans2 {
	sound(120);
	delay(70);
	sound(220);
	delay(70);
	sound(320);
	delay(70);
	sound(420);
	delay(70);
	sound(0);
};

//--------------------------------------------------------------------------//
// GWScreenSaver                                                            //
// Manages the screensaver timer events.                                    //
// This macro is only intended to be run by assigning it as a timer         //
// event macro.                                                             //
//                                                                          //
// EXAMPLE: nHandle = set_timer_event('GWScreenSaver /T=1000', 90, EDIT);   //
//                                                                          //
//          /T= : the number of ticks to wait before the screensaver is     //
//                activated                                                 //
//--------------------------------------------------------------------------//
macro GWScreenSaver TRANS2 {
	int iSaver, iDelay=18 * Parse_Int('/T=',MParm_Str),
			iLeft = (last_keypress_time+iDelay) - system_timer;

	if( iLeft < 0 ) {
		iSaver = global_int('!GWScreenSaver');  // The desired screensaver
		if( iSaver == 1 ) {
			// No Screensaver, do nothing
		}
		else if( iSaver == 2 ) {
			// Blank screen only
			RM('_BlankScreen');
		}
		else if( iSaver == 3 ) {
			// Column Drop and Float screensaver
			RM('_DropFloat');
		}
		else if( iSaver == 4 ) {
			// Time Saver screensaver
			RM('_TimeSaver');
		}
		else if( iSaver == 5 ) {
			// WhirlPool screensaver
			RM('_WhirlPool');
		}
		update_status_line;
	}
}

//--------------------------------------------------------------------------//
// _BlankScreen - Just do a clear screen                                    //
//--------------------------------------------------------------------------//
macro _BlankScreen TRANS {
	int nTimeRow = time_row, nStatus = status_row;

	if (global_int('!SCREENSAVERLOADED')) {   // Prevents calling the saver multiple times
		goto exit;
	};

	set_global_int('!SCREENSAVERLOADED', 1);  // Indicate the saver is loaded
	time_row = 0;                             // Switch time/date display off
	status_row = 0;                           // Switch status display off

	save_box(1,1,screen_width,screen_length); // Save the current screen
	clear_screen(0);
	read_key;
	kill_box;                                 // Restore the screen
	set_global_int('!SCREENSAVERLOADED', 0);  // Make the saver available again
	time_row = nTimeRow;                      // Switch time/date display on
	status_row = nStatus;                     // Switch status display on

exit:
}

//--------------------------------------------------------------------------//
// _DropFloat - the Column Drop and Float macro                             //
//--------------------------------------------------------------------------//
macro _DropFloat TRANS {
	int i, j, lExit = 0, nTimeRow = time_row, nUp, nTimer = system_timer, nStatus = status_row;

	if (global_int('!SCREENSAVERLOADED')) {  // Prevents calling the saver multiple times
		goto exit;
	};

	set_global_int('!SCREENSAVERLOADED', 1); // Indicate the saver is loaded
	time_row = 0;                            // Switch time/date display off
	status_row = 0;                          // Switch status display off

	while (!lExit) {
		// Save the current screen
		save_box(1,1,screen_width,screen_length);
		// Scroll the screen regions up or down
		for (i = 1; i <= screen_width; i=i+10) {
			val(nUp, copy(time, 7, 2));      // Get current seconds
			nUp = nUp % 2;                   // Determine if it's even
	 		for (j = 1; j <= screen_length; j++) {
				// Prevent that the screensaver runs to fast, so delay at least 1 tick
				while (nTimer >= system_timer) {
					Delay(1);
				}
				nTimer = system_timer;
				if (nUp) {                     // If even the scroll up
					scroll_box_up(i,1,i+9,screen_length,0);
				}
				else {                         // otherwise scroll down
					scroll_box_dn(i,1,i+9,screen_length,0);
				}
					if (check_key) {             // Check if a key was pressed
					lExit = 1;
					break;
				}
			}
			if (lExit) {
				break;
			}
		}
		kill_box;                          // Restore the screen
	}
	set_global_int('!SCREENSAVERLOADED', 0); // Make the saver available again
	time_row = nTimeRow;                     // Switch time/date display on
	status_row = nStatus;                    // Switch status display on

exit:
}

//--------------------------------------------------------------------------//
// _TimeSaver - the Time Saver macro                                        //
//--------------------------------------------------------------------------//
macro _TimeSaver TRANS {
	int iHorSkip=2, iVertSkip=1, iHor, iVert=1, iRows=4, iCols=27, lExit = 0, iColor=1,
			nTimeRow = time_row, nTimer = system_timer, nStatus = status_row, nExplo = explosions;

	if (global_int('!SCREENSAVERLOADED')) {  // Prevents calling the saver multiple times
		goto exit;
	};

	set_global_int('!SCREENSAVERLOADED', 1); // Indicate the saver is loaded
	time_row = 0;                            // Switch time/date display off
	status_row = 0;                          // Switch status display off
	explosions = 0;                          // Switch explosion off

	val(iHor, copy(time,7,2));           // Start column of time window
	if( iHor > 52 ){                     // Max start column is 52
		iHor = 52;
	}
	while (!lExit) {
		// Save the current screen
		save_box(1,1,screen_width,screen_length);
		clear_screen(0);
		while (!lExit) {
			if (check_key) {                 // Check if a key was pressed
				lExit = 1;
				break;
			}
			// Reverse horizontal movement
			if( ((iHor + iHorSkip + iCols) > screen_width) or ((iHor + iHorSkip) < 1)) {
				iHorSkip = iHorSkip * -1;
				// Change color
				if (iColor < 7) {
					iColor++;
				}
				else {
					iColor = 1;
				}
			}
			// Reverse vertical movement
			if( ((iVert + iVertSkip + iRows) > screen_length) or ((iVert + iVertSkip) < 1)) {
				iVertSkip = iVertSkip * -1;
				// Change color
				if (iColor < 7) {
					iColor++;
				}
				else {
					iColor = 1;
				}
			}
			// New upper left corner of time window
			iHor = iHor + iHorSkip;
			iVert = iVert + iVertSkip;
			put_box(iHor, iVert, iHor+iCols, iVert+iRows,0,(iColor*16)+iColor,'',0);
			write(date+' / '+copy(time, 1, 5), iHor+6, iVert+2,0,(iColor*16)+15);

			// Prevent that the screensaver runs to fast, so delay 1000 milliseconds
			Delay(1000);
			kill_box;
		}
		kill_box;                          // Restore the screen
	}
	set_global_int('!SCREENSAVERLOADED', 0); // Make the saver available again
	time_row = nTimeRow;                     // Switch time/date display on
	status_row = nStatus;                    // Switch status display on
	explosions = nExplo;                     // Switch explosion on

exit:
}

//--------------------------------------------------------------------------//
// _WhirlPool - the WhirlPool macro                                         //
//--------------------------------------------------------------------------//
macro _WhirlPool TRANS {
	int iHorSkip=2, iVertSkip=1, iTop, iBottom, iLeft, iRight, lExit = 0, j,
			nTimeRow = time_row, nTimer = system_timer, nStatus = status_row, nExplo = explosions;

	if (global_int('!SCREENSAVERLOADED')) {  // Prevents calling the saver multiple times
		goto exit;
	};

	set_global_int('!SCREENSAVERLOADED', 1); // Indicate the saver is loaded
	time_row = 0;                            // Switch time/date display off
	status_row = 0;                          // Switch status display off
	explosions = 0;                          // Switch explosion off

	while (!lExit) {
		iTop=1; iBottom=screen_length; iLeft=1; iRight=screen_width-1 ;
		// Save the current screen
		save_box(1,1,screen_width,screen_length);
		while (!lExit) {

			for (j = iLeft; j <= iRight; j=j+iHorSkip) {
				write('  ', j, iTop, 0, 0);
				Delay(20);
			}
			iTop = iTop + iVertSkip ;

			if (check_key) {                 // Check if a key was pressed
				lExit = 1;
				break;
			}

			for (j = iTop; j <= iBottom; j=j+iVertSkip) {
				write('  ', iRight, j, 0, 0);
				Delay(20);
			}
			iRight = iRight - iHorSkip ;

			if (check_key) {                 // Check if a key was pressed
				lExit = 1;
				break;
			}

			for (j = iRight; j >= iLeft; j=j-iHorSkip) {
				write('  ', j, iBottom, 0, 0);
				Delay(20);
			}
			iBottom = iBottom - iVertSkip ;

			if (check_key) {                 // Check if a key was pressed
				lExit = 1;
				break;
			}

			for (j = iBottom; j >= iTop; j=j-iVertSkip) {
				write('  ', iLeft, j, 0, 0);
				Delay(20);
			}
			iLeft = iLeft + iHorSkip ;

			if (check_key) {                 // Check if a key was pressed
				lExit = 1;
				break;
			}
			if ((iTop>iBottom) or (iLeft>iRight)) {
				break ;
			}
		}
		kill_box;                          // Restore the screen
	}
	set_global_int('!SCREENSAVERLOADED', 0); // Make the saver available again
	time_row = nTimeRow;                     // Switch time/date display on
	status_row = nStatus;                    // Switch status display on
	explosions = nExplo;                     // Switch explosion on

exit:
}

//--------------------------------------------------------------------------//
// String2CHR - Make a CHR call for a string                                //
//--------------------------------------------------------------------------//
macro String2CHR DUMP {                // Macro is dumped out of memory
	str cText  = Parse_Str('/TEXT=',MParm_Str), cChar, cNewText='' ;
	int iLen = svl(cText),               // Length of the text
			i, iInsert;

	if (!iLen) {
		// cText is empty so ask for a string
		set_global_str('GWISTR_1', '');
		set_global_str('GWIPARM_1','/T=Enter some text :/C=2/W=40/L=2/TP=0/QK=0');
		rm('UserIn^Data_In /X=5/W=64/PRE=GW/S=1/#=1/T= String to CHR call ');
		cText = remove_space(global_str('GWISTR_1'));
		iLen = svl(cText);                 // Length of the text
		set_global_str('GWISTR_1', '');
		set_global_str('GWIPARM_1','');
	}

	// If ESCape was pressed then exit
	if (return_int) {
		for (i = 1; i <= iLen; i++) {
			cChar = str_char(cText, i);        // Get a character
			cNewText += 'CHR('+ str(ascii(cChar)) +')';
			if (i<iLen) {
				cNewText += '+';
			}
		}
		// Write the CHR call to the current window
		eol; iInsert=insert_mode; insert_mode=1; cr; home;
		text('// CHR() call of text '''+cText+''''); cr;
		text(cNewText);
		insert_mode=iInsert; home;
	}
}

//--------------------------------------------------------------------------//
// FunConvert - 'Fun convert' a string                                      //
//--------------------------------------------------------------------------//
macro FunConvert DUMP {                // Macro is dumped out of memory
	    // Table with characters to convert
	str sTable = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ',
			// 'fun conversion' table
			sFun   = 'G׭KMP$W*[g׳kmp$wx',
			// Text to 'fun convert'
			cText  = Parse_Str('/TEXT=',MParm_Str),
			cChar;
	int iLen = svl(cText),               // Length of the text
			i, iFind;

	if (!iLen) {
		// cText is empty so ask for a string
		set_global_str('GWISTR_1', '');
		set_global_str('GWIPARM_1','/T=Enter some text :/C=2/W=40/L=2/TP=0/QK=0');
		rm('UserIn^Data_In /X=5/W=64/PRE=GW/S=1/#=1/T= ''Fun-convert'' a string ');
		cText = remove_space(global_str('GWISTR_1'));
		iLen = svl(cText);                 // Length of the text
		set_global_str('GWISTR_1', '');
		set_global_str('GWIPARM_1','');
	}

	// If ESCape was pressed then do not 'fun convert'
	if (return_int) {
		for (i = 1; i <= iLen; i++) {
			cChar = str_char(cText, i);        // Get a character
			iFind = xpos(cChar, sTable, 1);    // Find the character in sTable
			if (iFind) {
				// Character found, stuff 'fun character' in text
				// Delete the character to 'fun convert'
				cText = str_del(cText, i, 1);
				// Insert the 'fun character'
				cText = str_ins(str_char(sFun, iFind), cText, i);
			}
		}
		// Write the 'fun-converted' string to the current window
		eol; iFind=insert_mode; insert_mode=1; cr; home; text(cText); insert_mode=iFind;
	}
}
