/*
 * File......: GTCLOCK.PRG
 * Author....: Brian Dukes
 * BBS.......: The Dark Knight Returns
 * Net/Node..: 050/069
 * User Name.: Brian Dukes
 * Date......: $Date$
 * Revision..: $Revision$
 * Log file..: $Logfile$
 *
 * This is an original work by Brian Dukes and is placed in the public
 * domain.
 *
 * Modification history:
 * ---------------------
 *
 * $Log$
 *
 */

/*  $DOC$
 *  $FUNCNAME$
 *      GT_CLOCK()
 *  $CATEGORY$
 *      Screen Saver
 *  $ONELINER$
 *      Use a BIG Clock as a screen saver
 *  $SYNTAX$
 *      GT_Clock([<nDelay>],[<cColor>],[<lCLS>],[<nTop>],[<nLeft>], ;
 *               [<nXMove>],[<nYMove>]) --> NIL
 *  $ARGUMENTS$
 *      <nDelay> is an optional numeric parameter that is the delay to
 *      use when updating the display. This is in 1/18ths of a second.
 *      If not supplied this parameter defaults to 1.
 *
 *      <cColor> is an optional colour string.  If not supplied this
 *      defaults to "G+/N"  (Bright Green on Black)
 *
 *      <lCLS> is an optional logical parameter that instructs the saver
 *      to clear the screen first, if the value is .T. then the screen is
 *      cleared, if .F. then the clock will be displayed over the current
 *      screen contents.  This defaults to .T.
 *
 *      <nTop> is an optional numeric parameter that tells the save where
 *      to start displaying the clock.  (Top Line)
 *
 *      <nLeft> is an optional numeric parameter that tells the saver where
 *      to start displaying the clock.  (1st Column)
 *
 *      <nXMove> is an optional numeric parameter that tells the saver how
 *      many columns to move on each increment.  If this is set to 0, and
 *      nYMove is set to 1, the clock will just bounce up and down. This
 *      defaults to 1.
 *
 *      <nYMove> is an optional numeric parameter that tells the saver how
 *      many lines to move on each increment.  If this is set to 0, and
 *      nXmove is set to 1, the clock will just bounce right to left. This
 *      defaults to 1.
 *
 *
 *  $RETURNS$
 *      Nothing.
 *
 *  $DESCRIPTION$
 *      GT_Clock() is a screen saver function.  It has the ability of being
 *      altered a great deal by the passed parameters.  The screen saver
 *      will terminate if a GT_Interrupt() codeblock has been sets up and
 *      evaluates TRUE; or if a default codeblock of any key pressed becomes
 *      true.
 *
 *      This function calls GT_BigTime()   to display the clock
 *                          GT_Interrupt() to interrogate completion
 *                          GT_SaveScr()   to save the screen
 *                          GT_RestScr()   to restore the screen
 *
 *  $EXAMPLES$
 *      // Call the screen saver with a delay of 1 second, displaying the
 *      // clock over the current screen.  The Clock will be displayed at
 *      // line 10, and will bounce from right to left.
 *
 *      GT_Clock(1,,.f.,10,,1,0)
 *  $END$
 */

#include "gt_lib.ch"

function GT_Clock(nDelay,cColor,lCLS,nTop,nLeft,nXMove,nYMove)
   local cScrCap    := ""                ,;
         nBottom    := 3                 ,;
         nRight     := 31                ,;
         nTimeout   := 0                 ,;
         hOldScreen := GT_SaveScr()      ,;
         nOldCursor := setcursor(SC_NONE),;
         cOldColour := setcolor("w/n")   ,;
         bQuit      := GT_Interrupt()

   default cColor to "G+/N"
   default lCLS   to .t.
   default nDelay to 1
   default bQuit  to {|| inkey() != 0 }
   default nTop   to 0
   default nLeft  to 0
   default nXMove to 1
   default nYMove to 1

   nBottom := nTop + 3        // Four lines deep
   nRight  := nLeft + 31      // 32 Chars wide (HH:MM:SS * 4chars)


   if lCLS
      clear screen
   endif

   do while !eval(bQuit)
      nTimeOut := seconds()
      cScrCap := GT_SaveScr(nTop,nLeft,nBottom,nRight)
      GT_BigTime(nTop,nLeft,cColor)

      if nXMove == 1
         nXMove := if(nRight == maxcol(),-1,nXMove)
      else
         nXMove := if(nLeft == 0,1,nXMove)
      endif
      if nYMove == 1
         nYMove := if(nBottom == maxrow(),-1,nYMove)
      else
         nYMove := if(nTop == 0,1,nYMove)
      endif
      nTop    += nYMove
      nLeft   += nXMove
      nBottom += nYMove
      nRight  += nXMove

      do while Seconds() - nTimeOut < nDelay ; enddo    // DELAY
      GT_RestScr(cScrCap)
   enddo

   setcolor(cOldColour)
   GT_RestScr(hOldScreen)
   setcursor(nOldCursor)

return NIL
