/*
 * File......: FLOCATE.PRG
 * Author....: Dave Pearson
 * BBS.......: The Dark Knight Returns
 * Net/Node..: 050/069
 * User Name.: Dave Pearson
 * Date......: 13/04/93
 * Revision..: 1.0
 *
 * This is an original work by Dave Pearson and is placed in the public
 * domain.
 *
 * Modification history:
 * ---------------------
 *
 * $Log$
 *
 */

#include "gt_lib.ch"

#define MAX_BUFFER_SIZE 8192
#define ERR_READ_ERROR  -1
#define ERR_NOT_FOUND   -2

/*  $DOC$
 *  $FUNCNAME$
 *      GT_FLOCATE()
 *  $CATEGORY$
 *      File I/O
 *  $ONELINER$
 *      Find a string in a low level file.
 *  $SYNTAX$
 *      GT_FLocate(<nFileHandle>,<cString>,[<lCaseSensitive>],;
 *                 [<nStartPos>]) --> nLocation
 *  $ARGUMENTS$
 *      <nFileHandle> is the handle of a file that has been opened or
 *      created with one of Clipper's low level file functions.
 *
 *      <cString> is the string you are looking for.
 *
 *      <lCaseSensitive> is an optional logical value to describe the
 *      type of search. If True (.T.) the search will be a case sensitive
 *      search. The default is False (.F.)
 *
 *      <nStartPos> is an optional numeric value to describe the position
 *      at which the search will start. The default is 0.
 *  $RETURNS$
 *      The byte location of the string if a match is found. -1 if there
 *      was an error reading the file. -2 if a match could not be found.
 *  $DESCRIPTION$
 *      GT_FLocate() can be used to find a text string in a file that has
 *      been opened with one of Clipper's low level file functions.
 *
 *      Note that as well as returning the location of the hit GT_Flocate()
 *      will also place the file pointer at the first byte of the match. If
 *      no match was found the file pointer will be in it's original
 *      location.
 *  $EXAMPLES$
 *      // The following example opens a file and prints every byte location
 *      // of the target word.
 *
 *      local nFile := fopen("LIFE.42")
 *      local nHit  := 0
 *      do while nHit >= 0
 *         if (nHit := GT_FLocate(nFile,"Meaning")) >= 0
 *            ? nHit
 *         endif
 *      enddo
 *      fclose(nFile)
 *  $SEEALSO$
 *      GT_FTELL() GT_FSIZE()
 *  $END$
 */

function GT_FLocate(nFileHandle,cString,lCaseSensitive,nStartPos)
local nOffset     := ERR_NOT_FOUND,;
      nOldOffset  := 0            ,;
      nBufferSize := 0            ,;
      cBuffer     := NULL         ,;
      cTextBuff   := NULL         ,;
      nPosition   := 0            ,;
      nFileLength := 0            ,;
      nIsInBuff   := 0
default lCaseSensitive to FALSE
default nStartPos      to 0
if valtype(nFileHandle) == TYPE_NUMERIC .and. valtype(cString) == TYPE_CHAR
   nFileLength := GT_FSize(nFileHandle)
   if nStartPos < nFileLength
      nOldOffset  := GT_FTell(nFileHandle)
      nBufferSize := min(MAX_BUFFER_SIZE,nFileLength-nStartPos)
      cBuffer     := space(nBufferSize)
      cTextBuff   := space(len(cString))
      nPosition   := fseek(nFileHandle,nStartPos,FS_SET)
      do while nPosition < nFileLength
         if fread(nFileHandle,@cBuffer,nBufferSize) == F_ERROR
            nOffset := ERR_READ_ERROR
            exit
         endif
         cTextBuff := right(cTextBuff,len(cString))+if(!lCaseSensitive,upper(cBuffer),cBuffer)
         if (nIsInBuff := at(if(!lCaseSensitive,upper(cString),cString),cTextBuff)) != 0
            nOffset := nPosition + nIsInBuff - len(cString) - 1
            exit
         endif
         nPosition := GT_FTell(nFileHandle)
      enddo
      if nOffset < 0
         fseek(nFileHandle,nOldOffset,FS_SET)
      else
         fseek(nFileHandle,nOffset,FS_SET)
      endif
   endif
endif
return(nOffset)
