/*
 * File......: BITFLAGS.C
 * Author....: Dave Pearson
 * BBS.......: The Dark Knight Returns
 * Net/Node..: 050/069
 * User Name.: Dave Pearson
 * Date......: 31/03/93
 * Revision..: 1.0
 *
 * This is an original work by Dave Pearson and is placed in the public
 * domain.
 *
 * Modification history:
 * ---------------------
 *
 * $Log$
 *
 */

// NOTE: This code has been written for and compiled with Borland C++
//       Version 3.1
//

#include <extend.h>

#define _GT_MAX(x,y)    (x > y ? x : y)

/*  $DOC$
 *  $FUNCNAME$
 *      GT_NEWFLAG()
 *  $CATEGORY$
 *      General
 *  $ONELINER$
 *      Create a new bit flag string.
 *  $SYNTAX$
 *      GT_NewFlag(<nFlagCount>) --> cFlagString
 *  $ARGUMENTS$
 *      <nFlagCount> is the number of flags you wish to store.
 *  $RETURNS$
 *      A string to hold the bit flags. All flags are set to FALSE.
 *  $DESCRIPTION$
 *      GT_NewFlag() is used to construct a bit flag string. The bit flag
 *      functions can be used for storing a large number of logical values
 *      is a small space.
 *
 *      To create a bit flag string you need to pass GT_NewFlag() a value
 *      that is equal to or greater than the number of flags required (you
 *      may want to allow for future expansion). Each character in the
 *      string returned from GT_NewFlag() will hold 8 logical values.
 *  $EXAMPLES$
 *      cFlags := GT_NewFlag(20)   // Create a bit flag string for 20
 *                                 // logical values.
 *  $SEEALSO$
 *      GT_SETFLAG() GT_CLRFLAG() GT_ISFLAG()
 *  $END$
 */

CLIPPER GT_NewFlag(void)
{
        char     *FlagString = NULL;
        unsigned ByteCount   = 0;
        unsigned FlagCount   = 1;
        int      Byte;

        if (PCOUNT)
        {
                if (ISNUM(1))
                {
                        FlagCount = (unsigned) _parni(1);
                }
        }
        if (FlagCount > 0)
        {
                ByteCount = (unsigned)((FlagCount / 8) + 1);
                if (!(FlagCount % 8))
                {
                        --ByteCount;
                }
                FlagString = _xgrab(ByteCount);
                for (Byte = 0; Byte < ByteCount; Byte++)
                {
                        FlagString[Byte] = 0;
                }
                _retclen(FlagString,ByteCount);
                _xfree(FlagString);
                _xunlock();
        }
        else
        {
                _retc("");
        }
}

/*  $DOC$
 *  $FUNCNAME$
 *      GT_SETFLAG()
 *  $CATEGORY$
 *      General
 *  $ONELINER$
 *      Set a number of flags to TRUE in a bit flag string.
 *  $SYNTAX$
 *      GT_SetFlag(<cFlagString>,[<nStart>],[<nEnd>]) --> cFlagString
 *  $ARGUMENTS$
 *      <cFlagString> is a bit flag string created with GT_NewFlag()
 *
 *      <nStart> is the starting flag. This is an optional numeric value.
 *      If not supplied it defaults to 1.
 *
 *      <nEnd> is the ending flag. This is an optional numeric value. If
 *      not supplied it defaults to <nStart>.
 *  $RETURNS$
 *      The bit map string with the new flag settings.
 *  $DESCRIPTION$
 *      GT_SetFlag() is used to turn flags within the flag string on.
 *  $EXAMPLES$
 *      cFlags := GT_NewFlag(20)   // Create a bit flag string for 20
 *                                 // logical values.
 *
 *      // Now set flags 10 to 15 to true.
 *
 *      cFlags := GT_SetFlag(cFlags,10,15)
 *
 *      // And set flag 18 to true.
 *
 *      cFlags := GT_SetFlag(cFlags,18)
 *
 *      // And set flag 1 to true.
 *
 *      cFlags := GT_SetFlag(cFlags)
 *  $SEEALSO$
 *      GT_NEWFLAG() GT_CLRFLAG() GT_ISFLAG()
 *  $END$
 */

CLIPPER GT_SetFlag(void)
{
        char     *FlagString = NULL;
        unsigned StartBit    = 1;
        unsigned EndBit      = 1;
        unsigned BitCount;
        unsigned BitPointer;
        unsigned BytePointer;

        if (PCOUNT && ISCHAR(1))
        {
                FlagString = _parc(1);
                if (PCOUNT >= 2 && ISNUM(2))
                {
                        StartBit = _parni(2);
                }
                if (PCOUNT >= 3 && ISNUM(3))
                {
                        EndBit = _parni(3);
                }
                EndBit = _GT_MAX(StartBit,EndBit);
                if (StartBit > 0 && EndBit <= (_parclen(1) * 8))
                {
                        for (BitCount = StartBit; BitCount <= EndBit; BitCount++)
                        {
                                BitPointer  = BitCount % 8;
                                BytePointer = (unsigned) (BitCount / 8);
                                if (!BitPointer)
                                {
                                        BitPointer = 8;
                                        --BytePointer;
                                }
                                FlagString[BytePointer] |= 1 << (BitPointer - 1);
                        }
                }
                _retclen(FlagString,_parclen(1));
                _xunlock();
        }
        else
        {
                _retc("");
        }
}

/*  $DOC$
 *  $FUNCNAME$
 *      GT_CLRFLAG()
 *  $CATEGORY$
 *      General
 *  $ONELINER$
 *      Set a number of flags to FALSE in a bit flag string.
 *  $SYNTAX$
 *      GT_ClrFlag(<cFlagString>,[<nStart>],[<nEnd>]) --> cFlagString
 *  $ARGUMENTS$
 *      <cFlagString> is a bit flag string created with GT_NewFlag()
 *
 *      <nStart> is the starting flag. This is an optional numeric value.
 *      If not supplied it defaults to 1.
 *
 *      <nEnd> is the ending flag. This is an optional numeric value. If
 *      not supplied it defaults to <nStart>.
 *  $RETURNS$
 *      The bit map string with the new flag settings.
 *  $DESCRIPTION$
 *      GT_ClrFlag() is used to turn flags within the flag string off.
 *  $EXAMPLES$
 *      cFlags := GT_NewFlag(20)   // Create a bit flag string for 20
 *                                 // logical values.
 *
 *      // Now, turn them all on.
 *
 *      cFlags := GT_SetFlag(cFlags,1,20)
 *
 *      // Now set flags 10 to 15 to false.
 *
 *      cFlags := GT_ClrFlag(cFlags,10,15)
 *
 *      // And set flag 18 to false.
 *
 *      cFlags := GT_ClrFlag(cFlags,18)
 *
 *      // And set flag 1 to false.
 *
 *      cFlags := GT_ClrFlag(cFlags)
 *  $SEEALSO$
 *      GT_NEWFLAG() GT_SETFLAG() GT_ISFLAG()
 *  $END$
 */

CLIPPER GT_ClrFlag(void)
{
        char     *FlagString = NULL;
        unsigned StartBit    = 1;
        unsigned EndBit      = 1;
        unsigned BitCount;
        unsigned BitPointer;
        unsigned BytePointer;

        if (PCOUNT && ISCHAR(1))
        {
                FlagString = _parc(1);
                if (PCOUNT >= 2 && ISNUM(2))
                {
                        StartBit = _parni(2);
                }
                if (PCOUNT >= 3 && ISNUM(3))
                {
                        EndBit = _parni(3);
                }
                EndBit = _GT_MAX(StartBit,EndBit);
                if (StartBit > 0 && EndBit <= (_parclen(1) * 8))
                {
                        for (BitCount = StartBit; BitCount <= EndBit; BitCount++)
                        {
                                BitPointer  = BitCount % 8;
                                BytePointer = (unsigned) (BitCount / 8);
                                if (!BitPointer)
                                {
                                        BitPointer = 8;
                                        --BytePointer;
                                }
                                FlagString[BytePointer] &= 0xff - (1 << (BitPointer - 1));
                        }
                }
                _retclen(FlagString,_parclen(1));
                _xunlock();
        }
        else
        {
                _retc("");
        }
}

/*  $DOC$
 *  $FUNCNAME$
 *      GT_ISFLAG()
 *  $CATEGORY$
 *      General
 *  $ONELINER$
 *      Test the setting of a flag in a bit flag string.
 *  $SYNTAX$
 *      GT_IsFlag(<cFlagString>,[<nFlag>]) --> lSetting
 *  $ARGUMENTS$
 *      <cFlagString> is a bit flag string created with GT_NewFlag()
 *
 *      <nFlag> is the flag to be tested.
 *  $RETURNS$
 *      A boolean value, TRUE if the flag is on, FALSE if it's off.
 *  $DESCRIPTION$
 *      GT_IsFlag() is used to test the state of a flag with a bit flag
 *      string.
 *  $EXAMPLES$
 *
 *      // Print the setting of the flags in a flag string called ``cDave''
 *
 *      for nFlag := 1 to (len(cDave)*8)
 *         ? "Flag number ",nFlag," == ",GT_IsFlag(cDave,nFlag)
 *      next
 *  $SEEALSO$
 *      GT_NEWFLAG() GT_SETFLAG() GT_CLRFLAG()
 *  $END$
 */

CLIPPER GT_IsFlag(void)
{
        Boolean         FlagStatus = FALSE;
        unsigned        Bit        = 1;
        unsigned        BitPointer;
        unsigned        BytePointer;
        char            *FlagString;

        if (PCOUNT && ISCHAR(1))
        {
                FlagString = _parc(1);
                if (PCOUNT >= 2 && ISNUM(2))
                {
                        Bit = _parni(2);
                }
                if (Bit > 0 && Bit <= (_parclen(1) * 8))
                {
                        BitPointer  = Bit % 8;
                        BytePointer = (unsigned) (Bit / 8);
                        if (!BitPointer)
                        {
                                BitPointer = 8;
                                --BytePointer;
                        }
                        FlagStatus = FlagString[BytePointer] & (1 << (BitPointer - 1));
                }
        }
        _retl(FlagStatus);
}
