/* Copyright (C) 1989, 1990 Aladdin Enterprises.  All rights reserved.
   Distributed by Free Software Foundation, Inc.
   
   This file is not part (yet) of Ghostscript 2.0.
   
   Ghostscript is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
   to anyone for the consequences of using it or for whether it serves any
   particular purpose or works at all, unless he says so in writing.  Refer
   to the Ghostscript General Public License for full details.
   
   Everyone is granted permission to copy, modify and redistribute
   Ghostscript, but only under the conditions described in the Ghostscript
   General Public License.  A copy of this license is supposed to have been
   given to you along with Ghostscript so you can know your rights and
   responsibilities.  It should be in a file named COPYING.  Among other
   things, the copyright notice and this notice must be preserved on all
   copies.  */

/* gdevpjet.c */
/* Printer driver for GhostScript, locally modified for the paintjet. */
/* Page bitmap is stored in a temporary file on disk */
/* Only a part of page bitmap is store in memory     */
/* We use virtual memory between disk and memory     */
#define DRIVER				/* see gdevprn.h */
#include "gdevpjet.h"

private long mem_blk=0; /* number of block in memory */

/* Macro for casting gx_device argument */
#define paintjet_dev ((paintjet_device *)dev)

/* Open the printer device */
int
paintjet_open(gx_device *dev)
{
  int number;

  /* specific initialization of the printer device */
  paintjet_dev->is_open = 1 ;	/* TRUE */

#ifdef __MSDOS__
  
  /* create temporary file and close it*/
  if((paintjet_dev->output_file = fopen("{prntmp}\0","w")) == NULL)
    {
      /* try again */
      if((paintjet_dev->output_file = fopen("{prntmp}\0","w")) == NULL)
	{
	  fprintf(stderr,"can`t open temporary file for printer device\n");
	  fprintf(stderr,"make sure there is enought room on disk to store ghostscript page\n");
	  exit(1);
	}
    }
  fclose(paintjet_dev->output_file);
#else
  if (open_printer_stream_as_pipe)
    {
      if ((paintjet_dev->output_file =
	   popen (prn_output_stream_name, "w")) == NULL)
	{
	  perror("gs (paintjet_open)") ;
	  exit (1) ;
	}
    }
  else
    {
      /* assumes that the contents of prn_output_stream_name is a valid filename */
      paintjet_dev->output_file = fopen (prn_output_stream_name,"w") ;
    }
  if (paintjet_dev->output_file == NULL)
    {
      fprintf(stderr,
	      "gs (paintjet_open): Can`t open printer as file %s\n",
	      prn_output_stream_name);
      exit(1);
    }
  else if (! open_printer_stream_as_pipe)
    {
      fprintf (stdout,
	       "\nPrinting into file %s. \n",
	       prn_output_stream_name) ;
    }
#endif				

  /* Create virtual memory block */
  paintjet_dev->blk = (vm_blk *)malloc(paintjet_dev->max_blk * sizeof(vm_blk));
  for(number=0; number<paintjet_dev->max_blk; number++)
    {
      paintjet_dev->blk[number].status = FREE;
      paintjet_dev->blk[number].mem_ptr = UNKNOWN;
    }
#ifdef __MSDOS__
  printf("Printing in memory: ---- Be patient ----\n");
#endif
  return 0;
}


/* Close the printer device */
int
paintjet_close(gx_device *dev)
{
  int number;

#ifdef __MSDOS__
  /* Remove temporary files */
  remove("{prntmp}\0");
#else
  if (open_printer_stream_as_pipe)
    pclose(paintjet_dev->output_file) ;
  else
    fclose(paintjet_dev->output_file) ;
#endif


  for(number=0; number<paintjet_dev->max_blk; number++)
    {
      /* free memory use by block to store page description*/
      if(paintjet_dev->blk[number].status == MEMORY)
	free((void *)paintjet_dev->blk[number].mem_ptr);
    }
  /* free virtual memory block */
  free((void *)paintjet_dev->blk);

  /* closing of the paintjet device */
  return 0 ;
}

#define BLACK		0
#define RED		1
#define GREEN		2
#define YELLOW		3
#define BLUE		4
#define MAGENTA		5
#define CYAN		6
#define WHITE_COLOR	7

/* r, g, b are between 0 and 1. */
private gx_color_index rgb_color_PaintJet[2][2][2] =
{
  {
    { BLACK, BLUE },
    { GREEN, CYAN }
  },
  {
    { RED, MAGENTA },
    { YELLOW, WHITE_COLOR }
  }
} ;


gx_color_index 
paintjet_map_rgb_color (gx_device *dev, unsigned short r, unsigned short g, unsigned short b)
{
  if(dev->has_color)  /* 8 colors device */ /* local */
    {
      return rgb_color_PaintJet[r][g][b];
    }
  else  /* BLACK and WHITE device */
    {
      if(max(max(r,g),b) > 0)
	return WHITE; /* white */
      else
	return BLACK; /* black */
    }
}

/* Map a color code to r-g-b. This is NOT algorithmic. (local) */
int paintjet_map_color_rgb(dev, color, prgb)
     gx_device *dev;
     gx_color_index color;
     ushort *prgb;
{
  switch (color)
    {
    case BLACK :
      prgb[0] = 0;
      prgb[1] = 0;
      prgb[2] = 0;
      break ;
    case RED :
      prgb[0] = 1;
      prgb[1] = 0;
      prgb[2] = 0;
      break ;
    case GREEN :
      prgb[0] = 0;
      prgb[1] = 1;
      prgb[2] = 0;
      break ;
    case YELLOW :
      prgb[0] = 1;
      prgb[1] = 1;
      prgb[2] = 0;
      break ;
    case BLUE :
      prgb[0] = 0;
      prgb[1] = 0;
      prgb[2] = 1;
      break ;
    case MAGENTA :
      prgb[0] = 1;
      prgb[1] = 0;
      prgb[2] = 1;
      break ;
    case CYAN :
      prgb[0] = 0;
      prgb[1] = 1;
      prgb[2] = 1;
      break ;
    case WHITE_COLOR :
      prgb[0] = 1;
      prgb[1] = 1;
      prgb[2] = 1;
      break ;
    }
  return (0) ;
}



/* Print a page description to the printer */
/* if operator is showpage you erase the page after printing */
int paintjet_output_page(dev)
gx_device *dev;
{
#ifdef __MSDOS__
  FILE *paintjet_stream = fopen("prn","w");
  if(paintjet_stream == NULL)
    {
      /* try again */
      paintjet_stream = fopen("prn","w");
      if(paintjet_stream == NULL)
	{
	  fprintf(stderr,"Can`t open printer as a file\n");
	  exit(1);
	}
    }
  
  printf("Finally, printing from memory to printer \n");
#else
  FILE *paintjet_stream = paintjet_dev->output_file ;
#endif
  /* print all the block use to store page description */
  (paintjet_dev->local_procs->print_blk) (dev, paintjet_stream);

  return 0;
}

/* Copy a monochrome bitmap.  The colors are given explicitly. */
/* Color = -1 means transparent (no effect on the image). */
int
  paintjet_copy_mono(dev, base, sourcex, raster, x, y, w, h, zero, one)
gx_device *dev;
byte *base;
int sourcex;  /* bit offset from base of first dot */
int raster; /* bytes per scan line */
int x, y, w, h;
gx_color_index zero, one;
{
  byte *ptr_source, *ptr_line;
  int shift_number, border, pixel, dest_x = x, end_x = x+w, dest_y = y;
  int number=-1;

  border = 7 - (sourcex & 7);     /* border of the next byte */
  ptr_line = base + (sourcex >> 3);
  while(h--)
    { /* for each line */
      ptr_source = ptr_line;
      
      /* make sure block is set in memory */
      if(number != (dest_y >> 6))
	{
	  number = dest_y >> 6;
	  if(paintjet_dev->blk[number].status != MEMORY)
	    set_blk(dev, number);
	}
      
      while(dest_x < end_x)
	{ /* quit when each bits in a line are done */
	  for(shift_number=border; shift_number>=0; shift_number--)
	    {
	      if(dest_x < end_x)
		{
		  pixel = (int)(((*ptr_source) >> shift_number) & 1);
		  if((pixel)&&(one != -1))
		    (paintjet_dev->local_procs->write_bit_blk)(dev, number, dest_x, dest_y, one);
		  if((!pixel)&&(zero != -1))
		    (paintjet_dev->local_procs->write_bit_blk)(dev, number, dest_x, dest_y, zero);
		  dest_x++;
		}
	    }
	  ptr_source++;
	  for(shift_number=7; shift_number>border; shift_number--)
	    {
	      if(dest_x < end_x)
		{
		  pixel = (int)(((*ptr_source) >> shift_number) & 1);
		  if((pixel)&&(one != -1))
		    (paintjet_dev->local_procs->write_bit_blk)(dev, number, dest_x, dest_y, one);
		  if((!pixel)&&(zero != -1))
		    (paintjet_dev->local_procs->write_bit_blk)(dev, number, dest_x, dest_y, zero);
		  dest_x++;
		}
	    }
	}
      ptr_line += raster;
      dest_x = x;
      dest_y++;
    }
  return 0;
}



/* Copy a color pixel map.  This is just like a bitmap, */
/* Each pixel takes 3 bits instead of 1 when device driver has color. */
int paintjet_copy_color(dev, base, sourcex, raster, x, y, w, h) /*, color_transparent)*/
     gx_device *dev;
     byte *base;   /* location of the color pixel map */
     int sourcex;  /* bit offset from base of first dot */
     int raster;   /* bytes per scan line */
     int x, y;     /* first pixel on the screen */
     int w, h;     /* width and height */
     /*int color_transparent; *//* transparent color */ /* local */
{
  int color_transparent = -1 ;
  int number = -1;

  if(dev->has_color == HAS_COLOR)     /* color device driver */
    {
      byte *ptr_source, *ptr_line;
      int dest_y = y, dest_x = x, end_x = x+w;
      int color;
      
      ptr_line = base + (sourcex >> 1);
      while(h--)
	{              /* for each line */
	  ptr_source = ptr_line;
	  
	  /* make sure block is set in memory */
	  if(number != (dest_y >> 6))
	    {
	      number = dest_y >> 6;
	      if(paintjet_dev->blk[number].status != MEMORY)
		set_blk(dev, number);
	    }
	  
	  if(sourcex & 1)    /* find out wich side of the byte we begin with */
	    {
	      while(dest_x < end_x)
		{            /* quit when each pixel in a line are done */
		  color =  *ptr_source++ & 0xf;
		  if(color != color_transparent)
		    (paintjet_dev->local_procs->write_bit_blk) (dev, number,
								dest_x,
								dest_y,
								color);
		  dest_x++;
		  if(dest_x < end_x)
		    {
		      color = *ptr_source >> 4;
		      if(color != color_transparent)
			(paintjet_dev->local_procs->write_bit_blk) (dev, number,
								    dest_x,
								    dest_y,
								    color);
		      dest_x++;
		    }
		}
	    }
	  else
	    {
	      while(dest_x < end_x)
		{            /* quit when each pixel in a line are done */
		  color = *ptr_source >> 4;
		  if(color != color_transparent)
		    (paintjet_dev->local_procs->write_bit_blk) (dev, number,
								dest_x,
								dest_y,
								color);
		  dest_x++;
		  if(dest_x < end_x)
		    {
		      color =  *ptr_source++ & 0xf;
		      if(color != color_transparent)
			(paintjet_dev->local_procs->write_bit_blk) (dev, number,
								    dest_x,
								    dest_y,
								    color);
		      dest_x++;
		    }
		}
	    }
	  dest_x = x;
	  dest_y++;
	  ptr_line += raster;
	}
    }
  else /* BLACK and WHITE device driver: one bit by pixel */
    {
      /* bitmap is the same than copy_mono: one bit by pixel */
      int zero = 0, one = 1;
      
      switch ( color_transparent )
	{
	case 0: zero = -1; break;
	case 1: one = -1; break;
	}
      paintjet_copy_mono(dev, base, sourcex, raster, x, y, w, h, zero, one);
    }
  return 0;
}


/* Fill a rectangle. */
int
  paintjet_fill_rectangle(dev, x, y, w, h, color)
gx_device *dev;
int x, y, w, h;
gx_color_index color;
{
  int dest_x = x, end_x = x+w, dest_y =y;
  int number=-1, white ;

  white = (2 << (paintjet_dev->bits_per_color_pixel-1)) - 1 ;
  while(h--)
    {
      /* make sure block is set in memory */
      if(number != (dest_y >> 6))
	{
	  number = (dest_y >> 6);
	  
	  /* When the block is in memory, it is ready to process. If we want
	     to put 0 (white) on an empty block, it is wasted time. Thus, the
	     the corresponding part of the white rectangle is skipped.
	     (This typically occurs when erasing the page before starting; this operation
	     is not required for a printer and takes a lot of time to set all the
	     pixels in memory to 0 one by one). Other than for that special case, the
	     block must be read from disk, or created, when it is not in memory.	*/
	  
	  if(paintjet_dev->blk[number].status != MEMORY)
	    { if(paintjet_dev->blk[number].status == FREE &&
		 color == white) /* WHITE */
		{ dest_y++;
		  while(h > 0 && (dest_y >> 6) == number)
		    { h--; dest_y++;
		    }
		  continue;
		}
	    else set_blk(dev, number);
	    }
	}
      
      /* Do beginning of a line that is not a complete byte */
      while(((dest_x & 7) != 0) && (dest_x < end_x))
	{
	  (paintjet_dev->local_procs->write_bit_blk)(dev, number, dest_x, dest_y, color);
	  dest_x++;
	}
      
      /* Do complete byte */
      while((dest_x + 8) < end_x)
	{
	  (paintjet_dev->local_procs->write_byte_blk)(dev, number, dest_x, dest_y, color);
	  dest_x += 8;
	}
      
      /* Do end of line that is not a complete byte */
      while(dest_x < end_x)
	{
	  (paintjet_dev->local_procs->write_bit_blk)(dev, number, dest_x, dest_y, color);
	  dest_x++;
	}
      dest_x = x;
      dest_y++;
    }
  return 0;
}

/* Tile a rectangle.  Note that the two colors must both be
   supplied, i.e. neither one can be -1 (transparent). */
int
  paintjet_tile_rectangle(dev, tile, x, y, w, h, zero, one)
gx_device *dev;
gx_bitmap *tile;
int x, y, w, h;
gx_color_index zero, one;
{
  while ( h > 0 )
    {
      int yr = y % tile->height;
      byte *data = tile->data + yr * tile->raster;
      int ny = tile->height - yr;
      int dx = x, dw = w;
      if ( ny > h ) ny = h;
      while ( dw > 0 )
	{
	  int xr = dx % tile->width;
	  int nx = tile->width - xr;
	  if ( nx > dw ) nx = dw;
	  paintjet_copy_mono(dev, data, xr, tile->raster, dx, y, nx, ny, zero, one);
	  dx += nx, dw -= nx;
	}
      y += ny, h -= ny;
    }
  return 0;
}

/****************************************************************************/
/* Virtual memory utility */
/* Use only by printer device */

/* DESCRIPTION DES VARIABLES LOCALES UTILISEES DANS write_bit_blk_PaintJet
   ET write_byte_blk_PaintJet. : 
   
   start_byte_ptr : adresse du debut de la portion de memoire continue ou 
   sont stockees les rangees logiques correspondantes aux rangees de 
   pixels sur l'imprimante. 
   plane: numero du plan de couleur courant.  0 <= plane < nb_of_planes 
   nb_of_planes: nombre de plan de couleur de l'imprimante 
   nb_of_planes = paintjet_dev->bits_per_color_pixel 
   NOTE: si il y a 3 plans de couleur, une rangee de pixels a 
   l'imprimante necessite 3 rangees dans la memoire (3 rangees 
   logiques) 
   byte_per_plane: nombre de byte dans une rangee en memoire.  Si il y a 3
   bits par pixel de couleur (3 plans de couleur), le nombre de byte
   pour une rangee en memoire est : (nombre de byte pour une rangee de
   pixels a l'imprimante)/3, soit paintjet_dev->byte_in_row / nb_of_planes.
   byte_target_ptr: adresse du groupe de 8 bits parmi lesquels sera entrepose
   le bit correspondant a la valeur de la couleur courante selon le
   plan de couleur courant. 
   bit_target: position du bit correspondant au pixel courant dans le 
   groupe de 8 bits byte_target. 
   color: couleur desiree du pixel (0 <= color < 2**nb_of_planes).
   offset_in_byte: decalage, a l'interieur du byte, qu'il faut appliquer pour
   obtenir la position du bit correspondant au pixel vise.
   vertical_offset: decalage vertical, a l'interieur du bloc de 64 lignes, de
   la rangee (ligne) de pixel visee.
   horizontal_offset: decalage horizontal dans la rangee de pixel visee.
   */

/*
  ::NOTE: swaping colors::
  
  First, a block vm_block (paintjet_dev->blk[number]) is initially full of
  zeros.  When there is something to draw, we draw into memory (into these
  blocks).  Note that operators like erasepage fill the page (the
  corresponding memory blocks) with zeros.
  
  Now suppose that we would want to print a empty page (full of zeros in
  memory). If we print on a X window, we would obtain a white page, but if
  we print such an empty page description on the Paintjet, we would get a
  black page, e.g. the printer would deposit black ink all over the sheet,
  which is not what we wanted.
  To circonvent such a behaviour of the printer we take the following
  steps :
  1) we tell the printer that the color white (no deposit of
  ink ==> color of the paper) is entry 0; so the printing of
  an empty page will yield the expected result (no ink
  deposit) ;
  2) we change the last entry in the color palette of the
  printer to now contain the color black.  This entry was
  previously the one for white ;
  
  These two steps complete the swapping of black and white colors as
  far as the printer is concerned.  These actions are taken in the
  function print_blk_PaintJet.
  However, we have to make sure that the black or white colors
  received from the imaging portion of Ghostscript are interpreted
  correctly.  So the following additionnals steps are to be taken
  into the functions write_bit_blk_PaintJet and
  write_byte_blk_PaintJet : 
  
  3) when we receive a request to draw in black, which is the
  color 0, we have to map this to the right entry in the
  printer's modified color palette : the last entry.
  Remember that the last entry is now the color black for the
  printer;
  4) similarily, when we receive a request to draw in white,
  which is the highiest numbered color (7 when there is 3
  bits per color pixel), we have to map this to the first
  entry in the printer's color palette : the entry 0.  Again,
  remember that the entry 0 is now the color white for the
  printer; 
  
  So, it appears as we've again swapped the black and white colors.
  This completes the manipulations required to make everything being
  printed as expected.
  Note that the other colors are not changed nor affected by this.
  */


/* write a bit in a block */
int write_bit_blk_PaintJet(dev, number, x, y, color)
     gx_device *dev;
     int number;
     int x, y, color;
{
  byte *byte_target_ptr, *start_byte_ptr;
  byte bit_target, mask ;
  byte offset_in_byte = x&7, vertical_offset = y&63, horizontal_offset = x>>3;
  int nb_of_planes, plane, byte_per_plane ;
  
  start_byte_ptr = (byte *) paintjet_dev->blk[number].mem_ptr ;
  nb_of_planes = paintjet_dev->bits_per_color_pixel ;
  byte_per_plane = paintjet_dev->byte_in_row / nb_of_planes ;
  bit_target = 128 >> offset_in_byte ;
  
  /* swapping white color with black color. */
  /* See the ::NOTE: swaping colors:: just before this function */
  if (color == 0) /* black --> white */
    color = (1<<nb_of_planes)-1 ;
  else if (color == (1<<nb_of_planes)-1) /* white --> black */
    color = 0 ;

  byte_target_ptr = start_byte_ptr
    + vertical_offset * paintjet_dev->byte_in_row
      + horizontal_offset ;
  
  for (plane=0; plane < nb_of_planes; plane++)
    {
      /* ((color>>plane)&1 is true if the bit for */
      /* the current color plane is 1 */
      if ((color >> plane) & 1)
	*byte_target_ptr = (*byte_target_ptr) | bit_target ;	
      
      /* next byte containing the following bit for the next color plane */
      byte_target_ptr += byte_per_plane ;
    }
}

/* write a byte in a block */
int write_byte_blk_PaintJet(dev, number, x, y, color)
     gx_device *dev;
     int number;
     int x, y, color;
{
  byte *byte_target_ptr, *start_byte_ptr, mask ;
  byte vertical_offset = y&63, horizontal_offset = x>>3 ;
  int nb_of_planes, plane, byte_per_plane ;
  
  start_byte_ptr = (byte *) paintjet_dev->blk[number].mem_ptr ;
  nb_of_planes = paintjet_dev->bits_per_color_pixel ;
  byte_per_plane = paintjet_dev->byte_in_row / nb_of_planes ;
  if (color == 0) /* black */
    color = (1<<nb_of_planes)-1 ;
  else if (color == (1<<nb_of_planes)-1) /* white */
    color = 0 ;
  
  byte_target_ptr = start_byte_ptr
    + (vertical_offset) * paintjet_dev->byte_in_row
      + horizontal_offset ;
  for (plane=0; plane < nb_of_planes; plane++)
    {
      *byte_target_ptr = ( ((color>>plane)&1) ? (0xff) : (0) ) ;
      /* ((color>>plane)&1 is true if the bit for */
      /* the current color plane is 1 */
      byte_target_ptr += byte_per_plane ;
    }
}


/* set block from disk file to ram memory */

int set_blk(dev, number)
     gx_device *dev;
     int number;
{
  int top, bottom;
  int not_allocate=1, not_find=1;

  /* allocate memory for a block */
  while(not_allocate)
    {
#ifdef __MSDOS__
      /* only 128Kb of RAM (amount store in vm_mem) for virtual memory */
      if((mem_blk*paintjet_dev->size_of_blk) < paintjet_dev->vm_mem)
	{ /* try to allocate memory */
	  paintjet_dev->blk[number].mem_ptr = (char *)calloc((unsigned)paintjet_dev->size_of_blk, sizeof(char));
	}
      else
	{ /* fail to allocate memory */
	  paintjet_dev->blk[number].mem_ptr = NULL;
	}
#else
      /* UNIX: just try to allocate memory */
      paintjet_dev->blk[number].mem_ptr = (char *)calloc((unsigned)paintjet_dev->size_of_blk, sizeof(char));
#endif
      /* verify result of allocation */
      if(paintjet_dev->blk[number].mem_ptr == NULL)
	{ /* allocation failed: write a block on disk and free some memories */
	  /* first, make sure there is block in memory */
	  if(mem_blk < 1)
	    {
	      fprintf(stderr,"Out of ram memory, Can`t allocate 20Kb block for device printer\n");
	      fprintf(stderr,"Use switch -M to shrink size of block allocated by GhostScript\n");
	      exit(1);
	    }
	  /* free a block in memory */
	  top = paintjet_dev->max_blk - 1;
	  bottom = 0;
	  not_find = 1;
	  while(not_find)
	    { /* We pick the farest block from number */
	      if((top - number) > (number - bottom))
		{ /* top is the farest */
		  if(paintjet_dev->blk[top].status == MEMORY)
		    {
		      write_blk(dev, top);
		      free((void *)paintjet_dev->blk[top].mem_ptr);
		      paintjet_dev->blk[top].status = DISK;
		      mem_blk--;
		      not_find = 0; /* exit not_find loop */
		    }
		  top--;
		}
	      else
		{ /* bottom is the farest */
		  if(paintjet_dev->blk[bottom].status == MEMORY)
		    {
		      write_blk(dev, bottom);
		      free((void *)paintjet_dev->blk[bottom].mem_ptr);
		      paintjet_dev->blk[bottom].status = DISK;
		      mem_blk--;
		      not_find = 0; /* exit not_find loop */
		    }
		  bottom++;
		}
	    }
	}
      else /* allocation succeed */
	{
	  not_allocate = 0; /* exit not_allocate loop */
	  mem_blk++;
	}
    }
  /* Now that allocation succeed, set content of block */
  if(paintjet_dev->blk[number].status == DISK)
    { /* read block from disk */
      read_blk(dev, number);
    }
  /* This block now is in memory */
  paintjet_dev->blk[number].status = MEMORY;

  return 0;
}

#ifndef __MSDOS__
#   define SEEK_SET 0
#endif

/* write a block in temporary file */
int
  write_blk(dev, number)
gx_device *dev;
int number;
{
  unsigned num_write;
  FILE *file_stream;
  
  /* open temporary file */
  if((file_stream = fopen("{prntmp}\0","r+")) == NULL)
    {
      /* try again */
      if((file_stream = fopen("{prntmp}\0","r+")) == NULL)
	{
	  fprintf(stderr,"can`t open temporary file for printer device\n");
	  fprintf(stderr,"make sure there is enought room on disk to store ghostscript page\n");
	  exit(1);
	}
    }
  /* set file cursor */
  if( fseek(file_stream, (long)number*paintjet_dev->size_of_blk, SEEK_SET) != NULL)
    {
      /* try again */
      if( fseek(file_stream, (long)number*paintjet_dev->size_of_blk, SEEK_SET) != NULL)
	{
	  fprintf(stderr,"can't write block from disk in printer device: wrong fseek 2\n");
	  exit(1);
	}
    }
  /* write block */
  num_write = fwrite((void *)paintjet_dev->blk[number].mem_ptr, sizeof(char), (int)paintjet_dev->size_of_blk, file_stream);
  if(num_write < paintjet_dev->byte_in_row)
    {
      /* try again */
      num_write = fwrite(paintjet_dev->blk[number].mem_ptr, sizeof(char), (int)paintjet_dev->size_of_blk, file_stream);
      if(num_write < paintjet_dev->byte_in_row)
	{
	  fprintf(stderr,"can`t write block to disk in printer device: wrong fwrite: num_write = %ld\n",num_write);
	  exit(1);
	}
    }
  /* Close temporary files */
  fclose(file_stream);
  
  return 0;
}

/* read a block in temporary file */
int
  read_blk(dev, number)
gx_device *dev;
int number;
{
  unsigned num_read;
  FILE *file_stream;

  /* open temporary file */
  if((file_stream = fopen("{prntmp}\0","r")) == NULL)
    {
      /* try again */
      if((file_stream = fopen("{prntmp}\0","r")) == NULL)
	{
	  fprintf(stderr,"can`t open temporary file for printer device\n");
	  fprintf(stderr,"make sure there is enought room on disk to store ghostscript page\n");
	  exit(1);
	}
    }
  /* set file cursor */
  if( fseek(file_stream, (long)number*paintjet_dev->size_of_blk, SEEK_SET) != NULL)
    {
      /* try again */
      if( fseek(file_stream, (long)number*paintjet_dev->size_of_blk, SEEK_SET) != NULL)
	{
	  fprintf(stderr,"can't read block from disk in printer device: wrong fseek\n");
	  exit(1);
	}
    }
  /* read block */
  num_read = fread((void *)paintjet_dev->blk[number].mem_ptr, sizeof(char), (int)paintjet_dev->size_of_blk, file_stream);
  if(num_read < paintjet_dev->byte_in_row)
    {
      /* try again */
      num_read = fread(paintjet_dev->blk[number].mem_ptr, sizeof(char), (int)paintjet_dev->size_of_blk, file_stream);
      if(num_read < paintjet_dev->byte_in_row)
	{
	  fprintf(stderr,"can`t read block from disk in printer device: wrong fread num_read = %u\n", num_read);
	  exit(1);
	}
    }
  /* Close temporary files */
  fclose(file_stream);

  return 0;
}

/* Send all blocks to printer (PaintJet): goal, print a page description */
/* (local) */
int print_blk_PaintJet(dev, paintjet_stream)
     gx_device *dev;
     FILE *paintjet_stream;
{
  int number;
  unsigned char rlebuf[360];     /* for RLE encoded scan lines */
  int numrle;                    /* number of bytes in above */

  byte *end_row;		/* last byte in the row */
  
  /* number of byte per color plane for one raster line */
  int byte_per_plane = paintjet_dev->byte_in_row
    / paintjet_dev->bits_per_color_pixel ;
  
  /* ends raster graphics to set raster graphics resolution */
  fprintf(paintjet_stream,"%c*r%dB",27,0);
  
  /* set raster graphics resolution 180 dpi */
  fprintf(paintjet_stream,"%c*t%dR",27,180);
  
  /* set the number of color plane per pixel to the number of bits */
  /* per pixels of the device */
  fprintf(paintjet_stream,"%c*r%dU",27,
	  paintjet_dev->bits_per_color_pixel) ;
  
  /* Swap the entry for black with the one for white. */
  /* SEE the note ::NOTE: swapping colors:: in this file (probably just */
  /* before the code for the function write_bit_blk_PaintJet) for more */
  /* details. */
  /* Note that if there is only one bit per color pixel, the device is */
  /* functionning in black in white and this swapping will have no effect. */
  /* For more details on color palette customizing, refer to the */
  /* documentation of the HP Paintjet pp. 5-17, 5-18, the tables on pages */
  /* 5-30 and 5-31. */
  fprintf(paintjet_stream,"%c*v%dA%c*v%dB%c*v%dC%c*v%dI",
	  27,90,		/* red component for white */
	  27,88,		/* green component for white */
	  27,85,		/* blue component for white */
	  27,0);		/* assign white to the entry previously for black */
  fprintf(paintjet_stream,"%c*v%dA%c*v%dB%c*v%dC%c*v%dI",
	  27,4,			/* red component for black */
	  27,4,			/* green component for black */
	  27,6,			/* blue component for black */
	  /* assign black to the entry previously for white */
	  27,
	  (1<<paintjet_dev->bits_per_color_pixel)-1);
  
  
  /* move to top left of page */
  fprintf(paintjet_stream,"%c&a%dH",27,0);
  fprintf(paintjet_stream,"%c&a%dV",27,0);

  /* set rle mode */
  fprintf(paintjet_stream,"%c*b1M",27);
  
  /* start raster graphics */
  fprintf(paintjet_stream,"%c*r%dA",27,1);
  
  /* send each block */
  for(number=0; number<paintjet_dev->max_blk; number++)
    {
      int row, max_row=64;	/* 64 rows in each block */
      
      /* last block can have less than 64 rows */
      if(number == paintjet_dev->max_blk - 1) 
	{
	  max_row = paintjet_dev->row_last_blk;
	  if(paintjet_dev->blk[number].status == FREE)
	    {
	      break;		/* nothing to print : exit the for loop */
	    }
	}
      
      if(paintjet_dev->blk[number].status == FREE) /* block is all white */
	{	
	  /* leave raster graphic mode for sending the escape sequence for */
	  /* skiping lines */
	  fprintf(paintjet_stream,"%c*r%dB",27,0);	  

	  /* move 64 lines down from current print position */
	  /* The escape sequence is ESC&a#V where # is in */
	  /* decipoints. For the PaintJet, 4 decipoints = */
	  /* 1/180 inches then 64 lines at 180 dpi = 4*64 */
	  /* decipoints because 1 line = 1/180 inches. */
	  fprintf(paintjet_stream,"%c&a+%dV",27,max_row*4); 

	  /* restart raster graphics */
	  fprintf(paintjet_stream,"%c*r%dA",27,1);
	}
      if(paintjet_dev->blk[number].status == DISK)
	{
	  /* bring block in memory */
	  set_blk(dev, number);
	}
      if(paintjet_dev->blk[number].status == MEMORY)
	{
	  byte *start = (byte *)paintjet_dev->blk[number].mem_ptr ;
	      
	  /* print rows */
	  for(row=0; row<max_row; row++)
	    {
	      int i_color_plane ;

	      /* for each row, there is paintjet_dev->bits_per_color_pixel to */
	      /* transfer.  All but the last requires */
	      /* ESC*b#V, the last one requires ESC*b#W, where # is the */
	      /* number of bytes to transmit. */
	      for (i_color_plane = 0 ;
		   i_color_plane < paintjet_dev->bits_per_color_pixel-1;
		   i_color_plane ++, start += byte_per_plane)
		{
		  /* transfer raster graphics */
		  /* encode it */
		  numrle = rle(start,rlebuf,byte_per_plane);
		  fprintf(paintjet_stream,"%c*b%dV",27,numrle);
		  /* send a row of bits for the current color plane */
		  fwrite(rlebuf, sizeof(char), numrle, paintjet_stream); 
		}
	      numrle = rle(start,rlebuf,byte_per_plane);
	      fprintf(paintjet_stream,"%c*b%dW",27,numrle);
	      /* send the last row of bits for the current color plane */
	      fwrite(rlebuf, sizeof(char), numrle, paintjet_stream);

	      /* place the poiter to the begining of the next block of */
	      /* three rows in memory, e.g. the begining of the next */
	      /* "raster" line. */
	      start += byte_per_plane ;
	    }
	  /* operator showpage erase page while operator copypage don`t */
	  free((void *)paintjet_dev->blk[number].mem_ptr);
	  paintjet_dev->blk[number].status = FREE;
	  mem_blk--;
	}
    }
      
  /* end raster graphics */
  fprintf(paintjet_stream,"%c*r%dB",27,0);
      
  /* eject page */
  fprintf(paintjet_stream,"%c",12) ;

  return 0;
}

/* rle() -- Run-Length-Encodes 'n' bytes from buffer 'a' into buffer 'b'
   and returns the number of bytes that are in buffer 'b'.
   Author: Bill Thorson, Colorado State University
           thorson@typhoon.atmos.colostate.edu
*/
int rle(a,b,n)
  unsigned char *a, *b;
  int n;
{
  unsigned char count,pattern, *ptr;
  int ia=0, ib=0, num=0;
  while (ia<n)
  {
    count = 0;
    pattern = *(a+ia);
    ptr = a+ia+1;
    while (*ptr == pattern)
    {
     ptr++;
     count++;
    }
    *(b+ib++) = count;
    *(b+ib++) = pattern;
    num += 2;
    ia += count+1;
  }
  return (num);
}


/* ---------- Debugging routine ------------- */

/* dump virtual memory block */
int
  dump_vm_blk(dev)
gx_device *dev;
{
  int number;
  
  fprintf(stderr,"address of vm_blk is %p\n\n", paintjet_dev->blk);
  fprintf(stderr,"number   status   mem_ptr  \n");
  
  for(number=0; number<paintjet_dev->max_blk; number++)
    {
      fprintf(stderr,"   %d    ", number);
      if(paintjet_dev->blk[number].status == FREE)
	fprintf(stderr," FREE    ");
      if(paintjet_dev->blk[number].status == DISK)
	fprintf(stderr," DISK    ");
      if(paintjet_dev->blk[number].status == MEMORY)
	fprintf(stderr,"MEMORY   ");
      
      if(paintjet_dev->blk[number].mem_ptr == UNKNOWN)
	fprintf(stderr,"UNKNOWN\n");
      else
	fprintf(stderr,"%p\n",paintjet_dev->blk[number].mem_ptr);
    }
  return 0;
}

/* test read_blk and write_blk */
void disk_io(dev)
     
     gx_device *dev;
{
  int count;
  int number=0;
  char *start;
  
  set_blk(dev, number);
  start = paintjet_dev->blk[number].mem_ptr;
  for(count=0; count<paintjet_dev->size_of_blk; count++)
    {
      *start = 255;
      start++;
    }
  write_blk(dev, number);
  number = 1;
  set_blk(dev, number);
  start = paintjet_dev->blk[number].mem_ptr;
  for(count=0; count<paintjet_dev->size_of_blk; count++)
    {
      *start = 255;
      start++;
    }
  write_blk(dev, number);
  
  set_blk(dev,2);
  read_blk(dev,0);
  write_blk(dev,2);
}


