UNIT MOUSE;   { uses $33 IRQ to access MicroSoft Mouse Driver functions }
INTERFACE
TYPE
  IntegerArray32 = ARRAY[1..32] of Integer;

FUNCTION MouseDriverFound : BOOLEAN;
FUNCTION MouseReset : BOOLEAN;
PROCEDURE ShowCursor;
PROCEDURE HideCursor;
PROCEDURE GetButtonsAndPosition(VAR Buttons, Xpos, Ypos : Integer);
PROCEDURE SetCursorPosition(Xpos, Ypos : Integer);
PROCEDURE GetButtonPressInfo(ButtonNum : Integer;
                        VAR Buttons, Times, Xpos, Ypos : Integer);
PROCEDURE GetButtonReleaseInfo(ButtonNum : Integer;
                         VAR Buttons, Times, Xpos, Ypos : Integer);
PROCEDURE SetHorizCursorBounds(Xmin, Xmax : INTEGER);
PROCEDURE SetVertCursorBounds(Ymin, Ymax : INTEGER);
PROCEDURE DefineGraphicsCursor(Xhot, Yhot : Integer;
                               VAR Masks : IntegerArray32);
PROCEDURE SetTextCursor(TypeCursor, Value1, Value2 : Integer);
PROCEDURE ReadMickeyCount(VAR Xcount, Ycount : Integer);
PROCEDURE SetMickeyPixelRatio(Xratio, Yratio : Integer);
FUNCTION SoftwareReset : BOOLEAN;
PROCEDURE GetMouseSysInfo(VAR MajorVer, MinorVer, MouseType, IRQnum : Integer);
FUNCTION LeftButtonPressed(VAR Xpos, Ypos : Integer): BOOLEAN;
FUNCTION LeftButtonReleased(VAR Xpos, Ypos : Integer): BOOLEAN;
FUNCTION RightButtonPressed(VAR Xpos, Ypos : Integer): BOOLEAN;
FUNCTION RightButtonReleased(VAR Xpos, Ypos : Integer): BOOLEAN;
FUNCTION BothButtonsPressed(VAR Xpos, Ypos : Integer): BOOLEAN;


IMPLEMENTATION

USES DOS;

VAR
  Regs : registers;
  Visible : Integer; { used to keep image of internal mouse cursor flag }


FUNCTION MouseDriverFound : BOOLEAN;
VAR
  Vec : Pointer;
BEGIN
  GetIntVec($33,Vec);
  If (LongInt(Vec) = 0) OR (Byte(Vec^) = 207) THEN
    MouseDriverFound := FALSE
  ELSE
    MouseDriverFound := TRUE;
END;

FUNCTION MouseReset : BOOLEAN;
BEGIN
  Regs.AX := 0;
  Intr($33,Regs);
  IF Integer(Regs.AX) = -1 THEN  { word to integer for negative value }
    MouseReset := TRUE  { => MOUSE HARDWARE FOUND AND RESET }
  ELSE
    MouseReset := FALSE;
  Visible := -1;
END;

PROCEDURE ShowCursor;
BEGIN
  IF Visible <> 0 THEN
    REPEAT
      Regs.AX := 1;
      Intr($33,Regs);
      Inc(Visible);
    UNTIL Visible = 0;
END;

PROCEDURE HideCursor;
BEGIN
  Regs.AX := 2;
  Intr($33,Regs);
  Dec(Visible);
END;

PROCEDURE GetButtonsAndPosition(VAR Buttons, Xpos, Ypos : Integer);
BEGIN
  Regs.AX := 3;
  Intr($33,Regs);
  Buttons := Regs.BX;  { 0 neither button, 1 left, 2 right, 3 both }
  Xpos := Regs.CX;
  Ypos := Regs.DX;
END;

PROCEDURE SetCursorPosition(Xpos, Ypos : Integer);
BEGIN
  Regs.AX := 4;
  Regs.CX := Xpos;
  Regs.DX := Ypos;
  Intr($33,Regs);
END;

PROCEDURE GetButtonPressInfo(ButtonNum : Integer;
                        VAR Buttons, Times, Xpos, Ypos : Integer);
BEGIN                { ButtonNum : button to check: 0 left, 1 right }
  Regs.AX := 5;
  Regs.BX := ButtonNum;
  Intr($33,Regs);
  Buttons := Regs.AX; { CURRENTLY:  0 neither button, 1 left, 2 right, 3 both }
  Times := Regs.BX;   { times button ButtonNum PRESSED since last call}
  Xpos := Regs.CX;    { AT LAST PRESS : X position button ButtonNum }
  Ypos := Regs.DX;                    { Y position button ButtonNum }
END;

PROCEDURE GetButtonReleaseInfo(ButtonNum : Integer;
                         VAR Buttons, Times, Xpos, Ypos : Integer);
BEGIN                { ButtonNum : button to check: 0 left, 1 right }
  Regs.AX := 6;
  Regs.BX := ButtonNum;
  Intr($33,Regs);
  Buttons := Regs.AX; { CURRENTLY: 0 neither button, 1 left, 2 right, 3 both }
  Times := Regs.BX;   { times button ButtonNum RELEASED since last call}
  Xpos := Regs.CX;    { AT LAST RELEASE: X position button ButtonNum }
  Ypos := Regs.DX;                     { Y position button ButtonNum }
END;

PROCEDURE SetHorizCursorBounds(Xmin, Xmax : INTEGER);
BEGIN
  Regs.AX := 7;
  Regs.CX := Xmin;
  Regs.DX := Xmax;
  Intr($33,Regs);
END;

PROCEDURE SetVertCursorBounds(Ymin, Ymax : INTEGER);
BEGIN
  Regs.AX := 8;
  Regs.CX := Ymin;
  Regs.DX := Ymax;
  Intr($33,Regs);
END;

PROCEDURE DefineGraphicsCursor(Xhot, Yhot : Integer;
                               VAR Masks : IntegerArray32);
BEGIN
  Regs.AX := 9;
  Regs.BX := Xhot;
  Regs.CX := Yhot;
  Regs.DX := Ofs(Masks);
  Regs.ES := Seg(Masks);
  Intr($33,Regs);
END;

PROCEDURE SetTextCursor(TypeCursor, Value1, Value2 : Integer);
BEGIN      { TypeCursor: 0 software cursor, 1 hardware cursor }
  Regs.AX := 10;
  Regs.BX := TypeCursor;
  Regs.CX := Value1; { Software cursor: SCREEN MASK; }
                     { Hardware cursor: FIRST SCAN LINE }
  Regs.DX := Value2; { Software cursor: CURSOR MASK; }
                     { Hardware cursor: LAST SCAN LINE }
  Intr($33,Regs);
END;

PROCEDURE ReadMickeyCount(VAR Xcount, Ycount : Integer);
BEGIN     { counts in Mickeys = 1/200" increments SINCE LAST CALL }
  Regs.AX := 11;
  Intr($33,Regs);
  Xcount := Regs.CX;
  Ycount := Regs.DX;
END;

(****************************************************)

PROCEDURE SetIrqMaskAndSubAddr;
                                { $33 Intr with Regs.AX = 12 NOT IMPLEMENTED}
BEGIN
END;

PROCEDURE LightPenOn;
                                { $33 Intr with Regs.AX = 13 NOT IMPLEMENTED}
BEGIN
END;

PROCEDURE LightPenOff;
                                { $33 Intr with Regs.AX = 14 NOT IMPLEMENTED}
BEGIN
END;
(****************************************************)

PROCEDURE SetMickeyPixelRatio(Xratio, Yratio : Integer);
BEGIN                   { ratio is Mickeys/(8 virtual screen pixels }
  Regs.AX := 15;     { default for horiz is 8 Mickeys/8 virtual screen pixels }
  Regs.CX := Xratio; { default for vert is 16 Mickeys/8 virtual screen pixels }
  Regs.DX := Yratio;
  Intr($33,Regs);
END;

(************** Interrupt $33 functions 16 .. 32 not implemented ********)

FUNCTION SoftwareReset : BOOLEAN;   { use if cursor in text mode and go }
BEGIN                               { into graphics mode                }
  Regs.AX := 33;
  Intr($33,Regs);
  IF (Integer(Regs.AX) = -1) AND (Regs.BX = 2) THEN
    BEGIN
      SoftwareReset := TRUE;
      Visible := -1;
    END
  ELSE
    SoftwareReset := FALSE;
END;

(************** Interrupt $33 functions 34 .. 35 not implemented ********)

PROCEDURE GetMouseSysInfo(VAR MajorVer, MinorVer, MouseType, IRQnum : Integer);
BEGIN
  Regs.AX := 36;
  Intr($33,Regs);
  MajorVer := Regs.Bh;    { to be read in HEX }
  MinorVer := Regs.Bl;    { to be read in HEX }
  MajorVer := (MajorVer DIV 16)*10 + (MajorVer MOD 16);
  MinorVer := (MinorVer DIV 16)*10 + (MinorVer MOD 16);
  MouseType := Regs.Ch; {1 bus, 2 serial, 3 InPort, 4 PS/2, 5 Hewlett-Packard}
  IRQnum := Regs.Cl;    { IRQ type : 0 PS/2, 2..5 or 7 mouse }
END;

FUNCTION LeftButtonPressed(VAR Xpos, Ypos : Integer): BOOLEAN;
VAR
  Buttons : Integer;
BEGIN
  GetButtonsAndPosition(Buttons,Xpos,Ypos);
  IF Buttons = 1 THEN
    LeftButtonPressed := TRUE
  ELSE
    LeftButtonPressed := FALSE;
END;

FUNCTION LeftButtonReleased(VAR Xpos, Ypos : Integer): BOOLEAN;
VAR
  Buttons, Times : Integer;
BEGIN
  GetButtonReleaseInfo(0,Buttons,Times,Xpos,Ypos);
  IF Times <> 0 THEN
    LeftButtonReleased := TRUE
  ELSE
    LeftButtonReleased := FALSE;
  END;

FUNCTION RightButtonPressed(VAR Xpos, Ypos : Integer): BOOLEAN;
VAR
  Buttons : Integer;
BEGIN
  GetButtonsAndPosition(Buttons,Xpos,Ypos);
  IF Buttons = 2 THEN
    RightButtonPressed := TRUE
  ELSE
    RightButtonPressed := FALSE;
END;

FUNCTION RightButtonReleased(VAR Xpos, Ypos : Integer): BOOLEAN;
VAR
  Buttons, Times : Integer;
BEGIN
  GetButtonReleaseInfo(1,Buttons,Times,Xpos,Ypos);
  IF Times <> 0 THEN
    RightButtonReleased := TRUE
  ELSE
    RightButtonReleased := FALSE;
  END;

FUNCTION BothButtonsPressed(VAR Xpos, Ypos : Integer): BOOLEAN;
VAR
  Buttons : Integer;
BEGIN
  GetButtonsAndPosition(Buttons,Xpos,Ypos);
  IF Buttons = 3 THEN
    BothButtonsPressed := TRUE
  ELSE
    BothButtonsPressed := FALSE;
END;

END.