/*

	FX munge (FX Uucp), copyright (c) 1994 Jorge Cwik
	munge uucp job's Unix filenames to DOS

	This routine munges filenames the same way as FX Uucp. Use this
	code or equivalent if you want to interface with FX Uucp.

	LEGAL:
	You are allowed to use this code verbatim or, in any modified
	form for implementing software add-ons for FX Uucp. The copy-
	right notice must be preserved in all cases.

	DISCLAIMER:
	Even though we tested the program and tried to make it full
	compatible with FX Uucp, this software is offered "as is".
	We cannot guarantee that the code performs well and exactly
	as intended. We also reserve the right to change the munging
	algorythm in FX Uucp.


	Unix filenames are of the form:
		D.host<grade>jobId	(e.g.: D.laserC1234)

	There are a few variations on the above format. But in any
	case it is not a valid DOS filename and must be munged.

	The munging algorythm must preserve the original uniqueness.
	Because each host has its own spool directory, there is no
	possibility of conflict between differents hosts. Thus, the
	host name is not needed and discarded. However, two different
	jobs may differ only in the case, the munging must still result
	in different DOS filename (that doesn't preserve cases).


	Detailed description:

	o Discard the prefix and the host name (the prefix will be used later)
	o Take the last five letters and compute a bitmask to preserve the case
	o Put the bitmask in the first letter of the munged filename
	o Append up to seven of the last letters of the original Unix name
	o Add the original prefix as the DOS filename extension.

	Discarding the host name:

	The routine must be passed the remote uucpname to be able to
	separate it and discard it. Otherwise it is not possible to do
	that. The length of the subfields are not fixed and the job ID
	may be encoded in alphanumerical characters, not only digits.
	Some systems use the remote uucpname (our local one) instead of
	their own name. The code must check for this possibility. It
	scans the Unix file name for the longer piece that matches _either_
	the local or remote uucpname. This scan is done with case regarding
	comparison.
	
	Sometimes the host name doesn't match anyone, neither the local
	neither the remote. In this case, we just take the last seven
	letters. This might happen, for example, if the uucpname we
	are passed as parameter have the wrong case.


	Computing the bitmask:

	Only the last five letters are used to compute the bitmask
	Lowercase letters have a bit set, all the other characters a zero.
	The rightmost (first) letter correspond to the less significant bit.
	The whole byte is taken as a byte (whose range is 0-31) and enco-
	ded in a single alphanumeric character. If is less than 10, use
	digits 0-9, otherwise use uppercase letter.


	This implementation takes the following parameter:

	char 		*munged: pointer to a previouslly allocated buffer
	const char	*unixname : the original Unix job's filename
	const char	*remotname: the remot uucpname
	const char	*localname: the local uucpname

	It returns 0 on ivalid input, 1 otherwise.
*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>


int fxMunge( char *munged, const char *unixname,
	const char *remotname, const char *localname)
{
	const char *s, *s2;
	unsigned bitmask;
	int i;

	/* Simple sanity check, must be D. or X. */
	if( unixname[1] != '.' || ( unixname[0] != 'D' && unixname[0] != 'X') )
		return 0;

	/* Find the longest hostname match */
	s2 = s = unixname + 2;
	while( *s && *s == *remotname++)
		s++;
	while( *s2 && *s2 == *localname++)
		s2++;
	if( s2 > s)
		s = s2;


	/* 
	Now 's' splits the Unix filename between the host name and
	the rest, the jobID (plus grade) that we are interested.

	You may want to add other sanity checks based in the lenght
	of the remaining string.
	*/

	s2 = s;		/* Remember pointer */

	/* Only the last five characters build the bitmask */
	i = strlen(s) - 5;
	if( i > 0)
		s += i;

	for( bitmask = 0; *s; s++)
	{
		bitmask <<= 1;
		if( islower( *s))
			bitmask |= 1;
	}

	/* You may use bitmask as an index in a table if you want */
	/* Slightly faster and longer                             */

	bitmask += '0';
	if( bitmask > '9')
		bitmask += 'A' - '9' - 1;

	/* Take at most seven characters */
	i = strlen(s2) - 7;
	if( i > 0)
		s2 += i;

#if 1
	sprintf( munged, "%c%s.%c", bitmask, s2, unixname[0]);
#else

	/* 
	It may be overkill to use sprintf if it's never called in
	any other sections of the program. This code is equivalent.
	*/

	{
		char *ptr; /* We need a non-const pointer */

		munged[0] = bitmask;
		ptr = stpcpy( munged + 1, s2);	/* similar to strcpy  		*/
		*ptr++ = '.';					/* but returns end of string */
		*ptr++ = unixname[0];
		*ptr++ = 0;
	}
#endif

	return 1;
}
