/*
 *  Filename    : dict.c
 *  Author      : David Harvey-George
 *  Start Date  : 13 May 1994
 *
 *  Description:
 *  Some generic dictionary manipulation routines.
 *
 *  Copyright (C) 1994	DB Harvey-George
 *
 *  $Log$
 */

static char rcsid[]="$Id$";

#include <stdio.h>
#include "dict.h"

/*
 *  initDict
 *
 *  Initialise dict structure.
 *
 *  Returns
 *  none
 */

void
initDict(
    Dict_t *Dict,
    int (*func)() )
{
    Dict->Count = 0;
    Dict->Head = NULLNODE;
    Dict->Compare = func;
}


/*
 *  addNode
 *
 *  Adds an entry to a sorted dict.
 *
 *  RETURNS
 *  1		Duplicate Item
 *  0		Okay
 */

int
addNode(
    Dict_t *Dict,
    Node_t *Node)
{
    if (Dict->Head == NULLNODE)
    {
        Dict->Head = Node;
    }
    else
    {
        Node_t **pHead;
        int    Cmp;

        pHead = &Dict->Head;
        do
        {
            Cmp = Dict->Compare(*pHead, Node);
            if (Cmp == 0)
            {
                return 1;  /* Error, duplicate entry */
            }
            if (Cmp > 0)
            {
                /*
                 * Add before
                 */
                 Node->Next = *pHead;
                 *pHead = Node;
                 break;
            }
            else
            {
                 if(((*pHead)->Next) == NULLNODE)
                 {
                     /*
                      * end of the dict, add after
                      */
                      (*pHead)->Next = Node;
                      break;
                 }
                 
            }
        } while ((pHead = &(*pHead)->Next) != 0);
    }
    Dict->Count++;
    return 0;
}


/*
 *  getNode
 *
 *  Generic function to return a node in a dict.
 *
 *  Returns:
 *  Specified Node
 *  or NULLNODE - node not in dict.
 */

Node_t *
getNode(
    Dict_t *pDict,
    Id Name )
{
    Node_t   **pHead;

    pHead = &pDict->Head;
    while (*pHead != 0)
    {
        if (pDict->Compare(*pHead, Name) == 0)
        {
            return *pHead;
        }
        pHead = &(*pHead)->Next;
    }

    return NULLNODE;
}


/*
 *  removeNode
 *
 *  Generic function to remove and return a node.
 *
 *  Returns:
 *  A pointer to the removed node,
 *  or NULLNODE if the node was not found.
 */

Node_t *
removeNode(
    Dict_t *Dict,
    Id Name )
{
    Node_t   **pHead;
    Node_t   *tNode;

    pHead = &Dict->Head;
    while (*pHead != 0)
    {
        if (Dict->Compare(*pHead, Name) == 0)
        {
            tNode = *pHead;
            *pHead = tNode->Next;	/* remove node from Dict */
            tNode->Next = NULL;		/* remove access to Dict */

            Dict->Count--;
            return tNode;
        }
        pHead = &(*pHead)->Next;
    }

    return NULLNODE;
}


/*
 *  getFirstNode
 *
 *  Returns a reference to the first node in a dict.
 * 
 *  Returns:
 *  Pointer to node.
 *  or NULLNODE - dict is empty
 */

Node_t *
getFirstNode(
    Dict_t *Dict )
{
    if (Dict->Head != 0)
    {
        return Dict->Head;
    }

    return NULLNODE;
}


/*
 *  getNextNode
 *
 *  Returns a reference to Next node after the one given.
 */

Node_t *
getNextNode(
    Dict_t *Dict,
    Id Name )
{
    Node_t   *Head;

    Head = Dict->Head;
    while (Head != 0)
    {
        if (Dict->Compare(Head, Name)  > 0)
        {
            return Head;
        }
        Head = Head->Next;
    }

    return NULLNODE;
}
