// gcp.h -- Garbage Collecting Pointer
// (C) Copyright 1994  John Webster Small
// All rights reserved

#ifndef gcp_h
#define gcp_h

#include <stddef.h>    // size_t
#ifndef GCP_NDEBUG
    #include <stdlib.h>  // abort()
    #include <limits.h>  // UCHAR_MAX
    #include <iostream.h>
    #include <iomanip.h>
    #include <strstrea.h>
#endif

// heap alignment type
#ifndef GCO_ALIGN
    #define GCO_ALIGN long
#endif
#ifndef GCP_NDEBUG
    #ifdef GCO_FENCE_OBJS
        #if GCO_FENCE_OBJS < 0
            #undef GCO_FENCE_OBJS
            #define GCO_FENCE_OBJS 0
        #endif
    #else
        #define GCO_FENCE_OBJS 1
    #endif
    #ifndef GCO_FENCE_CHAR_MASK
        #define GCO_FENCE_CHAR_MASK UCHAR_MAX
    #endif
#endif

// allow for overloaded operator->()
struct GCptr    { int selectable; };
struct GCNSptr  { int not_selectable; };
struct SPptr    { int selectable; };
struct SPNSptr  { int not_selectable; };

#define GC_FLAG          0
#define SP_FLAG          1
#define GCP_ARRAY_FLAG   2
#define SP_DEL_FLAG      4

// garbage collecting and self policing pointer macros
#ifdef GCP_NDEBUG
  #define gcp(T)   GCP<T,T,GCptr>
  #define gcp_init(T,V) gcp(T) V; V
  #define new_gcp(V) new(V)
  #define gcnsp(T) GCP<T,GCNSptr,T>
  #define gcnsp_init(T,V) gcnsp(T) V; V
  #define new_gcnsp(V) new(V)
  #define spp(T)   T *
  #define spp_init(T,V) spp(T) V
  #define new_spp(V) new
  #define spnsp(T) T *
  #define spnsp_init(T,V) spnsp(T) V
  #define new_spnsp(V) new
  #define delete_sp(P) delete P
  #define delete_spa(P) delete[] P
  #define GCP_FILE_LINE_FPARAMS
  #define GCP_FLAGS_FILE_LINE_FPARAMS
  #define GCP_FLAGS_FILE_LINE_APARAMS
  #define GCP_AFLAGS_FILE_LINE_APARAMS
#else
  #define GCP_FILE_LINE_FPARAMS  \
      , const char * FILE, int LINE
  #define GCP_FLAGS_FILE_LINE_FPARAMS \
      , unsigned FLAGS, const char * FILE, int LINE
  #define GCP_FLAGS_FILE_LINE_APARAMS  \
      , FLAGS, FILE, LINE
  #define GCP_AFLAGS_FILE_LINE_APARAMS  \
      , FLAGS | GCP_ARRAY_FLAG, FILE, LINE
  #define gcp(T) GCP<T,T,GCptr>
  #define gcp_init(T,V) gcp(T) V(__FILE__,__LINE__); V
  #define new_gcp(V) new(V,GC_FLAG,__FILE__,__LINE__)
  #define gcnsp(T) GCP<T,GCNSptr,T>
  #define gcnsp_init(T,V) \
      gcnsp(T) V(__FILE__,__LINE__); V
  #define new_gcnsp(V) new(V,GC_FLAG,__FILE__,__LINE__)
  #define spp(T) GCP<T,T,SPptr>
  #define spp_init(T,V) spp(T) V(__FILE__,__LINE__); V
  #define new_spp(V) new(V,SP_FLAG,__FILE__,__LINE__)
  #define spnsp(T) GCP<T,SPNSptr,T>
  #define spnsp_init(T,V) \
      spnsp(T) V(__FILE__,__LINE__); V
  #define new_spnsp(V) new(V,SP_FLAG,__FILE__,__LINE__)
  #define delete_sp(P) P.del(0,__FILE__,__LINE__)
  #define delete_spa(P) P.del(1,__FILE__,__LINE__)
#endif

// allow logging of errors and warnings
#if !defined(GCP_NDEBUG) && defined(GCP_LOG_ERRORS)
    #include <fstream.h>
    #include <time.h>
    class  GCP_ERR {
        ofstream err;
    public:
        GCP_ERR(const char * filename)
        {
            time_t t;
            err.open(filename,ios::out | ios::app);
            if (err)  {
               time_t t; time(&t);
               err << "\n\n" << ctime(&t);
            }
        }
        ~GCP_ERR() { err.close(); }
        ofstream& operator()() { return err; }
    };
    extern GCP_ERR gcp_err;
    #define GCP_ERROR_LOG(filename)  \
        GCP_ERR gcp_err(filename);
#else
    #define GCP_ERROR_LOG(filename)
#endif

// garbage collectable block object
union GCO  {
    struct {
        size_t nref;
        size_t nobj;
      #ifndef GCP_NDEBUG
        unsigned flags;
        size_t fence_bytes;
        size_t back_fence_idx;
        const char * file;
        int line;
      #endif
    } header;
    GCO_ALIGN align[1];
};

#ifdef GCP_NDEBUG
  #define GCP_TEST(TEST,FILE,LINE,MSG)
  #define GCP_ASSERT(TEST,FILE,LINE,MSG)
  #define GCP_ASSERT_SUBSCRIPT(GCO,FILE,LINE,\
    SCALAR,INDEX)
  #define GCP_STAMP(gcp,expr) expr
#else
  #define GCP_TEST(TEST,FILE,LINE,MSG)    \
    test(TEST,FILE,LINE,MSG)
  #define GCP_ASSERT(TEST,FILE,LINE,MSG)  \
    assert(TEST,FILE,LINE,MSG)
  #define GCP_ASSERT_SUBSCRIPT(GCO,FILE,LINE,\
    SCALAR,INDEX)  \
    assert_subscript(GCO,FILE,LINE,SCALAR,INDEX)
  #define GCP_STAMP(gcp,expr)  \
    (gcp.stamp(__FILE__,__LINE__),expr)
#endif

template <class A>
class GCO_Services  {
    static size_t GCO_ALIGN_CELLS_REQ(size_t size)
    {
        return (size / sizeof(GCO_ALIGN)
            + ((size % sizeof(GCO_ALIGN))
            ? 1 : 0));
    }
protected:
  #ifdef GCP_NDEBUG
    static size_t unlink(GCO * gco)
    {
        if(!--gco->header.nref)
            return gco->header.nobj;
        return 0;
    }
    static void * object(GCO * gco)
    {
        if (gco)
            return &(((GCO_ALIGN *)gco)
                [GCO_ALIGN_CELLS_REQ(sizeof(GCO))]);
        return 0;
    }
  #else
    static void test(int expr, const char * file,
        int line, const char * msg)
    {
        if (!expr)  {
            #ifdef GCP_LOG_ERRORS
                gcp_err()
            #else
                cerr
            #endif
                << "\nWarning "
                << (file? file : "file unknown")
                << " " << line << ": "
                << msg << flush;
            #ifdef GCP_ABORT_ON_WARN
                #ifdef GCP_LOG_ERRORS
                    gcp_err()
                #else
                    cerr
                #endif
                    << endl;
                abort();
            #endif
        }
    }
    static void assert(int expr, const char * file,
        int line, const char * msg)
    {
        if (!expr)  {
            #ifdef GCP_LOG_ERRORS
              gcp_err()
            #else
              cerr
            #endif
                << "\nError "
                << (file? file : "file unknown")
                << " " << line << ": "
                << msg << endl;
            abort();
        }
    }
    static void assert_subscript(GCO * gco,
        const char * file, int line,
        size_t scalar, size_t index);

    static unsigned char * frontFence(GCO * gco)
    {
        return (unsigned char *)&(((GCO_ALIGN *)gco)
            [GCO_ALIGN_CELLS_REQ(sizeof(GCO))]);
    }
    static void * object(GCO * gco)
    {
        if (gco)
            return &(((unsigned char *)gco)
                [GCO_ALIGN_CELLS_REQ(sizeof(GCO))
                * sizeof(GCO_ALIGN)
                + gco->header.fence_bytes]);
        return 0;
    }
    static unsigned char * backFence(GCO * gco)
    {
        return &(((unsigned char *)gco)
            [gco->header.back_fence_idx]);
    }
    static size_t unlink
        (GCO * gco GCP_FILE_LINE_FPARAMS);
    static void del(GCO * gco, int array,
      const char * FILE, int LINE);
  #endif
    static GCO * alloc(size_t req_size, size_t obj_size
        GCP_FLAGS_FILE_LINE_FPARAMS);
};

#ifndef GCP_NDEBUG

template <class A>
void GCO_Services<A>::assert_subscript(GCO * gco,
    const char * file, int line,
    size_t scalar, size_t index)
{
    GCP_ASSERT(gco != 0,file,line,
        "attempted NULL pointer assignment/access"
        "\n\tuse GCP_STAMP() to find exact location!");
    GCP_ASSERT(!(SP_DEL_FLAG & gco->header.flags),
        file,line,"attempt to access deleted object"
        "\n\tuse GCP_STAMP() to find exact location!");
    GCP_ASSERT(scalar + index >= scalar,
        file,line,"ptr wrap, out of range"
        "\n\tuse GCP_STAMP() to find exact location!");
    GCP_ASSERT(scalar + index >= index,
        file,line,"ptr wrap, out of range"
        "\n\tuse GCP_STAMP() to find exact location!");
    GCP_ASSERT(scalar + index < gco->header.nobj,
        file,line,"out of range access"
        "\n\tuse GCP_STAMP() to find exact location!");
}

template <class A>
size_t GCO_Services<A>::unlink
    (GCO * gco GCP_FILE_LINE_FPARAMS)
{
    if (!--gco->header.nref)  {
        GCP_TEST(!(SP_FLAG & gco->header.flags)
            || (SP_DEL_FLAG & gco->header.flags),
            FILE,LINE,
            "memory leak - ptr never deleted!");
        unsigned char * front = frontFence(gco);
        unsigned char * back  = backFence(gco);
        int frontBreach = 0, backBreach = 0;
        for (size_t i = gco->header.fence_bytes; i--;)
        {
            frontBreach |=
                (*front++ != GCO_FENCE_CHAR_MASK);
            backBreach |=
                (*back++ != GCO_FENCE_CHAR_MASK);
        }
        GCP_TEST(!frontBreach,
            gco->header.file,gco->header.line,
            "object's front fence breached!");
        GCP_TEST(!backBreach,
            gco->header.file,gco->header.line,
            "object's back fence breached!");
        return gco->header.nobj;
    }
    return 0;
}

template <class A>
void GCO_Services<A>::del(GCO * gco, int array,
    const char * FILE, int LINE)
{
    GCP_TEST(gco != 0,FILE,LINE,
        "deleting NULL pointer!");
    if (!gco) return;
    GCP_ASSERT(SP_FLAG & gco->header.flags,
        FILE,LINE,
        "only self policing pointers can be deleted "
        "\n\t- not garbage collecting pointers!");
    GCP_TEST(
        (array && (GCP_ARRAY_FLAG & gco->header.flags))
        || (!array
        && !(GCP_ARRAY_FLAG & gco->header.flags)),
        FILE,LINE,"use delete[] for arrays,"
        " e.g. delete_spa()!");
    GCP_TEST(!(SP_DEL_FLAG & gco->header.flags),
        FILE,LINE,"object already deleted "
        "being deleted again!");
    gco->header.flags |= SP_DEL_FLAG;
    char msg[160];
    ostrstream os(msg,sizeof(msg));
    os << "other pointers still point to object "
       << "being deleted!\n  "
       << " Object";
    if (GCP_ARRAY_FLAG & gco->header.flags)
       os << "[" << gco->header.nobj
           << "]";
    os << " with " << gco->header.nref
       << " links allocated on line "
       << gco->header.line << " of "
       << gco->header.file << "!";
    msg[os.pcount()] = '\0';
    GCP_TEST(gco->header.nref == 1,
        FILE,LINE,msg);
}

#endif

template <class A>
GCO * GCO_Services<A>::alloc(size_t req_size,
    size_t obj_size GCP_FLAGS_FILE_LINE_FPARAMS)
{
    GCO * gco = (GCO *) new GCO_ALIGN[
            GCO_ALIGN_CELLS_REQ(sizeof(GCO))
        #ifndef GCP_NDEBUG
            + GCO_ALIGN_CELLS_REQ(obj_size)
            * GCO_FENCE_OBJS * 2
        #endif
            + GCO_ALIGN_CELLS_REQ(req_size)];
    if (gco)  {
        gco->header.nref = 1;
        gco->header.nobj = req_size / obj_size;
        #ifndef GCP_NDEBUG
            gco->header.flags = FLAGS;
            gco->header.fence_bytes =
                GCO_ALIGN_CELLS_REQ(obj_size)
                * sizeof(GCO_ALIGN) * GCO_FENCE_OBJS;
            gco->header.back_fence_idx =
                GCO_ALIGN_CELLS_REQ(sizeof(GCO))
                * sizeof(GCO_ALIGN)
                + gco->header.fence_bytes + req_size;
            gco->header.file = FILE;
            gco->header.line = LINE;
            unsigned char * front = frontFence(gco);
            unsigned char * back  = backFence(gco);
            for (size_t i = gco->header.fence_bytes;
                i--; )
                *front++ = *back++ 
                    = GCO_FENCE_CHAR_MASK;
        #endif
    }
    return gco;
}

// Parameter types for op+, op+=, op-, op-=
#if sizeof(size_t) == sizeof(long int)
    #define GCO_OFFSET_TYPE  long int
#elif sizeof(size_t) == sizeof(int)
    #define GCO_OFFSET_TYPE  int
#else
    #define GCO_OFFSET_TYPE  short int
#endif

template <class T, class ST, class F>
class GCP : GCO_Services<GCO_ALIGN>  {
    GCO * gco;
    T * obj;
    size_t scalar;
    #ifndef GCP_NDEBUG
        const char * file;
        int line;
    #endif
    void destruct();
public:
    #ifndef GCP_NDEBUG
        void stamp(const char * FILE, int LINE)
            { file = FILE; line = LINE; }
        GCP(const char * FILE, int LINE)
        {
            gco = 0; obj = 0; scalar = 0;
            file = FILE; line = LINE;
        }
    #endif
    GCP()
    {
        gco = 0; obj = 0; scalar = 0;
        #ifndef GCP_NDEBUG
            file = 0; line = 0;
        #endif
    }
    GCP(const GCP<T,ST,F>& gcp)
    { 
        gco = 0; obj = 0; scalar = 0;
        #ifndef GCP_NDEBUG
            file = 0; line = 0;
        #endif
        *this = gcp;
    }
    ~GCP()  { destruct(); }
    void * alloc(size_t size
        GCP_FLAGS_FILE_LINE_FPARAMS)
    {
        if (gco) destruct();
        return object(gco=GCO_Services<GCO_ALIGN>::
            alloc(size,sizeof(T)
            GCP_FLAGS_FILE_LINE_APARAMS));
    }

    // pointer relational operators
    int operator==(const GCP<T,ST,F>& gcp) const
    {
        return ((obj + scalar) ==
            (gcp.obj + gcp.scalar));
    }
    int operator==(const T * tobj) const
    {  return ((obj + scalar) == tobj); }
    int operator>(const GCP<T,ST,F>& gcp) const
    {
        return ((obj + scalar) >
            (gcp.obj + gcp.scalar));
    }
    int operator>(const T * tobj) const
        { return ((obj + scalar) > tobj); }
    int operator<(const GCP<T,ST,F>& gcp) const
        { return (!(*this > gcp) && !(*this == gcp)); }
    int operator<(const T * tobj) const
      { return (!(*this > tobj) && !(*this == tobj)); }
    int operator<=(const GCP<T,ST,F>& gcp) const
        { return !(*this > gcp); }
    int operator<=(const T * tobj) const
        { return !(*this > tobj); }
    int operator>=(const GCP<T,ST,F>& gcp) const
        { return !(*this < gcp); }
    int operator>=(const T * tobj) const
        { return !(*this < tobj); }

    // pointer assignment
    GCP<T,ST,F>& operator=(const GCP<T,ST,F>& gcp);
    GCP<T,ST,F>& operator=(T * tobj);

    // pointer deferencing and scaling
    T & operator[](size_t index) const
    {
      GCP_ASSERT(gco && obj,file,line,
        "attempted access via uninitialized or "
        "deleted pointer!\n\tuse GCP_STAMP() to "
        "find exact location!");
      GCP_ASSERT_SUBSCRIPT(gco,file,line,scalar,index);
      return obj[scalar+index];
    }
    ST * operator->() const
      { return (ST*)&(*this)[0]; }
      // Use gcnsp/spnsp macro forms for
      // non-selectable objs!
    operator T *() const { return obj; }
    T & operator*() const  { return (*this)[0]; }
    GCP<T,ST,F>& operator+=(GCO_OFFSET_TYPE i)
        { scalar += i; return *this; }
    GCP<T,ST,F> operator+(GCO_OFFSET_TYPE i);
    GCP<T,ST,F>& operator++()
        { ++scalar; return *this; }
    GCP<T,ST,F> operator++(int);
    GCP<T,ST,F>& operator-=(GCO_OFFSET_TYPE i)
        { scalar -= i; return *this; }
    GCP<T,ST,F> operator-(GCO_OFFSET_TYPE i);
    GCP<T,ST,F>& operator--()
        { --scalar; return *this; }
    GCP<T,ST,F> operator--(int);
    #ifndef GCP_NDEBUG
      void del(int array, const char * FILE, int LINE)
      {
        GCO_Services<GCO_ALIGN>::
            del(gco,array,FILE,LINE);
        GCP_TEST(!scalar,FILE,LINE,
            "attempting to delete non-base pointer!");
        destruct();
      }
    #endif

};

template <class T, class ST, class F>
void GCP<T,ST,F>::destruct()
{
    if (gco)  {
        size_t nobj = unlink(gco
            #ifndef GCP_NDEBUG
                , file,line
            #endif
            );
        if (nobj)  {
            GCP_TEST(obj != 0,gco->header.file,
                gco->header.line,
                "allocated memory left dangling - "
                "never assigned to a pointer!");
            if (obj)  while (nobj--)
                (&obj[nobj])->T::~T();
            delete gco;
        }
        gco = 0;
    }
    obj = 0; scalar = 0;
}

template <class T, class ST, class F>
GCP<T,ST,F>& GCP<T,ST,F>::
    operator=(const GCP<T,ST,F>& gcp)
{
    GCP_ASSERT(gcp.gco && gcp.obj
        || !gcp.gco && !gcp.obj,
        gcp.file,gcp.line,
        "assignment of pointer that was never "
        "properly initialized!"
        "\n\tuse GCP_STAMP() to find exact location!");
    destruct();
    if ((gco = gcp.gco) != 0)  {
        gco->header.nref++;
        #ifndef GCP_NDEBUG
            if (!file)  {
                file = gco->header.file;
                line = gco->header.line;
            }
        #endif
    }
    obj = gcp.obj;
    scalar = gcp.scalar;
    return *this;
}

template <class T, class ST, class F>
GCP<T,ST,F>& GCP<T,ST,F>::operator=(T * tobj)
{
    if (!tobj)  {
        destruct();
        return *this;
    }
    GCP_ASSERT(gco && !obj,file,line,
        "only dynamic objects allocated via "
        "new_gcp/gcnsp/spp/spnsp(V) can be "
        "assigned to V!");
    GCP_ASSERT(((T*)gco) < tobj
        && tobj < ((T*)backFence(gco)),
        file,line,
        "only dynamic objects allocated via "
        "new_gcp/gcnsp/spp/spnsp(V) can be "
        "assigned to V!");
    obj = tobj;
    #ifndef GCP_NDEBUG
        if (!file)  {
            file = gco->header.file;
            line = gco->header.line;
        }
    #endif
    return *this;
}

template <class T, class ST, class F>
GCP<T,ST,F> GCP<T,ST,F>::operator+(GCO_OFFSET_TYPE i)
{
    GCP<T,ST,F> tmp = *this;
    return (tmp += i);
}

template <class T, class ST, class F>
GCP<T,ST,F> GCP<T,ST,F>::operator++(int)
{
    GCP<T,ST,F> tmp = *this;
    scalar++;
    return tmp;
}

template <class T, class ST, class F>
GCP<T,ST,F> GCP<T,ST,F>::operator-(GCO_OFFSET_TYPE i)
{
    GCP<T,ST,F> tmp = *this;
    return (tmp -= i);
}

template <class T, class ST, class F>
GCP<T,ST,F> GCP<T,ST,F>::operator--(int)
{
    GCP<T,ST,F> tmp = *this;
    scalar--;
    return tmp;
}

#define GCSP_DECL(gcp_T)                       \
inline void * operator new(size_t size,        \
    gcp_T& gcp GCP_FLAGS_FILE_LINE_FPARAMS)    \
{                                              \
    return gcp.alloc(size                      \
        GCP_FLAGS_FILE_LINE_APARAMS);          \
}                                              \
inline void * operator new[](size_t size,      \
     gcp_T& gcp GCP_FLAGS_FILE_LINE_FPARAMS)   \
{                                              \
    return gcp.alloc(size                      \
        GCP_AFLAGS_FILE_LINE_APARAMS);         \
}

#define GCP_DECL(T) GCSP_DECL(gcp(T))
#define GCNSP_DECL(T) GCSP_DECL(gcnsp(T))
#ifdef GCP_NDEBUG
    #define SPP_DECL(T)
    #define SPNSP_DECL(T)
#else
    #define SPP_DECL(T) GCSP_DECL(spp(T))
    #define SPNSP_DECL(T) GCSP_DECL(spnsp(T))
#endif

#endif   //#ifndef gcp_h
