#include "FiveWin.ch"

#define MF_ENABLED       0
#define MF_GRAYED        1
#define MF_DISABLED      2
#define MF_BITMAP        4
#define MF_CHECKED       8
#define MF_POPUP        16  // 0x0010
#define MF_BYPOSITION 1024  // 0x0400
#define MF_SEPARATOR  2048  // 0x0800
#define MF_HELP      16384  // 0x4000
#define MF_HILITE      128  // 0x0080
#define MF_UNHILITE      0

static oLastItem
static hClass

//----------------------------------------------------------------------------//

CLASS TMenu

   DATA   hMenu
   DATA   aItems
   DATA   oWnd
   DATA   lSysMenu, lPopup

   METHOD New( lPopup, oWnd )  CONSTRUCTOR
   METHOD ReDefine( cResName, lPopup ) CONSTRUCTOR
   METHOD NewSys( oWnd )       CONSTRUCTOR
   METHOD Add( oMenuItem )
   METHOD Command( nCommand )

   METHOD LastItem() INLINE oLastItem

   METHOD GetMenuItem( nItemId ) INLINE  SearchItem( ::aItems, nItemId )

   METHOD Activate( nRow, nCol, oWnd )

   METHOD Refresh() INLINE ::oWnd:SetMenu( Self )

   METHOD Reset() INLINE If( ::lSysMenu, GetSystemMenu( ::oWnd:hWnd, .t. ),;
                         DestroyMenu( ::hMenu ) ), ::hMenu := CreateMenu(),;
                         ::aItems := {}

   METHOD Destroy() INLINE DestroyItems( ::aItems )

   METHOD End() INLINE If( ::oWnd != nil, SetMenu( ::oWnd:hWnd, 0 ),),;
                           ::Destroy(), DestroyMenu( ::hMenu )

   METHOD Hilite( nPopUp ) INLINE ;
                           HiliteMenuItem( ::oWnd:hWnd, ::hMenu, nPopUp - 1,;
                                           nOr( MF_HILITE, MF_BYPOSITION ) )

   METHOD UnHilite( nPopUp ) INLINE ;
                           HiliteMenuItem( ::oWnd:hWnd, ::hMenu, nPopUp - 1,;
                                           nOr( MF_UNHILITE, MF_BYPOSITION ) )
   METHOD AddMdi()

ENDCLASS

//----------------------------------------------------------------------------//

METHOD New( lPopup, oWnd ) CLASS TMenu

   DEFAULT lPopup := .f.

   ::hMenu    = If( lPopup, CreatePopupMenu(), CreateMenu() )
   ::aItems   = {}
   ::lSysMenu = .f.
   ::lPopup   = lPopup

   if oWnd != nil
      oWnd:SetMenu( Self )
   endif

return nil

//----------------------------------------------------------------------------//

METHOD ReDefine( cResName, lPopup ) CLASS TMenu

   local hMenu := LoadMenu( GetResources(), cResName )
   local n

   ::hMenu    = hMenu
   ::aItems   = {}
   ::lSysMenu = .f.
   ::lPopup   = lPopup

   if lPopup
      // Windows does not provides a way of storing only Popups in resources
      // so we are going to create one on the fly copying it from the
      // one placed at resources
      ::hMenu = CreatePopupMenu()
      MenuClone( ::hMenu, hMenu )
      DestroyMenu( hMenu )
   endif

   ResBuild( Self )

return nil

//----------------------------------------------------------------------------//

METHOD NewSys( oWnd ) CLASS TMenu

   local n

   if oWnd != nil
      ::oWnd  = oWnd
      ::hMenu = GetSystemMenu( oWnd:hWnd, .f. )
   endif
   ::aItems   = {}
   ::lSysMenu = .t.

return nil

//----------------------------------------------------------------------------//

METHOD Add( oMenuItem ) CLASS TMenu

   AAdd( ::aItems, oMenuItem )

   oMenuItem:oMenu = Self

   if oMenuItem:bAction:ClassName() == "TMENU"
      if oMenuItem:hBitmap == 0
         AppendMenu( ::hMenu, MF_POPUP, oMenuItem:bAction:hMenu,;
                     oMenuItem:cPrompt )
      else
         AppendMenu( ::hMenu, nOR( MF_POPUP, MF_BITMAP ),;
                     oMenuItem:bAction:hMenu, oMenuItem:hBitmap )
      endif
   else
      if oMenuItem:cPrompt != nil
         AppendMenu( ::hMenu,;
                     nOR( If( oMenuItem:lActive, MF_ENABLED,;
                          nOR( MF_DISABLED, MF_GRAYED ) ),;
                          If( oMenuItem:lChecked, MF_CHECKED, 0 ),;
                          If( oMenuItem:lHelp, MF_HELP, 0 ) ),;
                     oMenuItem:nId, oMenuItem:cPrompt )
      else
         if oMenuItem:hBitmap != 0
            AppendMenu( ::hMenu, MF_BITMAP,;
                        oMenuItem:nId, oMenuItem:hBitmap )
         else
            AppendMenu( ::hMenu, MF_SEPARATOR, oMenuItem:nId, "" )
         endif
      endif
   endif

return nil

//----------------------------------------------------------------------------//

METHOD Command( nCommand ) CLASS TMenu

   local oMenuItem := ::GetMenuItem( nCommand )

   if oMenuItem != nil
      if ValType( oMenuItem:bAction ) == "B"
         oLastItem = oMenuItem
         Eval( oMenuItem:bAction )
      endif
   endif

return nil

//----------------------------------------------------------------------------//

METHOD Activate( nRow, nCol, oWnd ) CLASS TMenu

   if oWnd != nil
      if oWnd:oPopup != nil
         oWnd:oPopup:End()
      endif
      oWnd:oPopup = Self
      TrackPopup( ::hMenu, 1, nRow, nCol, 0, oWnd:hWnd )
      SysRefresh()
      ::End()
      oWnd:oPopup = nil
   endif

return nil

//----------------------------------------------------------------------------//

static function SearchItem( aItems, nId )

   local n      := 1
   local lFound := .f.
   local oReturn

   if hClass == nil
      hClass = TMenu():ClassH()
   endif

   while n <= Len( aItems ) .and. ! lFound
      if aItems[ n ]:nId == nId
         return aItems[ n ]
      else
         if aItems[ n ]:bAction:ClassH() == hClass
            if aItems[ n ]:bAction:hMenu == nId
               return nil
            else
               oReturn = SearchItem( aItems[ n ]:bAction:aItems, nId )
               if oReturn != nil
                  exit
               endif
            endif
         endif
      endif
      n++
   end

return oReturn

//----------------------------------------------------------------------------//

static function DestroyItem( aItems )

   local n := 1
   local oItem

   if hClass == nil
      hClass = TMenu():ClassH()
   endif

   while n <= Len( aItems )
      oItem = aItems[ n ]
      if oItem:bAction != nil
         if oItem:bAction:ClassH() == hClass
            DestroyItem( oItem:bAction:aItems )
         endif
      endif
      oItem:End()
      n++
   end

return nil

//----------------------------------------------------------------------------//

static function ResBuild( oMenu )

   local n
   local hMenu := oMenu:hMenu
   local hSubMenu
   local oSubMenu, oMenuItem

   for n = 1 to GetMItemCount( hMenu )
         oMenuItem          = TMenuItem()
         oMenuItem:nId      = GetMItemID( hMenu, n - 1 )
         oMenuItem:cPrompt  = GetMenuString( hMenu, n - 1, MF_BYPOSITION )
         oMenuItem:cMsg     = ""
         oMenuItem:lChecked = lAnd( GetMenuState( hMenu, n - 1, MF_BYPOSITION ), MF_CHECKED )
         oMenuItem:lActive  = lAnd( GetMenuState( hMenu, n - 1, MF_BYPOSITION ), MF_ENABLED )
         oMenuItem:oMenu    = oMenu
         oMenuItem:hBitmap  = 0
         AAdd( oMenu:aItems, oMenuItem )
         if ( hSubMenu := GetSubMenu( hMenu, n - 1 ) ) != 0
            oSubMenu          = TMenu()
            oSubMenu:hMenu    = hSubMenu
            oSubMenu:lSysMenu = .f.
            oSubMenu:aItems   = {}
            oMenuItem:bAction = oSubMenu
            ResBuild( oSubMenu )
         endif
   next

return nil

//----------------------------------------------------------------------------//

static function MenuClone( hTarget, hSource )

   local n
   local hSubTarget, hSubSource

   for n = 1 to GetMItemCount( hSource )
      hSubSource = GetSubMenu( hSource, n - 1 )
      if hSubSource != 0
         hSubTarget = CreatePopupMenu()
         MenuClone( hSubTarget, hSubSource )
      endif
      if lAnd( GetMenuState( hSource, n - 1, MF_BYPOSITION ), MF_SEPARATOR )
         AppendMenu( hTarget, MF_SEPARATOR, GetMItemId( hSource, n - 1 ), "" )
      else
         AppendMenu( hTarget,;
                     nLoByte( GetMenuState( hSource, n - 1, MF_BYPOSITION ) ),;
                     If( hSubSource != 0, hSubTarget,;
                         GetMItemId( hSource, n - 1 ) ),;
                     GetMenuString( hSource, n - 1, MF_BYPOSITION ) )
      endif
   next

return nil

//----------------------------------------------------------------------------//

METHOD AddMdi() CLASS TMenu

   MENUITEM "&Window"
   MENU
      MENUITEM "&Tile Vertical"   ACTION ::oWnd:Tile()
      MENUITEM "&Tile Horizontal" ACTION ::oWnd:Tile( .t. )
      MENUITEM "&Cascade"         ACTION ::oWnd:Cascade()
      MENUITEM "&Next Window"     ACTION ::oWnd:NextWindow()
      MENUITEM "&Arrange Icons"   ACTION ::oWnd:ArrangeIcons()
      MENUITEM "&Iconize All"     ACTION ::oWnd:IconizeAll()
      MENUITEM "&Close All"       ACTION ::oWnd:CloseAll()
   ENDMENU

return nil

//----------------------------------------------------------------------------//
