/****************************************************************************

    Module  : DockWin.c

    Version : v1.0alpha

    Date    : 20/8/93

    Changes :  20/8/93 - Changed code which distinguishes between Windows
               .EXEs & DOS .EXEs to correctly choose.
               20/8/93 - Added use of RePaintSlot func to individually repaint
               slots.

               13/9/93 - Changed code which sets the window position when the
               user starts an application which has it's window position
               stored. Code now finds top parent window of currently active
               window instead of currently active window.

               30/3/94 - corrected code to handle the mail box shrinking &
               getting the correct size after. Tidied up the mail timer when
               the app finished. Corrected a bug which allowed the right
               button click to be processed whilst a slot was being dragged.

*****************************************************************************/

/****************************************************************************

    FUNCTION: DockWinProc(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)

    VERSION : v1.0

    PURPOSE :

    MESSAGES:

    COMMENTS:

****************************************************************************/


#include <windows.h>            /* include standard windows header */
#include <shellapi.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys\stat.h>
#include <string.h>
#include "freedock.h"           /* include control message definitions */

/**********************************************************
	Global Data for the various system metrics we need
**********************************************************/
extern GLOBAL_METRICS gMetrics;
extern GLOBAL_OPTIONS gOptions;

FARPROC  lpfnAboutDockDlgProc;	// Used in dockutil.c


long FAR PASCAL DockWinProc(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	static SLOT_ENTRY    *Slot;
	static DOCK_STRUCT    Dock;
	CREATESTRUCT		 *create;

    PAINTSTRUCT     ps;         /* paint struct , used when painting window */
    static HDC      Win_hdc, Screen_hdc;
    static HCURSOR  hRemCur;
    UINT            SlotHit, FileIndex, nItem, Status;
    int             i, StartSlot, EndSlot;
    HANDLE          hDrop;
    int             SlotIndex, nDropSlot;
	UINT			nNumDropped;
    char            DroppedFileName[MAX_FPATH_LEN], TmpBuffer[MAX_FPATH_LEN];
    char           *TmpCharPtr;
    POINT           DropPoint;
    char            CommandLine[MAX_CMDLINE_LEN];
    static FARPROC  lpfnAppOptionsDlgProc, lpfnMainOptionsDlgProc, lpfnMailOptionsDlgProc, lpfnClockOptionsDlgProc,
                    lpfnWinExitOptionsDlgProc;
    static UINT     OrigSlot;
    int             NewX, NewY;
    static BOOL     bDragIcon, bRemoveSlot, bSlotOffDock, bSlotMove;
    static HBITMAP  hbmUnderIcon, hbmNewUnderIcon, hbmTmp;
    static HBITMAP  hbmOldUnderIcon, hbmOldNewUnderIcon, hbmOldTmp, hbmOldIconCache;
    static HDC      hdcUnderIcon, hdcNewUnderIcon, hdcTmp;
    struct stat     StatBuf;

    static int      OldLeft, OldTop, OffLeft, OffTop;  // Old coordinates and Offset coordinates
    static RECT     MoveRect, Rect;
    static BOOL     bDragDock;
    static BOOL     bWindowPositioned = FALSE;
	static HCURSOR  hCurHand, hCurOld;

    /* Select messages which are handled by this procedure */

    switch (message) {

        case WM_CREATE:             /* The window has just been created */
            /********************************************************
                malloc the memory for the maximum number of slots
            ********************************************************/
            Slot = (SLOT_ENTRY *)calloc(gOptions.MaxDockSize, sizeof(SLOT_ENTRY));
            if(Slot == NULL){
                MessageBox( NULL, "FreeDock Error",
                    "Cannot allocate memory for dock slots",
                    MB_ICONSTOP | MB_OK); 
                exit(-3);
            }
			Dock.Slot = Slot;

            /*********************************************
                No need to initialise all slot types to
                being free since SLOT_FREE is 0 and we
                used calloc to get the memory
            *********************************************/
			/***************************
				Read Dock Options
			***************************/
			ReadDockOptions( &Dock );

			/**********************************
				Initialise Dock Window var 
			**********************************/
			Dock.hwndDock = hwnd;

			/*******************************
				Is this the Root Dock ?
			*******************************/
			create = (CREATESTRUCT *)lParam;
			if( create->hwndParent == NULL ){
				Dock.DockType = ROOT_DOCK;
			}
			else{
				Dock.DockType = SUB_DOCK;
			}

           /**********************************************************************
                Create bitmaps and device contexts for the icon cache Item at end
                of cache is for TmpSlot/Icons scratch space
				These are always the default (correct) slot width and height so
				that the slot icons & buttons are stored at 1:1 and then scaled
				when painted on the dock
            **********************************************************************/
            Win_hdc = GetDC(hwnd);
            Dock.hbmIconCache = CreateCompatibleBitmap( Win_hdc,  DEF_SLOT_WIDTH * (gOptions.MaxDockSize+1), DEF_SLOT_HEIGHT );
            Dock.hdcIconCache = CreateCompatibleDC(Win_hdc);
            hbmOldIconCache = SelectObject(Dock.hdcIconCache, Dock.hbmIconCache);
            ReleaseDC( hwnd, Win_hdc );
			/**********************************************************            
            	Initialise each Slots Icon position within the cache
				and the pointer to the parent Dock
			**********************************************************/                                                                       
			for( i=0; i < gOptions.MaxDockSize; i++ ){
			 	Slot[i].IconIndex = i * DEF_SLOT_WIDTH;
				Slot[i].Dock = &Dock;
			}                                                                       
			
            /**************************
                Read in all slot info
            **************************/

            ReadAllSlotOptions(&Dock, Slot);

			/*****************************************************
				Limit Dock Size to max value set in .INI file
			*****************************************************/
			Dock.SlotCount = (Dock.SlotCount < gOptions.MaxDockSize)?Dock.SlotCount:gOptions.MaxDockSize;

            /****************************************************
                Create a bitmap to store what is under the icon
                when moving a slot, + A scratch pad of the same
                size. (Tmp)
            ****************************************************/
            Win_hdc = GetDC(hwnd);
			hbmUnderIcon = CreateCompatibleBitmap( Win_hdc,
                                                   MAX_SLOT_WIDTH,
                                                   MAX_SLOT_HEIGHT );
            hbmNewUnderIcon = CreateCompatibleBitmap( Win_hdc,
                                                      MAX_SLOT_WIDTH,
                                                      MAX_SLOT_HEIGHT );
            hbmTmp = CreateCompatibleBitmap( Win_hdc,
                                             MAX_SLOT_WIDTH,
                                             MAX_SLOT_HEIGHT );
            hdcUnderIcon    = CreateCompatibleDC(Win_hdc);
            hdcNewUnderIcon = CreateCompatibleDC(Win_hdc);
            hdcTmp          = CreateCompatibleDC(Win_hdc);
            ReleaseDC(hwnd, Win_hdc);

			hbmOldUnderIcon    = SelectObject(hdcUnderIcon,    hbmUnderIcon);
            hbmOldNewUnderIcon = SelectObject(hdcNewUnderIcon, hbmNewUnderIcon);
            hbmOldTmp          = SelectObject(hdcTmp,          hbmTmp);

            hCurHand = LoadCursor(gOptions.hAppInst, "Hand");

            /******************************************************
                Identify this window as a drag and drop acceptor
            ******************************************************/
            DragAcceptFiles(hwnd, TRUE);

            bDragIcon = FALSE;
            bDragDock = FALSE;
            /**************************************************
            * Create all proc instances for all dialog boxes *
            **************************************************/
            lpfnAppOptionsDlgProc = MakeProcInstance((FARPROC) AppOptionsDlgProc, gOptions.hAppInst);
            lpfnWinExitOptionsDlgProc = MakeProcInstance((FARPROC) WinExitOptionsDlgProc, gOptions.hAppInst);
            lpfnMailOptionsDlgProc = MakeProcInstance((FARPROC) MailOptionsDlgProc, gOptions.hAppInst);
            lpfnMainOptionsDlgProc = MakeProcInstance((FARPROC) MainOptionsDlgProc, gOptions.hAppInst);
            lpfnClockOptionsDlgProc = MakeProcInstance((FARPROC) ClockOptionsDlgProc, gOptions.hAppInst);
            lpfnAboutDockDlgProc = MakeProcInstance((FARPROC) AboutDockDlgProc, gOptions.hAppInst);

            if (gOptions.AlwaysOnTop) {
                SetWindowPos(hwnd, HWND_TOPMOST, 0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE | SWP_NOACTIVATE);
            }
            if (gOptions.MailActive) {
                SetTimer(hwnd, MAIL_TIMER, gOptions.MailFreq * 1000 * 60, NULL);
            }
            return 0;               /* end of WM_CREATE */


        case WM_PAINT:
            /** Make sure the dock is in the correct position **/
            /** Cannot do this from WM_CREATE Message         **/
    		if( !bWindowPositioned ){
    		    SetDockWinPos( &Dock, Dock.Orientation + QX_HORZ );
                bWindowPositioned = TRUE;
            }

			Win_hdc = BeginPaint(hwnd, &ps);
            StartSlot = FindSlotHit(Dock.Orientation, ps.rcPaint.left, ps.rcPaint.top);
            EndSlot = FindSlotHit(Dock.Orientation, ps.rcPaint.right, ps.rcPaint.bottom);

			for (i = StartSlot; i <= EndSlot; i++) {
                PaintSlot(Win_hdc, &Dock, &Slot[i]);
            }
			EndPaint(hwnd, &ps);
			return 0;


        case WM_TIMER:
			if(Dock.DockType != ROOT_DOCK) return 0;
            if (wParam == MAIL_TIMER) {
                stat(gOptions.MailPath, &StatBuf);
                if (StatBuf.st_size > gOptions.MailBoxSize) {
                    gOptions.MailBoxSize = (unsigned long) StatBuf.st_size;
                    i = 0;
                    while ((Slot[i].SlotType != SLOT_SPECIAL_MAIL) && (i < 4))
                        i++;
                    if (Slot[i].SlotType != SLOT_SPECIAL_MAIL){
                        KillTimer(hwnd, MAIL_TIMER);
                        return 0;
                    }
                    gOptions.bMailInBox = TRUE;
                    UtilLoadIcon( &Dock, &Slot[i] );
                    if (gOptions.MailSound) {
                        MessageBeep(MB_ICONHAND);
                    }
                    RePaintSlot(&Dock, &Slot[i], FALSE);
                } 
                else if (StatBuf.st_size < gOptions.MailBoxSize) {
                    gOptions.MailBoxSize = (unsigned long) StatBuf.st_size;
                }
            }
            return 0;

        case WM_LBUTTONDOWN:
            SetCapture(hwnd);
            OldLeft = LOWORD(lParam);
            OldTop = HIWORD(lParam);
            OrigSlot = FindSlotHit(Dock.Orientation, OldLeft, OldTop);
            if( Slot[OrigSlot].SlotType == SLOT_USED ){
                if(!gOptions.DockLocked) hCurOld = SetCursor(hCurHand);

                /***********************************************************
                    Delete slot temporaraly & FORCE a repaint before
                    going any further.
                ***********************************************************/
                Slot[OrigSlot].SlotType = SLOT_FREE;
                bDragIcon = TRUE;
                bSlotMove = FALSE;
                bRemoveSlot = FALSE;
                bDragDock = FALSE;
            }
            else if(Slot[OrigSlot].SlotType == SLOT_SPECIAL_TITLE){
				Screen_hdc = GetDC(NULL);
				bDragDock = TRUE;
				bDragIcon = FALSE;
				OffLeft = OldLeft; // Save offset of mouse
				OffTop = OldTop; // from top left of rect
				MoveRect.left   = Dock.DockLeft;
				MoveRect.top    = Dock.DockTop;
				if(Dock.Orientation == DOCK_HORZ){
					MoveRect.bottom = MoveRect.top + gOptions.SlotHeight;
					MoveRect.right  = MoveRect.left + Dock.SlotCount*gOptions.SlotHeight;
				}
				else{
					MoveRect.bottom = MoveRect.top + Dock.SlotCount*gOptions.SlotHeight;
					MoveRect.right  = MoveRect.left + gOptions.SlotWidth;
				}
				DrawFocusRect( Screen_hdc, &MoveRect );
            }
            else{
                bDragDock = FALSE;
                bDragIcon = FALSE;
                bRemoveSlot = FALSE;
            }
            return 0;

        case WM_LBUTTONUP:
            ReleaseCapture();
			if (bDragDock){
				DrawFocusRect( Screen_hdc, &MoveRect ); // Erase old frame
				bDragDock = FALSE;

				Dock.DockLeft += ((short int)LOWORD(lParam) - OffLeft);
				Dock.DockTop += ((short int)HIWORD(lParam) - OffTop);
				SetDockWinPos( &Dock, Dock.Orientation + QX_HORZ ); // Move Dock to new position

				ReleaseDC(NULL, Screen_hdc);
				WriteDockOptions( &Dock );	// Update Position entry in ini file
			}
			else if( bDragIcon ){
                if(!gOptions.DockLocked) SetCursor(hCurOld);

                if(bSlotMove){
                    Win_hdc = GetDC(hwnd);
					BitBlt( Win_hdc, OldLeft-(gOptions.SlotWidth/2), OldTop-(gOptions.SlotHeight/2), 
							gOptions.SlotWidth, gOptions.SlotHeight, hdcUnderIcon, 0, 0, SRCCOPY);
                    ReleaseDC(hwnd, Win_hdc);
				}

				// Reset Slot status flag to true state;
                Slot[OrigSlot].SlotType = SLOT_USED;
				// Find the slot hit when the button was released, if the dock is locked
				// fake this by setting the hit slot to the original slot
				if( gOptions.DockLocked ){
					SlotHit = OrigSlot;
				}
				else{
	                SlotHit = FindSlotHit(Dock.Orientation, LOWORD(lParam), HIWORD(lParam));
				}

                if(bRemoveSlot){
                	UtilEmptySlot( &Slot[OrigSlot] );
					WriteSlotOptions( &Dock, &Slot[OrigSlot] );	// Delete entry in ini file
                    if(bSlotMove)
                        RePaintSlot(&Dock, &Slot[OrigSlot], TRUE);
                }
                else if (SlotHit != OrigSlot){   // if the slot actually moved

                    if( (Slot[SlotHit].SlotType == SLOT_USED) ||
                    	(Slot[SlotHit].SlotType == SLOT_FREE)){

						SwapSlots( &Slot[OrigSlot], &Slot[SlotHit] );

                        if(bSlotMove)
                            RePaintSlot( &Dock, &Slot[OrigSlot], FALSE );
                    }
                    else {   // Slot must be a special, can't move a slot to here.
                        if(bSlotMove)
                            RePaintSlot( &Dock, &Slot[OrigSlot], FALSE );
                    }

					WriteSlotOptions( &Dock, &Slot[SlotHit] );	// Update entry in ini file
					WriteSlotOptions( &Dock, &Slot[OrigSlot] );	// Update entry in ini file
                }
				else{
					RePaintSlot( &Dock, &Slot[SlotHit], FALSE );
				}

                if(bSlotMove){
                    RePaintSlot(&Dock, &Slot[SlotHit-1], TRUE);
                    RePaintSlot(&Dock, &Slot[SlotHit]  , TRUE);
                    RePaintSlot(&Dock, &Slot[SlotHit+1], TRUE);
                }
                else{
                    /*******************************************
                        Slot did not move, so treat this
                        as a single left click message &
                        run the app, if SingleClickStart
                        is TRUE
                    *******************************************/
                    if( gOptions.SingleClickStart ){
                        SendMessage( hwnd, WM_LBUTTONDBLCLK, wParam, lParam );
                    }
                }
                bDragIcon = FALSE;
            }
            return 0;

        case WM_MOUSEMOVE:
			if (bDragDock){
				DrawFocusRect( Screen_hdc, &MoveRect ); // Erase old frame
				OldLeft = (short int)LOWORD(lParam);
				OldTop  = (short int)HIWORD(lParam);
				MoveRect.left   = (Dock.DockLeft + OldLeft) - OffLeft;
				MoveRect.top    = (Dock.DockTop + OldTop) - OffTop;

				if(Dock.Orientation == DOCK_HORZ){
					MoveRect.bottom = MoveRect.top + gOptions.SlotHeight;
					MoveRect.right  = MoveRect.left + Dock.SlotCount*gOptions.SlotHeight;
				}
				else{
					MoveRect.bottom = MoveRect.top + Dock.SlotCount*gOptions.SlotHeight;
					MoveRect.right  = MoveRect.left + gOptions.SlotWidth;
				}
				DrawFocusRect( Screen_hdc, &MoveRect );
			}
			else if(bDragIcon && !gOptions.DockLocked){ //:-)
                if( IsCursorOutOfDock( &Dock, (int)LOWORD(lParam), (int)HIWORD(lParam)) ){
                    if(!bRemoveSlot){
                        bSlotOffDock = FALSE;
                        bRemoveSlot = TRUE;
                        hRemCur = LoadCursor(gOptions.hAppInst, "RemoveSlot");
                        SetCursor( hRemCur );
                    }
                    if(bSlotOffDock) return(TRUE);
                }
                else if(bRemoveSlot){
                    bRemoveSlot = FALSE;
                    SetCursor(hCurHand);
                    DestroyCursor(hRemCur);
                }

                if( !bSlotMove ){
                    Win_hdc = GetDC(hwnd);

					/******************************************
                         Paint the area under the blank slot
                    ******************************************/
					BlankSlot( Win_hdc, &Dock,  &Slot[OrigSlot] );

                    /*****************************************
                    Copy what will be under the Icon
                    *****************************************/
                    BitBlt( hdcUnderIcon, 0, 0, gOptions.SlotWidth, gOptions.SlotHeight,
                            Win_hdc, OldLeft-(gOptions.SlotWidth/2), OldTop-(gOptions.SlotHeight/2), SRCCOPY);
                    /********************
                        Draw the Icon
                    ********************/
                    UtilDrawIcon( Win_hdc, &Dock, &Slot[OrigSlot], OldLeft-(gOptions.SlotWidth/2), 
                    			  OldTop-(gOptions.SlotHeight/2), TRUE);

                    ReleaseDC(hwnd, Win_hdc);
					bSlotMove = TRUE;
                    bDragIcon = TRUE;
                    return(TRUE);
                }

                /*****************************************************************
                    The following is an attempt at getting flicker free dragging
                    of a slot around the dock, third buffer required to eliminate
                    trailing/leading edge flicker when dragging
                *****************************************************************/
                NewX = LOWORD(lParam);
                NewY = HIWORD(lParam);
                Win_hdc = GetDC(hwnd);
				// Blit what will be covered to Tmp
                BitBlt( hdcNewUnderIcon, 0, 0, gOptions.SlotWidth, gOptions.SlotHeight,
                        Win_hdc, NewX-(gOptions.SlotWidth/2), NewY-(gOptions.SlotHeight/2), SRCCOPY);

                // Blit UnderIcon to old-new in tmp
                BitBlt( hdcNewUnderIcon, (OldLeft)-NewX, (OldTop)-NewY, gOptions.SlotWidth, gOptions.SlotHeight,
                        hdcUnderIcon, 0, 0, SRCCOPY);

                // Draw complete Icon into buffer & then blit to screen
                UtilDrawIcon( hdcTmp, &Dock, &Slot[OrigSlot], 0, 0, TRUE );

                // Draw Icon into Under Icon Bmp at new-old
                UtilDrawIcon( hdcUnderIcon, &Dock, &Slot[OrigSlot], NewX-(OldLeft), NewY-(OldTop), TRUE);

                // Blit complete icon to screen
                BitBlt( Win_hdc, NewX-(gOptions.SlotWidth/2), NewY-(gOptions.SlotHeight/2), 
                		gOptions.SlotWidth, gOptions.SlotHeight, hdcTmp, 0, 0, SRCCOPY);

                // Replace what was covered by the icon, and is not covered by new pos
                BitBlt( Win_hdc, (OldLeft-(gOptions.SlotWidth/2)), (OldTop-(gOptions.SlotHeight/2)), 
                		gOptions.SlotWidth, gOptions.SlotHeight, hdcUnderIcon, 0, 0, SRCCOPY);

                // Save new image of what's under the icon.
                BitBlt( hdcUnderIcon, 0, 0, gOptions.SlotWidth, gOptions.SlotHeight,
                        hdcNewUnderIcon, 0, 0, SRCCOPY);

                // Save new posn
                OldLeft = NewX;
                OldTop = NewY;

                if(!bSlotOffDock) bSlotOffDock = TRUE;
                ReleaseDC(hwnd, Win_hdc);
			}
            return 0;


        case WM_LBUTTONDBLCLK:
            /*******************************************************************
               First need to calculate which button has been hit, by dividing
               the y coordinate by the height of the buttons and subtracting
               the number of special buttons operational and 1 for the title
               Icon.
            *******************************************************************/

            SlotHit = FindSlotHit(Dock.Orientation, LOWORD(lParam), HIWORD(lParam));

            switch (Slot[SlotHit].SlotType) {
                case SLOT_USED:
					RePaintSlot( &Dock, &Slot[SlotHit], FALSE);
                    ExecSlot( &Slot[SlotHit], Slot[SlotHit].CmdLine );
                    break;

                case SLOT_SPECIAL_MAIL:
                	gOptions.bMailInBox = FALSE;
                    UtilLoadIcon( &Dock, &Slot[SlotHit] );
                    RePaintSlot(&Dock, &Slot[SlotHit], FALSE);
                    break;

                case SLOT_SPECIAL_EXIT:
                    if (gOptions.WinExitConfirm) {
                        nItem = MessageBox(hwnd, "Do you really want to exit Windows ?",
                            "FreeDock", MB_OKCANCEL | MB_ICONSTOP);
                        if (nItem == IDOK) {
                            ExitWindows(0, 0);
                        }
                    } 
                    else {
                        ExitWindows(0, 0);
                    }
                    break;

                case SLOT_SPECIAL_TITLE:
                    DialogBox(gOptions.hAppInst, "ABOUTDOCKDLG", hwnd, lpfnAboutDockDlgProc);
                    break;

            }
            return 0;

        case WM_DROPFILES:
            hDrop = (HANDLE)wParam;

            /***********************************************************
               Find out if the drop point was in the client area.
               This must be the case, or an error has occurred.
            ***********************************************************/
			if( ! DragQueryPoint(hDrop, (LPPOINT)&DropPoint) ){
            	sprintf(TmpBuffer, "Error Occurred : DockWinProc()\n(WM_DROPFILES, Drop point not in client area).");
            	MessageBox(hwnd, TmpBuffer, "FreeDock", MB_OK | MB_ICONSTOP);
				break;
			}

            /*******************************************
                Find out how many files were dropped
            *******************************************/
            nNumDropped = DragQueryFile(hDrop, -1, (LPSTR)NULL, 0);

			if( nNumDropped < 1 ){
            	sprintf(TmpBuffer, "Error Occurred : DockWinProc()\n(WM_DROPFILES, DragQuery returned %d).", nNumDropped);
            	MessageBox(hwnd, TmpBuffer, "FreeDock", MB_OK | MB_ICONSTOP);
				break;
			}

            SlotHit = FindSlotHit(Dock.Orientation, DropPoint.x, DropPoint.y);

            switch (Slot[SlotHit].SlotType) {

                case SLOT_USED:
                    /***********************************************************
                        Build up the command line and then call the ExecSlot
                        function to execute the associated program
                    ***********************************************************/
                    sprintf(CommandLine, "%s ", Slot[SlotHit].CmdLine);
                    for (FileIndex = 0; FileIndex < nNumDropped; FileIndex++) {
                        DragQueryFile(hDrop, FileIndex, DroppedFileName, MAX_FPATH_LEN);
                        /************************************************
                            Check if this file has an extension, if not
                            add a trailing '.'
                        ************************************************/
                        if (!strchr(&DroppedFileName[strlen(DroppedFileName) - 4], '.')) {
                            DroppedFileName[strlen(DroppedFileName) + 1] = '\0';
                            DroppedFileName[strlen(DroppedFileName)] = '.';
                        }

						// Add Filename to Commandline if it will fit
						if( (strlen(CommandLine) + strlen(DroppedFileName)+1) < MAX_CMDLINE_LEN ) {
                        	strcat(CommandLine, DroppedFileName);
                        	strcat(CommandLine, " ");
						}
						else{
            			    MessageBox(hwnd, "Too many files dropped to fit on application command line, ignoring extras.",
                 			           "FreeDock Error", MB_OK | MB_ICONSTOP);
						 	break;
						}
                    }

                    ExecSlot( &Slot[SlotHit], CommandLine );

                    break;

                case SLOT_FREE:
                    nDropSlot = SlotHit;
                    SlotIndex = SlotHit;
                    for (FileIndex = 0; FileIndex < nNumDropped; FileIndex++) {
                        DragQueryFile(hDrop, FileIndex, DroppedFileName, MAX_FPATH_LEN);
                        Slot[SlotIndex].SlotType = SLOT_USED;
                        Slot[SlotIndex].StartState = START_NORMAL;
                        Slot[SlotIndex].WinX = DEF_STORE_X;
                        Slot[SlotIndex].WinY = DEF_STORE_Y;
                        Slot[SlotIndex].WinWidth = DEF_STORE_W;
                        Slot[SlotIndex].WinHeight = DEF_STORE_H;
                        strcpy(Slot[SlotIndex].AppName, DroppedFileName);
                        strcpy(Slot[SlotIndex].RunTimeDir, DroppedFileName);
                        strcpy(Slot[SlotIndex].IconFile, DroppedFileName);
                        Slot[SlotIndex].IconPos = 0;

                        /************************************************
                           Set Run Time Dir to be same path as filename
                        ************************************************/
                        TmpCharPtr = strrchr(Slot[SlotIndex].RunTimeDir, '\\');
                        if (*(TmpCharPtr-1) == ':')
                            *(TmpCharPtr+1) = '\0';
                        else
                            *TmpCharPtr = '\0';

                        /*****************************************************************
                           Check What type of file has been dropped, is it a .PIF, .COM
                           or a .BAT ?
                        *****************************************************************/
                        if (!stricmp(&DroppedFileName[strlen(DroppedFileName) - 4], ".PIF") ||
                            !stricmp(&DroppedFileName[strlen(DroppedFileName) - 4], ".COM") ||
                            !stricmp(&DroppedFileName[strlen(DroppedFileName) - 4], ".BAT")) {


		                    Status = (UINT)FindExecutable("progman.exe",
    		                                              NULL,
        		                                          Slot[SlotIndex].IconFile);
            		        if( Status <= 31 ){
                		            MessageBox(hwnd, "Program Manager not found, cannot select default Icon.",
                    		                   "FreeDock Error", MB_OK | MB_ICONSTOP);
		                    }
                            Slot[SlotIndex].IconPos = 1;
                            // Load the Icon into the slot's own bitmap
                            UtilLoadIcon( &Dock, &Slot[SlotIndex] );
                        }
                        /******************************************************************
                            Is is a .EXE, if so is it a Windows EXE or a DOS EXE ?
                        ******************************************************************/
                        else if (!stricmp(&DroppedFileName[strlen(DroppedFileName) - 4], ".EXE")) {
                            // Load the Icon into the slot's own bitmap
                            UtilLoadIcon( &Dock, &Slot[SlotIndex] );
                            if (Slot[SlotIndex].IconTotal == 0) {

			                    Status = (UINT)FindExecutable("progman.exe",
    			                                              NULL,
        			                                          Slot[SlotIndex].IconFile);
            			        if( Status <= 31 ){
                			            MessageBox(hwnd, "Program Manager not found, cannot select default Icon.",
                    			                   "FreeDock Error", MB_OK | MB_ICONSTOP);
		                    	}

                                Slot[SlotIndex].IconPos = 1;
                                // Load the Icon into the slot's own bitmap
                                UtilLoadIcon( &Dock, &Slot[SlotIndex] );

                            }
                        }
                        /******************************************************************
                            It must be a document file, try to find an association for it
                        ******************************************************************/
                        else {
                            strcpy(Slot[SlotIndex].CmdLine, Slot[SlotIndex].AppName);
                            // Read the executable name in,
                            // then call SearchPath to get entire path for application.
                            Status = (UINT)FindExecutable(Slot[SlotIndex].CmdLine,
                                Slot[SlotIndex].RunTimeDir,
                                Slot[SlotIndex].AppName);
                                
                            // Now call find executable to ensure we have a complete
                            // path to the application
                            if( FindExecutable(Slot[SlotIndex].AppName,
                                Slot[SlotIndex].RunTimeDir,
                                TmpBuffer) > (HINSTANCE)32 ){
	                          	strcpy( Slot[SlotIndex].AppName, TmpBuffer);
	                        }
                            
                            if (Status > 32) {
                                strcpy(Slot[SlotIndex].IconFile, Slot[SlotIndex].AppName);
                                Slot[SlotIndex].IconPos = 0;
                                // Load the Icon into the slot's own bitmap
                                UtilLoadIcon( &Dock, &Slot[SlotIndex] );
                            } 
                            else {
                                sprintf(TmpBuffer, "%s\nis not an executable or an associated file", Slot[SlotIndex].CmdLine);
                                MessageBox(hwnd, TmpBuffer, "FreeDock", MB_OK | MB_ICONSTOP);
                                Slot[SlotIndex].SlotType = SLOT_FREE;
                            }
                        }

                        RePaintSlot(&Dock, &Slot[SlotIndex], FALSE);

                        WriteSlotOptions(&Dock, &Slot[SlotIndex]);    // Save the new slot

                        SlotIndex++;
                        while ((SlotIndex != nDropSlot) && (Slot[SlotIndex].SlotType != SLOT_FREE)) {
                            SlotIndex++;
                            if (SlotIndex >= Dock.SlotCount)
                                SlotIndex = 1;
                        }
                        if (SlotIndex == nDropSlot)
                            break;
                    }
                    break;

            }

            DragFinish(hDrop);

            return 0;

        case WM_RBUTTONDOWN:
            /**********************************************************
              If we are currently dragging a slot or the dock  around
              then ignore any buttons being pressed
          ************************************************************/
            if( bDragIcon || bDragDock ) return 0;

            SlotHit = FindSlotHit(Dock.Orientation, LOWORD(lParam), HIWORD(lParam));

            switch (Slot[SlotHit].SlotType) {
                case SLOT_USED:
                case SLOT_FREE:
                    Status = DialogBoxParam( gOptions.hAppInst, "APPOPTIONSDLG", hwnd, 
                    						lpfnAppOptionsDlgProc, (LPARAM)&Slot[SlotHit] );
                    if(Status) WriteSlotOptions(&Dock, &Slot[SlotHit]);
                    break;

                case SLOT_SPECIAL_TITLE:
                    Status = DialogBoxParam( gOptions.hAppInst, "MAINOPTIONSDLG", hwnd, 
                    						 lpfnMainOptionsDlgProc, (LPARAM)&Dock);
                    if(Status){
                    	WriteGlobalOptions();
						WriteDockOptions( &Dock );
					}
                    break;

                case SLOT_SPECIAL_EXIT:
                    Status = DialogBox(gOptions.hAppInst, "WINEXITOPTIONSDLG", hwnd, lpfnWinExitOptionsDlgProc);
                    if(Status) WriteGlobalOptions();
                    break;

                case SLOT_SPECIAL_MAIL:
                    Status = DialogBox(gOptions.hAppInst, "MAILOPTIONSDLG", hwnd, lpfnMailOptionsDlgProc);
                    if(Status) WriteGlobalOptions();
                    break;

                case SLOT_SPECIAL_CLOCK:
                    Status = DialogBox(gOptions.hAppInst, "CLOCKOPTIONSDLG", hwnd, lpfnClockOptionsDlgProc);
                    if(Status) WriteGlobalOptions();
                    break;
            }

            return 0;


        case WM_CLOSE:
            DestroyWindow(hwnd);
			return 0;

        case WM_QUIT:
        case WM_DESTROY:
            FreeProcInstance(lpfnAppOptionsDlgProc);
            FreeProcInstance(lpfnAboutDockDlgProc);
            FreeProcInstance(lpfnWinExitOptionsDlgProc);
            FreeProcInstance(lpfnMailOptionsDlgProc);
            FreeProcInstance(lpfnMainOptionsDlgProc);
            FreeProcInstance(lpfnClockOptionsDlgProc);
            KillTimer(Dock.hwndDock, MAIL_TIMER);
            /**************************************
            	Release all the Device Contexts
            	and bitmaps used
            **************************************/
            SelectObject(Dock.hdcIconCache, hbmOldIconCache);
			SelectObject(hdcUnderIcon,      hbmOldUnderIcon);
            SelectObject(hdcNewUnderIcon,   hbmOldNewUnderIcon);
            SelectObject(hdcTmp,            hbmOldTmp);

            Status = 0;
            
			Status += DeleteObject(hbmUnderIcon);
            Status += DeleteObject(hbmNewUnderIcon);
            Status += DeleteObject(hbmTmp);
            Status += DeleteObject(Dock.hbmIconCache);
			Status += DeleteDC(hdcUnderIcon);
            Status += DeleteDC(hdcNewUnderIcon);
            Status += DeleteDC(hdcTmp);
			Status += DeleteDC(Dock.hdcIconCache);

            DestroyCursor(hCurHand);
            // Delete UtilLoadIcon() internal bitmap handle.
            UtilLoadIcon( NULL, NULL );

			if( Status != 0 ){
		      	MessageBox(hwnd, "Error releasing Resources, you may need to restart Windows.", 
		      			   "FreeDock Resource Error", MB_OK | MB_ICONSTOP);
        	}    

            free(Slot);
            PostQuitMessage(0);
			return 0;
    }

    /* return all unused mesages to the system */
    return DefWindowProc(hwnd, message, wParam, lParam);
}
