// ==========================================================================
// 							Class Specification : CPathSpec
// ==========================================================================

// Header file : path.h

// Source : Periphere NV (R.Mortelmans)
// Creation Date : 	   2nd November. 1995
// Last Modification : 2nd November. 1995
                          
// //////////////////////////////////////////////////////////////////////////

// Properties:
//	NO	Abstract class (does not have any objects)
//	YES	Derived from CDirSpec and CFileSpec

//	NO	Is a Cwnd.                     
//	NO	Two stage creation (constructor & Create())
//	NO	Has a message map
//	NO Needs a resource (template)

//	NO	Persistent objects (saveable on disk)      
//	NO	Uses exceptions

// //////////////////////////////////////////////////////////////////////////

// Desciption :         
//		This class encapsulates a path specification
//		A file specification consists of a directory (drive and subdirectory)
//		 and a file name (base name and extender) and file attributes (length, ...)
//		All member functions starting with "Do" will physically change something
//		 on disk.  Other functions only change the internal data.
//		E.g. The normal destructor only destroys the internal representation,
//		 to actually remove a directory use DoRemove()

// Remark:
//		

// Prerequisites (necessary conditions):
//		

/////////////////////////////////////////////////////////////////////////////
#ifndef __PATH_H__
#define __PATH_H__

#include "dir.h"
#include "file.h"
#include "pathiter.h"

class CPathSpec : public CDirSpec, public CFileSpec
{
DECLARE_DYNAMIC(CPathSpec)

// Data members -------------------------------------------------------------
public:
	
protected:

private:
	
// Member functions ---------------------------------------------------------
public:
	CPathSpec();
	// --- In  :
	// --- Out : 
	// --- Returns :
	// --- Effect : Contructor of object
	//				It will initialize the internal state

	CPathSpec(const char* pszPath);
	// --- In  : pszPath : The new path name of the object
	// --- Out : 
	// --- Returns : 
	// --- Effect : 
	// --- Effect : Contruction of an object together with a SetPath()
	//				Notice that whether the path specification is valid or not
	//				cannot be checked in this way
	
    CPathSpec(const CPathSpec& pathSrc);
	// --- In :	pathSrc : Path object which will be copied
	// --- Out : 
	// --- Returns :
	// --- Effect : Copy contruction.
	
	const CPathSpec& operator=(const CPathSpec& pathSrc);
	// --- In :	pathSrc : Path object which will be assign to 'this' file object
	// --- Out:
	// --- Returns: 
	// --- Effect : Assignment operator
	
	CString GetPath() const;
	// --- In  :
	// --- Out : 
	// --- Returns :The path specification  of this object
	// --- Effect : 
	BOOL SetPath(const char* pszPath);
	// --- In  : pszPath : The new path specification of the object
	// --- Out : 
	// --- Returns : Whether path has a correct format
	//				 Wild characters only allowed in the file specification
	// --- Effect : 
	void ForceSetPath(const char* pszPath);
	// --- In  : pszPath : The new path specification of the object
	// --- Out : 
	// --- Returns : 
	// --- Effect : This function extracts illegal characters and thus
	//				will always succeeds
	
	BOOL SetPath(const CDirSpec dirSpec, const CFileSpec fileSpec);
	// --- In  : dirSpec : The directory part of the new path
	//           fileSpec : The file part of the new path
	// --- Out : 
	// --- Returns : Whether path has a correct format
	// --- Effect : 
	
	CString GetShortDescription();
	// --- In  :
	// --- Out : 
	// --- Returns :The path specification with all but the last subdirectory 
	//              replaced by a double full stop
	//				(E.g. C:\ONE\TWO\THREE\test.txt becomes C:\..\THREE\test.txt)
	// --- Effect : 
		
	BOOL MakeTemp(BOOL bCreateEmpty = TRUE, const char* pszPrefix = TEXT("TMP"));
	// --- In  : bCreateEmpty : Whether to create an empty file with the resulting name
	//			 pszPrefix : Prefix that will be used in the temporary file name
	//						 Only the first three characters will be used
	// --- Out : 
	// --- Returns : Whether a unique path spec could be determined
	// --- Effect : Replaces the path specification by that of the unique
	//				temporary file
	//				By default an empty temporary file with that name will be created on disk
	
	BOOL MakeAbsolute();
	// --- In  : 
	// --- Out : 
	// --- Returns : Whether path has a correct format
	// --- Effect : Replaces the path specification by an absolute path specification
	
	BOOL MakeUnique();
	// --- In  : 
	// --- Out : 
	// --- Returns : Whether a unique file name could be made
	// --- Effect : Replaces the file specification by another file specification
	//				that does not yet exist on disk
	//				So this can be used to make unique file names
	
	BOOL Exists() const;
	// --- In  : 
	// --- Out : 
	// --- Returns : Whether the path physically exists on disk
	// --- Effect : 
	
	BOOL IsEmpty() const;
	// --- In  : 
	// --- Out : 
	// --- Returns : Whether the path specification is empty
	// --- Effect : 
	
	void Empty();
	// --- In  : 
	// --- Out : 
	// --- Returns : 
	// --- Effect : CLears the path specification
	
	BOOL GetFirstFile(CPathIterator& Iterator) const;
	// --- In  :
	// --- Out : A CPathIterator value that can be used for iteration or object pointer
	//				 retrieval; Invalid iterator if the the dir is empty.
	// --- Returns : Whether the iterator is valid or not
	// --- Effect : Gets the CPathIterator pointer to the first file
	//				of this dir.
	
    CFileSpec GetNextFile(CPathIterator& Iterator) const;
	// --- In  : 
	// --- Out : FIterator : a reference to a CPathIterator returned by a previuos call
	//					  to GetFirstFile or GetNextFile
	// --- Returns : the file specification that represents the CPathIterator
	// --- Effect : Gets the list element identified by CPathIterator, then sets
	//				CPathIterator to the CPathIterator value of the next file in the dir
	//				You can use GetNextFile in a forward iteration loop if
	//				you establish the initial CPathIterator with a call to
	//				GetFirstFile.
	//				If the retrieved CFileSpec is the last in the dir, then
	//				CPathIterator is in his INVALID state (check the IsItValid function) .
	
	BOOL GetFirstDir(CPathIterator& Iterator) const;
	// --- In  :
	// --- Out :
	// --- Returns : A void pointer value that can be used for iteration or object pointer
	//				 retrieval; NULL if the the dir is empty.
	// --- Effect : Gets the void pointer to the fileinfo of the first subdir
	//				of this dir.
	
    CDirSpec GetNextDir(CPathIterator& Iterator) const;
	// --- In  : 
	// --- Out : FIterator : a reference to a fileinfo returned by a previuos call
	//					  to GetFirstDir or GetNextDir
	// --- Returns : the dir specification that represents the pFInfo
	// --- Effect : Gets the dir element identified by pFInfo, then sets
	//				pFInfo to the fileinfo value of the next subdir in the dir
	//				You can use GetNextDir in a forward iteration loop if
	//				you establish the initial fileinfo with a call to
	//				GetFirstDir.
	//				If the retrieved CDirSpec is the last in the dir, then
	//				CPathIterator is in his INVALID state (check the IsItValid function) .
	
	BOOL DoSearch(CFileSpec fileName, CDirSpec startingDir = CDirSpec(), BOOL bRecursively = FALSE);
	// --- In  : fileName : The name of the file to search for (must not include wild chars)
	//			 startingDir : The directory in which to search
	//			 bRecursively : Whether subdirectories of the specified directory 
	//   						should be searched as well
	// --- Out : 
	// --- Returns : Whether specified file was found
	// --- Effect : Searches for the specified file and assigns the result to this object
	//				The search order is :
	//					1 - The specified directory if the directory specification is not empty
	//					2 - Subdirectories of the specified directory if
	//						the directory specification is not empty and bRecursively is TRUE
	//						(This is not yet implemented)
	//					3 - The directory that contains the executable file of this running program
	//					4 - The current directory
	//					5 - The Windows directory
	//					6 - The System directory
	//					7 - The directories specified in the PATH-environment variable
	
	BOOL DoCopy(CPathSpec destinationPath) const;	
	// --- In  : destinationPath : To destination to where the file must be
	//			 				   copied.  The filename may be empty.
	// --- Out : 
	// --- Returns : Whether it succeeded or not
	// --- Effect : Physically copies the contents of one file into another
	//				When the detination file already exists, it will be truncated
	
	BOOL DoMove(CPathSpec destinationPath) const;	
	// --- In  : destinationPath : To destination to where the file must be
	//			 				   renamed or moved.  The filename must not be empty.
	// --- Out : 
	// --- Returns : Whether it succeeded or not
	// --- Effect : Physically renames or moves the file to another name or location
	//				When the destination file already exists, it will be truncated
	//				Directories cannot be renamed on moving across disks is not supported
	
	BOOL DoRemove(BOOL bIgnoreReadOnly = FALSE) const;	
	// --- In  : bIgnoreReadOnly : Whether the file should be removed,
	//		   			even when it is read-only
	// --- Out : 
	// --- Returns : Whether it succeeded or not
	// --- Effect : Removes the directory

	BOOL DoGetInfo();
	// --- In  : 
	// --- Out : 
	// --- Returns : Whether it succeeded or not
	// --- Effect : Get the real time, file length and attributes from disk
	//				and puts them in this path specification

	BOOL DoSetTime();
	// --- In  : 
	// --- Out : 
	// --- Returns : Whether it succeeded or not
	// --- Effect : Makes the real  time of the file on disk equal to the
	//				time specified in the path specification
		
	BOOL DoSetLength();
	// --- In  : 
	// --- Out : 
	// --- Returns : Whether it succeeded or not
	// --- Effect : Makes the real length of the file on disk equal to the
	//				length specified in the path specification
		
	BOOL DoSetAttributes();
	// --- In  : 
	// --- Out : 
	// --- Returns : Whether it succeeded or not
	// --- Effect : Makes the real attributes of the file on disk equal to the
	//				attributes specified in the path specification
	//				Notice that other function may change the attribute as a
	//				side effect (archive attribute)
	
#ifdef _DEBUG
	virtual void Dump(CDumpContext&) const;
	virtual void AssertValid() const;
#endif //_DEBUG

	virtual ~CPathSpec();
	// --- In  :
	// --- Out : 
	// --- Returns :
	// --- Effect : Destructor of object

	void* operator new(size_t nSize);
	void operator delete(void* p);
	// This functions must be overloaded, because both direct parents of CPathSpec,
	// namely CDirSpec and CFileSpec are derived from CObject
	// So this is done to unambiguize
protected:      
	static HMODULE GetThisModule();
	static BOOL SearchEnvironment(const char* pszFileName, const char* pszVarName, char* pszPathName);
	static DWORD GetEnvironmentVar(const char* pszVarName, char* pszValue, DWORD nLength);

private:
                   
};

#endif
// ==========================================================================
