// ==========================================================================
// 							Class Implementation : CPathSpec
// ==========================================================================

// Source file : path.cpp

// Source : Periphere NV (R.Mortelmans)
// Creation Date : 	   2nd November 1995
// Last Modification : 2nd November 1995
                          
// //////////////////////////////////////////////////////////////////////////

#include "stdafx.h"		// standard MFC include
#include "path.h"		// class specification
#include "xstring.h"	// for string-int conversion
#ifndef WIN32
#include "toolhelp.h"	// To determine the module handle
#endif

#include <direct.h>		// For directory functions (_fullpath, ...)
#include <dos.h>		// For _dos_setfileattr, ...
#include <io.h>			// For _chsize()

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

IMPLEMENT_DYNAMIC(CPathSpec, CDirSpec)

#define new DEBUG_NEW

/////////////////////////////////////////////////////////////////////////////
// Definition of static members


// Data members -------------------------------------------------------------
// protected:

// private:

// Member functions ---------------------------------------------------------
// public:

CPathSpec::CPathSpec()
	{
	}
	
CPathSpec::CPathSpec(const char* pszPath)
	{
	if (!SetPath(pszPath))
		{
		TRACE(TEXT("CPathSpec::CPathSpec : An invalid path (%s) was specified, clearing object\n"),
			pszPath);
		SetPath(TEXT(""));
		}
	}
	
CPathSpec::CPathSpec(const CPathSpec& pathSrc)
	:
	CDirSpec(pathSrc),
	CFileSpec(pathSrc)
	{
	}
	
const CPathSpec& CPathSpec::operator=(const CPathSpec& pathSrc)
	{
	CDirSpec::operator=(pathSrc);
	CFileSpec::operator=(pathSrc);
	return *this;
	}
	
CString CPathSpec::GetPath() const
	{
	CString sDir = GetDirectory();
	CString sFile = GetFileName();
	CString sPath;
	if (!sFile.IsEmpty())
		if (!sDir.IsEmpty())
			if (sDir.Right(1) != CString(TEXT("\\")))
				// \\DIR and AAA.BBB
				sPath = sDir + TEXT("\\") + sFile;
			else
				// \\ and AAA.BBB
				sPath = sDir + sFile;
		else
			// empty and AAA.BBB
			sPath = sFile;
	else
		// \\DIR and empty
		sPath = sDir;
	return sPath;;
	}
	
BOOL CPathSpec::SetPath(const char* pszPath)
	{            
	char pszDrive[_MAX_DRIVE];
	char pszSubdir[_MAX_DIR];
	char pszBaseName[_MAX_FNAME];
	char pszExtender[_MAX_EXT];
	
	_splitpath(pszPath, pszDrive, pszSubdir, pszBaseName, pszExtender);
	int nSubDir = strlen(pszSubdir);
	if (1 < nSubDir)
		// ... When not empty and not root, remove trailing back slash
		pszSubdir[nSubDir - 1] = '\0';
	int nExt = strlen(pszExtender);
	if (1 <= nExt)
		// ... Remove leading full stop
		strcpy(pszExtender, pszExtender + 1);
	
	return SetDrive(pszDrive) && SetSubdirectory(pszSubdir) &&
		   SetBaseName(pszBaseName) && SetExtender(pszExtender);
	}
	
void CPathSpec::ForceSetPath(const char* pszPath)
	{            
	char pszDrive[_MAX_DRIVE];
	char pszSubdir[_MAX_DIR];
	char pszBaseName[_MAX_FNAME];
	char pszExtender[_MAX_EXT];
	
	_splitpath(pszPath, pszDrive, pszSubdir, pszBaseName, pszExtender);
	int nSubDir = strlen(pszSubdir);
	if (1 < nSubDir)
		// ... When not empty and not root, remove trailing back slash
		pszSubdir[nSubDir - 1] = '\0';
	int nExt = strlen(pszExtender);
	if (1 <= nExt)
		// ... Remove leading full stop
		strcpy(pszExtender, pszExtender + 1);
	
	ForceSetDrive(pszDrive);
	ForceSetSubdirectory(pszSubdir);
	ForceSetBaseName(pszBaseName);
	ForceSetExtender(pszExtender);
	}
	
BOOL CPathSpec::SetPath(const CDirSpec dirSpec, const CFileSpec fileSpec)
	{
	CDirSpec::operator=(dirSpec);
	CFileSpec::operator=(fileSpec);
	return TRUE;
	}
	
CString CPathSpec::GetShortDescription()
	{
	// ... If path spec is empty, just return
	if (GetPath().IsEmpty())
		return TEXT("");

	CPathSpec tempPath(*this);
	CString sLastSubdir;
	CString sEliminatedDirs = TEXT("\\..\\");

	// First try to make absolute path
	if (!tempPath.MakeAbsolute())
		{
		TRACE(TEXT("CPathSpec::GetShortDescription : Could not make absolute path, returning full path spec\n"));
		return GetPath();
		}

	// Get the last subdir
	sLastSubdir = tempPath.GetLastSubdirectory().GetSubdirectory();
	// ... Last subdir should never contain back slashes
	ASSERT(sLastSubdir.Find('\\') == -1);
	// ... If last subdir is empty (root) or equals dir itself, 
	//     no subdirs have been eliminated
	if (sLastSubdir.IsEmpty())
		sEliminatedDirs.Empty();
		else if ((TEXT("\\") + sLastSubdir) == tempPath.GetSubdirectory())
			sEliminatedDirs = TEXT("\\");

	// Return the composed short description
	return tempPath.GetDrive() + sEliminatedDirs + sLastSubdir + TEXT("\\") + tempPath.GetFileName();
	}

BOOL CPathSpec::MakeTemp(BOOL bCreateEmpty /* = TRUE */,  const char* pszPrefix /* = TEXT("TMP") */)
	{
	char path_buffer[_MAX_PATH + 1];
	*path_buffer = '\0'; 

#ifdef WIN32
	// Get temp path
	CString sTempPath;
	BOOL bSucces = ::GetTempPath(_MAX_PATH, sTempPath.GetBuffer(_MAX_PATH));
	sTempPath.ReleaseBuffer();

	if (!bSucces)
		return FALSE;

	if (!::GetTempFileName(sTempPath,	// Use the default drive
		    		       pszPrefix,    // Temporary file name prefix
					       0, 			// Generate number and create file
				    	   path_buffer)) // Result
		return FALSE;

#else
	// Get temp path
	::GetTempFileName(0,			// Use the default drive
				      pszPrefix,    // Temporary file name prefix
				      0, 			// Generate number and create file
				      path_buffer); // Result
#endif

	if (*path_buffer != '\0')
		{
		BOOL bResult = SetPath(path_buffer);
		if (bCreateEmpty)
			{
			// Empty file should have been created by ::GetTempFileName
			ASSERT(Exists());
			}
		else
			{
			// Delete empty file created by ::GetTempFileName
			VERIFY(DoRemove());
			ASSERT(!Exists());
			}
		return bResult;
		}
	else
		{
		TRACE(TEXT("CPathSpec::MakeTemp : Could not make temporary path spec\n"));
		return FALSE;
		}
	}

BOOL CPathSpec::MakeAbsolute()
	{
	// If no file name was specified, just return
	// Apparently in WIN32 _fullpath returns the apllications full path
	//  in this case (??)
	if (GetFileName().IsEmpty())
		return TRUE;

	char pszFullPath[_MAX_PATH];
	if (_fullpath(pszFullPath, (const char*)GetPath(), _MAX_PATH) != NULL)
		return SetPath(pszFullPath);
	else
		return FALSE;
	}
	
BOOL CPathSpec::MakeUnique()
	{
	if (GetBaseName().IsEmpty())
		SetBaseName(TEXT("unique"));
		
	if (!Exists())
		return TRUE;  

	// Change the name by first adding underscores, until 8 characters are used.
	// Then the last character(s) are replaced by a number starting from 2,
	// until a unique name is found.
	CXString sNumber = TEXT("2");
	CString sBaseName = GetBaseName().Left(8);
	while (sBaseName.GetLength() < 8)
		sBaseName +=  TEXT("_");          
		
	ASSERT(sBaseName.GetLength() == 8);
	ASSERT(Exists());
	do
		{
		sBaseName = sBaseName.Left(8 - sNumber.GetLength());
		sBaseName += sNumber;
		VERIFY(SetBaseName(sBaseName));
		sNumber = sNumber.GetInt() + 1;		// Implicit conversion to int
		}
		while(Exists());
		
	return TRUE;
	}
	
BOOL CPathSpec::Exists() const
	{
	CFileStatus fileStatus;
	return CFile::GetStatus(GetPath(), fileStatus);
	}
	
BOOL CPathSpec::IsEmpty() const
	{
	return CDirSpec::IsEmpty() && CFileSpec::IsEmpty();
	}
	
void CPathSpec::Empty()
	{
	CDirSpec::Empty();
	CFileSpec::Empty();
	}
	
BOOL CPathSpec::DoSearch(CFileSpec fileName, CDirSpec startingDir /* = CDirSpec() */,
		 BOOL bRecursively /* = FALSE */)
	{
	CPathSpec resultPath;
	CDirSpec currentDir;
	
	// First store the current dir
	VERIFY(currentDir.DoGetCurrentDir());
	
	// 1. Check the specified directory
	if (!startingDir.IsEmpty())
		{
		resultPath.SetPath(startingDir, fileName);
		if (resultPath.Exists())
			{
			*this = resultPath;
			return TRUE;
			}
		}
		
	// 2. Check the subdirectories of the specified directory
	
	// ... Recursive search is not yet implemented
	ASSERT(!bRecursively);

	// 3  Check the EXE-directory
	char pszModulePath[_MAX_PATH];
	if (::GetModuleFileName(GetThisModule(), pszModulePath, _MAX_PATH) != 0)
		{
		VERIFY(resultPath.SetPath(pszModulePath));
		resultPath.CFileSpec::operator=(fileName);
		if (resultPath.Exists())
			{
			*this = resultPath;
			return TRUE;
			}
		}
	else
		{
		TRACE(TEXT("CPathSpec::DoSearch : Could not retrieve the path of the running application\n"));
		}
	
	// 4. Check the current directory
	if (!currentDir.IsEmpty())
		{
		resultPath.SetPath(currentDir, fileName);
		if (resultPath.Exists())
			{
			*this = resultPath;
			return TRUE;
			}
		}

	// 5. Check the Windows directory
	char pszWinDir[_MAX_DIR];
	if (::GetWindowsDirectory(pszWinDir, _MAX_DIR) != 0)
		{
		VERIFY(resultPath.SetDirectory(pszWinDir));
		resultPath.CFileSpec::operator=(fileName);
		if (resultPath.Exists())
			{
			*this = resultPath;
			return TRUE;
			}
		}
	else
		{
		TRACE(TEXT("CPathSpec::DoSearch : Could not retrieve the Windows directory\n"));
		}

	// 6. Check the System directory
	char pszSystemDir[_MAX_DIR];
	if (::GetSystemDirectory(pszSystemDir, _MAX_DIR) != 0)
		{
		VERIFY(resultPath.SetDirectory(pszSystemDir));
		resultPath.CFileSpec::operator=(fileName);
		if (resultPath.Exists())
			{
			*this = resultPath;
			return TRUE;
			}
		}
	else
		{
		TRACE(TEXT("CPathSpec::DoSearch : Could not retrieve the Windows directory\n"));
		}

	// 7. Check the directories of PATH-environment variable
	char pszResultPath[_MAX_PATH];
	if (SearchEnvironment(fileName.GetFileName(), TEXT("PATH"), pszResultPath))
		{
		ASSERT(*pszResultPath != '\0');
		VERIFY(SetPath(pszResultPath));
		return TRUE;
		}

	// If still not returned, the file was not found
	return FALSE;
	}
	
BOOL CPathSpec::DoCopy(CPathSpec destinationPath) const
	{
	ASSERT(!GetFileName().IsEmpty());		// Source file must be specified
	CString sSourcePath;
	CString sDestPath;
	if (destinationPath.GetFileName().IsEmpty())
		// ... Make destination file name equal to source
		destinationPath.SetFileName(GetFileName());      
	sSourcePath = GetPath();
	sDestPath = destinationPath.GetPath();
	
#ifdef _DEBUG
	if (!Exists())
		{
		TRACE(TEXT("CPathSpec::DoCopy : Source file %s does not exist\n"), sSourcePath);
		return FALSE;
		}
	else
		if (destinationPath.Exists())
			TRACE(TEXT("CPathSpec::DoCopy : Destination file %s already exists, truncating ...\n"), sDestPath);
#endif	

	const nBufferLength = 2048;
	BYTE pBuffer[nBufferLength + 1];
	int nLengthRead;
	CFile source;
	CFile dest;
	BOOL bSuccess = TRUE;
	TRY
		{
		if ( (source.Open(sSourcePath,CFile::modeRead | CFile::shareCompat) != 0) &&
		     (dest.Open(sDestPath, CFile::modeCreate | CFile::modeWrite | CFile::shareExclusive) != 0) )
		     {
			do
				{
				nLengthRead = source.Read(pBuffer, nBufferLength);
				dest.Write(pBuffer, nLengthRead);
				}
				while (nLengthRead == nBufferLength);	// So while not EOF
			source.Close();
			dest.Close();
			}
		else
			{
			TRACE(TEXT("CPathSpec::DoCopy : Could not open files\n"));
			bSuccess = FALSE;
			}
		}
	CATCH(CFileException, pxFile)
		{
		TRACE(TEXT("CPathSpec::DoCopy : Catching FileException (%XH)\n"), pxFile->m_cause);
		bSuccess = FALSE;
		}
	END_CATCH      
	
	return bSuccess;			
	}
	
BOOL CPathSpec::DoMove(CPathSpec destinationPath) const
	{
	if (destinationPath.GetDrive() == TEXT(""))
		VERIFY(destinationPath.SetDrive(GetDrive()));
	if (destinationPath.GetSubdirectory() == TEXT(""))
		VERIFY(destinationPath.SetSubdirectory(GetSubdirectory()));
	if (destinationPath.GetFileName() == TEXT(""))
		VERIFY(destinationPath.SetFileName(GetFileName()));
	if (*this == destinationPath)
		{
		TRACE(TEXT("CPathSpec::DoMove : Source and destination files are the same %s\n"), 
			(const char*)GetPath());
		// ... Nothing to do
		return TRUE;
		}
		
#ifdef _DEBUG
	if (!Exists())
		TRACE(TEXT("CPathSpec::DoMove : Source file %s does not exist\n"), 
			(const char*)GetPath());
	if (destinationPath.Exists())
		TRACE(TEXT("CPathSpec::DoMove : Destination file %s does not exist\n"), 
			(const char*)destinationPath.GetPath());
#endif
	
	// ... Assume success
	BOOL bSuccess = TRUE;
	TRY
		{
		CFile::Rename(GetPath(), destinationPath.GetPath());
		}
	CATCH(CFileException, px)
		{
		TRACE(TEXT("CPathSpec::DoMove : CFile::Rename(%s, %s) failed with CFileException cause %i\n"), 
			(const char*)GetPath(), (const char*)destinationPath.GetPath(), px->m_cause);
		bSuccess = FALSE;
		}
	END_CATCH
	return bSuccess;
	}
	
BOOL CPathSpec::DoRemove(BOOL bIgnoreReadOnly /* = FALSE */) const
	{
	CString sPath = GetPath();
	
	// ... This function may only be used to remove a file,
	//	   not a directory. So the file name must not be empty
	ASSERT(!GetFileName().IsEmpty());

#ifdef _DEBUG
	if (!Exists())
		TRACE(TEXT("CPathSpec::DoRemove : File %s does not exist\n"), sPath);
#endif	
	if (bIgnoreReadOnly)
#ifdef WIN32
		if (!SetFileAttributes(sPath, CFile::normal))
#else
		if(_dos_setfileattr(sPath, CFile::normal))
#endif
			{
			TRACE(TEXT("CPathSpec::DoRemove : File not found or cannot remove R/O attribute of %s\n"), sPath);
			return FALSE;
			}
			
	if (remove(sPath))
		{
		TRACE(TEXT("CPathSpec::DoRemove : Cannot remove file %s\n"), sPath);
		return FALSE;
		}            
	return TRUE;
	}
	
BOOL CPathSpec::DoGetInfo()
	{
	CFileStatus fileStatus;
	if (CFile::GetStatus(GetPath(), fileStatus))
		{
		m_time = 		fileStatus.m_mtime;
		m_lnLength = 	fileStatus.m_size;
		m_eAttributes = fileStatus.m_attribute;
		return TRUE;
		}
	else
		{
		TRACE(TEXT("CPathSpec::DoGetInfo : Could not get file status of %s\n"),
			GetPath());
		return FALSE;
		}
	}
	
BOOL CPathSpec::DoSetTime()
	{
	CFileStatus fileStatus;
	
	if (!CFile::GetStatus(GetPath(), fileStatus))
		{
		TRACE(TEXT("CPathSpec::DoSetTime : Could not even get the present status, failing\n"));
		return FALSE;
		}

	// Set new time, (DOS only knows one type of time)		
	fileStatus.m_ctime = m_time;;
	fileStatus.m_mtime = fileStatus.m_ctime;
	fileStatus.m_atime = fileStatus.m_ctime;
	
	BOOL bSuccess = TRUE;
	TRY
		{
		CFile::SetStatus(GetPath(), fileStatus);
		}
	CATCH(CFileException, pxFile)
		{
		TRACE(TEXT("CPathSpec::DoSetTime : Catching file exception (cause %XH)"),
				((CFileException*)pxFile)->m_cause);
		bSuccess = FALSE;
		}
	END_CATCH
	return bSuccess;
	}
	
BOOL CPathSpec::DoSetLength()
	{
	// SetStatus does not set the file length, so we use direct dos access
#ifdef WIN32
	HANDLE hFile;
	hFile = CreateFile(GetPath(), GENERIC_READ | GENERIC_WRITE, 
			0,NULL, OPEN_EXISTING, 
			FILE_ATTRIBUTE_NORMAL | FILE_FLAG_OVERLAPPED, NULL);
	if (hFile == INVALID_HANDLE_VALUE)
		{
		TRACE(TEXT("CPathSpec::DoSetLength : Could not open file %s, failing\n"),
			(const char*)GetPath());
		return FALSE;
		}
	SetFilePointer(hFile,m_lnLength, NULL,FILE_BEGIN);
	if(GetLastError() != ERROR_SUCCESS)
		{
		TRACE(TEXT("CPathSpec::DoSetLength : Could set new length of %s, failing\n"),
			(const char*)GetPath());
		return FALSE;
		}
	SetEndOfFile(hFile);
	if(GetLastError() != ERROR_SUCCESS)
		{
		TRACE(TEXT("CPathSpec::DoSetLength : Could not set end of %s, failing\n"),
			(const char*)GetPath());
		return FALSE;
		}
	if (!CloseHandle(hFile))
		{
		TRACE(TEXT("CPathSpec::DoSetLength : Could not close file %s, failing\n"),
			(const char*)GetPath());
		return FALSE;
		}
	
#else
	UINT nErr;
	int handle;
	if ((nErr = _dos_open(GetPath(), CFile::modeReadWrite, &handle)) != 0)
		{
		TRACE(TEXT("CPathSpec::DoSetLength : Could not open file %s, failing\n"),
			(const char*)GetPath());
		return FALSE;
		}
	if ((nErr = _chsize(handle, m_lnLength)) != 0)
		{
		TRACE(TEXT("CPathSpec::DoSetLength : Could set new length of %s, failing\n"),
			(const char*)GetPath());
		return FALSE;
		}
	if ((nErr = _dos_close(handle)) != 0)
		{
		TRACE(TEXT("CPathSpec::DoSetLength : Could not close file %s, failing\n"),
			(const char*)GetPath());
		return FALSE;
		}
#endif
	
	return TRUE;
	}
	
BOOL CPathSpec::DoSetAttributes()
	{
	CFileStatus fileStatus;
	
	if (!CFile::GetStatus(GetPath(), fileStatus))
		{
		TRACE(TEXT("CPathSpec::DoSetAttributes : Could not even get the present status, failing\n"));
		return FALSE;
		}

	// Set new attributes
	fileStatus.m_attribute = m_eAttributes;;
	
	BOOL bSuccess = TRUE;
	TRY
		{
		CFile::SetStatus(GetPath(), fileStatus);
		}
	CATCH(CFileException, pxFile)
		{
		TRACE(TEXT("CPathSpec::DoSetAttributes : Catching file exception (cause %XH)"),
				((CFileException*)pxFile)->m_cause);
		bSuccess = FALSE;
		}
	END_CATCH
	return bSuccess;
	}

// To avoid conflict between '#define new DEBUG_NEW' and 'operator new'
#undef new 

void* CPathSpec::operator new(size_t nSize)
	{
	return CDirSpec::operator new(nSize);
	}
	
void CPathSpec::operator delete(void* p)
	{
	CDirSpec::operator delete(p);
	}
	
#ifdef _DEBUG
void CPathSpec::Dump(CDumpContext& dc) const
	{
	CDirSpec::Dump(dc);
	CFileSpec::Dump(dc);
	}

void CPathSpec::AssertValid() const
	{
	CDirSpec::AssertValid();
	CFileSpec::AssertValid();
	}
#endif

CPathSpec::~CPathSpec()
	{
	}
	
// protected:
HMODULE CPathSpec::GetThisModule()
	// --- In  : 
	// --- Out : 
	// --- Returns : The module handle of the running task
	// --- Effect : 
	{
	HMODULE hThisModule = NULL;

#ifdef WIN32
	hThisModule = GetModuleHandle(NULL);
#else	
	// WIN16 does not have a direct function which returns the module handle of the running application
	// Therefor TOOLHELP is used to calculate it from the running task
    HTASK hCurrentTask;
    TASKENTRY taskEntry;
    taskEntry.dwSize = sizeof(taskEntry);
    hCurrentTask = GetCurrentTask();
    if (hCurrentTask == NULL)
            return NULL;
    TaskFindHandle(&taskEntry, hCurrentTask);
	hThisModule = taskEntry.hModule;

	// An alternative (but undocumented) ways is the following
	// HMODULE hThisModule2 = (HMODULE)*(LPWORD)MAKELP(GetCurrentTask(), 0x1E);
#endif
	
	return hThisModule;
	}

BOOL CPathSpec::SearchEnvironment(const char* pszFileName, const char* pszVarName, char* pszPathName)
	// --- In  : pszFileName : The name of the file to search for
	//			 pszVarName : The name of the environment variable
	// --- Out : pszPathName : The full path of the file if found, otherwise empty
	// --- Returns : Whether the specified file was found
	// --- Effect : Searches all the directories of the specified environment variable
	//				for the specified file
	//				An environment variable is case-sensitive
	// --- Remark : This function is an implementation of the C-RunTime function _searchenv
	//				 This function also works when called from a DLL, which _searchenv does not
	{
	const int nMaxEntriesLength(300);
	char szEntries[nMaxEntriesLength + 1];
	char* pszToken;
	DWORD dwEntriesLength;
	
	// ... Assume failure, so initialize out-parameter
	ASSERT(pszPathName != NULL);
	ASSERT(AfxIsValidAddress(pszPathName, 1));
	*pszPathName = '\0';	
	
	// ... First get the environment value
	dwEntriesLength = GetEnvironmentVar(pszVarName, szEntries, nMaxEntriesLength);
	if (dwEntriesLength == 0)
		// ... Failure : Environment variable does not exist
		return FALSE;
#ifdef _DEBUG	
	if (nMaxEntriesLength < dwEntriesLength)
		TRACE(TEXT("CPathSpec::SearchEnv : Environment variable value length (%u) exceeds maximum length (%u) and will be truncated\n"),
			dwEntriesLength, nMaxEntriesLength);
#endif	
	
	// Iterate all the directory entries, which are seperated by a semi-colon
	BOOL bFound = FALSE;
	CFileStatus fileStatus;
	char szPath[_MAX_PATH];
	char* pszConcat;
	pszToken = strtok(szEntries, TEXT(";"));
	while (!bFound && (pszToken != NULL))
		{
		strcpy(szPath, pszToken);
		// ... String cannot be empty because (pszToken != '\0')
		ASSERT(1 <= strlen(szPath));
		// ... Examine the last char of the directory spec, 
		//     if it is not back slash and not a colon (relative path), add a back slash
		pszConcat = &szPath[strlen(szPath) - 1];
		if ( (*pszConcat != '\\') && (*pszConcat != ':') )
			*(++pszConcat) = '\\';
		// ... Position after last char of directory spec
		pszConcat++;
		// ... Add file name
		strcpy(pszConcat, pszFileName);
		bFound = CFile::GetStatus(szPath, fileStatus);
		pszToken = strtok(NULL, TEXT(";"));
		}
	if (bFound)
		{
		ASSERT(AfxIsValidAddress(pszPathName, strlen(szPath) + 1));
		strcpy(pszPathName, szPath);
		}
	return bFound;
	}

DWORD CPathSpec::GetEnvironmentVar(const char* pszVarName, char* pszValue, DWORD nLength)
	// --- In  : pszVarName : The name of the environment variable 
	//			 pszValue : Buffer into which the result will be stored
	//			 nLength : The length of the buffer
	// --- Out : pszValue : The result (the environment table entry containing 
	//				 the current string value of pszVarName)
	// --- Returns : The number of characters stored into the buffer pointed to by pszValue, 
	//				  not including the terminating null character. 
	//				 When the variable name was not found the return value is zero. 
	//				 If the buffer pointed to by pszValue is not large enough, 
	//				  the return value is the buffer size, in characters, 
	//				  required to hold the value string and its terminating null character. 
	// --- Effect : Searches the list of environment variables for the specified entry
	//				An environment variable is case-sensitive
	// --- Remark : This function is an implementation of the C-RunTime function getenv
	//				 This function also works when called from a DLL, which in WIN16 getenv does not
	//				See Also MS Developers Network Q78542 : 
	//				 Retreiving MS-DOS Environment Vars from a Windows DLL
	{
#ifdef WIN32
	// Functionality exists in WIN32, so just call Windows API function
	return ::GetEnvironmentVariable(pszVarName, pszValue, nLength);
#else	
	// ... Assume failure, so initialize out-parameter
	ASSERT(pszValue != NULL);
	ASSERT(AfxIsValidAddress(pszValue, (UINT)nLength));
	*pszValue = '\0';	

	char* lpEnvSearch;
	const char* lpszVarSearch;
	
	// ... Check for empty var
	if (*pszVarName == '\0')
		{
		TRACE(TEXT("CPathSpec::GetEnvironmentVar : Empty environment variable, returning 0\n"));
    	return 0;
    	}
    	
	//  ... Get a pointer to the MS-DOS environment block
	lpEnvSearch = GetDOSEnvironment();
	// ... Iterat all strings in the environment table
	while (*lpEnvSearch != '\0')
		{
        //  ... Check to see if the variable names match
    	lpszVarSearch = pszVarName;
    	while ( (*lpEnvSearch != '\0') && (*lpszVarSearch != '\0') &&
                (*lpEnvSearch == *lpszVarSearch) )
	    	{
	      	lpEnvSearch++;
	      	lpszVarSearch++;
	      	}
        // ... If the names match, the lpEnvSearch pointer is on the "="
        //     character and lpszVarSearch is on a null terminator.
        //     Increment and return lpszEnvSearch, which will point to the
        //     environment variable's contents.
		if ((*lpEnvSearch == '=') && (*lpszVarSearch == '\0'))
			{
			lpEnvSearch++;
			strncpy(pszValue, lpEnvSearch, (size_t)nLength);
			pszValue[nLength] = '\0';
			return strlen(lpEnvSearch);
			}
        // ... If the names do not match, increment lpEnvSearch until it
        //     reaches the end of the current variable string.
	    else
      		while (*lpEnvSearch != '\0')
        		lpEnvSearch++;
		// ... At this point the end of the environment variable's string
        // 	   has been reached. Increment lpEnvSearch to move to the
        //     next variable in the environment block. If it is NULL,
        //     the end of the environment block has been reached.
	    lpEnvSearch++;
	    }
	    
	// If this section of code is reached, the variable was not found.
	TRACE(TEXT("CPathSpec::GetEnvironmentVar : Environment variable (%s) not found, returning NULL\n"), pszVarName);
  	return 0; 
#endif // WIN32  	
	}
	
BOOL CPathSpec::GetFirstFile(CPathIterator& FIterator) const
	{
	CPathSpec searchPath(*this);
	if (searchPath.GetFileName().IsEmpty())
		VERIFY(searchPath.SetFileName(TEXT("*.*")));

	BOOL bGoodFileFound = FALSE;	
	BOOL bJustFileFound = TRUE;	
    FIterator.m_bValid = TRUE;

#ifdef WIN32
    FIterator.m_hFindFile = FindFirstFile(searchPath.GetPath(), &FIterator.m_FindFileData);

	if (FIterator.m_hFindFile == INVALID_HANDLE_VALUE)
    	{
    	FindClose(FIterator.m_hFindFile);
    	FIterator.m_hFindFile = NULL;
    	FIterator.m_bValid = FALSE;
    	}
	else
		{
		while (!bGoodFileFound && bJustFileFound)
			{
			if (!IsChildDir(&FIterator.m_FindFileData) &&
			   (FIterator.m_FindFileData.cFileName[0] != __TEXT('.')))
				bGoodFileFound = TRUE;
			
			if (!bGoodFileFound)
				bJustFileFound = FindNextFile(FIterator.m_hFindFile, &FIterator.m_FindFileData);
			}
		}
#else
	
    bGoodFileFound = !_dos_findfirst(searchPath.GetPath(), _A_NORMAL | _A_ARCH, &FIterator.m_FileInfo);
    	
#endif

   	FIterator.m_bValid = bGoodFileFound;
    return FIterator.m_bValid;
	}
	
CFileSpec CPathSpec::GetNextFile(CPathIterator& FIterator) const
	{
	ASSERT_VALID(&FIterator);

	CFileSpec ActualFile;

	BOOL bDirFound(TRUE);
#ifdef WIN32
	BOOL bFileFound;
	ActualFile.SetFileName(FIterator.m_FindFileData.cFileName);
	ActualFile.SetTime(CTime(FIterator.m_FindFileData.ftLastWriteTime));
	ASSERT(FIterator.m_FindFileData.nFileSizeHigh == 0);
	ActualFile.SetLength(FIterator.m_FindFileData.nFileSizeLow);
	ActualFile.SetAttributes((CFile::Attribute)FIterator.m_FindFileData.dwFileAttributes);

	bFileFound = FindNextFile(FIterator.m_hFindFile, &FIterator.m_FindFileData);
	while (bFileFound && bDirFound)
		{
		if (!IsChildDir(&FIterator.m_FindFileData) &&
		   (FIterator.m_FindFileData.cFileName[0] != __TEXT('.')))
			bDirFound = FALSE;
		
		if (bDirFound)
			bFileFound = FindNextFile(FIterator.m_hFindFile, &FIterator.m_FindFileData);
			}
		
	if (!bFileFound)	
		{
    	FindClose(FIterator.m_hFindFile);
    	FIterator.m_hFindFile = NULL;
    	FIterator.m_bValid = FALSE;
		}

#else
	ActualFile.SetFileName(FIterator.m_FileInfo.name);
	ActualFile.SetTime(CTime((WORD)FIterator.m_FileInfo.wr_date, (WORD)FIterator.m_FileInfo.wr_time));
	ActualFile.SetLength(FIterator.m_FileInfo.size);
	ActualFile.SetAttributes((CFile::Attribute)FIterator.m_FileInfo.attrib);
	
	if (_dos_findnext(&FIterator.m_FileInfo) != 0)
		{
	   	FIterator.m_bValid = FALSE;
		}
#endif                

	return ActualFile;
	}
	
BOOL CPathSpec::GetFirstDir(CPathIterator& FIterator) const                 
	{
	CPathSpec searchPath(*this);
	if (searchPath.GetFileName().IsEmpty())
		VERIFY(searchPath.SetFileName(TEXT("*.*")));

    FIterator.m_bValid = TRUE;
	BOOL bFileFound(TRUE);	
	BOOL bDirFound(FALSE);	

#ifdef WIN32
    FIterator.m_hFindFile = FindFirstFile(searchPath.GetPath(), &FIterator.m_FindFileData);

	if (FIterator.m_hFindFile == INVALID_HANDLE_VALUE)
    	{
    	FindClose(FIterator.m_hFindFile);
    	FIterator.m_hFindFile = NULL;
    	FIterator.m_bValid = FALSE;
    	}
	else
		{
		while (!bDirFound && bFileFound)
			{
			if (IsChildDir(&FIterator.m_FindFileData))
				bDirFound = TRUE;
			
			if (!bDirFound)
				bFileFound = FindNextFile(FIterator.m_hFindFile, &FIterator.m_FindFileData);
			}
		}
    
#else
    bFileFound = !_dos_findfirst(searchPath.GetPath(), _A_NORMAL | _A_ARCH | _A_SUBDIR, &FIterator.m_FileInfo);
   	while (bFileFound && !bDirFound)
		{
		if ((FIterator.m_FileInfo.attrib & _A_SUBDIR) && (strcmp(FIterator.m_FileInfo.name, TEXT(".")) != 0) &&
			(strcmp(FIterator.m_FileInfo.name, TEXT("..")) != 0))
			bDirFound = TRUE;
		
		if (!bDirFound)	
			bFileFound = (_dos_findnext(&FIterator.m_FileInfo) == 0);
		}		
   	
#endif

   	FIterator.m_bValid = bDirFound;
    return FIterator.m_bValid;	
	}

CDirSpec CPathSpec::GetNextDir(CPathIterator& FIterator) const
	{
	ASSERT_VALID(&FIterator);
	 
	CDirSpec ActualDir;

	BOOL bFileFound;	
	BOOL bDirFound(FALSE);	

#ifdef WIN32
	ActualDir.SetDrive(m_sDrive);
	ActualDir.SetSubdirectory(m_sSubdirectory);
	ActualDir.AppendDirectory(CDirSpec(FIterator.m_FindFileData.cFileName));

	bFileFound = FindNextFile(FIterator.m_hFindFile, &FIterator.m_FindFileData);
	while (bFileFound && !bDirFound)
		{
		if (IsChildDir(&FIterator.m_FindFileData))
			bDirFound = TRUE;

		if (!bDirFound)
			bFileFound = FindNextFile(FIterator.m_hFindFile, &FIterator.m_FindFileData);
		}
		
	if (!bFileFound)	
		{
    	FindClose(FIterator.m_hFindFile);
    	FIterator.m_hFindFile = NULL;
    	FIterator.m_bValid = FALSE;
		}

#else

	ActualDir.SetDrive(m_sDrive);
	ActualDir.SetSubdirectory(m_sSubdirectory);
	ActualDir.AppendDirectory(CDirSpec(FIterator.m_FileInfo.name));

	bFileFound = !_dos_findnext(&FIterator.m_FileInfo);
   	while (bFileFound && !bDirFound)
		{
		if ((FIterator.m_FileInfo.attrib & _A_SUBDIR) && (strcmp(FIterator.m_FileInfo.name, TEXT(".")) != 0) &&
			(strcmp(FIterator.m_FileInfo.name, TEXT("..")) != 0))
			bDirFound = TRUE;
		
		if (!bDirFound)	
			bFileFound = (_dos_findnext(&FIterator.m_FileInfo) == 0);
		}		

   	FIterator.m_bValid = bDirFound;
#endif                

	return ActualDir;	
	}
	
// private:

// ==========================================================================
