// Calendar Control

#include <windows.h>
#include <mem.h>
#include <string.h>
#include "calendar.h"

static char DAY_NAMES[] = "Su Mo Tu We Th Fr Sa";

char *MONTH_NAME[13] =
{
  "???", "JAN", "FEB", "MAR", "APR", "MAY", "JUN", "JUL",
  "AUG", "SEP", "OCT", "NOV", "DEC"
};

extern "C" {
  long far pascal _export CalendarProc(HWND, UINT, WPARAM, LPARAM);
}

/*-----------------------------------------------------------------------------
This function registers the CALENDAR class. It is called automatically when
the program starts up.

The external variables _hPrev and _hInstance duplicate the arguments
hPrevInstance and hInstance, which are passed to WinMain(). If the startup
code does not supply these external variables, you must pass the arguments to
this function and call it explicitly before invoking any CALENDAR control.
-----------------------------------------------------------------------------*/

extern HINSTANCE _hPrev, _hInstance;

static void register_calendar(void)
{
  if (!_hPrev)
  {
    WNDCLASS w;
    w.cbClsExtra = 0;
    w.cbWndExtra = sizeof(void *);
    w.hbrBackground = (HBRUSH) COLOR_WINDOW + 1;
    w.hCursor = LoadCursor(NULL, IDC_ARROW);
    w.hIcon = NULL;
    w.hInstance = _hInstance;
    w.lpfnWndProc = CalendarProc;
    w.lpszClassName = "CALENDAR";
    w.lpszMenuName = NULL;
    w.style = 0;
    RegisterClass(&w);
  }
}

#pragma startup register_calendar

const int JAN = 31;
const int FEB = 28;
const int MAR = 31;
const int APR = 30;
const int MAY = 31;
const int JUN = 30;
const int JUL = 31;
const int AUG = 31;
const int SEP = 30;
const int OCT = 31;
const int NOV = 30;
const int DEC = 31;

const int DAYS_PER_WEEK = 7;

union cal_date
{
  DWORD dword;
  struct
  {
    BYTE day;
    BYTE month;
    WORD year;
  } x;
};

static BOOL leap_year(union cal_date d)
{
  return d.x.year%4 == 0 && !(d.x.year%100 == 0 && d.x.year%400 != 0);
}

static unsigned days_in_month(union cal_date d)
{
  static BYTE DAYS_IN_MONTH[13] =
  {
    0, JAN, FEB, MAR, APR, MAY, JUN, JUL, AUG, SEP, OCT, NOV, DEC
  };
  return DAYS_IN_MONTH[d.x.month] + (d.x.month == 2 && leap_year(d));
}

static unsigned day_of_week(union cal_date d)
{
  static WORD DAYS_BEFORE_MONTH[13] =
  {
    0,
    0,
    JAN,
    JAN+FEB,
    JAN+FEB+MAR,
    JAN+FEB+MAR+APR,
    JAN+FEB+MAR+APR+MAY,
    JAN+FEB+MAR+APR+MAY+JUN,
    JAN+FEB+MAR+APR+MAY+JUN+JUL,
    JAN+FEB+MAR+APR+MAY+JUN+JUL+AUG,
    JAN+FEB+MAR+APR+MAY+JUN+JUL+AUG+SEP,
    JAN+FEB+MAR+APR+MAY+JUN+JUL+AUG+SEP+OCT,
    JAN+FEB+MAR+APR+MAY+JUN+JUL+AUG+SEP+OCT+NOV
  };
  unsigned y = d.x.year - 1;
  return (y + y/4 - y/100 + y/400 + DAYS_BEFORE_MONTH[d.x.month] + d.x.day +
    (leap_year(d) && d.x.month > 2)) % DAYS_PER_WEEK;
}

unsigned day_of_week(DWORD d)
{
  union cal_date dd;
  dd.dword = d;
  return day_of_week(dd);
}

static unsigned spaces_at_beginning_of_month(union cal_date d)
{
  return (day_of_week(d) - d.x.day + (5*DAYS_PER_WEEK+1)) % DAYS_PER_WEEK;
}

static cal_date next_day(union cal_date d)
{
  if (++d.x.day > days_in_month(d))
  {
    d.x.day = 1;
    if (++d.x.month > 12)
    {
      d.x.month = 1;
      d.x.year++;
    }
  }
  return d;
}

static cal_date previous_day(union cal_date d)
{
  if (--d.x.day < 1 )
  {
    if (--d.x.month < 1)
    {
      d.x.month = 12;
      d.x.year--;
    }
    d.x.day = days_in_month(d);
  }
  return d;
}

struct calendar
{
  union cal_date current;
  union cal_date first;
  union cal_date last;
  HFONT font;
  WORD height;
  WORD width;
  WORD character_width;
  WORD character_height;
  BOOL disabled : 1;
  BOOL focus : 1;
};

/*-----------------------------------------------------------------------------
This function computes the rectangle containing a specified day and
invalidates it.
-----------------------------------------------------------------------------*/

static void invalidate_day(HWND handle, calendar *p, unsigned day)
{
  RECT r;
  unsigned n = day - 1 + spaces_at_beginning_of_month(p->current);
  r.left = 3 + (n%DAYS_PER_WEEK) * 3 * p->character_width;
  r.right = r.left + 2 * p->character_width;
  r.top = 3 + (2 + n/DAYS_PER_WEEK) * p->character_height;
  r.bottom = r.top + p->character_height;
  InvalidateRect(handle, &r, TRUE);
}

/*-----------------------------------------------------------------------------
This function changes the date.
-----------------------------------------------------------------------------*/

static void change_date(HWND handle, calendar *p, union cal_date new_date,
  BOOL notify)
{
  if (new_date.dword < p->first.dword)
    new_date = p->first;
  else if (p->last.dword < new_date.dword)
    new_date = p->last;
  if (new_date.dword != p->current.dword)
  {
    if (new_date.x.year == p->current.x.year && new_date.x.month == p->current.x.month)
    {
      invalidate_day(handle, p, p->current.x.day);
      invalidate_day(handle, p, new_date.x.day);
    }
    else
      InvalidateRect(handle, NULL, TRUE);
    p->current = new_date;
    if (notify)
      SendMessage(GetParent(handle), WM_COMMAND, GetMenu(handle),
        MAKELONG(handle, CAL_CHANGE));
  }
}

/*-----------------------------------------------------------------------------
This function outputs the specified string, using the specified character width
as the character spacing.
-----------------------------------------------------------------------------*/

static void text_out(HDC dc, int x, int y, int width, char *s)
{
  while (*s != 0)
  {
    TextOut(dc, x, y, s++, 1);
    x += width;
  }
}

/*-----------------------------------------------------------------------------
This function receives all messages directed to the control.
-----------------------------------------------------------------------------*/

long far pascal _export CalendarProc(HWND handle, UINT message, WPARAM wParam,
  LPARAM lParam)
{
  calendar *p = (calendar *)
    #if sizeof(void *) == sizeof(short)
      GetWindowWord(handle, 0);
    #else
      GetWindowLong(handle, 0);
    #endif
  switch (message)
  {
    case WM_CREATE:
    {
      p = new calendar;
      if (p == NULL)
        return -1;
      {
        LOGFONT f;
        memset(&f, 0, sizeof(LOGFONT));
        f.lfHeight = (((CREATESTRUCT far *) lParam)->cy - 6) / 8;
        f.lfWidth = (((CREATESTRUCT far *) lParam)->cx - 6) / 20;
        f.lfWeight = FW_MEDIUM;
        f.lfPitchAndFamily = FIXED_PITCH | FF_ROMAN;
        f.lfQuality = PROOF_QUALITY;
        p->font = CreateFontIndirect(&f);
        if (p->font == NULL)
        {
          delete p;
          return -1;
        }
      }
      {
        HDC dc = GetDC(handle);
        HFONT old_font = SelectObject(dc, p->font);
        TEXTMETRIC tm;
        GetTextMetrics(dc, &tm);
        p->character_width = tm.tmAveCharWidth;
        p->character_height = tm.tmHeight + tm.tmExternalLeading;
        SelectObject(dc, old_font);
        ReleaseDC(handle, dc);
      }
      p->height = ((CREATESTRUCT far *) lParam)->cy;
      p->width = ((CREATESTRUCT far *) lParam)->cx;
      p->focus = FALSE;
      p->disabled = (((CREATESTRUCT far *) lParam)->style & WS_DISABLED) != 0;
      p->first.dword = MAKEDATE(1, 1, 1800);
      p->last.dword = MAKEDATE(12, 31, 2099);
      p->current.dword = 0L;
      #if sizeof(void *) == sizeof(short)
        SetWindowWord(handle, 0, (WORD) p);
      #else
        SetWindowLong(handle, 0, (DWORD) p);
      #endif
      return 0;
    }
    case WM_PAINT:
    {

      COLORREF gray_color = GetSysColor(COLOR_GRAYTEXT);
      COLORREF normal_text_color = p->disabled ? gray_color :
        GetSysColor(COLOR_WINDOWTEXT);
      COLORREF normal_background_color = GetSysColor(COLOR_WINDOW);
      COLORREF selected_text_color = GetSysColor(COLOR_HIGHLIGHTTEXT);
      COLORREF selected_background_color = GetSysColor(COLOR_HIGHLIGHT);
      PAINTSTRUCT paint;
      HDC dc = BeginPaint(handle, &paint);
      HPEN pen = CreatePen(PS_SOLID, 1, normal_text_color);
      HPEN old_pen = SelectObject(dc, pen);
      HBRUSH brush = CreateSolidBrush(normal_background_color);
      HBRUSH old_brush = SelectObject(dc, brush);
      HFONT old_font = SelectObject(dc, p->font);
      union cal_date date;
      date = p->current;
      Rectangle(dc, 0, 0, p->width, p->height);
      if (p->focus)
        Rectangle(dc, 1, 1, p->width - 1, p->height - 1);
      if (date.dword != 0L)
      {
        SetTextColor(dc, normal_text_color);
        SetBkColor(dc, normal_background_color);
        {
          char s[8];
          strcpy(s, (date.dword & 0xFFFFFF00L) >
            (p->first.dword & 0xFFFFFF00L) ? "<-" : "  ");
          strcpy(s+2, MONTH_NAME[date.x.month]);
          strcpy(s+5, (date.dword & 0xFFFFFF00L) <
            (p->last.dword & 0xFFFFFF00L) ? "->" : "  ");
          text_out(dc, 3, 3, p->character_width, s);
        }
        {
          unsigned y = date.x.year;
          char s[9];
          strcpy(s, y > p->first.x.year ? "<-": "  ");
          s[5] = y%10 + '0';
          y /= 10;
          s[4] = y%10 + '0';
          y /= 10;
          s[3] = y%10 + '0';
          s[2] = y/10 + '0';
          strcpy(s+6, y < p->last.x.year ? "->" : "  ");
          text_out(dc, 3 + 12 * p->character_width, 3, p->character_width, s);
        }
        text_out(dc, 3, 3 + p->character_height, p->character_width, DAY_NAMES);
        {
          unsigned horizontal_offset = spaces_at_beginning_of_month(date);
          int y = 3 + 2 * p->character_height;
          unsigned days_in_this_month = days_in_month(date);
          union cal_date d;
          for (d.dword = (date.dword & 0xFFFFFF00L) + 1;
            d.x.day <= days_in_this_month; d.x.day++)
          {
            char s[2];
            s[0] = d.x.day < 10 ? ' ' : d.x.day/10 + '0';
            s[1] = d.x.day%10 + '0';
            if (d.dword == date.dword)
            {
              SetTextColor(dc, selected_text_color);
              SetBkColor(dc, selected_background_color);
            }
            else if (d.dword < p->first.dword || p->last.dword < d.dword)
            {
              SetTextColor(dc, gray_color);
              SetBkColor(dc, normal_background_color);
            }
            else
            {
              SetTextColor(dc, normal_text_color);
              SetBkColor(dc, normal_background_color);
            }
            TextOut(dc, 3 + p->character_width * 3 * horizontal_offset, y, s, 2);
            if (++horizontal_offset == DAYS_PER_WEEK)
            {
              horizontal_offset = 0;
              y += p->character_height;
            }
          }
        }
      }
      SelectObject(dc, old_font);
      SelectObject(dc, old_pen);
      DeleteObject(pen);
      SelectObject(dc, old_brush);
      DeleteObject(brush);
      EndPaint(handle, &paint);
      return 0;
    }
    case CAL_GETDATE:
      return p->current.dword;
    case CAL_SETDATE:
      change_date(handle, p, *(union cal_date *)(&lParam), FALSE);
      return 0;
    case CAL_FIRSTDATE:
      p->first.dword = lParam;
      return 0;
    case CAL_LASTDATE:
      p->last.dword = lParam;
      return 0;
    case WM_KEYDOWN:
    {
      union cal_date new_date;
      new_date = p->current;
      if (wParam == VK_LEFT)
        new_date = previous_day(new_date);
      else if (wParam == VK_RIGHT)
        new_date = next_day(new_date);
      else if (wParam == VK_UP)
      {
        unsigned n = DAYS_PER_WEEK;
        do new_date = previous_day(new_date); while (--n != 0);
      }
      else if (wParam == VK_DOWN)
      {
        unsigned n = DAYS_PER_WEEK;
        do new_date = next_day(new_date); while (--n != 0);
      }
      else if (wParam == VK_PRIOR)
      {
        if (GetKeyState(VK_CONTROL) & 0x8000)
        {
          if (new_date.x.month == 1 && new_date.x.day == 1)
            new_date.x.year--;
          else
            new_date.x.month = new_date.x.day = 1;
        }
        else
          do new_date = previous_day(new_date); while (new_date.x.day != 1);
      }
      else if (wParam == VK_NEXT)
      {
        if (GetKeyState(VK_CONTROL) & 0x8000)
        {
          new_date.x.year++;
          new_date.x.month = new_date.x.day = 1;
        }
        else
          do new_date = next_day(new_date); while (new_date.x.day != 1);
      }
      change_date(handle, p, new_date, TRUE);
      return 0;
    }
    case WM_ENABLE:
      if (wParam && p->disabled || !wParam && !(p->disabled))
      {
        p->disabled = !p->disabled;
        InvalidateRect(handle, NULL, TRUE);
      }
      return 0;
    case WM_LBUTTONDOWN:
      if (3 <= HIWORD(lParam) &&
        HIWORD(lParam) <= 8 * p->character_height + 3 &&
        3 <= LOWORD(lParam) && LOWORD(lParam) <= 20 * p->character_width + 3)
      {
        unsigned i = (HIWORD(lParam) - 3) / p->character_height;
        union cal_date new_date;
        new_date = p->current;
        if (i == 0)
        {
          unsigned j = (LOWORD(lParam) - 3) / p->character_width;
          if (j == 0 || j == 1)
          {
            do new_date = previous_day(new_date); while (new_date.x.day != 1);
          }
          else if (j == 5 || j == 6)
          {
            do new_date = next_day(new_date); while (new_date.x.day != 1);
          }
          else if (j == 12 || j == 13)
          {
            if (new_date.x.month == 1 && new_date.x.day == 1)
              new_date.x.year--;
            else
              new_date.x.month = new_date.x.day = 1;
          }
          else if (j == 18 || j == 19)
          {
            new_date.x.year++;
            new_date.x.month = new_date.x.day = 1;
          }
        }
        else if (i >= 2)
        {
          unsigned j = (LOWORD(lParam) - 3 + p->character_width/2) / (3*p->character_width);
          if (j < DAYS_PER_WEEK)
          {
            int d = 1 + (i - 2) * DAYS_PER_WEEK + j -
              spaces_at_beginning_of_month(p->current);
            if (1 <= d && d <= days_in_month(p->current))
              new_date.x.day = d;
          }
        }
        SetFocus(handle);
        change_date(handle, p, new_date, TRUE);
        return 0;
      }
      break;
    case WM_SETFOCUS:
      p->focus = TRUE;
      InvalidateRect(handle, NULL, TRUE);
      return 0;
    case WM_KILLFOCUS:
      p->focus = FALSE;
      InvalidateRect(handle, NULL, TRUE);
      SendMessage(GetParent(handle), WM_COMMAND, GetMenu(handle),
        MAKELONG(handle, CAL_KILLFOCUS));
      return 0;
    case WM_GETDLGCODE:
      return DLGC_WANTARROWS;
    case WM_ERASEBKGND:
      return 0;
    case WM_DESTROY:
      DeleteObject(p->font);
      delete p;
      return 0;
  }
  return DefWindowProc(handle, message, wParam, lParam);
}

