/*
 * FidoNet(tm) Address Parsing Class (full support for 5D format)
 *
 * Copyright (c) 1995, 1997 by Branislav L. Slantchev
 * A fine product of Silicon Creations, Inc. (gargoyle)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the License which accompanies this
 * software. This library is distributed in the hope that it will
 * be useful, but without any warranty; without even the implied
 * warranty of merchantability or fitness for a particular purpose.
 *
 * You should have received a copy of the License along with this
 * library, in the file LICENSE.DOC; if not, write to the address
 * below to receive a copy via electronic mail.
 *
 * You can reach Branislav L. Slantchev (Silicon Creations, Inc.)
 * at bslantch@cs.angelo.edu. The file SUPPORT.DOC has the current
 * telephone numbers and the postal address for contacts.
*/

#include "fidoaddr.h"

#ifndef EOS
	#define EOS '\0'
#endif

#ifndef PB_SDK
	#include <stdio.h>
	#include <string.h>
	#include <stdlib.h>
#else
	#include "pblibc.h"
#endif

fido_address::fido_address()
	:m_zone(0)
	,m_net(0)
	,m_node(0)
	,m_point(0)
{
	m_domain[0] = EOS;
}

fido_address::fido_address(int zone, int net, int node, int point)
	:m_zone(zone)
	,m_net(net)
	,m_node(node)
	,m_point(point)
{
	m_domain[0] = EOS;
}

fido_address::fido_address(int z, int net, int node, int p, const char *d)
	:m_zone(z)
	,m_net(net)
	,m_node(node)
	,m_point(p)
{
	strcpy(m_domain, d);
}

fido_address::fido_address(const fido_address &address)
	:m_zone(address.m_zone)
	,m_net(address.m_net)
	,m_node(address.m_node)
	,m_point(address.m_point)
{
	strcpy(m_domain, address.m_domain);
}

fido_address::fido_address(const char *ascii_address)
{
	split(ascii_address);
}

fido_address&
fido_address::operator=(const fido_address &address)
{
	m_zone  = address.m_zone;
	m_net   = address.m_net;
	m_node  = address.m_node;
	m_point = address.m_point;
	strcpy(m_domain, address.m_domain);
	return *this;
}

char*
fido_address::merge(char *buf) const
{
	buf[0] = EOS;
	if( m_zone ) sprintf(buf, "%u:", m_zone);
	if( m_net ) sprintf(buf + strlen(buf), "%u/", m_net);
	sprintf(buf + strlen(buf), "%u", m_node);
	if( m_point ) sprintf(buf + strlen(buf), ".%u", m_point);
	if( EOS != m_domain[0] ) sprintf(buf + strlen(buf), "@%s", m_domain);

	return buf;
}

void
fido_address::split(const char *ascii_address)
{
	const char *p, *p2;

	m_zone = m_net = m_node = m_point = 0;
	m_domain[0] = EOS;

	// check for zone first and set it, of found; note that
	// atol() stops on invalid chars, which is the ':' in our case
	if( 0 != (p = strchr(ascii_address, ':')) )
	{
		m_zone = (int)atol(ascii_address);
		p++;
	}
	else p = ascii_address;

	// here 'p' points either past the zone in the address
	// or to the beginning of the string address, check for net
	if( 0 != (p2 = strchr(p, '/')) )
	{
		m_net = (int)atol(p);
		p = p2 + 1;
	}

	// here we have 'p' pointing to the node (which must
	// be present in any of the valid address combinations)
	m_node = (int)atol(p);

	// now try to get the point if there is any
	if( 0 != (p2 = strchr(p, '.')) ) m_point = (int)atol(p2+1);

	// now see if we have a domain designation
	if( 0 != (p2 = strchr(p, '@')) ) strcpy(m_domain, p2 + 1);
}

char*
fido_address::merge2d(char *buf) const
{
	sprintf(buf, "%u/%u", m_net, m_node);
	return buf;
}

char*
fido_address::merge3d(char *buf) const
{
	sprintf(buf, "%u:%u/%u", m_zone, m_net, m_node);
	return buf;
}

char*
fido_address::merge4d(char *buf) const
{
	sprintf(buf, "%u:%u/%u.%u", m_zone, m_net, m_node, m_point);
	return buf;
}

char*
fido_address::merge5d(char *buf) const
{
	if( EOS == m_domain[0] )
	{
		merge4d(buf);
	}
	else
	{
		sprintf(buf, "%u:%u/%u.%u@%s", m_zone,m_net,m_node,m_point,m_domain);
	}
	return buf;
}

char*
fido_address::domain(char *buf) const
{
	return strcpy(buf, m_domain);
}

void
fido_address::set_domain(const char *domain)
{
	strcpy(m_domain, domain);
}
