/*
** Program fdb - family tree database generation and manipulation
**
** Copyright (C) 1994 Andy Burrows 
**
**            email: cadellin@corum.me.man.ac.uk (130.88.29.14)
**
** This program is free software; you can redistribute it and/or modify it
** under the terms of the GNU General Public Licence as published by the Free
** Software Foundation; either version 1, or any later version.
** 
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
** or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
** more details. It should be included in this distribution in a file 
** called COPYING
**
*/

/*
   header file for the family tree database code - cadellin `94
*/

/*
   macros definitions
*/

/* version of the program */

#define  FDB_VERSION        "2.3.1"

/* max. number of indices in the database */

#define  MAX_INDICES          200 

/* max. number of spouses per person and children per person */

#define  MAX_SPOUSES          7            /* max. number of spouses per person 
                                              N.B. the actual limit is 1 less than the
                                              value specified because allowance has
                                              to be made for a possible dummy spouse
                                              for gathering "unattached" children
                                              in build_node */
#define  MAX_CHILDREN         12           /* max. number of children per person */

/* max. lengths of various text strings */

#define  MAX_SURNAME         42            /* max. length of a surname string */
#define  MAX_FORENAMES       62            /* max. length of a forenames string */
#define  MAX_TITLE           15            /* max. length of a title string */
#define  MAX_DATE           255            /* max. length of a date string 
                                              it's that big to cope with multiple
                                              marriages, for example */
#define  MAX_STRING_LENGTH  255            /* max. length of a general string */
#define  MAX_OCCUPATION     255            /* max. length of an occupation string */

/* positioning of elements of entry/display in panel */

#define  X_OFFSET            10    /* offset from left side of panel in pixels */
#define  Y_OFFSET            10    /* offset from top of panel in pixels */
#define  Y_SEPARATION        30    /* vertical separation between lines in pixels */

/* male and female (and an either entry) to make the code more readable */

#define  MALE        ((int) 0)
#define  FEMALE      ((int) 1)
#define  ANY_GENDER  ((int) -1)
 
/* status variables */

#define  LIVING    ((int) 0)
#define  DECEASED  ((int) 1)

/* definition of the allowable selection types for database entries */

#define  SELECTION   ((int) 0)
#define  DESELECTION ((int) 1)

#define  MOTHER  ((int) 0)
#define  FATHER  ((int) 1)
#define  SPOUSE  ((int) 2)
#define  CHILD   ((int) 3)
#define  ANYBODY ((int) 4)

/* the following help define the layout of the graphical tree displays */

#define  LARGER_FONT            14      /* larger font to use for display */
#define  SMALLER_FONT           12      /* smaller font to use for display */
#define  VERTICAL_SEPARATION     8      /* vertical spacing between generations 
                                           specified in number of text line heights */
#define  HORIZONTAL_SEPARATION  35      /* horizontal separation between members
                                           of a generation in pixels */
#define  VIEW_MARGIN            50      /* margin around display */
#define  DELTA_Y                 5      /* number of pixels between lines in a 
                                           persons entry on the canvas */

/* the following help in the PostScript plot generation */

#define  A4_HEIGHT_PTS          ((float) 841.9)
#define  A4_WIDTH_PTS           ((float) 598.1)

/* 
   the minimum allowed separation in seconds between consecutive selections
   in the same scrolling list - below this threshold the second selection
   is ignored
   this is also used for certain choice selections after a report from a user
   of problems (I have been unable to recreate the situation though!)
   the FILE value is larger to allow for creation of each new display
   of files when traversing the directory structure
*/

#define  MULTICLICK_TOLERANCE        0.2
#define  FILE_MULTICLICK_TOLERANCE   1.5

/* the possible values of the output_type_flag */

#define  LATEX                       0
#define  POSTSCRIPT                  1

/* the first time flag */

int  first_time;

/* the number of clock ticks per second */

double  ticks_per_sec;

/* the maximum permitted window width and height */

int  max_window_width;
int  max_window_height;

/*
   create an array of pointers to database record structs
*/

struct db_record
    {
    char  surname[MAX_SURNAME];               /* surname string           */
    char  maiden_name[MAX_SURNAME];           /* maiden name string       */
    char  forenames[MAX_FORENAMES];           /* forenames string         */
    char  title[MAX_TITLE];                   /* title string             */
    int   gender;                             /* 0 = male, 1 = female     */
    int   status;                             /* 0 = living, 1 = deceased */

    char  birth_date[MAX_DATE];               /* date of birth     */
    long int encoded_date;                    /* date of birth coded as a long int
                                                 YYYYMMDD where YYYY is the year
                                                                MM   is the month
                                                                DD   is the day
                                                 used for ordering children for views */
    char  baptism_date[MAX_DATE];             /* date of baptism   */
    char  marriage_date[MAX_DATE];            /* date of marriage  */
    char  death_date[MAX_DATE];               /* date of death     */
    
    char  birth_place[MAX_STRING_LENGTH];     /* place of birth     */
    char  baptism_place[MAX_STRING_LENGTH];   /* place of baptism   */
    char  marriage_place[MAX_STRING_LENGTH];  /* place of marriage  */
    char  death_place[MAX_STRING_LENGTH];     /* place of death     */
    char  resting_place[MAX_STRING_LENGTH];   /* place of rest      */
    
    int   father;                          /* index of father's record */
    int   mother;                          /* index of mother's record */

    int   number_of_spouses;               /* number of spouses           */
    int   spouse[MAX_SPOUSES];             /* indices of spouse's records */

    int   number_of_children;              /* number of children            */
    int   child[MAX_CHILDREN];             /* indices of children's records */

    char  occupation[MAX_OCCUPATION];      /* occupation */

    char  notes[MAX_STRING_LENGTH];        /* miscellaneous notes */

    char  birth_source[MAX_STRING_LENGTH];    /* source of information on birth */
    char  baptism_source[MAX_STRING_LENGTH];  /* source of information on baptism */
    char  marriage_source[MAX_STRING_LENGTH]; /* source of information on marriage */
    char  death_source[MAX_STRING_LENGTH];    /* source of information on death */
    } *entry[MAX_INDICES];

/* 
   array used to store the indices of the entries in the entry 
   selection scrolling list
*/

int     entries_index[MAX_INDICES];

/*
   structs used for graphically displaying family trees
*/

struct  db_generation
    {
    int    number_of_children;       /* number of people in this generation */
    int    children[MAX_CHILDREN];   /* indices of members of this generation */
    char   forenames[MAX_CHILDREN][MAX_FORENAMES]; /* forenames of generation members */
    char   surname[MAX_CHILDREN][2 * MAX_SURNAME + 7]; /* surname of generation members */
    char   birth_date[MAX_CHILDREN][MAX_DATE + 3]; /* date of birth of generation members */
    char   death_date[MAX_CHILDREN][MAX_DATE + 3]; /* date of death of generation members */
    int    x[MAX_CHILDREN];          /* x location of generation member */
    int    y[MAX_CHILDREN];          /* y location of generation member */
    int    person;                   /* index of the person through whom the line
                                        is being traced */ 
    int    location;                 /* position of person in child array */
    int    spouse;                   /* index of the spouse of the person through
                                        whom the line is being traced */
    char   spouse_name[MAX_FORENAMES + 2 * MAX_SURNAME + 10]; /* spouses name */
    } *generation[MAX_INDICES];

struct  point
    {
    int   x;   /* x coordinate */
    int   y;   /* y coordinate */
    };

struct  db_descendant
    {
    /* index of descendant */

    int    index;

    /* number of children of descendant */

    int    number_of_children;

    /* number of spouses of descendant */

    int    number_of_spouses;

    /* indices of the spouses */

    int    spouse[MAX_SPOUSES]; 
 
    /* spouses names */

    char   spouse_name[MAX_SPOUSES][MAX_FORENAMES + 2 * MAX_SURNAME + 10];

    /* location of spouse name on display */

    struct point  spouse_location[MAX_SPOUSES];

    /* number of children per spouse */

    int    num_children_of_spouse[MAX_SPOUSES];

    /* indices of the children of each spouse */

    int    children_of_spouse[MAX_SPOUSES][MAX_CHILDREN];

    /*
       pointers to structs of children, related to the relevant other 
       parent i.e. spouse 
    */

    struct db_descendant   *children[MAX_SPOUSES][MAX_CHILDREN];

    /* forenames of descendant */

    char   forenames[MAX_FORENAMES];

    /* surname of descendant */

    char   surname[2 * MAX_SURNAME + 7];

    /* date of birth of descendant */

    char   birth_date[MAX_DATE + 3];

    /* date of death of descendant */

    char   death_date[MAX_DATE + 3];

    /* location of descendant */

    struct point  location;

    } *descendant[MAX_INDICES];

/*
   total number of descendant structs currently defined
*/

int  number_of_descendants;

/*
   the current x location of the right most defined position 
   during the creation of a descendant tree
*/

int  current_x;

/*
   the maximum number of generations to be plotted in the descent line
   and descendants tree graphical displays
   also the variables to store the old values for when the sliders are
   deactivated due to the "Use All Available" generations check boxes
   being set
*/

int  max_gen_generations;
int  max_des_generations;
int  old_max_gen_generations;
int  old_max_des_generations;

/*
   limits of the descendants plot
*/

int  max_desc_x, min_desc_x, max_desc_y, min_desc_y;

/* 
   the current selection nature (SELECTION or DESELECTION) 
   and type (MOTHER, FATHER, SPOUSE, CHILD or ANYBODY)
   being performed 
*/

int     selection_nature;
int     selection_type;

/* the current file name and temporary new file name */

char    file_name[MAX_STRING_LENGTH - 24];
char    new_file_name[MAX_STRING_LENGTH - 24];

/* the left footer string, the status string and the error message string */

char    footer_string[MAX_STRING_LENGTH];
char    status_string[MAX_STRING_LENGTH];
char    error_message[MAX_STRING_LENGTH];

/* index of the currently active database record */

int     current_index;

/* the maximum index currently in use */

int     max_index;

/* the number of entries in the database */

int     num_entries;

/* dimensions of the canvas which displays the descent line */

int gen_canvas_width, gen_canvas_height;

/* dimensions of the canvas which displays the descendants tree */

int des_canvas_width, des_canvas_height;

/* 
   the vertical separation between lines of text in a person entry 
   and between generations on the canvas 
*/

int vertical_text_separation;
int vertical_generation_separation;

/*
   the minimum allowed text width for a leaf during creation of
   the descendants tree
*/

int min_text_width;

/*
   the current output format (LATEX or POSTSCRIPT)
*/

int output_type_flag;

/* Xlib variables used when drawing onto the canvas */

Display      *display, *des_display, *gen_display;
XGCValues     gc_values;
GC            gc, gc_small;
int           screen_number;

/* XView variables used when drawing onto the canvas etc. */

Xv_Screen          screen;
Xv_Font            font, small_font;
Font_string_dims   dimensions;

/* 
   array for use in generating a line of descent for viewing parts of family tree
   holds the indices of the people through whom the current line is being traced
   and the number of people in the current line (including the current person)
*/

int descent_array[MAX_INDICES];
int number_of_generations;

/* 
   flag used to determine whether spaces are to be left in hard copy output
   to allow the user to write the data in while investigating records etc.
*/

int leave_spaces_in_output;

/* array used to check for circular paths in the database */

int in_use[MAX_INDICES];

/* define the available titles and their number */

static char    titles[][MAX_TITLE] = { "", "Mr.", "Mrs.", "Miss", "Ms.", "Dr.",
                         "Prof.", "Rev.", "Sir", "Dame", "Lord", "Lady", 
                         "Rev. Dr.", "Eur. Ing.", "Eur. Ing. Dr.", "Cdr.",
                         "Lt.", "Capt.", "Col.", "Gen.", "Brig.", "Prince",
                         "Princess", "King", "Queen", "Flt. Lt.", "Sqdr. Ldr.",
                         "Wng. Cdr.", "Comm.", "Adm."};
static int     num_titles = 30;

/* 
   declare the global variables which will be needed by various
   callback routines
*/

Rect            image_rect, label_rect;
Server_image    main_image, descent_image, tree_image;
Icon            main_icon, descent_icon, tree_icon;

Frame       fdb_frame;           /* the base window which holds the fdb display */
Frame       gen_frame;           /* the base window which holds the descent line display */
Frame       des_frame;           /* the base window which holds the descendants display */
Frame       source_frame;        /* the base window which holds the date sources display */
Frame       save_frame;          /* the base window which holds the save as file selector */
Frame       load_frame;          /* the base window which holds the load from file selector */
Frame       entries_frame;       /* the base window which holds the entry selection list */
Frame       error_frame;         /* the base window which holds the error popup */
Frame       settings_frame;      /* the base window which holds the settings menu */
Panel       fdb_panel;           /* the main database entry control panel */
Panel       gen_panel;           /* the descent display control panel */
Panel       des_panel;           /* the descendants display control panel */
Panel       source_panel;        /* the date source entry control panel */
Panel       menu_panel;          /* the control panel which holds the row of menu buttons */
Panel       save_panel;          /* the control panel which holds the save as selectors */
Panel       load_panel;          /* the control panel which holds the load from selectors */
Panel       entries_panel;       /* the control panel which holds the entry selection list */
Panel       error_panel;         /* the control panel which holds the error popup */
Panel       settings_panel;      /* the control panel which holds the settings menu */
Canvas      fdb_canvas;          /* the area where the family tree is drawn */
Canvas      gen_canvas;          /* the area where the descent line is drawn */
Canvas      des_canvas;          /* the area where the descendants tree is drawn */
Menu        file_menu;           /* file handling menu                      */
Menu        edit_menu;           /* edit handling menu                      */
Menu        select_menu;         /* select relative handling menu           */
Menu        deselect_menu;       /* deselect relative handling menu         */
Menu        output_menu;         /* output handling menu                    */
Menu        view_menu;           /* database viewing menu                   */
Menu_item   load_file_item;      /* the menu item associated with the "Load" file
                                    option of the file menu - used to allow
                                    manual calling of it's callback routine */
Menu_item   save_file_item;      /* the menu item associated with the "Save as..." file
                                    option of the file menu - used to allow
                                    manual calling of it's callback routine */
Menu_item   plot_des_line_item;       
Menu_item   plot_gen_tree_item;  /* the two menu items associated with hard copy 
                                    of the descent line and generation tree respectively
                                    these handles are used to deactivate these options
                                    when hard copy type is set to LaTeX */
Panel_item  depth_gen_choice;    /* choice of max. generations in a descent line */
Panel_item  use_all_gen_choice;  /* use all available generations in a descent line */
Panel_item  depth_des_choice;    /* choice of max. generations in a descendants tree */
Panel_item  use_all_des_choice;  /* use all available generations in a descendants tree */
Panel_item  title_choice;        /* choice of title                         */
Panel_item  surname_text;        /* text entry/display for surname          */
Panel_item  forenames_text;      /* text entry/display for forenames        */
Panel_item  gender_choice;       /* choice of gender                        */
Panel_item  status_choice;       /* choice of status                        */
Panel_item  output_choice;       /* choice of hard copy output format       */
Panel_item  add_spaces_choice;   /* choice of spaces in text output         */
Panel_item  maiden_name_text;    /* text entry/display for maiden name      */
Panel_item  birth_date_text;     /* text entry/display for birth date       */
Panel_item  birth_place_text;    /* text entry/display for birth place      */
Panel_item  baptism_date_text;   /* text entry/display for baptism date     */
Panel_item  baptism_place_text;  /* text entry/display for baptism place    */
Panel_item  marriage_date_text;  /* text entry/display for marriage date    */
Panel_item  marriage_place_text; /* text entry/display for marriage place   */
Panel_item  death_date_text;     /* text entry/display for death date       */
Panel_item  death_place_text;    /* text entry/display for death place      */
Panel_item  resting_place_text;  /* text entry/display for resting place    */
Panel_item  occupation_text;     /* text entry/display for occupation       */
Panel_item  notes_text;          /* text entry/display for notes            */
Panel_item  display_mother_button;  /* display mothers info button          */
Panel_item  display_father_button;  /* display fathers info button          */
Panel_item  spouse_list;            /* display spouse(s) scrolling list     */
Panel_item  child_list;             /* display spouse(s) scrolling list     */
Panel_item  sources_message;        /* title for the information sources    */
Panel_item  birth_source_text;    /* text entry/display for source of birth info    */
Panel_item  baptism_source_text;  /* text entry/display for source of baptism info  */
Panel_item  marriage_source_text; /* text entry/display for source of marriage info */
Panel_item  death_source_text;    /* text entry/display for source of death info    */

Panel_item  load_list;           /* Load From File selection scrolling list */
Panel_item  load_file_text;      /* Load From File selection text entry     */
Panel_item  load_cancel_button;  /* Load From File selection cancel button  */
Panel_item  load_message;        /* Displays the current working directory  */
Panel_item  save_list;           /* Save as File selection scrolling list   */
Panel_item  save_file_text;      /* Save as File selection text entry       */
Panel_item  save_cancel_button;  /* Save as File selection cancel button    */
Panel_item  save_message;        /* Displays the current working directory  */
Panel_item  entries_list;        /* Select Entry selection scrolling list   */
Panel_item  cancel_select_button;/* Entry selection cancel button           */

Panel_item  error_message_item;  /* error message display                   */
Panel_item  error_ok_button;     /* dismiss error display button            */
Panel_item  settings_ok_button;  /* dismiss settings menu display button    */
Panel_item  dismiss_gen_button;  /* dismiss descent display button          */
Panel_item  dismiss_des_button;  /* dismiss descendants display button      */
Panel_item  min_text_width_slider; /* slider to allow control of horizontal
                                     spacing of leaves in descendants tree  */
Panel_item  descent_depth_slider;  /* slider to allow control of maximum depth
                                     of a descent line plot */
Panel_item  descendants_depth_slider; /* slider to allow control of maximum 
                                     depth of a descendants tree plot */
Xv_Window   surname_window;      /* window associated with surname text entry */

