#include <stdio.h>
#include <string.h>

/* Defines and macros */
#define EVAL_NONE	0
#define EVAL_BAD	1
#define	EVAL_INT	2
#define EVAL_DUB	3

#define ISVARSTART(d) ((d[0] & 0x40) == 0)
#define ISVAREND(d) (d[0] & 0x80)

/* Used to hold move */
struct Move {
	char FromFile, FromRank;
	char ToFile, ToRank;
	unsigned char Eval;
	char StartVar;	
};

/* List of moves seen */
struct Move Moves[256]; 

/* Input and Output files */
FILE *fileFBK, *fileRBM;

/* Function prototypes */
static const char *GetEvalStr(char Eval);

int
main(int argc, char **argv)
{
	unsigned char data[2];
	struct Move *pMove = Moves;
	unsigned long lines = 0;

	argc--;
	argv++;
	if (argc != 2) {
		fprintf(stderr, "usage: FBK2RBM fbk-file rbm-file\n");
		fprintf(stderr, "This program converts Fritz powerbook files,\n");
		fprintf(stderr, "created and maintained using ChessBase-for-Windows,\n");
		fprintf(stderr, "to REBEL opening book files.\n");
		fprintf(stderr, "Note: lines will be truncated, if necessary, to 50 halfmoves\n");
		return 1;
	}
	
	/* Open files */
	fileFBK = fopen(*argv, "rb");
	if (!fileFBK) {
		fprintf(stderr, "Error opening fbk-file \"%s\"\n", *argv);
		return 1;
	}
	fileRBM = fopen(*++argv, "wt");
	if (!fileRBM) {
		fprintf(stderr, "Error opening rbm-file \"%s\"\n", *argv);
		fclose(fileFBK);
		return 1;
	}

	/* Print banner */
	printf("FBK2RBM V1.0  Copyright 1995 Andy Duplain (duplaina@syntegra.bt.co.uk)\n");
	printf("Public Domain\n");

	/* Initialise moves array */	
	memset(pMove, 0, sizeof(Moves));

	/* Keep reading until EOF */	
	while (fread((char *)data, 1, 2, fileFBK) == 2) {
	
		/* Get "from" and "to" squares */
		pMove->FromFile = (data[0] & 7) + 'a';
		pMove->FromRank = ((data[0] >> 3) & 7) + '1';
		pMove->ToFile = (data[1] & 7) + 'a';
		pMove->ToRank = ((data[1] >> 3) & 7) + '1';
		
		/* Get variation start marker */
		pMove->StartVar = ISVARSTART(data);
		
		/* Get move evaluation */
		switch (data[1] & 0xc0) {
		case 0x80:
			pMove->Eval = EVAL_BAD;
			break;
		case 0x40:
			pMove->Eval = EVAL_INT;
			break;
		case 0xc0:
			pMove->Eval = EVAL_DUB;
			break;
		default:
			pMove->Eval = EVAL_NONE;
			break;
		}
		
		/* If this is the end of the line, print it */
		if (ISVAREND(data)) {
			struct Move *pTemp;
			int nMoves;
			for (pTemp = Moves, nMoves = 0; pTemp <= pMove; pTemp++, nMoves++) {
				if (nMoves < 50) {
					fprintf(fileRBM, "%c%c%c%c%s", pTemp->FromFile, pTemp->FromRank,
						pTemp->ToFile, pTemp->ToRank, GetEvalStr(pTemp->Eval));
				}
			}
			putc('\n', fileRBM);
			lines++;
			
			if (lines % 10 == 0) {
				printf("lines: %ld\r", lines);
			}
			
			/* Now find the start of the line, destroying later moves as we go... */
			while (pMove > Moves) {
				if (pMove->StartVar) {
					memset(pMove, 0, sizeof(struct Move));
					break;
				}
				memset(pMove, 0, sizeof(struct Move));
				pMove--;
			}
		} else {
			pMove++;
		}
	}

	printf("lines: %lu\n", lines);
		
	fclose(fileRBM);
	fclose(fileFBK);

	return 0;
}                                        

const char *
GetEvalStr(char Eval)
{
	switch(Eval) {
	case EVAL_BAD: 		return "-";	
	case EVAL_INT:		return " ";
	case EVAL_DUB:		return "-";
	case EVAL_NONE:
	default:			return " ";	
	}
}