//
// $Header: D:/ext2-os2/minifsd/vfs/RCS/buffer.c,v 9.0 1996/06/02 16:39:25 Willm Exp Willm $
//

// Linux ext2 file system driver for OS/2 2.x and WARP - Allows OS/2 to
// access your Linux ext2fs partitions as normal drive letters.
// Copyright (C) 1995, 1996 Matthieu WILLM
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#define INCL_DOSERRORS
#define INCL_DOS
#include <os2.h>

#include <dos.h>

#include <os2/types.h>
#include <os2/magic.h>
#include <linux/fs.h>
#include <linux/fs_proto.h>
#include <os2/os2proto.h>

extern struct buffer_head * (*bread)();
extern void (*brelse)();
extern struct buffer_head * (*getblk)();

#define NR_BUFFERS 30


char b_data[NR_BUFFERS  * BLOCK_SIZE];
struct buffer_head  bhlist[NR_BUFFERS];
static struct buffer_head *hash_table[NR_BUFFERS];
struct buffer_head *used_list;
struct buffer_head *free_list;

void buffer_init(void) {
    int i;
    for (i = 0; i < NR_BUFFERS; i++) {
        hash_table[i]         = 0;
        bhlist[i].b_data      = b_data + i * BLOCK_SIZE;
        bhlist[i].b_blocknr   = 0xFFFFFFFF;
        bhlist[i].b_uptodate  = 0;
        bhlist[i].b_count     = 0;
        bhlist[i].b_magic     = BUFFER_HEAD_MAGIC;
        bhlist[i].b_prev      = 0;
        bhlist[i].b_next      = 0;
        bhlist[i].b_prev_free = (i == 0 ? bhlist + NR_BUFFERS - 1 : bhlist + i - 1);
        bhlist[i].b_next_free = (i == NR_BUFFERS - 1 ? bhlist : bhlist + i + 1);
        bhlist[i].b_dev       = 0xFFFF;
        bhlist[i].b_lock      = 0;
    }
    used_list = 0;
    free_list = bhlist;
}

#define _hashfn(dev,block) (((unsigned long)(((unsigned long)(dev))^block))%NR_BUFFERS)
#define hash(dev,block) hash_table[_hashfn(dev,block)]

void remove_from_hash_queue(struct buffer_head * bh) {
    if (!bh)
        ext2_os2_panic(0, "bh is NULL");
    if (bh->b_magic != BUFFER_HEAD_MAGIC)
        ext2_os2_panic(0, "remove_from_hash_queue - invalid magic number");
    if (bh->b_blocknr == -1)
        ext2_os2_panic(0, "remove_from_hash_queue - not in hash queue");

    _disable();
    if (bh->b_next)
        bh->b_next->b_prev = bh->b_prev;
    if (bh->b_prev)
        bh->b_prev->b_next = bh->b_next;
    if (hash(bh->b_dev, bh->b_blocknr) == bh)
        hash(bh->b_dev, bh->b_blocknr) = bh->b_next;
    bh->b_next = 0;
    bh->b_prev = 0;
    _enable();

}

void add_to_hash_queue(struct buffer_head *bh) {

    if (!bh)
        ext2_os2_panic(0, "bh is NULL");
    if (bh->b_magic != BUFFER_HEAD_MAGIC)
        ext2_os2_panic(0, "add_to_hash_queue - invalid magic number");
    if (bh->b_next || bh->b_prev)
        ext2_os2_panic(0, "add_to_hash_queue - already in hash queue");

    _disable();
    bh->b_next = hash(bh->b_dev, bh->b_blocknr);
    hash(bh->b_dev, bh->b_blocknr) = bh;
    if (bh->b_next)
        bh->b_next->b_prev = bh;
    _enable();

}

struct buffer_head *get_hash_table(dev_t dev, blk_t block, unsigned long size) {
        struct buffer_head * tmp;

        for (tmp = hash(dev, block) ; tmp ; tmp = tmp->b_next) {
                if ((tmp->b_blocknr == block) && (tmp->b_dev == dev)) {
                    return tmp;
                }
        }
        return 0;
}

void remove_from_used_list(struct buffer_head *bh) {

    if (!bh)
        ext2_os2_panic(0, "bh is NULL");
    if (bh->b_magic != BUFFER_HEAD_MAGIC)
        ext2_os2_panic(0, "remove_from_used_list - invalid magic number");
    if (!(bh->b_prev_free) || !(bh->b_next_free))
        ext2_os2_panic(0, "remove_from_used_list - used block list corrupted");
    if(!used_list)
        ext2_os2_panic(0, "remove_from_used_list - used list empty");

    _disable();
    bh->b_prev_free->b_next_free = bh->b_next_free;
    bh->b_next_free->b_prev_free = bh->b_prev_free;

    if (used_list == bh)
        used_list = bh->b_next_free;
    if(used_list == bh)
         used_list = 0;
    bh->b_next_free = 0;
    bh->b_prev_free = 0;
    _enable();

}
void remove_from_free_list(struct buffer_head * bh) {
    if (!bh)
        ext2_os2_panic(0, "remove_from_free_list - bh is NULL");
    if (bh->b_magic != BUFFER_HEAD_MAGIC)
        ext2_os2_panic(0, "remove_from_free_list - invalid magic number");
    if (!bh->b_prev_free || !bh->b_next_free)
        ext2_os2_panic(0, "remove_from_free_list - free block list corrupted");
    if(!free_list)
        ext2_os2_panic(0, "remove_from_free_list - free list empty");

    _disable();
    bh->b_prev_free->b_next_free = bh->b_next_free;
    bh->b_next_free->b_prev_free = bh->b_prev_free;

    if (free_list == bh)
        free_list = bh->b_next_free;
    if(free_list == bh)
         free_list = 0;
    bh->b_next_free = 0;
    bh->b_prev_free = 0;
    _enable();
}

void put_last_used(struct buffer_head * bh) {
    if (!bh)
        ext2_os2_panic(0, "bh is NULL");
    if (bh->b_magic != BUFFER_HEAD_MAGIC)
        ext2_os2_panic(0, "put_last_used - invalid magic number");
    if (bh->b_prev_free || bh->b_next_free)
        ext2_os2_panic(0, "put_last_used - used block list corrupted");

    _disable();
    if(!used_list) {
        used_list = bh;
        used_list->b_prev_free = bh;
    };

    bh->b_next_free = used_list;
    bh->b_prev_free = used_list->b_prev_free;
    used_list->b_prev_free->b_next_free = bh;
    used_list->b_prev_free = bh;
    _enable();
}



void put_last_free(struct buffer_head * bh) {
    if (!bh)
        ext2_os2_panic(0, "put_last_free - bh is NULL");
    if (bh->b_magic != BUFFER_HEAD_MAGIC)
        ext2_os2_panic(0, "put_last_free - invalid magic number");
    if (bh->b_prev_free || bh->b_next_free)
        ext2_os2_panic(0, "put_last_free - free block list corrupted");

    _disable();
    if(!free_list) {
        free_list = bh;
        bh->b_prev_free = bh;
    };

    bh->b_next_free = free_list;
    bh->b_prev_free = free_list->b_prev_free;
    free_list->b_prev_free->b_next_free = bh;
    free_list->b_prev_free = bh;
    _enable();
}


struct buffer_head * stage1_getblk(dev_t dev, blk_t block, unsigned long size) {
    struct buffer_head *bh;

    bh = get_hash_table(dev, block, size);

    if (bh) {
        if (bh->b_count) {
            /*
             * buffer was on LRU list
             */
            remove_from_used_list(bh);
        } else {
            /*
             * buffer was on free list
             */
            remove_from_free_list(bh);
        }
        put_last_used(bh);
        bh->b_count ++;
        return bh;
    }

    bh = free_list;
    if (!bh)
        ext2_os2_panic(0, "No more buffers !");
    remove_from_free_list(bh);
    put_last_used(bh);
    if (bh->b_blocknr != -1) {
        /*
         * buffer was on hash queue
         */
        remove_from_hash_queue(bh);
    }
    bh->b_count    = 1;
    bh->b_dev      = dev;
    bh->b_blocknr  = block;
    bh->b_uptodate = 0;
    add_to_hash_queue(bh);
    return bh;
}

void stage1_brelse(struct buffer_head *buf) {
    if (buf) {
        if (buf->b_count) {
            buf->b_count --;
            if (!buf->b_count) {
                remove_from_used_list(buf);
                put_last_free(buf);
            }
        } else {
            ext2_os2_panic(0, "brelse: Trying to free free buffer");
        }
    }
}

void bforget(struct buffer_head *buf) {
    if (buf) {
        if (buf->b_count) {
            buf->b_count --;
            if (!buf->b_count) {
                remove_from_used_list(buf);
                put_last_free(buf);
                remove_from_hash_queue(buf);
                buf->b_blocknr  = 0xFFFFFFFF;
                buf->b_uptodate = 0;
            }
        } else {
            ext2_os2_panic(0, "brelse: Trying to free free buffer");
        }
    }
}

struct buffer_head *stage1_bread(dev_t dev, blk_t block, unsigned long size) {
    struct buffer_head *bh;

    bh = getblk(dev, block, size);            /* failure in getblk      = panic */
    if (bh->b_uptodate)
        return bh;
    ll_rw_block(READ, 1, &bh);                /* failure in ll_rw_block = panic */
    bh->b_uptodate = 1;
    return bh;
}

struct buffer_head * (*bread)()  = stage1_bread;
void (*brelse)() = stage1_brelse;
struct buffer_head * (*getblk)() = stage1_getblk;


void __wait_on_buffer(struct buffer_head * bh) {

    _disable();
    while (bh->b_lock) {
        ProcBlock((unsigned long)bh, -1, 1);
        _disable();
    }
    _enable();
}

void buffer_stage1_to_stage2(void) {
    buffer_init();
}
